
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: param.c,v 1.12 2000/04/28 20:46:34 dr Exp $ 
*/

#include <pthread.h>
#include <string.h>
#include "param.h"
#include "scanner.h"

#define LOCK()            pthread_mutex_lock (&_param_mutex_)
#define UNLOCK()          pthread_mutex_unlock (&_param_mutex_)

#define next_token(scanner) g_scanner_get_next_token (scanner)

static void            param_init ();
static void            free_value (ParamValue *value, ParamType type);
static void            set_value (ParamValue *value, ParamType type, ParamValue *new_value);
static void            set_param_value (Param *param, ParamValue *value, int is_default_value);
static ParamValue *    get_param_value (Param *param);
static void            indent (FILE *file, int level);
static void            param_node_dump_recurse (GNode *node, FILE *file, int level);
static GNode *         node_find_child (GNode *parent, char *name);
static GNode *         node_find (GNode *parent, char *path, int insert);
static GNode *         param_node_add_child_nolock (GNode *parent, GNode *child);

static int             match_token (GScanner *scanner, GTokenType token);
static void            parse_node (GScanner *scanner, GNode *parent);
static void            parse_params (GScanner *scanner, GNode *parent);

static pthread_mutex_t _param_mutex_     = PTHREAD_MUTEX_INITIALIZER;
static GScanner *      param_scanner     = NULL;
static GNode *         param_root_node   = NULL;

static void
param_init () {
  if (!param_scanner) {
    param_scanner = g_scanner_new (&param_scanner_config);
    param_root_node = param_node_new_nl ("Root");
  }
}

/*
 * 
 * Public functions
 *
 */

Param *
param_new (char *name, int flags, char *label, char *help,
	   ParamType type, ParamValue value) {
  Param *param;

  g_return_val_if_fail (name, NULL);

  param               = g_new (Param, 1);
  param->refcount     = 1;
  param->flags        = flags;
  param->name         = g_strdup (name);
  param->label        = label ? g_strdup (label) : NULL;
  param->help         = help ? g_strdup (help) : NULL;
  param->type         = type;
  
  set_param_value (param, &value, 0);

  return param;
}

Param *
param_ref (Param *param) {
  g_return_val_if_fail (param, NULL);

  g_return_val_if_fail (param->refcount < UINT_MAX, NULL);
  param->refcount ++;

  return param;
}

void
param_unref (Param *param) {
  g_return_if_fail (param);
  g_return_if_fail (param->refcount);

  param->refcount --;

  if (param->refcount == 0) {
    if (param->name)         g_free (param->name);
    if (param->label)        g_free (param->label);
    if (param->help)         g_free (param->help);
    
    /* XXX */
    g_free (param);
  }
}

ParamValue *
param_get_value (Param *param, ParamType type) {

  g_return_val_if_fail (param, NULL);
  g_return_val_if_fail (param->type == type, NULL);

  return get_param_value (param);
}

void
param_set_value (Param *param, ParamType type, ParamValue value) {
  g_return_if_fail (param);
  g_return_if_fail (param->type == type);

  set_param_value (param, &value, 0);
}

void
param_clear_value (Param *param) {
  g_return_if_fail (param);
  
  set_param_value (param, NULL, 0);
}

Param *
param_find (GNode *parent, char *path) {
  GNode *node;
  g_return_val_if_fail (path, NULL);
  
  node = node_find (parent, path, 0);
  if (!node)
    return NULL;

  return PARAM_NODE (node)->type == PARAM_TYPE_NODE ? NULL : PARAM_NODE (node);
}

GNode *
param_get_root_node () {
  return param_root_node;
}

ParamI *                    
parami_new (Param *param, ParamType type, ParamValue value) {
  ParamI *parami;
  
  g_return_val_if_fail (param, NULL);

  parami         = g_new (ParamI, 1);
  parami->param  = param_ref (param);
  set_value (&parami->value, param->type, &value);

  return parami;
}

ParamValue * 
parami_get_value (ParamI *parami, ParamType type) {
  
  g_return_val_if_fail (parami, NULL);
  g_return_val_if_fail (parami->param->type == type, NULL);

  return &parami->value;
}

void
parami_set_value (ParamI *parami, ParamType type, ParamValue value) {
  g_return_if_fail (parami);
  g_return_if_fail (parami->param->type == type);

  set_value (&parami->value, parami->param->type, &value);
}

void                        
parami_free (ParamI *parami) {
  g_return_if_fail (parami);

  free_value (&parami->value, parami->param->type);
  param_unref (parami->param);
  g_free (parami);
}


GNode *                     
param_node_parse (GNode *parent, FILE *file) {
  g_return_val_if_fail (file, NULL);

  LOCK ();
  
  param_init ();

  if (!parent)
    parent = param_root_node;

  g_scanner_input_file (param_scanner, fileno (file));
  /* initialize the parser */
  next_token (param_scanner);

  parse_node (param_scanner, parent);

  UNLOCK ();

  return parent;
}

GNode *                     
param_node_parse_from_string (GNode *parent, char *string) {
  g_return_val_if_fail (string, NULL);

  LOCK ();
  
  param_init ();

  if (!parent)
    parent = param_root_node;

  g_scanner_input_text (param_scanner, string, strlen (string));
  /* initialize the parser */
  next_token (param_scanner);

  parse_node (param_scanner, parent);
  
  UNLOCK ();

  return parent;
}

GNode *
param_node_find (GNode *parent, char *path) {
  g_return_val_if_fail (path, NULL);

  return node_find (parent, path, 0);
}


static void
free_value (ParamValue *value, ParamType type) {

  if (type == PARAM_TYPE_STRING &&
      value->v_string)
    g_free (value->v_string);
}

static void
set_value (ParamValue *value, ParamType type, ParamValue *new_value) {

  if (type == PARAM_TYPE_STRING) 
    value->v_string = new_value && new_value->v_string ? 
      g_strdup (new_value->v_string) : NULL;
  else if (value)
    *value = *new_value;
}

static void
set_param_value (Param *param, ParamValue *value, int is_default_value) {

  if (is_default_value) {

    if (param->flags & PARAM_DEFAULT_VALUE_SET)
      free_value (&param->default_value, param->type);

    set_value (&param->default_value, param->type, value);

    if (value)
      param->flags |= PARAM_DEFAULT_VALUE_SET;
    else
      param->flags &= ~PARAM_DEFAULT_VALUE_SET;

  } else { /* the same thing for param->value */

    if (param->flags & PARAM_VALUE_SET)
      free_value (&param->value, param->type);

    set_value (&param->value, param->type, value);
    
    if (value)
      param->flags |= PARAM_VALUE_SET;
    else 
      param->flags &= ~PARAM_VALUE_SET;
  }
}

static ParamValue *
get_param_value (Param *param) {
  g_return_val_if_fail (param, NULL);

  if (param->flags & PARAM_VALUE_SET)
    return &param->value;
  else if (param->flags & PARAM_DEFAULT_VALUE_SET)
    return &param->default_value;
  
  return NULL;
}

void
param_dump (Param *param, FILE *file) {
  g_return_if_fail (param && file);
  g_return_if_fail (param->type != PARAM_TYPE_NODE);

  fprintf (file, "%s \"%s\" \"%s\" ", param->name, param->label, param->help);

  switch (param->type) {
  case PARAM_TYPE_LONG:
    fprintf (file, "%lu\n", param->value.v_long);
    break;
  case PARAM_TYPE_DOUBLE:
    fprintf (file, "%f\n", param->value.v_double);
    break;
  case PARAM_TYPE_STRING:
    fprintf (file, "\"%s\"\n", param->value.v_string);
    break;
  default:
    fprintf (file, "0\n");
    break;
  }
}

void
param_node_dump (GNode *node, FILE *file) {
  g_return_if_fail (file);

  LOCK ();

  fputs ("# Automaticaly generated file.\n"
	 "# DO NOT EDIT\n\n", file);

  if (node)
    param_node_dump_recurse (node, file, 0);
  else
    param_node_dump_recurse (param_root_node->children, file, 0);

  UNLOCK ();
}

static void
indent (FILE *file, int level) {
  while (level --)
    fputc ('\t', file);
}

static void
param_node_dump_recurse (GNode *node, FILE *file, int level) {

  if (!node)
    return;

  if (PARAM_NODE (node)->type == PARAM_TYPE_NODE) {

    indent (file, level);
    fprintf (file, "( %s", PARAM_NODE (node)->name);

    if (node->children && node->children->next == NULL &&
	PARAM_NODE (node->children)->type == PARAM_TYPE_NODE) {
      /*
	merge nodes which contains 1 non leaf node:
	( node1
	    ( node2
	      ...
	    )
        )

	will give
	( node1/node2
	   ...
	)
      */

      GNode *echild = node->children;

      do {
	fputc ('/', file);
	fputs (PARAM_NODE (echild)->name, file);

	echild = echild->children;

      } while (echild && echild->next == NULL &&
	       PARAM_NODE (echild)->type == PARAM_TYPE_NODE);
    
      fputc ('\n', file);
      
      param_node_dump_recurse (echild, file, level + 1);
      
    } else {      
      fputc ('\n', file);
      
      param_node_dump_recurse (node->children, file, level + 1);
    }

    indent (file, level);
    fputc (')', file);
    fputc ('\n', file);

  } else {
    indent (file, level);
    param_dump (PARAM_NODE (node), file);
  }

  param_node_dump_recurse (node->next, file, level);
}

/*
 *
 * Searching functions
 *
 */

/* assumes the caller olds the lock */
static GNode *
node_find_child (GNode *parent, char *name) {
  GNode *node;
  g_return_val_if_fail (parent && name, NULL);

  node = parent->children;
  while (node) {
    if (!strcmp (name, PARAM_NODE (node)->name))
      return node;

    node = node->next;
  }

  return NULL;
}

static GNode *                     
node_find (GNode *parent, char *path, int insert) {
  GNode *node = NULL;
  char buf [PATH_MAX], *name;
  
  LOCK ();

  memcpy (buf, path, strlen (path) + 1);

  param_init ();

  if (!parent) 
    parent = param_root_node;

  name = strtok (buf, "/");

  while (name) {
    node = node_find_child (parent, name);
    if (!node)
      break;
    parent = node;
    name   = strtok (NULL, "/");
  }

  if (insert && name) {
    Param *param;

    node = parent;
    do {
      param = param_new_nl (name);
      node = param_node_add_child_nolock (node, g_node_new (param));
      param_unref (param);
      name = strtok (NULL, "/");
    } while (name);
  }

  UNLOCK ();

  return node;
}

GNode *
param_node_add_child (GNode *parent, GNode *child) {
  GNode *node;

  LOCK ();
  node = param_node_add_child_nolock (parent, child);
  UNLOCK ();

  return node;
}

static GNode *
param_node_add_child_nolock (GNode *parent, GNode *child) {
  GNode *sibling;
  g_return_val_if_fail (child, NULL);

  param_ref (PARAM_NODE (child));

  param_init ();
  if (!parent)
    parent = param_root_node;

  sibling = parent->children;

  while (sibling) {
    if (strcmp (PARAM_NODE (child)->name, PARAM_NODE (sibling)->name) < 0)
      return g_node_insert_before (parent, sibling, child);
    
    sibling = sibling->next;
  }

  return g_node_append (parent, child);
}

/*
 *
 * The Parser
 *
 */

static int
match_token (GScanner *scanner, GTokenType token) {

  g_return_val_if_fail (scanner, 0);

  if (scanner->token == token) 
    return 1;

  g_scanner_error (scanner, 
		   "Unexpected token %d (expected %d)", 
		   scanner->token, token);

  return 0;
}

static void
parse_node (GScanner *scanner, GNode *parent) {
  GNode *node;

  if (scanner->token != '(')
    return;

  next_token (scanner);

  if (!match_token (scanner, G_TOKEN_STRING))
    return;
  
  node = node_find (parent, scanner->value.v_string, 1);
  
  next_token (scanner);
  
  parse_params (scanner, node);
  parse_node (scanner, node);
    
  if (!match_token (scanner, ')'))
    return;

  next_token (scanner);

  parse_params (scanner, parent);
  parse_node (scanner, parent);
}

static void
parse_params (GScanner *scanner, GNode *parent) {
  Param *param;
  
  while (scanner->token == G_TOKEN_STRING) {
    
    param            = g_new (Param, 1);
    param->refcount  = 1;
    param->flags     = 0;
    param->name      = g_strdup (scanner->value.v_string);
    param->label     = NULL;
    param->help      = NULL;
    param->type      = PARAM_TYPE_UNKNOWN;
    next_token (scanner);

    if (!match_token (scanner, G_TOKEN_STRING)) {
      param_unref (param);
      return;
    }
    
    param->label = g_strdup (scanner->value.v_string);
    next_token (scanner);

    if (!match_token (scanner, G_TOKEN_STRING)) {
      param_unref (param);
      return;
    }

    param->help = g_strdup (scanner->value.v_string);
    next_token (scanner);

    switch (scanner->token) {
    case G_TOKEN_INT:
      param->type = PARAM_TYPE_LONG;
      param->value.v_long = scanner->value.v_int;
      break;

    case G_TOKEN_FLOAT:
      param->type = PARAM_TYPE_DOUBLE;
      param->value.v_double = scanner->value.v_float;
      break;
      
    case G_TOKEN_STRING:
      param->type = PARAM_TYPE_STRING;
      param->value.v_string = g_strdup (scanner->value.v_string);
      break;

    default:
      g_warning ("Unexpected parameter value type %d", scanner->token);
      param_unref (param);
      return;
    }

    next_token (scanner);
    
    param_node_add_child_nolock (parent, g_node_new (param));
    param_unref (param);
  }  
}



