
/*
    Axv: Another X Image Viewer
    Copyright (C) 2000 David RAMBOZ 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: xpm_module.c,v 1.5 2000/04/28 20:46:35 dr Exp $ 
*/


#include <axv_config.h>
#include <string.h>
#include <X11/xpm.h>
#include "xpm_module.h"

#undef RGB_VALUE
#define RGB_VALUE(r, g, b)     ((gulong) (r) << 24 | (gulong) (g) << 16 | (gulong) (b) << 8 | 0xFF)

typedef struct _RgbMapEntry RgbMapEntry;

struct _RgbMapEntry {
  char    *name;
  gulong  value;
};

static void                 xpm__class_init (ModuleClass *class);
static void                 xpm__init (Module *module);
static int                  xpm__can_read (Codec *codec);
static int                  xpm__read_image_infos (Codec *codec);
static int                  xpm__read_image (Codec *codec);
static int                  xpm__write_image (Codec *codec);
static gulong               xpm__parse_color (XpmColor *color);
static gulong               xpm__parse_hex_color (char *s);

static int                  rgb_skip_blank (FILE *file);
static void                 rgb_parse (char *path);

static GHashTable           *s_color_hash_table = NULL;

ModuleType 
xpm_get_type () {
  static ModuleType type = 0;

  if (!type) {
    static ModuleTypeInfo xpm_info = {
      "Xpm",
      NULL,
      NULL,
      sizeof (XpmModule),
      sizeof (XpmModuleClass),
      xpm__init,
      xpm__class_init,
      /* reserved 1 */ NULL,
      /* reserved 2 */ NULL
    };

    type = module_type_unique (codec_get_type (), &xpm_info);
  }

  return type;
}

static void                 
xpm__class_init (ModuleClass *class) {
  static char *extensions = ".xpm.";
  CodecClass *codec_class;
  CodecFunc  *operations;

  codec_class = (CodecClass *) class;

  codec_class->extensions         = extensions;
  operations = codec_class->operations;

  operations [CODEC_CAN_READ_IMAGE]     = xpm__can_read;
  operations [CODEC_READ_IMAGE_INFOS]   = xpm__read_image_infos;
  operations [CODEC_READ_IMAGE]         = xpm__read_image;
  operations [CODEC_WRITE_IMAGE]        = xpm__write_image;

  if (!s_color_hash_table) { /* create a hash table which maps symbolic colors name to their value */
    RgbMapEntry *rme;

    s_color_hash_table = g_hash_table_new (g_str_hash, g_str_equal);
    
    /* create the transparent colors (which aren't in the rgb database (?)) */
    rme        = g_new (RgbMapEntry, 1);
    rme->name  = g_strdup ("none");
    rme->value = 0;

    g_hash_table_insert (s_color_hash_table, rme->name, rme);

    rme        = g_new (RgbMapEntry, 1);
    rme->name  = g_strdup ("0");
    rme->value = 0;

    g_hash_table_insert (s_color_hash_table, rme->name, rme);

    rgb_parse (RGB_DATABASE_PATH);
  }
}

static void                 
xpm__init (Module *module) {
  
}

static int                  
xpm__can_read (Codec *codec) {
  char buff [12];

  if (!fread (buff, 9, 1, codec->file))
    return 0;

  return !strncmp (buff, "/* XPM */", 9);
}

static int                  
xpm__read_image_infos (Codec *codec) {

  return 0;
}

static int                  
xpm__read_image (Codec *codec) {
  XpmImage image;
  FilterTile *buffer;
  char path [PATH_MAX];
  guchar *dest;
  int *cmap, *src;
  int out_bpp, x, y;

  snprintf (path, PATH_MAX, "%s/%s", 
	    codec_get_image (codec)->directory, 
	    codec_get_image (codec)->file_name);

  if (XpmReadFileToXpmImage (path, &image, NULL) != XpmSuccess) {
    return 0;
  }

  /* build the color map */
  cmap = g_malloc (image.ncolors * sizeof (int));
 
  for (x = 0; x < image.ncolors; x++)
    cmap [x] = xpm__parse_color (&image.colorTable [x]);

  /* XXX : it shouldn't waste memory like that */
  out_bpp = 4;
 
  codec_init (codec, 
	      image.width, image.height, out_bpp,
	      image.width, image.height, out_bpp, 
	      image.width, 1);
  buffer = codec_get_buffer (codec);

  src = image.data;
  for (y = 0; y < image.height; y++) {

    dest = buffer->data;

    for (x = 0; x < image.width; x++) {
      dest [0] = ((char*) &cmap [src [x]]) [3];
      dest [1] = ((char*) &cmap [src [x]]) [2];
      dest [2] = ((char*) &cmap [src [x]]) [1];
      dest [3] = ((char*) &cmap [src [x]]) [0];
      dest += 4;
    }

    if (!codec_add_tile (FILTER (codec), buffer))
      break;
    src += image.width;
  }

  g_free (cmap);
  XpmFreeXpmImage (&image);

  return 1;
}

static int                  
xpm__write_image (Codec *codec) {

  return 0;
}

#undef PARSE_COLOR
#define PARSE_COLOR(str) G_STMT_START{                               \
                                                                     \
        if (*(str) == '#')                                           \
	  return xpm__parse_hex_color ((str) + 1);                   \
        else {                                                       \
          RgbMapEntry *rme;                                          \
          g_strdown ((str));                                         \
          rme = g_hash_table_lookup (s_color_hash_table, (str));     \
          if (rme)                                                   \
            return rme->value;                                       \
          else {                                                     \
             g_warning ("Xpm : unknown color %s.", (str));           \
             return 0;                                               \
           }                                                         \
        }                }G_STMT_END

    
static gulong
xpm__parse_color (XpmColor *color) {

  if (color->c_color)
    PARSE_COLOR(color->c_color);
  else if (color->symbolic)
    PARSE_COLOR(color->symbolic);
  else if (color->g_color)
    PARSE_COLOR(color->g_color);
  else if (color->g4_color)
    PARSE_COLOR(color->g4_color);
  else if (color->m_color)
    PARSE_COLOR(color->m_color);
 
  g_warning ("Xpm : Empty colormap  entry");
  
  return 0;
}

/*
 | xpm_parse_hex_color :
 | ---------------------
 | 
 | input  : (string) hex
 | output : (integer) RGB
 | 
*/

#define PARSE_HEX_BYTE(n, byte) G_STMT_START{                 \
    n <<= 4;                                                  \
    if ((byte) <= '9' && (byte) >= '0')                       \
      n += (byte) - '0';                                      \
    else if ((byte) <= 'F' && (byte) >= 'A')                  \
      n += (byte) - 'A' + 10;                                 \
    else if ((byte) <= 'f' && (byte) >= 'a')                  \
      n += (byte) - 'a' + 10;                                 \
    else {                                                    \
      g_warning ("Invalid hex value %c", (byte));             \
      return 0;                                               \
    }                           }G_STMT_END               

    
#undef PARSE_HEX_BYTE
#define PARSE_HEX_BYTE(byte) ((byte) <= '9' ?                                        \
                                 ((byte) >= '0' ? (byte) - '0' : -1) :               \
                                 ((byte) <= 'F' ?                                    \
				     ((byte) >= 'A' ? (byte) - 'A' + 10: -1) :       \
                                     ((byte) <= 'f' ?                                \
				         ((byte) >= 'a' ? (byte) - 'a' + 10: -1) :   \
				         -1)))

static gulong
xpm__parse_hex_color (char *hex) {
  gulong c = 0, i;
  int    bpc; /* bytes per component */
  char   *s;

  i = strlen (hex);
  if (i % 3 != 0)
    g_warning ("malformed color %s", hex);

  bpc = i / 3;
  s = hex;
  while (s [0] && s [1]) {
    c <<= 4;
    c += PARSE_HEX_BYTE (s [0]);
    c <<= 4;
    c += PARSE_HEX_BYTE (s [1]);

    for (i = 2; i < bpc; i++)
      if (!s [i]) {
	g_warning ("incomplete color");
	return 0;
      }
	
    s += bpc; /* go to the next component */
  }
    
  return (c << 8) | 0xFF;
}

/* rgb.txt parser */
static int
rgb_skip_blank (FILE *file) {
  int c;

  while ((c = fgetc (file)) == ' ' || c == '\t' || c == '\n')
    ;

  if (c == '!') { /* skip comment */
    while ((c = fgetc (file)) != '\n' && c != EOF)
      ;
    if (c == EOF)
      return 0;

    ungetc (c, file);
    rgb_skip_blank (file);
  } else
    ungetc (c, file);
  
  return c != EOF;
}

static void
rgb_parse (char *path) {
  FILE *file;
  RgbMapEntry *rme;
  int r, g, b, ret;
  char name [128];

  file = fopen (path, "r");
  g_return_if_fail (file);

  rgb_skip_blank (file);
  
  for (;;) {
    ret = fscanf (file, "%d %d %d %127[^\n]", &r, &g, &b, name);

    if (ret == EOF) 
      break;
    else if (ret != 4) {
      g_warning ("Can't parse rgb database");
      break;
    }

    g_strdown (name);

    rme         = g_new (RgbMapEntry, 1);
    rme->name   = g_strdup (name);
    rme->value  = RGB_VALUE (r, g, b);

    g_hash_table_insert (s_color_hash_table, rme->name, rme);
    
    rgb_skip_blank (file);
  }

  fclose (file);
}

