/*
 * magstep.c
 *	Magstep menu callbacks for mgv.
 *
 * Copyright (C) 1996  Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: mgv/magstep.c,v 1.46 $")

#include <assert.h>
#include <stdlib.h>
#include <Xm/ToggleB.h>
#include <Xm/TextF.h>

#include <wlib/typesanity.h>
#include <wlib/wlib.h>
#include <mine/mgv.h>
#include <mine/magstep.h>

/*
 * Set magstep and synchronize the widgets.
 *
 * If you are looking for a way to prevent high magsteps from griding your
 * machine into the ground, this is the place for the limits.  All of the
 * magstep setting goes through here.
 */
int
mgv_ui_magnify(MGV *m, int magstep)
{
	int	different;
	char	s[64];

	different = mgv_magnify(m, magstep);

	/*
	 * If the magstep is in the range of the radio buttons, we
	 * need to turn the correct one on and the old one off.
	 */
	if(magstep >= MGV_MAGMIN && magstep <= MGV_MAGMAX) {
		sprintf(s, "*menuBar*magstep_pd*%d", magstep);
		if(m->magsteptog != NULL)
			XmToggleButtonSetState(m->magsteptog, False, False);
		m->magsteptog = wl_find1(m->main, s);
		XmToggleButtonSetState(m->magsteptog, True, False);
	}
	else {
		if(m->magsteptog != NULL)
			XmToggleButtonSetState(m->magsteptog, False, False);
		m->magsteptog = NULL;
	}

	sprintf(s, "%+d", magstep);

	/*
	 * Update the status line thingy.
	 */
	XtVaSetValues(wl_find1(m->main, "*statusLine*magstepFrame*value"),
		XtVaTypedArg,	XmNlabelString,
				XtRString, s,
				strlen(s) + 1,
		NULL);

	/*
	 * Update the magstep->set dialog (if it exists).
	 */
	if(m->magstep != NULL)
		XtVaSetValues(wl_find1(m->magstep, "*text"),
			XmNvalue,	s,
			NULL);

	return different;
}

/*
 * Magstep->(increase|decrease)
 */
void
mgv_magstep(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	int	magstep;
	USEUP(call);
	assert(MgvOK(m));

	switch(*XtName(w)) {
	case 'd':	magstep = m->magnify - 1;		break;
	case 'i':	magstep = m->magnify + 1;		break;
	default:	assert("magstep is busted" != NULL);	return;
	}
	if(mgv_ui_magnify(m, magstep))
		mgv_show(m, m->page);
}

/*
 * magstep->set
 */
static WLW inner_form[] = {
	{"label",	wlLabelG,	0,	NULL},
	{"decr",	wlArrowBG,	0,	NULL},
	{"incr",	wlArrowBG,	0,	NULL},
	{"text",	wlTextField,	0,	NULL},
	{NULL}
};
static WLW magstep_form[] = {
	{"form",	wlForm,		WL_MANAGE,	inner_form},
	{"apply",	wlPushB,	0,		NULL},
	{"dismiss",	wlPushB,	0,		NULL},
	{"help",	wlPushB,	0,		NULL},
	{"sep",		wlSeparatorG,	0,		NULL},
	{NULL}
};
static WLW magstep_dlg[] = {
	{"magstepDialog",	wlFormDialog,	0,	magstep_form},
	{NULL}
};

static void
magstep_apply(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	String	s;
	USEUP(w); USEUP(call);
	assert(MgvOK(m));

	XtVaGetValues(wl_find1(m->magstep, "*text"), XmNvalue, &s, NULL);
	if(mgv_ui_magnify(m, atoi(s)))
		mgv_show(m, m->page);
	wl_dlg_popdown(m->magstep);
}

static void
magstep_updown(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	int	n;
	String	s;
	char	buf[64];
	USEUP(call);
	assert(MgvOK(m));

	switch(*XtName(w)) {
	case 'i':	n = +1;	break;
	case 'd':	n = -1; break;
	default:	assert("arrow buttons are busted!" != NULL);	return;
	}
	w = wl_find1(m->magstep, "*text");
	sprintf(buf, "%+d", atoi(s = XmTextFieldGetString(w)) + n);
	XtFree(s);
	XmTextFieldSetString(w, buf);
}

static WLC magstep_cb[] = {
	{"*incr",	XmNactivateCallback,	0,	NULL,	magstep_updown},
	{"*decr",	XmNactivateCallback,	0,	NULL,	magstep_updown},
	{"*apply",	XmNactivateCallback,	0,	NULL,	magstep_apply},
	{"*dismiss",	XmNactivateCallback,	0,	NULL,	wl_dismiss},
	{"*help",	XmNactivateCallback,	0,	NULL,	wl_help},
	{NULL}
};

void
mgv_magstep_set(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	USEUP(w); USEUP(call);
	assert(MgvOK(m));

	if(m->magstep == NULL) {
		char	s[64];
		m->magstep = wl_create(m->main, magstep_dlg);
		wl_callback(m->magstep, &magstep_cb[0], (XtPointer)m);
		sprintf(s, "%+d", m->magnify);
		XmTextFieldSetString(wl_find1(m->magstep, "*text"), s);
	}

	wl_dlg_popup(m->magstep);
}

void
mgv_magstep_edit(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	Widget	text;
	USEUP(w); USEUP(call);
	assert(MgvOK(m));

	text = wl_find1(m->main, "*statusLine*magstepFrame*text");
	XtManageChild(text);
	XtVaSetValues(wl_find1(m->main, "*statusLine*magstepFrame*label"),
		XmNrightWidget,	text,
		NULL);
	XtUnmanageChild(wl_find1(m->main, "*statusLine*magstepFrame*value"));
	XmProcessTraversal(text, XmTRAVERSE_CURRENT);
}

void
mgv_magstep_text(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	Widget	label, text, value;
	String	s;
	USEUP(w);
	assert(MgvOK(m));

	wl_find(m->main,
		&text,	"*statusLine*magstepFrame*text",
		&label,	"*statusLine*magstepFrame*label",
		&value,	"*statusLine*magstepFrame*value",
		NULL);
	XtUnmanageChild(text);
	XtManageChild(value);
	XtVaSetValues(label, XmNrightWidget, value, NULL);

	if(((XmAnyCallbackStruct *)call)->reason == XmCR_ACTIVATE) {
		XtVaGetValues(text, XmNvalue, &s, NULL);
		if(mgv_ui_magnify(m, atoi(s)))
			mgv_show(m, m->page);
	}
}

void
mgv_magstep_n(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m   = (MGV *)closure;
	XmTBS	*cbs = (XmTBS *)call;
	int	magstep;
	assert(MgvOK(m));

	if(!cbs->set)
		return;
	magstep = atoi(XtName(w));
	assert(magstep >= MGV_MAGMIN && magstep <= MGV_MAGMAX);
	if(mgv_ui_magnify(m, magstep))
		mgv_show(m, m->page);
}
