/*
 * page_control.c
 *	Page stack control.
 *
 * Copyright (C) 1998 Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: mgv/page_control.c,v 1.6 $")

#include <assert.h>
#include <Xm/List.h>

#include <mine/mgv.h>
#include <mine/page_control.h>
#include <mine/sens.h>

struct PGSTACK_s {
	MGV	*m;
	int	*list;
	int	size;		/* current size			*/
	int	current;	/* current position		*/
	int	used;		/* number of entries used	*/
};

static void
show(MGV *m, int page)
{
	int	top, vis;

	mgv_show(m, page);

	if(m->dsc == NULL)
		return;

	XtVaGetValues(m->list,
		XmNtopItemPosition,	&top,
		XmNvisibleItemCount,	&vis,
		NULL);

	/*
	 * Convert to Motif List numbering.
	 */
	++page;

	/*
	 * Check visibility.
	 */
	if(page < top) {
		XmListSetPos(m->list, page);
	}
	else if(page > top + vis - 1) {
		/*
		 * If we're near the end of the list we want to decrease
		 * the top position to keep the list's screen area full of
		 * entries (note:  if we XmListSetPos(m->list, last_page_pos)
		 * the list will end up with one visible item, we wish to
		 * avoid this situation).
		 */
		if(page > m->dsc->n_pages - vis)
			page = m->dsc->n_pages - vis + 1;
		XmListSetPos(m->list, page);
	}
}

PGSTACK
mgv_pgstack_alloc(MGV *m)
{
	PGSTACK	p = (PGSTACK)XtMalloc(sizeof(struct PGSTACK_s));

	p->size    = 10;
	p->list    = (int *)XtCalloc(p->size, sizeof(int));
	p->used    = 0;
	p->current = -1;
	p->m       = m;

	return p;
}

PGSTACK
mgv_pgstack_free(PGSTACK p)
{
	if(p == NULL)
		return p;
	XtFree((XtPointer)p->list);
	XtFree((XtPointer)p);
	return NULL;
}

static void
resize(PGSTACK p, int size)
{
	if(p->size >= size + 1)
		return;
	p->size = 10*((size + 9) / 10);
	p->list = (int *)XtRealloc((XtPointer)p->list, p->size * sizeof(int));
}

void
mgv_pgstack_forward(PGSTACK p)
{
	int	sens = p->m->sens;

	if(p->current == p->used - 1) {
		assert("Cannot go forward at the end of the list!" != NULL);
		return;
	}

	++p->current;

	if(p->current == p->used - 1)
		sens &= ~MgvSCANFORWARD;
	sens |= MgvSCANBACK;
	if(p->m->sens != sens)
		mgv_sens(p->m->main, p->m->sens = sens);

	show(p->m, p->list[p->current]);
}

void
mgv_pgstack_goto(PGSTACK p, int page)
{
	if(p == NULL)
		return;

	if(p->current == -1
	|| p->list[p->current] != page) {
		int	sens = p->m->sens;

		resize(p, p->current + 1);
		p->list[++p->current] = page;
		p->used = p->current + 1;

		if(!(sens & MgvSCANBACK)
		&& p->current != 0)
			sens |= MgvSCANBACK;
		sens &= ~MgvSCANFORWARD;
		if(p->m->sens != sens)
			mgv_sens(p->m->main, p->m->sens = sens);
	}

	show(p->m, page);
}

void
mgv_pgstack_back(PGSTACK p)
{
	int	sens = p->m->sens;

	if(p->current <= 0) {
		assert("Cannot go back at the beginning of the list!" != NULL);
		return;
	}

	--p->current;

	if(p->current == 0)
		sens &= ~MgvSCANBACK;
	sens |= MgvSCANFORWARD;
	if(sens != p->m->sens)
		mgv_sens(p->m->main, p->m->sens = sens);

	show(p->m, p->list[p->current]);
}

void
mgv_pgstack_prefill(PGSTACK p, int pages)
{
	int	i;

	resize(p, pages);
	p->used = pages;
	for(i = 0; i < pages; ++i)
		p->list[i] = i;
	p->current = 0;
	mgv_sens(p->m->main, p->m->sens |= MgvSCANFORWARD);
}
