/*
 * print.c
 *	Printing support functions for mgv.
 *
 * Copyright (C) 1996  Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@trends.net)
 */
#include <wlib/rcs.h>
MU_ID("$Mu: mgv/print.c,v 1.78 $")

#include <assert.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <Xm/MessageB.h>
#include <Xm/ToggleB.h>

#include <wlib/wlib.h>
#include <mine/mgv.h>
#include <mine/print.h>
#include <mine/log.h>
#include <mine/sens.h>
#include <mine/strings.h>
#include <mine/util.h>

/*
 * the command section is a little form so that we can get
 * the label and the text field to align in a pleasing fashion
 * (I could use the caption widget from Xbae but then everyone
 * would need to have the Xbae widgets kicking around)
 */
static WLW cmd_form[] = {
	{"cmd",		wlLabelG,	0,	NULL},
	{"cmdText",	wlTextField,	0,	NULL},
	{NULL}
};

/*
 * the widgets for the print dialog
 */
static WLW print_form[] = {
	{"cmdForm",	wlForm,		WL_MANAGE,	cmd_form},
	{"markedOnly",	wlToggleBG,	0,		NULL},
	{"unmark",	wlToggleBG,	0,		NULL},
	{"apply",	wlPushB,	0,		NULL},
	{"dismiss",	wlPushB,	0,		NULL},
	{"help",	wlPushB,	0,		NULL},
	{"sep",		wlSeparatorG,	0,		NULL},
	{NULL}
};

/*
 * the print dialog itself
 */
static WLW printdlg[] = {
	{"printDialog",	wlFormDialog,	0,		print_form},
	{NULL}
};

/*
 * popup a message to show the status, if any, of the print job
 *,,,I really should be using XtAppAddInput() for this (maybe later)
 */
static void
print_msg(MGV *m, char *tmp)
{
	FILE	*fp;
	char	buf[1024];

	if(*tmp == '\0')
		return;

	if(m->print_msg == NULL) {
		m->print_msg = XmCreateInformationDialog(m->main,
						"printMessage", NULL, 0);
		XtUnmanageChild(XmMessageBoxGetChild(m->print_msg,
						XmDIALOG_CANCEL_BUTTON));
	}

	fp = fopen(tmp, "r");
	remove(tmp);
	if(fp == NULL)
		return;
	memset((void *)&buf[0], '\0', sizeof(buf));
	fread((void *)&buf[0], 1, sizeof(buf) - 1, fp);
	fclose(fp);
	if(buf[0] == '\0')
		return;

	XtVaSetValues(m->print_msg,
		XtVaTypedArg,	XmNmessageString,
				XtRString, buf,
				strlen(buf) + 1,
		NULL);
	wl_dlg_popup(m->print_msg);
}

/*
 * print->apply
 */
static void
print_apply(Widget w, XtPointer closure, XtPointer call)
{
	struct sigaction	sa, osa;
	MGV	*m = (MGV *)closure;
	String	cmd;
	Boolean	marked, unmark;
	int	i;
	char	*buf = NULL;
	char	tmp[L_tmpnam];
	FILE	*fp;

	USEUP(call);
	assert(MgvOK(m));

	cmd     = NULL;
	marked  =
	unmark  = False;
	XtVaGetValues(wl_find1(m->printdlg, "*cmdText"),
		XmNvalue,	&cmd,
		NULL);

	/*
	 * sanity checks
	 */
	if(cmd == NULL || *cmd == '\0') {
		wl_error(m->printdlg, "%s", mgv_s(w, MgvSnoPrintCommand));
		return;
	}

	/*
	 * check the toggles (both toggles should always have the
	 * same sensitivity)
	 */
	w = wl_find1(m->printdlg, "*markedOnly");
	if(XtIsSensitive(w)) {
		marked = XmToggleButtonGetState(w);
		w      = wl_find1(m->printdlg, "*unmark");
		unmark = marked ? XmToggleButtonGetState(w) : False;
	}

	XtFree((XtPointer)m->printcmd);
	m->printcmd = XtNewString(cmd);

	buf = strcpy(XtMalloc(strlen(cmd) + L_tmpnam + 2 + 5 + 1), cmd);
	if(tmpnam(&tmp[0]) != NULL)
		strcat(strcat(strcat(buf, " >"), tmp), " 2>&1");
	else
		tmp[0] = '\0';

	/*
	 * popen() really isn't the best way to do this but it is really easy,
	 * pipe/dup2/fork/exec/XtAddInput would be better but that'll have
	 * to wait.
	 */
	memset((void *)&sa,  '\0', sizeof(sa));
	memset((void *)&osa, '\0', sizeof(osa));
	sa.sa_handler = SIG_IGN;
	sigemptyset(&sa.sa_mask);
	sigaction(SIGPIPE, &sa, &osa);
	if((fp = popen(buf, "w")) == NULL) {
		wl_error(m->printdlg, "%s %s: %s", mgv_s(w, MgvScantPrint),
							buf, strerror(errno));
		sigaction(SIGPIPE, &osa, &sa);
		XtFree(buf);
		return;
	}
	XtFree(buf);
	buf = NULL;
	mgv_write(m, fp, marked);
	pclose(fp);
	sigaction(SIGPIPE, &osa, &sa);

	print_msg(m, &tmp[0]);

	if(unmark && m->dsc != NULL) {
		for(i = 0; i < m->dsc->n_pages; ++i)
			mgv_mark(m, i, FALSE, FALSE);
		mgv_list_update(m, -1);
	}
	wl_dlg_popdown(m->printdlg);
}

/*
 * and all the callbacks
 */
static WLC print_cb[] = {
	{"*apply",	XmNactivateCallback,	0,	NULL,	print_apply},
	{"*dismiss",	XmNactivateCallback,	0,	NULL,	wl_dismiss},
	{"*help",	XmNactivateCallback,	0,	NULL,	wl_help},
	{NULL}
};

/*
 * Print
 */
void
mgv_print(Widget w, XtPointer closure, XtPointer call)
{
	MGV	*m = (MGV *)closure;
	USEUP(w); USEUP(call);
	assert(MgvOK(m));

	if(m->printdlg == NULL) {
		if((m->printdlg = wl_create(m->main, printdlg)) == NULL)
			return;
		wl_callback(m->printdlg, &print_cb[0], (XtPointer)m);
	}

	if(m->printcmd != NULL)
		XtVaSetValues(wl_find1(m->printdlg, "*cmdText"),
			XmNvalue,	m->printcmd,
			NULL);

	mgv_sens(m->main, m->sens);

	wl_dlg_popup(m->printdlg);
}
