
#define ZVIEW_VERSION "0.2"
#define ZVIEW_DATE "Dec 1 1997"
#define ZVIEW_AUTHOR "J. Crowe <crowej@eecs.umich.edu>"

#include "zview.h"
#include "ReadAnImage.h"

char *ProgramName;

/* X stuff */
Display *display;
int screen;
Visual *visual, *default_visual;
Window win_zview=0;
Window subw_image=0;
Window subw_plot_horiz=0;
Window subw_plot_vert=0;
Window subw_info=0;
Window win_help=0;

unsigned long background_pixel;
unsigned long foreground_pixel;
unsigned long red_pixel;
Colormap colormap_gray;

GC gc;     /* standard GC */
GC gc_dashes; /* GC for drawing dashed lines */
GC gc_fill;
GC gc_red;
XImage *image;
int display_bpp = 8;

float display_gamma=1.0;
float image_gamma=1.0;

/* pixels */
int current_pxl_x = 0;
int current_pxl_y = 0;
int curplot_pxl_x = 0;
int curplot_pxl_y = 0;        /* y index of pixel */

/* flags */
int flag_verbose = 0;
int flag_plots_on = 1;
int flag_TrueColor = 0;
int flag_ShowColor = 0;       /* 1 => colors can be used */
int flag_ColorDisplay = 0;    /* Color Hardware => 1 */
int flag_ColorInput = 1;      /* color input image => 1 */
int flag_MarkersOn = 1;    /* 1 => draw markers, 0 => dont */
int flag_PrivateColormap = 1;
int flag_ShowRGBPlots = 0;
int flag_ForceGrayDisplay = 1;
int xflag_IgnoreOneResize = 0; /* 1 => Ignore the next resize event, we caused it */

/* window sizes */
int plot_window_tall = 99;
int window_borderwidth = 1;
int window_separatorwidth = 1;
char *window_name = NULL;

float zoom_factor_x=1;
float zoom_factor_y=1;
int times_zoomed=1;
int nzoom=10;
int unzoomed_ul_x = 0;
int unzoomed_ul_y = 0;
int *last_unzoomed_ul_x;
int *last_unzoomed_ul_y;

/* image dim */
int image_width;
int image_height;
int nxpixels;
int nypixels;

/* image */
unsigned char **image_gray = 0;   /* holds intensity only version of input */
unsigned char **image_color = 0;  /* holds color version of input */
unsigned char **image_source = 0; /* this is the *original* input source */

XSegment *h_segment_i;
XSegment *h_segment_r;
XSegment *v_segment_i;
XSegment *v_segment_r;
float plot_scale;

int last_marker_x;
int last_marker_y;

/* --------------------------------------------------------------- */
char *image_infname = 0;
struct RAI_IMAGE AnImage;

/* --------------------------------------------------------------- */
void Refresh();
void ShowHelp();
void ShowSummary();
void MakeImageGray();
/* --------------------------------------------------------------- */
void
ShowSummary()
{
  char buff[48];
  int x=9;
  int y=15;
  XEvent event;

  if(flag_verbose)
    puts("Show_Summary()");
  
  XFillRectangle(display, subw_image, gc_fill,
                 0, 0,
                 nxpixels, nypixels);

  sprintf(buff, "Zview v%s %s by John Crowe", ZVIEW_VERSION, ZVIEW_DATE);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;
  sprintf(buff, "  (crowej@eecs.umich.edu)");
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;
  
  x+=18;

  sprintf(buff, "Input file: %s", image_infname);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;

  sprintf(buff, "Size: %d x %d (w x h)", image_width, image_height);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;

  sprintf(buff, "Displayed at: %d x %d (w x h)", nxpixels, nypixels);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;

  sprintf(buff, "%d bytes per pixel in input", AnImage.bytespp);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;
  
  sprintf(buff, "Zoom Factors  x=%f, =%f ", 1/zoom_factor_x, 1/zoom_factor_y);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;

  sprintf(buff, "Display Gamma: %2.1f%", display_gamma);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;

  sprintf(buff, "Image Gamma: %2.1f%", image_gamma);
  XDrawString(display, subw_image, gc, x, y, buff, strlen(buff));
  y+=15;

  do
    {
      XNextEvent(display, &event);
    } while ((event.type == KeyPress) || (event.type == ButtonPress) );

  Refresh();
  
  return;
}


static char *HelpMsg[]={
  "Valid in window:",
  " i   show info",
  " h,? show help",
  " p   toggle z plots",
  " s   screen dump Zview window",
  " Z,. zoom in",
  " z,, zoom out",
  " mouse-1 select pixel",
  " mouse-3 toggle markers",
  "",
  NULL
};

void
ShowHelp()
{
  char buff[48];
  int x=9;
  int y=15;
  XEvent event;
  
  char **c;

  if(flag_verbose)
    puts("Show_Help()");

    
  XFillRectangle(display, subw_image, gc_fill,
                 0, 0,
                 nxpixels, nypixels);

  for(c=HelpMsg;  **c; c++)
    {
      XDrawString(display, subw_image, gc, x, y, *c, strlen(*c));
      y+=15;
    }  
  x+=18;

  do
    {
      XNextEvent(display, &event);
    } while ((event.type == KeyPress) || (event.type == ButtonPress) );
      
  Refresh();
  
  return;
}

void
MakeImageGray()
{
  int ix, iy;
  int iix, iiy;

  /* input is rgb, convert to intensity */
 
  /* This is a leak!  the pointers to rows are not free'd */
  if(0!=image_gray)
    free(&image_gray[0][0]);  

  image_gray=(unsigned char **)
    Calloc2d(nypixels, nxpixels, sizeof(unsigned char));
  if(0==image_gray)
    Death(-1, "zv:  Could not allocate for gray image\n");

  for(ix=0; ix<nxpixels; ix++)
    for(iy=0; iy<nypixels; iy++)
      {
	iix=unzoomed_ul_x+(int)(ix*zoom_factor_x);
	iiy=unzoomed_ul_y+(int)(iy*zoom_factor_y);

	if(1==flag_ColorInput)
	  image_gray[iy][ix]=RGB2MONO(image_source[iiy][iix*3],
				      image_source[iiy][iix*3+1],
				      image_source[iiy][iix*3+2]);
	else
	  image_gray[iy][ix]=image_source[iiy][iix];
      }
}


void
NewXImage()
{
  /* Create our XImage */
  image=XCreateImage(display, visual,
		     display_bpp,
		     ZPixmap, 0, 
		     (flag_ShowColor ? &image_color[0][0] : &image_gray[0][0]),
		     nxpixels, nypixels, display_bpp, 0);
  XInitImage(image);
}

void
MarkersOff()
{
  flag_MarkersOn=0;
  XDrawLine(display, subw_image, gc_dashes,
	    0,last_marker_y, nxpixels-1,last_marker_y);
  XDrawLine(display, subw_image, gc_dashes,
	    last_marker_x,0, last_marker_x, nypixels-1);
  last_marker_x=-1;
  last_marker_y=-1;

}

void
MarkersOn()
{
  flag_MarkersOn=1;
  XDrawLine(display, subw_image, gc_dashes,
	    0,current_pxl_y, nxpixels-1,current_pxl_y);
  XDrawLine(display, subw_image, gc_dashes,
	    current_pxl_x,0, current_pxl_x, nypixels-1);
  last_marker_x=current_pxl_x;
  last_marker_y=current_pxl_y;
}
  
void
UpdateImage()
{

  if(flag_verbose)
    printf("UpdateImage()\n");

  if(!flag_MarkersOn)
    return;
  if(last_marker_x>0)
    {
      XDrawLine(display, subw_image, gc_dashes,
		0,last_marker_y, nxpixels-1,last_marker_y);
      XDrawLine(display, subw_image, gc_dashes,
		last_marker_x,0, last_marker_x, nypixels-1);
    }
  XDrawLine(display, subw_image, gc_dashes,
	    0,current_pxl_y, nxpixels-1,current_pxl_y);
  XDrawLine(display, subw_image, gc_dashes,
	    current_pxl_x,0, current_pxl_x, nypixels-1);
  last_marker_x=current_pxl_x;
  last_marker_y=current_pxl_y;
}

void
UpdateInfo()
{
  static char buf[257];

  int x=1;
  int y=15;
  int len;

  if(flag_verbose)
    puts("Update_Info()");

  if(!flag_plots_on)
    return;


  if( (current_pxl_x>=0) && (current_pxl_x < nxpixels) &&
      (current_pxl_y>=0) && (current_pxl_y < nypixels) )
    {
      XFillRectangle(display, subw_info, gc_fill,
		     0, 0,
		     plot_window_tall, plot_window_tall);

      /* replace Round with floor */
      sprintf(buf, "%d,%d",
	      unzoomed_ul_x + (int)(current_pxl_x * zoom_factor_x),
	      unzoomed_ul_y + (int)(current_pxl_y * zoom_factor_y));

      len=strlen(buf);

      x=9;
      y=15;
      XDrawImageString(display, subw_info, gc, x, y, buf, len);
      y+=15;
      sprintf(buf, " =%3d",
	      image_gray[current_pxl_y][current_pxl_x]);
      XDrawImageString(display, subw_info, gc, x, y, buf, strlen(buf));

    }

  
}
void
UpdateHoriz()
{
  int i;

  if(flag_verbose)
    puts("Update_Plot_Horiz()");

  if(!flag_plots_on)
    return;
    
  /* Clear the plot window */
  XFillRectangle(display, subw_plot_horiz, gc_fill,
		 0, 0,
		 nxpixels, plot_window_tall);
  for(i=0; i<nxpixels-2; i++)
    {
      h_segment_i[i].y1 = plot_window_tall - 1 -
	(int)Round(image_gray[current_pxl_y][i] * plot_scale );
      h_segment_i[i].y2 = plot_window_tall - 1 -
	(int)Round(image_gray[current_pxl_y][i+1] * plot_scale );
    }
  XDrawSegments(display, subw_plot_horiz, gc, h_segment_i, nxpixels-1);

  XDrawLine(display, subw_plot_horiz, gc_dashes,
	    current_pxl_x,0 , current_pxl_x,plot_window_tall-1);
	    
  if(0&&flag_ShowRGBPlots)
    {
      printf("%ld, %ld\n", 
	     (long)image_gray[current_pxl_y][current_pxl_x],
	     
	     (XGetPixel(image,
			current_pxl_x, current_pxl_y)));

      for(i=0; i<nxpixels-2; i++)
	{
	  h_segment_r[i].y1 = plot_window_tall - 1 -
	    (int)Round((XGetPixel(image,
				  i, current_pxl_y)) * plot_scale );
	  h_segment_r[i].y2 = plot_window_tall - 1 -
	    (int)Round((XGetPixel(image,
				  i+1, current_pxl_y)) * plot_scale);
	}
      XDrawSegments(display, subw_plot_horiz, gc_red, h_segment_r, nxpixels-1);
    }
}

void
UpdateVert()
{
  int i;

  if(flag_verbose)
    puts("Update_Plot_Vert()");

  if(!flag_plots_on)
    return;
    
  /* Clear the plot window */
  XFillRectangle(display, subw_plot_vert, gc_fill,
		 0, 0,
		 plot_window_tall, nypixels);

  for(i=0; i<nypixels-2; i++)
    {
      v_segment_i[i].x1 = 
	(int)Round(image_gray[i][current_pxl_x] * plot_scale );
      v_segment_i[i].x2 = 
	(int)Round(image_gray[i+1][current_pxl_x] * plot_scale );

    }
  XDrawSegments(display, subw_plot_vert, gc, v_segment_i, nypixels-1);
  
  if(0&&flag_ShowRGBPlots)
    {
      for(i=0; i<nypixels-2; i++)
	{
	  v_segment_r[i].x1 = 
	    (int)Round(image_gray[i][current_pxl_x] * plot_scale );
	  v_segment_r[i].x2 = 
	    (int)Round(image_gray[i+1][current_pxl_x] * plot_scale );

	}
      XDrawSegments(display, subw_plot_vert, gc_red, v_segment_r, nypixels-1);
    }

  XDrawLine(display, subw_plot_vert, gc_dashes,
	    0,current_pxl_y,  plot_window_tall-1,current_pxl_y);
}

void
UpdateAll()
{
  UpdateImage();
  UpdateInfo();
  UpdateHoriz();
  UpdateVert();
}


CreatePlotWindows()
{
  subw_plot_horiz = XCreateSimpleWindow (display, win_zview,
					 0,nypixels+window_borderwidth,
					 nxpixels, plot_window_tall,
					 0,
					 foreground_pixel,
					 background_pixel);

  subw_plot_vert = XCreateSimpleWindow (display, win_zview,
					nxpixels+window_borderwidth, 0,
					plot_window_tall, nypixels,
					0,
					foreground_pixel,
					background_pixel);

  /* window to hold info */
  subw_info = XCreateSimpleWindow (display, win_zview,
				   nxpixels+window_borderwidth,
				   nypixels+window_borderwidth,
				   plot_window_tall,
				   plot_window_tall,
				   0,
				   foreground_pixel,
				   background_pixel);
  XMapWindow (display, subw_plot_vert);
  XMapWindow (display, subw_plot_horiz);
  XMapWindow (display, subw_info); 
}
DestroyPlotWindows()
{
  /* Destroy the three windows, 2 plot and 1 info */
  if(subw_plot_vert)
    XDestroyWindow(display, subw_plot_vert);
  if(subw_plot_horiz)
    XDestroyWindow(display, subw_plot_horiz);
  if(subw_info)
    XDestroyWindow(display, subw_info);
}

NewPlotWindows()
{
  int i;

  if(flag_plots_on)
    DestroyPlotWindows();


  plot_scale=1.0/255.0 * (plot_window_tall-1);
  h_segment_i=(XSegment *)calloc(nxpixels, sizeof(XSegment));
  for(i=0; i<nxpixels-1; i++)
    {
      h_segment_i[i].x1=i;
      h_segment_i[i].x2=i+1;
    }
  v_segment_i=(XSegment *)calloc(nypixels, sizeof(XSegment));
  for(i=0; i<nypixels-1; i++)
    {
      v_segment_i[i].y1=i;
      v_segment_i[i].y2=i+1;
    }


  if(flag_ShowRGBPlots)
    {
      h_segment_r=(XSegment *)calloc(nxpixels, sizeof(XSegment));
      for(i=0; i<nxpixels-1; i++)
	{
	  h_segment_r[i].x1=i;
	  h_segment_r[i].x2=i+1;
	}
      v_segment_r=(XSegment *)calloc(nypixels, sizeof(XSegment));
      for(i=0; i<nypixels-1; i++)
	{
	  v_segment_r[i].y1=i;
	  v_segment_r[i].y2=i+1;
	}
    }
    
  CreatePlotWindows();

  XDrawLine(display, win_zview, gc,
	    nxpixels, 0, nxpixels, nypixels+plot_window_tall-1);
  XDrawLine(display, win_zview, gc,
	    0, nypixels, nxpixels+plot_window_tall-1, nypixels);

}

/* PlotsOn()  map in plot and info windows */
void
PlotsOn()
{
  int i;

  if(flag_verbose)
    puts("PlotsOn()");

  XResizeWindow(display, win_zview,
		nxpixels+plot_window_tall + window_borderwidth,
		nypixels+plot_window_tall + window_borderwidth);

  CreatePlotWindows();


  flag_plots_on=1;

  plot_scale=1.0/255.0 * (plot_window_tall-1);
  h_segment_i=(XSegment *)calloc(nxpixels, sizeof(XSegment));
  for(i=0; i<nxpixels-1; i++)
    {
      h_segment_i[i].x1=i;
      h_segment_i[i].x2=i+1;
    }
  v_segment_i=(XSegment *)calloc(nypixels, sizeof(XSegment));
  for(i=0; i<nypixels-1; i++)
    {
      v_segment_i[i].y1=i;
      v_segment_i[i].y2=i+1;
    }


  if(flag_ShowRGBPlots)
    {
      h_segment_r=(XSegment *)calloc(nxpixels, sizeof(XSegment));
      for(i=0; i<nxpixels-1; i++)
	{
	  h_segment_r[i].x1=i;
	  h_segment_r[i].x2=i+1;
	}
      v_segment_r=(XSegment *)calloc(nypixels, sizeof(XSegment));
      for(i=0; i<nypixels-1; i++)
	{
	  v_segment_r[i].y1=i;
	  v_segment_r[i].y2=i+1;
	}
    }
    
  XDrawLine(display, win_zview, gc,
	    nxpixels, 0, nxpixels, nypixels+plot_window_tall-1);
  XDrawLine(display, win_zview, gc,
	    0, nypixels, nxpixels+plot_window_tall-1, nypixels);

  return;
}

/* PlotsOff()  Turn off (and destroy) plot windows */
void
PlotsOff()
{
  XSetWindowAttributes attr;

  if(flag_verbose)
    puts("Plots_Off()");

  
  DestroyPlotWindows();

  /* Resize to image size */
  XResizeWindow(display, win_zview, nxpixels, nypixels);

  free(h_segment_i);
  free(v_segment_i);

  if(flag_ShowRGBPlots)
    {
      free(h_segment_r);
      free(v_segment_r);
    }
  /* Turn off marker lines, since there is no place to draw them */
  flag_MarkersOn=0;
  flag_plots_on=0;
}

void
NewWindows(char **argv, int argc, XSizeHints hints, XSetWindowAttributes attr,
	   unsigned long mask) 
{
  XWindowChanges xwc;
  XSetWindowAttributes xswa;

  if(flag_verbose)
    puts("NewWindows()");

  hints.max_width=nxpixels+( (flag_plots_on) ? plot_window_tall+window_borderwidth : 0);
  hints.max_height=nypixels+( (flag_plots_on) ? plot_window_tall+window_borderwidth : 0);
  hints.width=hints.max_width;
  hints.height=hints.max_height;

  if(win_zview)
    {
      XDestroySubwindows(display, win_zview);
      XDestroyWindow(display, win_zview);

      /* Zero all subw so we don't re-destroy */
      subw_image=0;
      subw_plot_horiz=0;
      subw_plot_vert=0;
      subw_info=0;
      subw_image=0;
    }
  win_zview = XCreateWindow (display, RootWindow (display, screen),
			     hints.x, hints.y,
			     hints.width, hints.height,
			     0,
			     display_bpp,
			     InputOutput, visual, mask,
			     &attr);
  XSetStandardProperties(display, win_zview, window_name, NULL, (Pixmap) 0,
			 argv, argc, &hints);
  XSetWindowColormap(display, win_zview, colormap_gray);


  /* window to hold image */
  subw_image = XCreateSimpleWindow(display, win_zview, 0, 0,
				   nxpixels, nypixels,
				   0,
				   attr.border_pixel,
				   attr.background_pixel);

  XMapWindow (display, subw_image);
  XMapWindow (display, win_zview);

  /* Standard GC for image */
  gc=XCreateGC(display, win_zview, 0, NULL);
  XSetForeground(display, gc, foreground_pixel);
  XSetBackground(display, gc, background_pixel);

  /* Standard GC for red */
  gc_red=XCreateGC(display, win_zview, 0, NULL);
  XSetForeground(display, gc, red_pixel);
  XSetBackground(display, gc, background_pixel);

  /* GC for drawing dashed lines */
  gc_dashes=XCreateGC(display, win_zview, 0, NULL);
  XCopyGC(display, gc, (long)0xeff, gc_dashes);
  XSetFunction(display, gc_dashes, GXxor);
  XSetLineAttributes(display, gc_dashes, 1, LineDoubleDash,
		     CapNotLast, JoinMiter);
  /* GC for fill */
  gc_fill=XCreateGC(display, win_zview, 0, NULL);
  XSetForeground(display, gc_fill, background_pixel);
  XSetBackground(display, gc_fill, foreground_pixel);

  if(flag_plots_on)
    NewPlotWindows();

}

void
Refresh()
{
  int save;
	    
  if(flag_verbose)
    printf("ExposeEvent\n");
  if(save=flag_MarkersOn)
    MarkersOff();
  XPutImage(display, subw_image, gc, image, 0, 0, 0, 0,
	    nxpixels, nypixels);
  if(flag_MarkersOn=save)
    MarkersOn();

  if(flag_plots_on)
    {
      XDrawLine(display, win_zview, gc,
		nxpixels, 0, nxpixels, nypixels+plot_window_tall-1);
      XDrawLine(display, win_zview, gc,
		0, nypixels, nxpixels+plot_window_tall-1, nypixels);
    }


  UpdateAll();
}

void
ZviewExit(int exit_val)
{
  
  char buff [1025];

  sprintf(buff, "/tmp/zview.%d", (int)getpid());
  if(0!=fopen(buff, "r"))
    remove(buff);

  exit(exit_val);
}

main(argc, argv)
     int argc;
     char **argv;
{
  float ftemp;
  char *displayname = NULL;
  char *geom = NULL;
  int i;
  XSizeHints hints;
  XSetWindowAttributes attr;
  XWindowAttributes wattr;
  unsigned long mask = 0L;
  int done;
  Bool reverse = False;
  XFontStruct *font_info;
  int ir, ic;
  XEvent event;
  XButtonEvent *buttonevent;
  XMotionEvent *motionevent;
  XKeyEvent *keyevent;
  XResizeRequestEvent *resizeevent;
  XColor *color;
  XVisualInfo visual_info;
  int erc;
  unsigned long *pixels;
  unsigned char **plot_horiz_data, **plot_vert_data;
  char buff[1024];
  KeySym keysym;
  unsigned char *cptr;
  int itemp;
  unsigned char ctemp;
  unsigned long ultemp;
  int flag_minusname = 0;
  int flag_showplots = 1;
  int *cindex;
  FILE *fh_in;
  FILE *fh;
  unsigned char *readbuf;
  int flag_stdin = 0;
  int flag_zcat = 0;
  int arg_nrows = -1;
  int arg_ncols = -1;
  struct stat stat_buff;
  int ii;
  int buf_offset;


  /* Parse the args */
  ProgramName = argv[0];
  for (i = 1; i < argc; i++)
    {
      if (argv[i][0] == '-')
	{
	  switch (argv[i][1])
	    {
	    case 'd':		/* -display host:display */
	      if (++i >= argc) Usage ();
	      displayname = argv[i];
	      continue;
	    case 'v':		/* -display host:display */
	      flag_verbose=1;
	      continue;
	    case 'g':
	      switch (argv[i][2])
		{
		case 'e':	/* -geometry geom */
		  if (++i >= argc) Usage ();
		  geom = argv[i];
		  continue;
		case 'a':	/* -gamma */
		  if (++i >= argc) Usage ();
		  image_gamma = atof(argv[i]);
		  continue;
		}
	      continue;
	    case 'n':
	      switch (argv[i][2])
		{
		case 'o':        /* -noplot */
		  if(strncmp(&argv[i][3], "plot", 4) == 0)
		    flag_plots_on=0;
		  break;
		case 'a':        /* -name */
		  if(++i >= argc) Usage();
		  window_name = argv[i];
		  break;
		case 'r':        /* -nrows */
		  if(++i >= argc) Usage();
		  arg_nrows = atoi(argv[i]);
		  break;
		case 'c':        /* -ncols */
		  if(++i >= argc) Usage();
		  arg_ncols = atoi(argv[i]);
		  break;
		}
	      continue;
	    case 'p':		/* -pz */
	      if(++i >= argc ) Usage();
	      plot_window_tall=atoi(argv[i]);
	      continue;
	    case 'r':		/* -rv */
	      reverse = True;
	      continue;
	    case '-':
	      switch(argv[i][2])
		{
		case '\0':
		  flag_minusname=1;
		  break;
		case 'h':
		  Usage();
		  exit(0);
		  break;
		}
	      continue;
	    case '\0':
	      image_infname=argv[i];
	      continue;
	      /*
		default:
		Usage ();
		*/
	    }
	}
      else 
	{
	  image_infname=strdup(argv[i]);
	  if(flag_minusname)
	    image_infname++;
	}
    }

  /* No input */
  if(image_infname==NULL)
    {
      Usage();
      exit(0);
    }

  /* Give us a default window name */
  if(window_name==NULL)
    {
      sprintf(buff, "ZView: %s", image_infname);
      window_name=strdup(buff);
    }

  /* default-ize the size of the plot windows */
  if(plot_window_tall<0)
    {
      Death(1, "invalid spec for -pz\n");
      Usage();
    }
  else if(plot_window_tall == 0)
    flag_plots_on = 0;

  /* --------------------------------------------------------------- */
  /* 
   * Input file
   *  - open, check readability
   *  - find out where beginning of file is in our private read buffer
   */
  if(0==strcmp(image_infname, "-"))
    {
      fh_in=stdin;
      flag_stdin=1;
    }
  else
    fh_in=fopen(image_infname, "r");
  if(0 == fh_in)
    Death(-1, "Could not open input file %s.\n", image_infname);


  if(-1!=arg_ncols)
    {
      AnImage.type=RAI_TYPE_GRAY8;
      AnImage.width=arg_ncols;
      AnImage.height=arg_nrows;
    }
  erc=ReadAnImage(fh_in, &AnImage);
  if(erc)
    Death(-1, "zview:  Could not read from \"%s\".  Use --help for usage\n",
	  image_infname);

  image_width=AnImage.width;
  image_height=AnImage.height;
  image_source=AnImage.data;
  flag_ColorInput=((AnImage.bytespp==3)? 1 : 0);
    
  fclose(fh_in);


  /* load file by here ... */

  /* --------------------------------------------------------------- */
  
  /* X
   *  - initialize X display 
   */
  

  display = XOpenDisplay (displayname);
  if (!display)
    {
      fprintf (stderr, "%s:  unable to open display '%s'\n",
	       ProgramName, XDisplayName (displayname));
      ZviewExit(1);
    }

  screen = DefaultScreen (display);

  /* --------------------------------------------------------------- */
  /* Visual -- try for a true color visual */

  {
    char *label[]={"StaticGray", "GrayScale", "StaticColor", "PseudoColor", "TrueColor", "DirectColor"};
    int erc, n;
    XVisualInfo *info, info_templ;
    info=XGetVisualInfo(display, VisualNoMask, &info_templ, &n);
    if(flag_verbose)
      {
	printf("%d visuals...\n", n);
	for(i=0; i<n; i++)
	  printf("Visual %d:  class=%d, depth=%d, cmap size=%d, %s\n",
		 i, info[i].class, info[i].bits_per_rgb, info[i].colormap_size,
		 label[info[i].class]);
      }

    /* Copy to get a decent default */
    visual=(Visual *)CopyFromParent;
    visual_info.visual=0;

    /* Which visual to use? */

    for(i=0; i<n; i++)  /* 8 bpp Pseudocolor */
      if((3==info[i].class) && (8==info[i].bits_per_rgb) 
	 && (256 <= info[i].colormap_size))
	visual_info=info[i];

    if(0==visual_info.visual)
      for(i=0; i<n; i++)  /* 24 bpp Pseudocolor */
	if((3==info[i].class) && (24==info[i].bits_per_rgb))
	  visual_info=info[i];

    if(0==visual_info.visual)
      for(i=0; i<n; i++)  /* 24 bit Truecolor */
	if((4==info[i].class) && (24==info[i].bits_per_rgb))
	  visual_info=info[i];

    if(0==visual_info.visual)
      for(i=0; i<n; i++)  /* 8 bpp GrayScale */
	if((1==info[i].class) && (8==info[i].bits_per_rgb))
	  {
	    visual_info=info[i];
	    flag_ShowColor=0;
	  }
    display_bpp=visual_info.bits_per_rgb;
    flag_TrueColor= ( visual_info.class == 4 );

    /* if not already specified private colormap, see if we need one anyway */
    if(0==flag_PrivateColormap)
      if( 1==(visual_info.class % 2))  /* Odd classes => must allocate */
	flag_PrivateColormap=1;
      
    if(flag_verbose)
      printf(" Using visual:  class=%d, depth=%d, cmap size=%d, %s\n",
	     visual_info.class, visual_info.bits_per_rgb,
	     visual_info.colormap_size,
	     label[visual_info.class]);
    XFree(info);
  }

  if(visual_info.visual != default_visual)
    if(flag_verbose)
      printf("Not using default visual\n");

  /* Our visual */
  visual=visual_info.visual;

  if(visual_info.class>1)
    flag_ColorDisplay=1;

  /* --------------------------------------------------------------- */
  /* Set initial size of image / window get our sizeing hints from the -geom arg */
  if(1==flag_plots_on)
    set_sizehints (&hints,
		   10 + window_borderwidth + plot_window_tall,
		   10 + window_borderwidth + plot_window_tall,
		   image_width + window_borderwidth + plot_window_tall,
		   image_height + window_borderwidth + plot_window_tall,
		   100,100,
		   geom);
  else
    set_sizehints (&hints,
		   10, 10,
		   image_width, image_height,
		   100,100,
		   geom);

  /* Size of image subwindow is nxpixels wide by nypixels tall */
  nxpixels=hints.width  - (flag_plots_on ? plot_window_tall + window_borderwidth : 0);
  nypixels=hints.height - (flag_plots_on ? plot_window_tall + window_borderwidth : 0);

  /* Initial zoom factor.  >0 shrinks, <0 enlarges */
  zoom_factor_x=(float)image_width/nxpixels;
  zoom_factor_y=(float)image_height/nypixels;

  /* Allocate room for our history of zoom factors.  We can move back down the list to 0th */
  last_unzoomed_ul_x=(int *)malloc(nzoom*sizeof(int));
  last_unzoomed_ul_y=(int *)malloc(nzoom*sizeof(int));

  if(flag_verbose)
    printf("zoom_factor_x=%f zoom_factor_y=%f\n", zoom_factor_x, zoom_factor_y);

  /* --------------------------------------------------------------- */

  /* We always need a gray image for intensity values */
  MakeImageGray();

  /* if we have a color display, and the input image was color, and
     we have not been asked to force grayscale output, then
     color output is OK */
  if(flag_ColorDisplay && flag_ColorInput && !flag_ForceGrayDisplay)
    flag_ShowColor=1;

  if(1==flag_PrivateColormap)
    {
      if(1==flag_verbose)
	printf("Private colormap\n");


      /* Need a colormap.  Make our own private gray one */
      colormap_gray=XCreateColormap(display, RootWindow(display,
							visual_info.screen),
				    visual_info.visual, AllocNone);

      pixels=(unsigned long *)malloc(256*sizeof(unsigned long));

      /* Get an 256 color colormap */
      XAllocColorCells(display, colormap_gray, True, (unsigned long *) NULL, 0,
		       pixels, 256);
      color=(XColor *)malloc(256*sizeof(XColor));

      cindex=(int *)malloc(256*sizeof(int));
      for(i=0; i<256; i++)
	cindex[i]=i;
 
      /* Allocate the 256 levels of gray */
      for(i=0; i<256; i++)
	{
	  if(reverse)
	    color[i].pixel=(unsigned long)pixels[256-1-i];
	  else
	    color[i].pixel=(unsigned long)pixels[i];
	  color[i].red=(unsigned short)cindex[i]*256;
	  color[i].green=(unsigned short)cindex[i]*256;
	  color[i].blue=(unsigned short)cindex[i]*256;
	  color[i].flags=DoRed|DoGreen|DoBlue;
	}
      XStoreColors(display, colormap_gray, color, 256);
	  
      /* foreground is black, background is white */
      background_pixel=pixels[255];
      foreground_pixel=pixels[0];

    }
  red_pixel=pixels[255/3];
  XInstallColormap(display, colormap_gray);

  /* Set attributes for the windows we will create */
  /* select for all relevant events */
  attr.event_mask = KeyPressMask | KeyReleaseMask | ButtonPressMask |
    ButtonReleaseMask | Button1MotionMask | Button2MotionMask |
    VisibilityChangeMask | ResizeRedirectMask |
    ExposureMask | ButtonMotionMask;
  attr.background_pixel = background_pixel;
  attr.border_pixel = foreground_pixel;
  attr.colormap = colormap_gray;
  mask |= (CWBackPixel | CWBorderPixel | CWEventMask | CWColormap);


  /* Make all New Windows */
  NewWindows(argv, argc, hints, attr, mask);

  /* Load a boring font we will use */
  if((font_info = XLoadQueryFont(display, "9x15")) == NULL)
    Death(-1, "Cannot open 9x15 font\n");

  /* Initially, are plot windows on or off? */
  (flag_plots_on) ? PlotsOn() : PlotsOff();

  /* Make a new XImage */
  NewXImage();

  /* and display it */
  XPutImage(display, subw_image, gc, image, 0, 0, 0, 0,
	    nxpixels, nypixels);

  /* Lets start the day off clean and fresh */
  XFlush(display);

  /* --------------------------------------------------------------- */

  /* Wait for events forever */
  for (done = 0; !done; )
    {
      XNextEvent (display, &event);

      switch (event.type)
	{
	case Expose:
        case VisibilityNotify:
	  if(flag_verbose)
	    printf("ExposeEvent\n");
	  Refresh();
	  break;
	case ResizeRequest:
	  resizeevent=(XResizeRequestEvent *)&event;
	  if(resizeevent->window=win_zview)
	    {
	      if(flag_verbose)
		printf("ResizeRequestEvent -> %d x %d\n",
		       resizeevent->width, resizeevent->height);

	      {
		int save;

		if(flag_verbose)
		  printf("ResizeRequestEvent -> %d x %d\n",
			 resizeevent->width, resizeevent->height);

		nxpixels=resizeevent->width -
		  ((flag_plots_on)? (plot_window_tall+window_borderwidth): 0);
		nypixels=resizeevent->height-
		  ((flag_plots_on)? (plot_window_tall+window_borderwidth): 0);
		if(flag_verbose)
		  printf(" new image size: %d x %d\n", nxpixels, nypixels);
		zoom_factor_x=(float)image_width/nxpixels;
		zoom_factor_y=(float)image_height/nypixels;

		NewWindows(argv, argc, hints, attr, mask);

		MakeImageGray();
		NewXImage();
		if(save=flag_MarkersOn)
		  MarkersOff();
		XPutImage(display, subw_image, gc, image, 0, 0, 0, 0,
			  nxpixels, nypixels);
		current_pxl_x=nxpixels/2-1;
		current_pxl_y=nypixels/2-1;
		if(flag_MarkersOn=save)
		  MarkersOn();

		UpdateAll();

	      }
	    }
	  break;
	case KeyRelease:
	  keyevent=(XKeyEvent *)&event;
	  itemp=XLookupString(keyevent, buff, 256, &keysym, NULL);
	  switch (itemp)
	    {
	    case 0:
	      strcpy(buff, XKeysymToString(keysym));
	      if(!strcmp(buff, "Up"))
		current_pxl_y-=(current_pxl_y!=0);
	      else if(!strcmp(buff, "Down"))
		current_pxl_y+=(current_pxl_y != nypixels-1);
	      else if(!strcmp(buff, "Left"))
		current_pxl_x-=(current_pxl_x!=0);
	      else if(!strcmp(buff, "Right"))
		current_pxl_x+=(current_pxl_x != nxpixels-1);
	      UpdateAll(image);
	      break;
	    case 1:
	      switch(buff[0])
		{
		case 'i':
		  /* Info about current pic */
		  ShowSummary();
		  break;
		case 'h':
		case '?':
		  /* Show help on keys / buttons */
		  ShowHelp();
		  break;
		case 'q':
		  /* Attempt a graceful exit */
		  XCloseDisplay (display);
		  ZviewExit(0);
		  break;
		case 'p':
		  /* Toggle state of plot windows (on or off) */
		  (flag_plots_on) ? PlotsOff() : PlotsOn();
		  break;
                case 's':
		  /* Screen dump to PGM file */
                  {
                    XImage *WholeImage;
                    XColor *colortemp;
                    int nw, nh, iw, ih;
                    unsigned char *cptr;

		    if(flag_verbose)
		      fprintf(stderr, "ScreenDump()\n");

                    nw=nxpixels + ((subw_plot_vert)?plot_window_tall : 0);
                    nh=nypixels + ((subw_plot_horiz)?plot_window_tall : 0);
                    if(flag_verbose)
		      fprintf(stderr, "  write PGM image: nw=%d nh=%d\n", nw, nh); 

                    colortemp=(XColor *)malloc(nxpixels*sizeof(XColor));
                    cptr=(char *)malloc(nw*sizeof(unsigned char));
                    WholeImage=XGetImage(display, win_zview,
                                         0, 0, nw, nh, 0xff, ZPixmap);
                    sprintf(buff, "%s", SCREEN_DUMP_FNAME);
                    if(0 != (fh=fopen(buff, "w")))
		      {
			fprintf(fh, "P5\n%d %d\n%d\n", nw, nh, 255);
			for(ih=0; ih<nypixels; ih++)
			  {
			    /* subw_image */
			    for(iw=0; iw<nxpixels; iw++)
			      cptr[iw]=XGetPixel(WholeImage, iw, ih); /* pixel index IS gray level */

			    /* subw_plot_vert */
			    for(iw=nxpixels; iw<nw; iw++)
			      cptr[iw]=(XGetPixel(WholeImage, iw, ih)==foreground_pixel)?0:255;

			    fwrite(cptr, sizeof(unsigned char), nw, fh);
			  }
			/* subw_plot_horiz */
			for(ih=nypixels; ih<nh; ih++)
			  {
			    for(iw=0; iw<nw; iw++)
			      cptr[iw]=(XGetPixel(WholeImage, iw, ih)!=background_pixel)?0:255;
			    fwrite(cptr, sizeof(unsigned char), nw, fh);
			  }
			fclose(fh);
		      }
		    else
		      fprintf(stderr, "zview:  Error, could not open %s for writing.\n",
			      SCREEN_DUMP_FNAME);

		    free(colortemp);
		    free(cptr);
		    XDestroyImage(WholeImage);
		  }
		break;
		case 'Z':
		  /* Zoom out (de-enlarge) */
		  if(times_zoomed>1)
		    {
		      int save;
		      unzoomed_ul_x=last_unzoomed_ul_x[times_zoomed];
		      unzoomed_ul_y=last_unzoomed_ul_y[times_zoomed];
		      times_zoomed--;
		      zoom_factor_x=zoom_factor_x*2;
		      if(flag_verbose)
			printf("zoom_x -> %f\n", zoom_factor_x);
		      zoom_factor_y=zoom_factor_y*2;
		      if(flag_verbose)
			printf("zoom_y -> %f\n", zoom_factor_y);
		      
		      MakeImageGray();
		      NewXImage();
		      
		      if(save=flag_MarkersOn)
			MarkersOff();
		      XPutImage(display, subw_image, gc, image, 0, 0, 0, 0,
				nxpixels, nypixels);
		      if(flag_MarkersOn=save)
			MarkersOn();

		      current_pxl_x=nxpixels/2-1;
		      current_pxl_y=nypixels/2-1;

		    }
		  break;
		case 'z':
		  /* Zoom in (enlarge) */
		  {
		    int save;

		    times_zoomed++;
		    /* preserve the current unzoomed locations */
		    if(nzoom==times_zoomed)
		      {
			nzoom+=10;
			last_unzoomed_ul_x=(int *)
			  realloc(last_unzoomed_ul_x, nzoom*sizeof(int));
			last_unzoomed_ul_y=(int *)
			  realloc(last_unzoomed_ul_y, nzoom*sizeof(int));
		      }
		    last_unzoomed_ul_x[times_zoomed]=unzoomed_ul_x;
		    last_unzoomed_ul_y[times_zoomed]=unzoomed_ul_y;

		    zoom_factor_x=zoom_factor_x/2;
		    zoom_factor_y=zoom_factor_y/2;
		    if(flag_verbose)
		      printf("zoom_x -> %f\n", zoom_factor_x);
		    unzoomed_ul_x=(int)((current_pxl_x*zoom_factor_x*2) -
					zoom_factor_x*nxpixels/2);
		    unzoomed_ul_y=(int)((current_pxl_y*zoom_factor_y*2) -
					zoom_factor_y*nypixels/2);
		    if(unzoomed_ul_x<0)
		      unzoomed_ul_x=0;
		    if(unzoomed_ul_y<0)
		      unzoomed_ul_y=0;
		    if((unzoomed_ul_x+nxpixels*zoom_factor_x)
		       >= image_width)
		      unzoomed_ul_x=image_width-nxpixels*zoom_factor_x-1;
		    if((unzoomed_ul_y+nypixels*zoom_factor_y)
		       >= image_height)
		      unzoomed_ul_y=image_height-nypixels*zoom_factor_y-1;
		    MakeImageGray();
		    NewXImage();
		    if(save=flag_MarkersOn)
		      MarkersOff();
		    XPutImage(display, subw_image, gc, image, 0, 0, 0, 0,
			      nxpixels, nypixels);
		    current_pxl_x=nxpixels/2-1;
		    current_pxl_y=nypixels/2-1;
		    if(flag_MarkersOn=save)
		      MarkersOn();
		  }
		break;
		}
	      break;
	    }
	case ButtonPress:
	  buttonevent=(XButtonEvent *)&event;
	  if(buttonevent->button==Button1)
	    {
	      /* Update our pixel location, pixel value, plots and marker lines */
	      if(buttonevent->subwindow == subw_image)
		{
		  if((buttonevent->x >=0) && (buttonevent->x < nxpixels))
		    current_pxl_x=buttonevent->x;
		  if((buttonevent->y >=0) && (buttonevent->y < nypixels))
		    current_pxl_y=buttonevent->y;
		  if(flag_verbose)
		    printf("(%d,%d)=%d,g=%d\n",
			   current_pxl_x, current_pxl_y,
			   (int)image_source[current_pxl_y][current_pxl_x],
			   (int)image_gray[current_pxl_y][current_pxl_x]);
		  UpdateAll();
		}
	    }
	  else
	    {
	      /* Toggle state of marker lines (on / off) */
	      if(buttonevent->button==Button3)
		flag_MarkersOn ? MarkersOff() : MarkersOn();		  
	    }
	  break;
	case MotionNotify:
	  motionevent=(XMotionEvent *)&event;
	  if((motionevent->subwindow == subw_image) &&
	     (motionevent->state & Button1Mask)
	     )
	    /* Button down and motion,
	       update our pixel location, pixel value, plots and marker lines */
	    {
	      if((motionevent->x >0) && (motionevent->x < nxpixels))
		current_pxl_x=motionevent->x;
	      if((motionevent->y >0) && (motionevent->y < nypixels))
		current_pxl_y=motionevent->y;
	      if(XPending(display)<1)
		UpdateAll();
	    }
	  break;
	case ButtonRelease:
	  break;
	}
    }

  /* Cannot happen.  Attempt a graceful exit anyways*/
  fprintf(stderr, "Cannot happen\n");
  ZviewExit(1);
 
}

void 
Usage()
{
  static char *UsageMessage[] = 
  {
    " Usage:  zview [...]",
    "  -display displayname      X server",
    "  -geometry geom            size and position of window",
    "  -name string              window name",
    "  -gamma gammaval           gamma value to use (def. 1.0)",
    "  -noplot                   no side Z plots",
    "  -pz n                     size Z plots n pixels tall (def. 100)",
    "  -rv                       reverse video",
    "  -nrows #rows -ncols #cols preset image dimensions",
    "       (must specify at least ncols for RAW input format, ignored otherwise)",
    "  -zcat                     specify un(compress/gzip)er, must accept -dc",
    "                            and decompress to stdout",
    "",
    "",
    NULL
  };

  char **c;

  fprintf(stderr, "ZView version %s, %s, %s\n",
	  ZVIEW_VERSION, ZVIEW_DATE, ZVIEW_AUTHOR);

  for(c=UsageMessage; **c; c++)
    fprintf(stderr, "%s\n", *c);

  fprintf(stderr, "  valid input formats include:\n\t");
  for(c=ReadAnImage_WhatTypes();  **c; c++)
    fprintf(stderr, "%s ", *c);
  fprintf(stderr, "\n  and gzipped or compressed versions of same.\n"); 

  for(c=HelpMsg; **c; c++)
    fprintf(stderr, " %s\n", *c);

  fprintf(stderr, " Screen dump filename \"%s\"\n", SCREEN_DUMP_FNAME);
}
void UsageE() { Usage(); exit(0); }


  
#ifdef UNDEF
gamma
(int)Round( pow((float)i/255.0, 1.0/image_gamma) * 255.0);
#endif

  
