/*
 * $Id: cube.c,v 1.4 1995/09/16 14:01:57 traister Exp traister $
 * cube.c: function MapToCube maps a 24-bit IMAGE to an 8-bit IMAGE
 * using a standard RGB 6x6x6 color cube.
 *
 * Copyright (c) 1995, Joe Traister
 *
 * $Log: cube.c,v $
 * Revision 1.4  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.3  1995/09/03  02:14:54  traister
 * Changed MapToCube to create new image rather than modifying
 * the old one.
 *
 * Revision 1.2  1995/07/30  19:20:47  traister
 * Updated copyright prior to first public release
 *
 * Revision 1.1  1995/05/13  01:36:18  traister
 * Initial revision
 *
 */

/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "vimage.h"
#include "cube.h"

#define FS_SCALE 1024

int MapToCube(IMAGE *image, IMAGE *newimage)
{
  UCHAR *pP;
  int i=0, *thisrerr, *nextrerr, *thisgerr, *nextgerr, *thisberr, *nextberr;
  int fs_direction, row, col, limitcol, sr, sg, sb, *temperr, err;
  int rbval[] = { 21, 64, 107, 149, 192, 235 }, pixwidth;
  int gval[] = { 18, 55, 91, 128, 164, 200, 237 };

  memcpy(newimage, image, sizeof(IMAGE));
  if (!(newimage->bits = (UCHAR*)malloc(image->width*image->height))) {
    strcpy(errmsg, strerror(errno));
    return -1;
  }
  pixwidth = image->palsize == -15 ? 2 : 3;

  /* Initialize Floyd-Steinberg error vectors */
  if (!(thisrerr = (int*)malloc((image->width+2)*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    return -1;
  }
  if (!(nextrerr = (int*)malloc((image->width+2)*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    free(thisrerr);
    return -1;
  }
  if (!(thisgerr = (int*)malloc((image->width+2)*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    free(thisrerr);
    free(nextrerr);
    return -1;
  }
  if (!(nextgerr = (int*)malloc((image->width+2)*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    free(thisrerr);
    free(nextrerr);
    free(thisgerr);
    return -1;
  }
  if (!(thisberr = (int*)malloc((image->width+2)*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    free(thisrerr);
    free(nextrerr);
    free(thisgerr);
    free(nextgerr);
    return -1;
  }
  if (!(nextberr = (int*)malloc((image->width+2)*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    free(thisrerr);
    free(nextrerr);
    free(thisgerr);
    free(nextgerr);
    free(thisberr);
    return -1;
  }
  srandom((int)(time(0)^getpid()));
  for (i=0; i<image->width+2; i++) {
    thisrerr[i] = random()%(FS_SCALE*2)-FS_SCALE;
    thisgerr[i] = random()%(FS_SCALE*2)-FS_SCALE;
    thisberr[i] = random()%(FS_SCALE*2)-FS_SCALE;
    /* (random errors in [-1 .. 1]) */
  }
  fs_direction = 1;

  for (row=0; row < image->height; row++) {
    for (i=0; i < image->width+2; i++)
      nextrerr[i] = nextgerr[i] = nextberr[i] = 0;
    if (fs_direction) {
      col = 0;
      limitcol = image->width;
      pP = image->bits+row*image->width*pixwidth;
    } else {
      col = image->width-1;
      limitcol = -1;
      pP = image->bits+row*image->width*pixwidth+(image->width-1)*pixwidth;
    }
    do {
      /* Apply Floyd-Steinberg error to pixel */
      if (image->palsize == -24) {
	sr = *pP+thisrerr[col+1]/FS_SCALE;
	sg = *(pP+1)+thisgerr[col+1]/FS_SCALE;
	sb = *(pP+2)+thisberr[col+1]/FS_SCALE;
      } else {
	sr = ((*((short*)pP)&0x7C00)>>7)+7+thisrerr[col+1]/FS_SCALE;
	sg = ((*((short*)pP)&0x03E0)>>2)+7+thisgerr[col+1]/FS_SCALE;
	sb = ((*((short*)pP)&0x001F)<<3)+7+thisberr[col+1]/FS_SCALE;
      }
      if (sr < 0) sr = 0;
      if (sr > 255) sr = 255;
      if (sg < 0) sg = 0;
      if (sg > 255) sg = 255;
      if (sb < 0) sb = 0;
      if (sb > 255) sb = 255;

      /* Compute 6x7x6 cube location */
      *(newimage->bits+row*image->width+col) = sr/43+(sg/37)*6+(sb/43)*42;

      /* Propogate Floyd-Steinberg errors */
      if (fs_direction) {
	err = (sr-rbval[sr/43])*FS_SCALE;
	thisrerr[col+2] += (err*7)/16;
	nextrerr[col] += (err*3)/16;
	nextrerr[col+1] += (err*5)/16;
	nextrerr[col+2] += err/16;
	err = (sg-gval[sg/37])*FS_SCALE;
	thisgerr[col+2] += (err*7)/16;
	nextgerr[col] += (err*3)/16;
	nextgerr[col+1] += (err*5)/16;
	nextgerr[col+2] += err/16;
	err = (sb-rbval[sb/43])*FS_SCALE;
	thisberr[col+2] += (err*7)/16;
	nextberr[col] += (err*3)/16;
	nextberr[col+1] += (err*5)/16;
	nextberr[col+2] += err/16;
      } else {
	err = (sr-rbval[sr/43])*FS_SCALE;
	thisrerr[col] += (err*7)/16;
	nextrerr[col+2] += (err*3)/16;
	nextrerr[col+1] += (err*5)/16;
	nextrerr[col] += err/16;
	err = (sg-gval[sg/37])*FS_SCALE;
	thisgerr[col] += (err*7)/16;
	nextgerr[col+2] += (err*3)/16;
	nextgerr[col+1] += (err*5)/16;
	nextgerr[col] += err/16;
	err = (sb-rbval[sb/43])*FS_SCALE;
	thisberr[col] += (err*7)/16;
	nextberr[col+2] += (err*3)/16;
	nextberr[col+1] += (err*5)/16;
	nextberr[col] += err/16;
      }
      if (fs_direction) {
	col++;
	pP += pixwidth;
      } else {
	col--;
	pP -= pixwidth;
      }
    } while (col != limitcol);
    temperr = thisrerr;
    thisrerr = nextrerr;
    nextrerr = temperr;
    temperr = thisgerr;
    thisgerr = nextgerr;
    nextgerr = temperr;
    temperr = thisberr;
    thisberr = nextberr;
    nextberr = temperr;
    fs_direction = ! fs_direction;
  }
  free(thisrerr);
  free(nextrerr);
  free(thisgerr);
  free(nextgerr);
  free(thisberr);
  free(nextberr);

  newimage->palsize = 252;
  memcpy(newimage->palette, std_colormap, sizeof(std_colormap));
  return 0;
}
