/*
 * $Id: filesel.c,v 1.5 1995/09/16 14:01:57 traister Exp traister $
 * filesel.c: a file selector for vimage using the curses library
 *
 * Copyright (c) 1995, Joe Traister
 *
 * $Log: filesel.c,v $
 * Revision 1.5  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.4  1995/09/03  02:14:54  traister
 * Now uses ReadFile.
 *
 * Revision 1.3  1995/08/06  03:05:55  traister
 * Fixed redraw when deleting a file
 *
 * Revision 1.2  1995/07/30  19:20:47  traister
 * Added delete, information display, slide show, slide file options
 * in file selector
 *
 * Revision 1.1  1995/05/13  01:37:38  traister
 * Initial revision
 *
 */

/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#ifdef NCURSES_SUPPORT

#include <ncurses.h>
#include <malloc.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include "vimage.h"

#define COPYRIGHT "Copyright (c) 1995, Joe Traister"

#define HELP_MSG " Arrows move cursor, enter views picture, "\
                 "'q' to quit, 'h' for help "

#define TITLE "Vimage v"VERSION

extern char *progname;

int fileSelector(char **names, int nnames)
{
  int i, j, colwidth=0, cols, vcols, firstvcol=0, ccol=0, crow=0, exit=0;
  int filetype, cmark=0, *marklist, c, rdrwmsg = 0;
  char *marks, **slidenames, ifilename[PATH_MAX], msg[160], *pch;
  WINDOW *namewin, *msgwin, *titlewin;
  IMAGE image, newimage;
  FILE *fpin, *ofile;

  if (!(marks=(char*)malloc(nnames))) {
    fprintf(stderr, "%s: Could not allocate marks array\n", progname);
    return -1;
  }
  memset(marks, ' ', nnames);
  if (!(marklist=(int*)malloc(sizeof(int)*nnames))) {
    fprintf(stderr, "%s: Could not allocate slide index array\n", progname);
    return -1;
  }

  if (InitVGA(modename)) {
    fprintf(stderr, "%s: %s\n", progname, errmsg);
    return -1;
  }

  initscr();
  cbreak();
  noecho();
  keypad(stdscr, TRUE);

  titlewin = newwin(1, COLS, 0, 0);
  wbkgd(titlewin, A_REVERSE);
  wattrset(titlewin, A_REVERSE);
  wclear(titlewin);
  mvwaddstr(titlewin, 0, 1, TITLE);
  mvwaddstr(titlewin, 0, COLS-strlen(COPYRIGHT)-1, COPYRIGHT);
  wrefresh(titlewin);

  for (i=0; i < nnames; i++) {
    if (strlen(names[i]) > colwidth)
      colwidth = strlen(names[i]);
  }
  colwidth += 2;
  cols = nnames/(LINES-2);
  if (nnames%(LINES-2))
    cols++;
  vcols = COLS/colwidth;

  namewin = newwin(LINES-2, COLS, 1, 0);
  if (!namewin) {
    endwin();
    fprintf(stderr, "%s: Could not open name window\n", progname);
    return -1;
  }

  for (i=0; i < LINES-2; i++)
    for (j=0; j < vcols; j++) {
      if (j*(LINES-2)+i >= nnames)
	break;
      if (i == crow && j == ccol)
	wattrset(namewin, A_REVERSE);
      mvwaddch(namewin, i, j*colwidth, marks[j*(LINES-2)+i]);
      waddstr(namewin, names[j*(LINES-2)+i]);
      wattrset(namewin, A_NORMAL);
    }
  wrefresh(namewin);

  msgwin = newwin(1, COLS, LINES-1, 0);
  if (!msgwin) {
    endwin();
    fprintf(stderr, "%s: Could not open message window\n", progname);
    return -1;
  }
  wattrset(msgwin, A_REVERSE);
  wbkgd(msgwin, A_REVERSE);

  wclear(msgwin);
  mvwaddstr(msgwin, 0, (COLS-strlen(HELP_MSG))/2, HELP_MSG);
  wrefresh(msgwin);

  do {
    c = getch();
    if (rdrwmsg) {
      rdrwmsg = 0;
      wclear(msgwin);
      mvwaddstr(msgwin, 0, (COLS-strlen(HELP_MSG))/2, HELP_MSG);
      wrefresh(msgwin);
    }
    switch (c) {
    case ' ':
      if (marks[(ccol+firstvcol)*(LINES-2)+crow] == '+') {
	marks[(ccol+firstvcol)*(LINES-2)+crow] = ' ';
	for (i=0; i < cmark-1; i++)
	  if (marklist[i] == (ccol+firstvcol)*(LINES-2)+crow)
	    memmove(&marklist[i], &marklist[i+1], (cmark-i-1)*sizeof(int));
	cmark--;
      } else {
	marks[(ccol+firstvcol)*(LINES-2)+crow] = '+';
	marklist[cmark]=(ccol+firstvcol)*(LINES-2)+crow;
	cmark++;
      }
      wattrset(namewin, A_REVERSE);
      mvwaddch(namewin, crow, ccol*colwidth,
	       marks[(ccol+firstvcol)*(LINES-2)+crow]);
      wattrset(namewin, A_NORMAL);
      wrefresh(namewin);

    case KEY_DOWN:
      if (ccol+firstvcol < cols-1 || 
	  (ccol+firstvcol == cols-1 && 
	   (crow < nnames%(LINES-2)-1 ||
	    (nnames%(LINES-2) == 0 && crow < LINES-3)))) {
	mvwaddch(namewin, crow, ccol*colwidth,
		 marks[(ccol+firstvcol)*(LINES-2)+crow]);
	waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	crow++;
	if (crow > LINES-3) {
	  crow = 0;
	  ccol++;
	}
	if (ccol >= vcols) {
	  ccol--;
	  firstvcol++;
	  wclear(namewin);
	  for (i=0; i < LINES-2; i++)
	    for (j=0; j < vcols; j++) {
	      if ((firstvcol+j)*(LINES-2)+i >= nnames)
		break;
	      if (i == crow && j == ccol)
		wattrset(namewin, A_REVERSE);
	      mvwaddch(namewin, i, j*colwidth,
		       marks[(j+firstvcol)*(LINES-2)+i]);
	      waddstr(namewin, names[(j+firstvcol)*(LINES-2)+i]);
	      wattrset(namewin, A_NORMAL);
	    }
	} else {
	  wattrset(namewin, A_REVERSE);
	  mvwaddch(namewin, crow, ccol*colwidth,
		   marks[(ccol+firstvcol)*(LINES-2)+crow]);
	  waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	  wattrset(namewin, A_NORMAL);
	}
	wrefresh(namewin);
      }
      break;

    case KEY_UP:
      if (ccol+firstvcol > 0 || crow > 0) {
	mvwaddch(namewin, crow, ccol*colwidth,
		 marks[(ccol+firstvcol)*(LINES-2)+crow]);
	waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	crow--;
	if (crow < 0) {
	  ccol--;
	  crow = LINES-3;
	}
	if (ccol < 0) {
	  ccol++;
	  firstvcol--;
	  wclear(namewin);
	  for (i=0; i < LINES-2; i++)
	    for (j=0; j < vcols; j++) {
	      if ((firstvcol+j)*(LINES-2)+i >= nnames)
		break;
	      if (i == crow && j == ccol)
		wattrset(namewin, A_REVERSE);
	      mvwaddch(namewin, i, j*colwidth,
		       marks[(j+firstvcol)*(LINES-2)+i]);
	      waddstr(namewin, names[(j+firstvcol)*(LINES-2)+i]);
	      wattrset(namewin, A_NORMAL);
	    }
	} else {
	  wattrset(namewin, A_REVERSE);
	  mvwaddch(namewin, crow, ccol*colwidth,
		   marks[(ccol+firstvcol)*(LINES-2)+crow]);
	  waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	  wattrset(namewin, A_NORMAL);
	}
	wrefresh(namewin);
      }
      break;

    case KEY_LEFT:
      if (firstvcol+ccol > 0) {
	mvwaddch(namewin, crow, ccol*colwidth,
		 marks[(ccol+firstvcol)*(LINES-2)+crow]);
	waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	ccol--;
	if (ccol < 0) {
	  ccol = 0;
	  firstvcol--;
	  wclear(namewin);
	  for (i=0; i < LINES-2; i++)
	    for (j=0; j < vcols; j++) {
	      if ((firstvcol+j)*(LINES-2)+i >= nnames)
		break;
	      if (i == crow && j == ccol)
		wattrset(namewin, A_REVERSE);
	      mvwaddch(namewin, i, j*colwidth,
		       marks[(j+firstvcol)*(LINES-2)+i]);
	      waddstr(namewin, names[(j+firstvcol)*(LINES-2)+i]);
	      wattrset(namewin, A_NORMAL);
	    }	
	} else {  
	  wattrset(namewin, A_REVERSE);
	  mvwaddch(namewin, crow, ccol*colwidth,
		   marks[(ccol+firstvcol)*(LINES-2)+crow]);
	  waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	  wattrset(namewin, A_NORMAL);
	}
	wrefresh(namewin);
      }
      break;

    case KEY_RIGHT:
      if (firstvcol+ccol < cols-1) {
	mvwaddch(namewin, crow, ccol*colwidth,
		 marks[(ccol+firstvcol)*(LINES-2)+crow]);
	waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	ccol++;
	if (ccol >= vcols) {
	  ccol = vcols-1;
	  firstvcol++;
	  if (firstvcol+ccol >= cols-1 && nnames%(LINES-2)
	      && crow >= nnames%(LINES-2))
	    crow = nnames%(LINES-2)-1;
	  wclear(namewin);
	  for (i=0; i < LINES-2; i++)
	    for (j=0; j < vcols; j++) {
	      if ((j+firstvcol)*(LINES-2)+i >= nnames)
		break;
	      if (i == crow && j == ccol)
		wattrset(namewin, A_REVERSE);
	      mvwaddch(namewin, i, j*colwidth,
		       marks[(j+firstvcol)*(LINES-2)+i]);
	      waddstr(namewin, names[(j+firstvcol)*(LINES-2)+i]);
	      wattrset(namewin, A_NORMAL);
	    }	
	} else {  
	  if (firstvcol+ccol == cols-1 && nnames%(LINES-2)
	      && crow >= nnames%(LINES-2))
	    crow = nnames%(LINES-2)-1;
	  wattrset(namewin, A_REVERSE);
	  mvwaddch(namewin, crow, ccol*colwidth,
		   marks[(ccol+firstvcol)*(LINES-2)+crow]);
	  waddstr(namewin, names[(ccol+firstvcol)*(LINES-2)+crow]);
	  wattrset(namewin, A_NORMAL);
	}
	wrefresh(namewin);
      }
      break;

    case 's':
      if (cmark > 0) {
	if (!(slidenames=(char **)malloc(sizeof(char*)*cmark))) {
	  wclear(msgwin);
	  mvwaddstr(msgwin, 0, 1, "Could not allocate slide names array");
	  wrefresh(msgwin);
	  rdrwmsg=1;
	  break;
	}
	for (i=0; i < cmark; i++)
	  slidenames[i] = names[marklist[i]];
	endwin();
	if (Slideshow(slidenames, cmark)) {
	  refresh();
	  wclear(msgwin);
	  mvwaddstr(msgwin, 0, 1, errmsg);
	  wrefresh(msgwin);
	  rdrwmsg=1;
	}
	else
	  refresh();
	free(slidenames);
      }
      break;

    case 'c':
      if (cmark) {
	cmark = 0;
	memset(marks, ' ', nnames);
	wclear(namewin);
	for (i=0; i < LINES-2; i++)
	  for (j=0; j < vcols; j++) {
	    if ((firstvcol+j)*(LINES-2)+i >= nnames)
	      break;
	    if (i == crow && j == ccol)
	      wattrset(namewin, A_REVERSE);
	    mvwaddch(namewin, i, j*colwidth,
		     marks[(j+firstvcol)*(LINES-2)+i]);
	    waddstr(namewin, names[(j+firstvcol)*(LINES-2)+i]);
	    wattrset(namewin, A_NORMAL);
	  }
	wrefresh(namewin);
      }
      break;

    case 'D':
      if (nnames > 0) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, "Delete file ");
	waddstr(msgwin, names[(firstvcol+ccol)*(LINES-2)+crow]);
	waddstr(msgwin, "? (Y/N): ");
	echo();
	i = wgetch(msgwin);
	noecho();
	if (i == 'Y' || i == 'y') {
	  rdrwmsg=1;
	  j = (firstvcol+ccol)*(LINES-2)+crow;
	  if (unlink(names[j])) {
	    wclear(msgwin);
	    mvwaddstr(msgwin, 0, 1, strerror(errno));
	    wrefresh(msgwin);
	    break;
	  }
	  pch = names[j];
	  if (j != nnames-1) {
	    memmove(&names[j], &names[j+1], sizeof(char*)*(nnames-j-1));
	    memmove(&marks[j], &marks[j+1], nnames-j-1);
	  }
	  nnames--;
	  for (i=0; i < cmark; i++)
	    if (marklist[i] > j)
	      marklist[i]--;
	    else if (marklist[i] == j) {
	      if (i != cmark-1) {
		memmove(&marklist[i], &marklist[i+1], sizeof(int)*cmark-i-1);
		i--;
	      }
	      cmark--;
	    }
	  cols=nnames/(LINES-2);
	  if (nnames%(LINES-2))
	    cols++;
	  if (cols > vcols && firstvcol > cols-vcols)
	    firstvcol = cols-vcols;
	  if (firstvcol+ccol > cols-1)
	    ccol = cols-firstvcol-1;
	  if (firstvcol+ccol == cols-1 && crow >= nnames%(LINES-2) &&
	      nnames%(LINES-2))
	    crow = nnames%(LINES-2)-1;
	  wclear(namewin);
	  for (i=0; i < LINES-2; i++)
	    for (j=firstvcol; j-firstvcol < vcols; j++) {
	      if (j*(LINES-2)+i >= nnames)
		break;
	      if (i == crow && j-firstvcol == ccol)
		wattrset(namewin, A_REVERSE);
	      mvwaddch(namewin, i, (j-firstvcol)*colwidth,
		       marks[j*(LINES-2)+i]);
	      waddstr(namewin, names[j*(LINES-2)+i]);
	      wattrset(namewin, A_NORMAL);
	    }
	  wrefresh(namewin);
	  wclear(msgwin);
	  mvwaddstr(msgwin, 0, 1, pch);
	  waddstr(msgwin, " deleted.");
	  wrefresh(msgwin);
	  free(pch);
	} else {
	  wclear(msgwin);
	  mvwaddstr(msgwin, 0, (COLS-74)/2, HELP_MSG);
	  wrefresh(msgwin);
	}
	if (!nnames) {
	  endwin();
	  fprintf(stderr, "%s: No more files\n", progname);
	  return -1;
	}
      }
      break;

    case 'S':
      if (cmark) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, "Enter file name for slide file: ");
	echo();
	wgetstr(msgwin, ifilename);
	noecho();
	if (strlen(ifilename)) {
	  if (!(ofile = fopen(ifilename, "w"))) {
	    wclear(msgwin);
	    mvwaddstr(msgwin, 0, 1, ifilename);
	    waddstr(msgwin, ": ");
	    waddstr(msgwin, strerror(errno));
	    wrefresh(msgwin);
	    rdrwmsg=1;
	    break;
	  }
	  for (i=0; i < cmark; i++) {
	    fputs(names[marklist[i]], ofile);
	    fputc('\n', ofile);
	  }
	  fclose(ofile);
	}
	wclear(msgwin);
	mvwaddstr(msgwin, 0, (COLS-74)/2, HELP_MSG);
	wrefresh(msgwin);
      }
      break;

    case 'x':
      if (!(fpin = fopen(names[(firstvcol+ccol)*(LINES-2)+crow], "r"))) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, strerror(errno));
	wrefresh(msgwin);
	rdrwmsg=1;
	break;
      }
      filetype = GetFileHead(fpin, &image);
      fclose(fpin);
      if (!filetype) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, names[(firstvcol+ccol)*(LINES-2)+crow]);
	waddstr(msgwin, ": Unknown file type");
	wrefresh(msgwin);
	rdrwmsg = 1;
	break;
      }
      sprintf(msg, "%s: %ix%ix%i %s", names[(firstvcol+ccol)*(LINES-2)+crow],
	      image.width, image.height,
	      image.palsize > 0 ? image.palsize : 1<<(-image.palsize),
	      ftypes[filetype].name);
      wclear(msgwin);
      mvwaddstr(msgwin, 0, 1, msg);
      wrefresh(msgwin);
      rdrwmsg = 1;
      break;

    case '\n':
      wclear(msgwin);
      mvwaddstr(msgwin, 0, 1, "Reading file: ");
      waddstr(msgwin, names[(firstvcol+ccol)*(LINES-2)+crow]);
      wrefresh(msgwin);
      if (ReadFile(names[(firstvcol+ccol)*(LINES-2)+crow], &image)) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, errmsg);
	wrefresh(msgwin);
	rdrwmsg=1;
	break;
      }
      SetModeInfo(&image);
      if (image.width > modeinfo->width || image.height > modeinfo->height) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, "Scaling...");
	wrefresh(msgwin);
	if (Scale(&image, &newimage, modeinfo->width, modeinfo->height)) {
	  free(image.bits);
	  wclear(msgwin);
	  mvwaddstr(msgwin, 0, 1, errmsg);
	  wrefresh(msgwin);
	  rdrwmsg=1;
	  break;
	}
	free(image.bits);
	memcpy(&image, &newimage, sizeof(IMAGE));
      }
      if (modeinfo->bytesperpixel == 1 && image.palsize < -8) {
	wclear(msgwin);
	mvwaddstr(msgwin, 0, 1, "Color quantizing...");
	wrefresh(msgwin);
	if (twopass) {
	  if (HeckbertMedianCut(&image, &newimage)) {
	    free(image.bits);
	    wclear(msgwin);
	    mvwaddstr(msgwin, 0, 1, errmsg);
	    wrefresh(msgwin);
	    rdrwmsg=1;
	    break;
	  }
	} else {
	  if (MapToCube(&image, &newimage)) {
	    free(image.bits);
	    wclear(msgwin);
	    mvwaddstr(msgwin, 0, 1, errmsg);
	    wrefresh(msgwin);
	    rdrwmsg=1;
	    break;
	  }
	}
	free(image.bits);
	memcpy(&image, &newimage, sizeof(IMAGE));
      }
      endwin();
      SetVideoMode();
      if (DisplayImage(&image)) {
	VGAShutdown();
	free(image.bits);
	fprintf(stderr, "%s: %s\n", progname, errmsg);
	return -1;
      }
      free(image.bits);
      getch();
      VGAShutdown();
      refresh();
      wclear(msgwin);
      mvwaddstr(msgwin, 0, (COLS-strlen(HELP_MSG))/2, HELP_MSG);
      wrefresh(msgwin);
      wrefresh(namewin);
      wrefresh(titlewin);
      break;

    case 'h':
    case 'H':
      wclear(titlewin);
      mvwaddstr(titlewin, 0, (COLS-18)/2, "File Selector Help");
      wrefresh(titlewin);
      wclear(namewin);
      mvwaddstr(namewin, (LINES-12)/2, (COLS-46)/2, 
		"Arrows   - Move cursor");
      mvwaddstr(namewin, (LINES-12)/2+1, (COLS-46)/2,
		"Enter    - View picture");
      mvwaddstr(namewin, (LINES-12)/2+2, (COLS-46)/2,
		"Spacebar - Toggle Marked/Unmarked");
      mvwaddstr(namewin, (LINES-12)/2+3, (COLS-46)/2,
		"c        - Clear all marks");
      mvwaddstr(namewin, (LINES-12)/2+4, (COLS-46)/2,
		"s        - View marked files in slideshow");
      mvwaddstr(namewin, (LINES-12)/2+5, (COLS-46)/2,
		"S        - Save marked names as slideshow file");
      mvwaddstr(namewin, (LINES-12)/2+6, (COLS-46)/2,
		"ESC      - Stop slideshow (while viewing)");
      mvwaddstr(namewin, (LINES-12)/2+7, (COLS-46)/2,
		"D        - Delete file");
      mvwaddstr(namewin, (LINES-12)/2+8, (COLS-46)/2,
		"x        - Display file statistics");
      mvwaddstr(namewin, (LINES-12)/2+9, (COLS-46)/2,
		"h        - Display this help screen");
      mvwaddstr(namewin, (LINES-12)/2+10, (COLS-46)/2,
		"q        - Quit Vimage");
      wrefresh(namewin);
      wclear(msgwin);
      mvwaddstr(msgwin, 0, (COLS-16)/2, "Press any key...");
      wrefresh(msgwin);
      getch();
      wclear(titlewin);
      mvwaddstr(titlewin, 0, 1, TITLE);
      mvwaddstr(titlewin, 0, COLS-strlen(COPYRIGHT)-1, COPYRIGHT);
      wrefresh(titlewin);
      wclear(namewin);
      for (i=0; i < LINES-2; i++)
	for (j=firstvcol; j-firstvcol < vcols; j++) {
	  if (j*(LINES-2)+i >= nnames)
	    break;
	  if (i == crow && j-firstvcol == ccol)
	    wattrset(namewin, A_REVERSE);
	  mvwaddch(namewin, i, (j-firstvcol)*colwidth,
		   marks[j*(LINES-2)+i]);
	  waddstr(namewin, names[j*(LINES-2)+i]);
	  wattrset(namewin, A_NORMAL);
	}
      wrefresh(namewin);
      wclear(msgwin);
      mvwaddstr(msgwin, 0, (COLS-strlen(HELP_MSG))/2, HELP_MSG);
      wrefresh(msgwin);
      break;

    case '\x1b': /* ESC */
    case 'Q':
    case 'q':
      exit = 1;
      break;
    }
  } while (!exit);
  endwin();
  putchar('\n');
  return 0;
}

#endif /* NCURSES_SUPPORT */
