/*
 * $Id: gif.c,v 1.4 1995/09/16 14:01:57 traister Exp traister $
 * gif.c: function ReadGIF reads a GIF file into an IMAGE struct
 * Copyright (c) 1995, Joseph J. Traister
 * 
 * $Log: gif.c,v $
 * Revision 1.4  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.3  1995/09/03  02:14:54  traister
 * Made screen and image descriptor conflict only produce a warning
 * rather than not viewing.
 *
 * Revision 1.2  1995/07/30  19:20:47  traister
 * Updated copyright prior to first public release
 *
 * Revision 1.1  1995/05/13  01:37:58  traister
 * Initial revision
 *
*/
/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <vga.h>
#include "vimage.h"

/* for decoding the color parameter in descriptors */
#define BITSPPIX	0x07
#define INTERLACE	0x40
#define COLORRES	0x70
#define COLORMAP	0x80
#define BUFSIZE		256*1024

#define MAXCODE		4095

typedef struct screen_desc_tag {
  char magic[6];
  int width, height;
  UCHAR color, background, eosd;
} SCREEN_DESC;

int prefix[MAXCODE+1];
UCHAR suffix[MAXCODE+1];
UCHAR stack[MAXCODE+1];
UCHAR lzwbuf[BUFSIZE];

int bitsleft, maxbytes, curbyte;
UCHAR icolor;

int DecompGIF(FILE*, IMAGE*);
int GetCode(FILE*, int);
UCHAR DecomposeCode(int, int, UCHAR**);

int ReadGIFHead(FILE *fpIn, IMAGE *image)
{
  SCREEN_DESC screen;
  int ileft, itop, i, c;
  
  fread(screen.magic, 1, sizeof(screen.magic), fpIn);
  screen.width = getc(fpIn);
  screen.width |= getc(fpIn)<<8;
  screen.height = getc(fpIn);
  screen.height |= getc(fpIn)<<8;
  screen.color = getc(fpIn);
  screen.background = getc(fpIn);
  screen.eosd = getc(fpIn);
  if (strncmp(screen.magic, "GIF87a", sizeof(screen.magic))) {
    if (strncmp(screen.magic, "GIF89a", sizeof(screen.magic)))
      return 0;
  }
  if (screen.color & COLORMAP) {
    for (i=0; i < (1 << ((screen.color & BITSPPIX)+1))*3; i++)
      image->palette[i] = getc(fpIn);
    image->palsize = 1<<((screen.color & BITSPPIX)+1);
  }
  do {
    while ((c=getc(fpIn)) != ',' && c != '!')
      ;
    if (c == '!') {
      getc(fpIn);
      while ((c=getc(fpIn)))
        fseek(fpIn, c, SEEK_CUR);
    }
  } while (c != ',');
  ileft = getc(fpIn);
  ileft |= getc(fpIn)<<8;
  itop = getc(fpIn);
  itop |= getc(fpIn)<<8;
  image->width = getc(fpIn);
  image->width |= getc(fpIn)<<8;
  image->height = getc(fpIn);
  image->height |= getc(fpIn)<<8;
  icolor = getc(fpIn);
  if (icolor & COLORMAP) {
    for (i=0; i < (1 << ((icolor & BITSPPIX)+1))*3; i++)
      image->palette[i] = getc(fpIn);
    image->palsize = 1<<((icolor & BITSPPIX)+1);
  }
  return 1;
}

int ReadGIF(FILE *fpIn, IMAGE *image)
{
  UCHAR *pch, *tempbuf;
  int i;

  if (!(image->bits = (UCHAR*)malloc(image->height*image->width+4096))) {
    strcpy(errmsg, strerror(errno));
    return -1;
  }
  if (DecompGIF(fpIn, image)) {
    free(image->bits);
    return -1;
  }
  else {
    if (icolor & INTERLACE) {
      if (!(tempbuf = (UCHAR*)malloc(image->width*image->height))) {
	strcpy(errmsg, strerror(errno));
	return -1;
      }
      pch = image->bits;
      for (i=0; i < image->height; i += 8) {
	memcpy(tempbuf+i*image->width, pch, image->width);
	pch += image->width;
      }
      for (i=4; i < image->height; i += 8) {
	memcpy(tempbuf+i*image->width, pch, image->width);
	pch += image->width;
      }
      for (i=2; i < image->height; i += 4) {
	memcpy(tempbuf+i*image->width, pch, image->width);
	pch += image->width;
      }
      for (i=1; i < image->height; i += 2) {
	memcpy(tempbuf+i*image->width, pch, image->width);
	pch += image->width;
      }
      free(image->bits);
      image->bits = tempbuf;
    }
  }
  return 0;
}

int DecompGIF(FILE *fpIn, IMAGE *image)
{
  int startbits, curbits, clearcode, nextcode, new, old;
  UCHAR *out;

  startbits = getc(fpIn);
  clearcode = 1<<startbits;
  curbits = ++startbits;
  nextcode = clearcode+2;
  out = image->bits;
  old = clearcode;
  curbyte = bitsleft = maxbytes = 0;
  while ((unsigned)out-(unsigned)image->bits <
         image->width*image->height) {
    new = GetCode(fpIn, curbits);
    if (new > MAXCODE)
      return -1;
    if (old == clearcode && new != clearcode) {
      *out++ = new;
      old = new;
      continue;
    }
    if (new == clearcode) {
      nextcode = clearcode+2;
      curbits = startbits;
      old = GetCode(fpIn, curbits);
      if (old > MAXCODE)
	return -1;
      *out++ = old;
      continue;
    }
    if (new == clearcode+1)
      break;
    if (new < clearcode) {
      *out++ = new;
      prefix[nextcode] = old;
      suffix[nextcode++] = new;
    } else if (new < nextcode) {
      prefix[nextcode] = old;
      suffix[nextcode++] = DecomposeCode(new, clearcode, &out);
    } else if (new == nextcode && old < clearcode) {
      *out++ = old;
      *out++ = old;
      prefix[nextcode] = old;
      suffix[nextcode++] = old;
    } else if (new == nextcode) { /* old > clearcode */
      prefix[nextcode] = old;
      suffix[nextcode] = DecomposeCode(old, clearcode, &out);
      *out++ = suffix[nextcode++];
    } else {
      strcpy(errmsg, "Bad code in GIF");
      return -1;
    }
    if (nextcode == 1<<curbits && curbits < 12)
      curbits++;
    old = new;
  }
  return 0;
}

int GetCode(FILE *fpIn, int bits)
{
  int c, bitstogo = bits, result = 0, goodbits;
  UCHAR mask[] = { 0x00, 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE, 0xFF };

  if (bits < 3 || bits > 12) {
    strcpy(errmsg, "Bad GIF code width");
    return MAXCODE+1;
  }
  while (bitstogo) {
    if (!bitsleft) {
      bitsleft = 8;
      curbyte++;
      if (curbyte >= maxbytes) {
        curbyte = 0;
        maxbytes = 0;
        while (maxbytes < BUFSIZE-300 && (c = getc(fpIn))) {
          if (fread(lzwbuf+maxbytes, 1, c, fpIn) != c) {
	    strcpy(errmsg, "Unexpected EOF in GIF");
	    return MAXCODE+1;
	  }
          maxbytes += c;
        }
      }
    }
    if (bitsleft <= bitstogo) {
      goodbits = lzwbuf[curbyte] & mask[bitsleft];
      bitstogo -= bitsleft;
      bitsleft = 0;
    } else {
      goodbits = (lzwbuf[curbyte]&mask[bitsleft])&(~mask[bitsleft-bitstogo]);
      bitsleft -= bitstogo;
      bitstogo = 0;
    }
    if (bits-bitstogo > 8-bitsleft)
      result |= goodbits<<((bits-bitstogo)-(8-bitsleft));
    else
      result |= goodbits>>((8-bitsleft)-(bits-bitstogo));
  }
  return result;    
}

UCHAR DecomposeCode(int code, int clearcode, UCHAR **out)
{
  int sp = 0;
  UCHAR startchar;

  while (code > clearcode) {
    stack[sp++] = suffix[code];
    code = prefix[code];
  }
  startchar = code;
  stack[sp++] = code;
  while (sp) {
    **out = stack[--sp];
    (*out)++;
  }
  return startchar;
}
