/*
 * $Id: hmcut.c,v 1.4 1995/09/16 14:01:57 traister Exp traister $
 * hmcut.c: function HeckbertMedianCut performs the cut on IMAGE
 *
 * Copyright (c) 1995, Joe Traister
 *
 * $Log: hmcut.c,v $
 * Revision 1.4  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.3  1995/09/03  02:14:54  traister
 * Fixed bug affecting choice of color for all boxes.
 *
 * Revision 1.2  1995/07/30  19:20:47  traister
 * Updated copyright prior to first public release
 *
 * Revision 1.1  1995/05/13  01:38:16  traister
 * Initial revision
 *
 */
/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "vimage.h"
#include "hash.h"

#define FS_SCALE 1024
#define MAX_COLORS 32767

typedef struct box_tag {
  int index, colors, sum;
} BOX;

int redcompare(const void*, const void*);
int greencompare(const void*, const void*);
int bluecompare(const void*, const void*);
int sumcompare(const void*, const void*);
COLORHIST *histogram(IMAGE*, int, int*);
COLORHIST *histogram32K(IMAGE*, int*);
COLORHISTLIST *hash_image(IMAGE*, int, int*);

int HeckbertMedianCut(IMAGE *image, IMAGE *newimage)
{
  UCHAR *pP;
  COLORHIST *chv;
  COLORHISTLIST *hashtable;
  BOX bv[256];
  int i, *thisrerr=NULL, *nextrerr=NULL, *thisgerr=NULL, *nextgerr=NULL;
  int *thisberr=NULL, *nextberr=NULL;
  int pixwidth, fs_direction=0, row, col, limitcol, sr, sg, sb, *temperr, err;
  int fcolors=0, boxes, colors, index, dist, newdist;
  int minr, maxr, ming, maxg, minb, maxb, lowersum, halfsum;
  int r, g, b, sum, bi;
  float bl, rl, gl;

  memcpy(newimage, image, sizeof(IMAGE));
  /* Build histogram */
  if (image->palsize == -15)
    chv = histogram32K(image, &fcolors);
  else
    chv = histogram(image, MAX_COLORS, &fcolors);
  if (!chv)
    return -1;

  /* Set up first box */
  bv[0].index = 0;
  bv[0].colors = fcolors;
  bv[0].sum = image->width*image->height;
  boxes = 1;
  
  /* Main loop: split boxes until we have enough */
  while (boxes < 256) {
    
    /* find the first splittable box */
    for (bi=0; bi < boxes; bi++)
      if (bv[bi].colors >= 2)
	break;
    if (bi == boxes)
      break; /* no more boxes to split */
    index = bv[bi].index;
    colors = bv[bi].colors;
    sum = bv[bi].sum;
    
    /* find the boundaries of the box */
    minr = maxr = chv[index].r;
    ming = maxg = chv[index].g;
    minb = maxb = chv[index].b;
    for (i=1; i < colors; i++) {
      if (chv[index+i].r < minr) minr = chv[index+i].r;
      if (chv[index+i].r > maxr) maxr = chv[index+i].r;
      if (chv[index+i].g < ming) ming = chv[index+i].g;
      if (chv[index+i].g > maxg) maxg = chv[index+i].g;
      if (chv[index+i].b < minb) minb = chv[index+i].b;
      if (chv[index+i].b > maxb) maxb = chv[index+i].b;
    }
    
    /* sort by largest dimension (by luminosity) */
    rl = 0.299*(maxr-minr);
    gl = 0.587*(maxg-ming);
    bl = 0.114*(maxb-minb);
    if (rl >= gl && rl >= bl)
      qsort((void*)&(chv[index]), colors, sizeof(COLORHIST), redcompare);
    else if (gl >= bl)
      qsort((void*)&(chv[index]), colors, sizeof(COLORHIST), greencompare);
    else
      qsort((void*)&(chv[index]), colors, sizeof(COLORHIST), bluecompare);

    /* find the median based on pixel (not color) counts */
    lowersum = chv[index].count;
    halfsum = sum/2;
    for (i=1; i < colors-1; i++) {
      if (lowersum >= halfsum)
	break;
      lowersum += chv[index+i].count;
    }

    /* split the box and sort by pixel count */
    bv[bi].colors = i;
    bv[bi].sum = lowersum;
    bv[boxes].index = index+i;
    bv[boxes].colors = colors-i;
    bv[boxes].sum = sum-lowersum;
    boxes++;
    qsort((void*)bv, boxes, sizeof(BOX), sumcompare);
  }

  /* choose representative color from box (by average pixel) */
  for (bi = 0; bi < boxes; bi++) {
    r = g = b = sum = 0;
    index = bv[bi].index;
    colors = bv[bi].colors;
    for (i=0; i < colors; i++) {
      r += chv[index+i].r*chv[index+i].count;
      g += chv[index+i].g*chv[index+i].count;
      b += chv[index+i].b*chv[index+i].count;
      sum += chv[index+i].count;
    }
    r /= sum;
    if (r > 255) r = 255;
    g /= sum;
    if (g > 255) g = 255;
    b /= sum;
    if (b > 255) b = 255;
    newimage->palette[bi*3] = r;
    newimage->palette[bi*3+1] = g;
    newimage->palette[bi*3+2] = b;
  }
  free(chv);

  newimage->bits = (UCHAR*)malloc(image->width*image->height);
  if (!newimage->bits) {
    strcpy(errmsg, strerror(errno));
    return -1;
  }
  pixwidth = image->palsize == -15 ? 2 : 3;

  if (floyd) {
    /* Initialize Floyd-Steinberg error vectors */
    thisrerr = (int*)malloc((image->width+2)*sizeof(int));
    if (!thisrerr) {
      strcpy(errmsg, strerror(errno));
      free(newimage->bits);
      return -1;
    }
    nextrerr = (int*)malloc((image->width+2)*sizeof(int));
    if (!nextrerr) {
      strcpy(errmsg, strerror(errno));
      free(newimage->bits);
      free(thisrerr);
      return -1;
    }
    thisgerr = (int*)malloc((image->width+2)*sizeof(int));
    if (!thisgerr) {
      strcpy(errmsg, strerror(errno));
      free(newimage->bits);
      free(thisrerr);
      free(nextrerr);
      return -1;
    }
    nextgerr = (int*)malloc((image->width+2)*sizeof(int));
    if (!nextgerr) {
      strcpy(errmsg, strerror(errno));
      free(newimage->bits);
      free(thisrerr);
      free(nextrerr);
      free(thisgerr);
      return -1;
    }
    thisberr = (int*)malloc((image->width+2)*sizeof(int));
    if (!thisberr) {
      strcpy(errmsg, strerror(errno));
      free(newimage->bits);
      free(thisrerr);
      free(nextrerr);
      free(thisgerr);
      free(nextgerr);
      return -1;
    }
    nextberr = (int*)malloc((image->width+2)*sizeof(int));
    if (!nextberr) {
      strcpy(errmsg, strerror(errno));
      free(newimage->bits);
      free(thisrerr);
      free(nextrerr);
      free(thisgerr);
      free(nextgerr);
      free(thisberr);
      return -1;
    }
    srandom((int)(time(0)^getpid()));
    for (i=0; i<image->width+2; i++) {
      thisrerr[i] = random()%(FS_SCALE*2)-FS_SCALE;
      thisgerr[i] = random()%(FS_SCALE*2)-FS_SCALE;
      thisberr[i] = random()%(FS_SCALE*2)-FS_SCALE;
      /* (random errors in [-1 .. 1]) */
    }
    fs_direction = 1;
  }

  hashtable = (COLORHISTLIST*)malloc(sizeof(COLORHISTLIST)*HASH_SIZE);
  if (!hashtable) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    if (floyd) {
      free(thisrerr);
      free(nextrerr);
      free(thisgerr);
      free(nextgerr);
      free(thisberr);
      free(nextberr);
    }
    return -1;
  }
  for (i=0; i < HASH_SIZE; i++) {
    hashtable[i].count = -1;
    hashtable[i].next = NULL;
  }
  for (row=0; row < image->height; row++) {
    if (floyd) {
      memset(nextrerr, 0, sizeof(int)*(image->width+2));
      memset(nextgerr, 0, sizeof(int)*(image->width+2));
      memset(nextberr, 0, sizeof(int)*(image->width+2));
    }
    if (!floyd || fs_direction) {
      col = 0;
      limitcol = image->width;
      pP = image->bits+row*image->width*pixwidth;
    } else {
      col = image->width-1;
      limitcol = -1;
      pP = image->bits+row*image->width*pixwidth+(image->width-1)*pixwidth;
    }
    do {
      if (image->palsize == -24) {
	sr = *pP;
	sg = *(pP+1);
	sb = *(pP+2);
      } else {
	sr = ((*((short*)pP)&0x7C00)>>7)+7;
	sg = ((*((short*)pP)&0x03E0)>>2)+7;
	sb = ((*((short*)pP)&0x001F)<<3)+7;
      }
      if (floyd) {
	/* Apply Floyd-Steinberg error to pixel */
	sr += thisrerr[col+1]/FS_SCALE;
	sg += thisgerr[col+1]/FS_SCALE;
	sb += thisberr[col+1]/FS_SCALE;
	if (sr < 0) sr = 0;
	if (sr > 255) sr = 255;
	if (sg < 0) sg = 0;
	if (sg > 255) sg = 255;
	if (sb < 0) sb = 0;
	if (sb > 255) sb = 255;
      }

      /* Compute closest match */
      if ((index = lookuphash(sr, sg, sb, hashtable)) < 0) {
	dist = 2000000000;
	for (i=0; i < boxes; i++) {
	  newdist = (sr-newimage->palette[i*3])*(sr-newimage->palette[i*3])+
	    (sg-newimage->palette[i*3+1])*(sg-newimage->palette[i*3+1])+
	      (sb-newimage->palette[i*3+2])*(sb-newimage->palette[i*3+2]);
	  if (newdist < dist) {
	    index = i;
	    dist = newdist;
	  }
	}
	if (addtohash(sr, sg, sb, index, hashtable)) {
	  strcpy(errmsg, strerror(errno));
	  free(newimage->bits);
	  freehash(hashtable);
	  if (floyd) {
	    free(thisrerr);
	    free(nextrerr);
	    free(thisgerr);
	    free(nextgerr);
	    free(thisberr);
	    free(nextberr);
	  }
	  return -1;
	}
      }
      *(newimage->bits+row*image->width+col) = index;
      
      /* Propogate Floyd-Steinberg errors */
      if (floyd && fs_direction) {
	err = (sr-newimage->palette[index*3])*FS_SCALE;
	thisrerr[col+2] += (err*7)>>4;
	nextrerr[col] += (err*3)>>4;
	nextrerr[col+1] += (err*5)>>4;
	nextrerr[col+2] += err>>4;
	err = (sg-newimage->palette[index*3+1])*FS_SCALE;
	thisgerr[col+2] += (err*7)>>4;
	nextgerr[col] += (err*3)>>4;
	nextgerr[col+1] += (err*5)>>4;
	nextgerr[col+2] += err>>4;
	err = (sb-newimage->palette[index*3+2])*FS_SCALE;
	thisberr[col+2] += (err*7)>>4;
	nextberr[col] += (err*3)>>4;
	nextberr[col+1] += (err*5)>>4;
	nextberr[col+2] += err>>4;
      } else if (floyd) {
	err = (sr-newimage->palette[index*3])*FS_SCALE;
	thisrerr[col] += (err*7)>>4;
	nextrerr[col+2] += (err*3)>>4;
	nextrerr[col+1] += (err*5)>>4;
	nextrerr[col] += err>>4;
	err = (sg-newimage->palette[index*3+1])*FS_SCALE;
	thisgerr[col] += (err*7)>>4;
	nextgerr[col+2] += (err*3)>>4;
	nextgerr[col+1] += (err*5)>>4;
	nextgerr[col] += err>>4;
	err = (sb-newimage->palette[index*3+2])*FS_SCALE;
	thisberr[col] += (err*7)>>4;
	nextberr[col+2] += (err*3)>>4;
	nextberr[col+1] += (err*5)>>4;
	nextberr[col] += err>>4;
      }
      if (!floyd || fs_direction) {
	col++;
	pP += pixwidth;
      } else {
	col--;
	pP -= pixwidth;
      }
    } while (col != limitcol);
    if (floyd) {
      temperr = thisrerr;
      thisrerr = nextrerr;
      nextrerr = temperr;
      temperr = thisgerr;
      thisgerr = nextgerr;
      nextgerr = temperr;
      temperr = thisberr;
      thisberr = nextberr;
      nextberr = temperr;
      fs_direction = ! fs_direction;
    }
  }
  if (floyd) {
    free(thisrerr);
    free(nextrerr);
    free(thisgerr);
    free(nextgerr);
    free(thisberr);
    free(nextberr);
  }
  freehash(hashtable);

  newimage->palsize = boxes;
  return 0;
}

int redcompare(const void *ch1, const void *ch2)
{
  return (int)((COLORHIST*)ch1)->r-((COLORHIST*)ch2)->r;
}

int greencompare(const void *ch1, const void *ch2)
{
  return (int)((COLORHIST*)ch1)->g-((COLORHIST*)ch2)->g;
}

int bluecompare(const void *ch1, const void *ch2)
{
  return (int)((COLORHIST*)ch1)->b-((COLORHIST*)ch2)->b;
}

int sumcompare(const void *b1, const void *b2)
{
  return ((BOX*)b2)->sum-((BOX*)b1)->sum;
}

COLORHIST *histogram32K(IMAGE *image, int *fcolors)
{
  COLORHIST *chv;
  int *counts, i;
  UCHAR *pP;

  *fcolors = 0;
  counts = (int*)malloc(sizeof(int)*MAX_COLORS);
  if (!counts) {
    strcpy(errmsg, strerror(errno));
    return NULL;
  }
  memset(counts, 0, sizeof(int)*MAX_COLORS);
  for (pP=image->bits; pP-image->bits < image->width*image->height*2; pP+=2) {
    if (!counts[*((short*)pP)])
      *fcolors += 1;
    counts[*((short*)pP)]++;
  }
  chv = (COLORHIST*)malloc(sizeof(COLORHIST)*(*fcolors));
  if (!chv) {
    strcpy(errmsg, strerror(errno));
    free(counts);
    return NULL;
  }
  *fcolors = 0;
  for (i=0; i < MAX_COLORS; i++)
    if (counts[i]) {
      chv[*fcolors].count = counts[i];
      chv[*fcolors].r = (i&0x7c00)>>7;
      chv[*fcolors].g = (i&0x3e0)>>2;
      chv[*fcolors].b = (i&0x1f)<<3;
      *fcolors += 1;
    }
  free(counts);
  return chv;
}

COLORHIST *histogram(IMAGE *image, int maxcolors, int *fcolors)
{
  COLORHISTLIST *cht, *chl;
  COLORHIST *chv;
  int i, c=0;
  
  cht = hash_image(image, maxcolors, fcolors);
  if (!(chv = (COLORHIST*)malloc(sizeof(COLORHIST)*(*fcolors)))) {
    strcpy(errmsg, strerror(errno));
    return NULL;
  }
  for (i=0; i < HASH_SIZE; i++)
    for (chl=&cht[i]; chl; chl = chl->next)
      if (chl->count) {
	chv[c].count = chl->count;
	chv[c].r = chl->r;
	chv[c].g = chl->g;
	chv[c].b = chl->b;
	c++;
      }
  freehash(cht);
  return chv;
}

COLORHISTLIST *hash_image(IMAGE *image, int maxcolors, int *fcolors)
{
  COLORHISTLIST *hashtable, *chl;
  int i, clearbits = 0;
  UCHAR *pP;

  do {
    hashtable = (COLORHISTLIST*)malloc(sizeof(COLORHISTLIST)*HASH_SIZE);
    if (!hashtable) {
      strcpy(errmsg, strerror(errno));
      return NULL;
    }
    for (i=0; i < HASH_SIZE; i++) {
      hashtable[i].count = 0;
      hashtable[i].next = NULL;
    }
    *fcolors = 0;
    for (pP=image->bits;
	 pP-image->bits < image->width*image->height*3; 
	 pP += 3) {
      i = hash(*pP|clearbits, *(pP+1)|clearbits, *(pP+2)|clearbits);
      if (hashtable[i].count) {
	for (chl = &hashtable[i]; chl; chl = chl->next)
	  if (chl->r == *pP|clearbits &&
	      chl->g == *(pP+1)|clearbits &&
	      chl->b == *(pP+2)|clearbits)
	    break;
	if (chl)
	  chl->count++;
	else {
	  chl = (COLORHISTLIST*)malloc(sizeof(COLORHISTLIST));
	  if (!chl) {
	    strcpy(errmsg, strerror(errno));
	    free(hashtable);
	    return NULL;
	  }
	  chl->count = 1;
	  chl->r = *pP|clearbits;
	  chl->g = *(pP+1)|clearbits;
	  chl->b = *(pP+2)|clearbits;
	  chl->next = hashtable[i].next;
	  hashtable[i].next = chl;
	  *fcolors += 1;
	}
      } else {
	hashtable[i].count = 1;
	hashtable[i].r = *pP|clearbits;
	hashtable[i].g = *(pP+1)|clearbits;
	hashtable[i].b = *(pP+2)|clearbits;
	*fcolors += 1;
      }
      if (*fcolors >= MAX_COLORS) {
	clearbits <<= 1;
	clearbits |= 1;
	freehash(hashtable);
	break;
      }
    }
  } while (*fcolors >= MAX_COLORS);
  return hashtable;
}
