/*
 * $Id: scale.c,v 1.3 1995/09/16 14:01:57 traister Exp traister $
 * scale.c: IMAGE scaling routines
 *
 * Copyright (c) 1995, Joe Traister
 *
 * $Log: scale.c,v $
 * Revision 1.3  1995/09/16  14:01:57  traister
 * Rewrote error reporting throughout.
 * Added file type table to facilitate adding new image types.
 *
 * Revision 1.2  1995/09/03  02:14:54  traister
 * Added aspect ratio correction for 320x200 modes.
 *
 * Revision 1.1  1995/08/17  03:54:29  traister
 * Initial revision
 *
 *
 */
/*   This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <string.h>
#include <stdlib.h>
#include "vimage.h"
#include "hash.h"

#define SCALE 4096
#define HALFSCALE 2048

int Scale(IMAGE *image, IMAGE *newimage, int xsize, int ysize)
{
  int *rs, *gs, *bs, i, j, k, rowleft, rowtofill, newrow, currow=0;
  int yscale, xscale, col, coltofill, colleft, r, g, b, newcol, x, y;
  int dist, newdist, index;
  UCHAR *temprow, *tempbits=NULL;
  COLORHISTLIST *hashtable;

  memcpy(newimage, image, sizeof(IMAGE));
  if (modeinfo->width == 320 && modeinfo->height == 200) {
    if (((float)xsize*0.82)/(float)image->width >
	(float)ysize/(float)image->height) {
      yscale = SCALE*((float)ysize/(float)image->height);
      xscale = 1.22*yscale;
    } else {
      xscale = SCALE*((float)xsize/(float)image->width);
      yscale = 0.82*xscale;
    }
  } else {
    if ((float)xsize/(float)image->width < (float)ysize/(float)image->height)
      yscale = xscale = SCALE*((float)xsize/(float)image->width);
    else
      xscale = yscale = SCALE*((float)ysize/(float)image->height);
  }
  newimage->width = ((float)image->width*(float)xscale)/(float)SCALE+0.999;
  newimage->height = ((float)image->height*(float)yscale)/(float)SCALE+0.999;
  if (image->palsize == -1) {
    newimage->palsize = -8;
    if (!(tempbits = (UCHAR*)malloc(image->width*image->height))) {
      strcpy(errmsg, strerror(errno));
      return -1;
    }
    for (y=0; y < image->height; y++)
      for (x=0; x < image->width; x++) {
	i = (image->width>>3)+(image->width&0x07 ? 1 : 0);
	if (image->bits[y*i+(x>>3)]&1<<(7-(x&0x07)))
	  tempbits[y*image->width+x] = 0;
	else
	  tempbits[y*image->width+x] = 255;
      }
    temprow = tempbits;
    tempbits = image->bits;
    image->bits = temprow;
  }
  if (image->palsize > 0 || image->palsize == -15)
    newimage->palsize = -24;
  if (!(newimage->bits = (UCHAR*)malloc(newimage->height*newimage->width*
					(newimage->palsize/-8)))) {
    strcpy(errmsg, strerror(errno));
    if (tempbits) {
      free(image->bits);
      image->bits = tempbits;
    }
    return -1;
  }
  if (!(rs = (int*)malloc(image->width*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(newimage->bits);
    if (tempbits) {
      free(image->bits);
      image->bits = tempbits;
    }
    return -1;
  }
  if (!(gs = (int*)malloc(image->width*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(rs);
    free(newimage->bits);
    if (tempbits) {
      free(image->bits);
      image->bits = tempbits;
    }
    return -1;
  }
  if (!(bs = (int*)malloc(image->width*sizeof(int)))) {
    strcpy(errmsg, strerror(errno));
    free(gs);
    free(rs);
    free(newimage->bits);
    if (tempbits) {
      free(image->bits);
      image->bits = tempbits;
    }
    return -1;
  }
  for (i=0; i < image->width; i++)
    rs[i] = bs[i] = gs[i] = HALFSCALE;
  if (!(temprow = (UCHAR*)malloc(image->width*(newimage->palsize/-8)))) {
    strcpy(errmsg, strerror(errno));
    free(bs);
    free(gs);
    free(rs);
    free(newimage->bits);
    if (tempbits) {
      free(image->bits);
      image->bits = tempbits;
    }
    return -1;
  }
  rowleft = yscale;
  rowtofill = SCALE;
  for (newrow=0; newrow < newimage->height; newrow++) {
    while (rowleft < rowtofill) {
      for (col=0; col < image->width; col++) {
	switch (image->palsize) {
	case -24:
	  rs[col] += rowleft*image->bits[(currow*image->width+col)*3];
	  gs[col] += rowleft*image->bits[(currow*image->width+col)*3+1];
	  bs[col] += rowleft*image->bits[(currow*image->width+col)*3+2];
	  break;

	case -15:
	  rs[col] += rowleft*
	    (((((short*)image->bits)[currow*image->width+col]&0x7c00)>>7)|7);
	  gs[col] += rowleft*
	    (((((short*)image->bits)[currow*image->width+col]&0x03e0)>>2)|7);
	  bs[col] += rowleft*
	    (((((short*)image->bits)[currow*image->width+col]&0x1f)<<3)|7);
	  break;

	case -1:
	case -8:
	  gs[col] += rowleft*image->bits[currow*image->width+col];
	  break;

	default:
	  rs[col] += rowleft*
	    image->palette[image->bits[currow*image->width+col]*3];
	  gs[col] += rowleft*
	    image->palette[image->bits[currow*image->width+col]*3+1];
	  bs[col] += rowleft*
	    image->palette[image->bits[currow*image->width+col]*3+2];
	  break;
	}
      }
      rowtofill -= rowleft;
      rowleft = yscale;
      if (currow < image->height-1)
	currow++;
    }
    if (rowtofill) {
      for (col=0; col < image->width; col++) {
	switch (image->palsize) {
	case -24:
	  rs[col] += rowtofill*image->bits[(currow*image->width+col)*3];
	  gs[col] += rowtofill*image->bits[(currow*image->width+col)*3+1];
	  bs[col] += rowtofill*image->bits[(currow*image->width+col)*3+2];
	  break;

	case -15:
	  rs[col] += rowtofill*
	    (((((short*)image->bits)[currow*image->width+col]&0x7c00)>>7)|7);
	  gs[col] += rowtofill*
	    (((((short*)image->bits)[currow*image->width+col]&0x03e0)>>2)|7);
	  bs[col] += rowtofill*
	    (((((short*)image->bits)[currow*image->width+col]&0x1f)<<3)|7);
	  break;

	case -1:
	case -8:
	  gs[col] += rowtofill*image->bits[currow*image->width+col];
	  break;

	default:
	  rs[col] += rowtofill*
	    image->palette[image->bits[currow*image->width+col]*3];
	  gs[col] += rowtofill*
	    image->palette[image->bits[currow*image->width+col]*3+1];
	  bs[col] += rowtofill*
	    image->palette[image->bits[currow*image->width+col]*3+2];
	  break;
	}
      }
    }
    for (col=0; col < image->width; col++) {
      switch (newimage->palsize) {
      case -24:
	temprow[col*3] = rs[col]/SCALE;
	temprow[col*3+1] = gs[col]/SCALE;
	temprow[col*3+2] = bs[col]/SCALE;
	rs[col] = gs[col] = bs[col] = HALFSCALE;
	break;

      case -8:
	temprow[col] = gs[col]/SCALE;
	gs[col] = HALFSCALE;
	break;
      }
    }
    rowleft -= rowtofill;
    if (!rowleft) {
      if (currow < image->height-1)
	currow++;
      rowleft = yscale;
    }
    rowtofill = SCALE;
    coltofill = SCALE;
    colleft = xscale;
    col = 0;
    for (newcol=0; newcol < newimage->width; newcol++) {
      r = g = b = HALFSCALE;
      while (colleft < coltofill) {
	switch (newimage->palsize) {
	case -24:
	  r += colleft*temprow[col*3];
	  g += colleft*temprow[col*3+1];
	  b += colleft*temprow[col*3+2];
	  break;

	case -8:
	  g += colleft*temprow[col];
	  break;
	}
	coltofill -= colleft;
	colleft = xscale;
	if (col < image->width-1)
	  col++;
      }
      if (coltofill)
	switch (newimage->palsize) {
	case -24:
	  r += coltofill*temprow[col*3];
	  g += coltofill*temprow[col*3+1];
	  b += coltofill*temprow[col*3+2];
	  break;

	case -8:
	  g += coltofill*temprow[col];
	  break;
	}
      switch (newimage->palsize) {
      case -24:
	newimage->bits[(newrow*newimage->width+newcol)*3] = r/SCALE;
	newimage->bits[(newrow*newimage->width+newcol)*3+1] = g/SCALE;
	newimage->bits[(newrow*newimage->width+newcol)*3+2] = b/SCALE;
	break;

      case -8:
	newimage->bits[newrow*newimage->width+newcol] = g/SCALE;
	break;
      }
      colleft -= coltofill;
      if (!colleft) {
	if (col < image->width-1)
	  col++;
	colleft = xscale;
      }
      coltofill = SCALE;
    }
  }
  free(rs);
  free(gs);
  free(bs);
  free(temprow);
  if (image->palsize == -1) {
    free(image->bits);
    image->bits = tempbits;
  }
  if (image->palsize > 0) {
    if (!(tempbits = (UCHAR*)malloc(newimage->width*newimage->height))) {
      strcpy(errmsg, strerror(errno));
      return -1;
    }
    if (!(hashtable =
	  (COLORHISTLIST*)malloc(sizeof(COLORHISTLIST)*HASH_SIZE))) {
      strcpy(errmsg, strerror(errno));
      return -1;
    }
    for (i=0; i < HASH_SIZE; i++) {
      hashtable[i].count = -1;
      hashtable[i].next = NULL;
    }
    for (i=0; i < newimage->height; i++) {
      for (j=0; j < newimage->width; j++) {
	r = newimage->bits[(i*newimage->width+j)*3];
	g = newimage->bits[(i*newimage->width+j)*3+1];
	b = newimage->bits[(i*newimage->width+j)*3+2];
	if ((index = lookuphash(r, g, b, hashtable)) < 0) {
	  dist = 2000000000;
	  for (k=0; k < image->palsize; k++) {
	    newdist = (r-image->palette[k*3])*(r-image->palette[k*3])+
	      (g-image->palette[k*3+1])*(g-image->palette[k*3+1])+
		(b-image->palette[k*3+2])*(b-image->palette[k*3+2]);
	    if (newdist < dist) {
	      index = k;
	      dist = newdist;
	    }
	  }
	  if (addtohash(r, g, b, index, hashtable)) {
	    strcpy(errmsg, strerror(errno));
	    free(tempbits);
	    free(newimage->bits);
	    freehash(hashtable);
	    return -1;
	  }
	}
	*(tempbits+i*newimage->width+j) = index;
      }
    }
    freehash(hashtable);
    free(newimage->bits);
    newimage->bits = tempbits;
    newimage->palsize = image->palsize;
    memcpy(newimage->palette, image->palette, (image->palsize*3)*sizeof(int));
  }
  return 0;
}



