/*
gdpc - a program for visualising molecular dynamic simulations
Copyright (C) 1999 Jonas Frantz

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <time.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <gdk_imlib.h>
#include "parameters.h"
#include <X11/Xlib.h>

/* Some internal variables declared next */
GtkWidget	*window;
GtkWidget	*drawing_area;
GtkWidget	*time_entry;
GtkWidget	*coord_entry;
GtkWidget	*maxx_entry,*maxy_entry,*maxz_entry,*xc_entry,*yc_entry,*zc_entry;
GdkPixmap	*pixmap;
GdkGC 		*gc;
GdkColor 	*colors;
Display		*disp;

FILE		*fp;

/************************************************************/
/* This function is called when the quit button is pressed. */
/************************************************************/
void quit (GtkWidget *widget, gpointer data)
{
    gtk_exit (0);
}


/*************************************************************/
/* This function is called when the pause button is pressed. */
/*************************************************************/
void pauseb (GtkWidget *widget,struct GlobalParams *params)
{

    if (params->pausecheck==TRUE) params->pausecheck=FALSE;
    else params->pausecheck=TRUE;
}


/*****************************************************************/
/* This function is called when the restart button is pressed.   */
/* It simply resets the filepointer and clears the drawingboard. */
/*****************************************************************/
void restart (GtkWidget *widget, struct GlobalParams *params)
{

    fseek(fp,SEEK_SET,0);
    params->atEnd=FALSE;
    cleardrawable(widget,pixmap,colors,params);
    params->numframe=1;
}


/**************************************************************/
/* This function is called when the setup button is pressed.  */
/* It stops the animation and calls the setupwindow function. */
/**************************************************************/
void setup (GtkWidget *widget, struct GlobalParams *params)
{
    params->setupstop=TRUE;
    params->oldxc=params->xcolumn;
    params->oldyc=params->ycolumn;
    params->oldzc=params->zcolumn;
    params->oldtc=params->tcolumn;
    params->oldxsize=params->absxsize;
    params->oldysize=params->absysize;
    setupwindow(params);
}


/******************************************************************/
/* This function is called when the window is moved, resized etc. */
/* It redraws the drawingboard.                                   */
/******************************************************************/
static gint expose_event (GtkWidget *widget, GdkEventExpose *event, struct GlobalParams *params)
{
  gdk_draw_pixmap(widget->window,widget->style->white_gc,pixmap,
          event->area.x, event->area.y,
          event->area.x, event->area.y,
          event->area.width, event->area.height);

  return FALSE;
}


/******************************************************************/
/* This function is called when the program is started.           */
/*   It creates a pixmap and clears it and puts it on the screen. */
/******************************************************************/
gint configure_event (GtkWidget *widget, GdkEventConfigure *event, struct GlobalParams *params)
{
    if (pixmap) gdk_pixmap_unref (pixmap);

    pixmap = gdk_pixmap_new (widget->window, widget->allocation.width,
			     widget->allocation.height, -1);

    if (!params->redrawcheck) {
	cleardrawable(widget,pixmap,colors,params);
    }
    else {
	cleardrawable(widget,pixmap,colors,params);
	rotateatoms(pixmap,colors,params);
	params->redrawcheck=FALSE;
    }
    params->rotated=TRUE;
    return TRUE;
}


/***********************************************************************/
/* This procedure is called when a mousebutton is pressed.             */
/* If the right button is pressed it quits, to be backwards compatible */
/* with dpc. When the left button is pressed this procedure saves the  */
/* position of the cursor.                                             */
/***********************************************************************/
gint button_press_event(GtkWidget *widget, GdkEventButton *event, struct GlobalParams *params)
{
    if (event->button == 1) {
	params->pressed=TRUE;
	params->xpress=event->x;
	params->ypress=event->y;
    } 
    else if (event->button == 2) {
    } 
    else if (event->button == 3) {
	gtk_exit(0);
    }
    return TRUE;
}


/***********************************************************************/
/* This procedure is called when a mousebutton is released.            */
/* When released the procedure ends the tracking of mouse movement and */
/* rotates the atoms to their final position.                          */
/***********************************************************************/
gint button_release_event(GtkWidget *widget, GdkEventButton *event, struct GlobalParams *params)
{
    if (event->button == 1) {
	params->pressed=FALSE;
	mouserotate(widget,(params->xpress-event->x),(params->ypress-event->y),params);
    } 
    else if (event->button == 2) {
    } 
    else if (event->button == 3) {
	gtk_exit(0);
    }
    return TRUE;
}


/**************************************************************************/
/* This function is called when the mouse is moved inside the window,     */
/* it sets the coordinates in the coordinateentry.                        */  
/* It also rotates the atoms according to the movement of the mouse while */
/* the left button is pressed down.					  */
/**************************************************************************/
gint motion_notify_event (GtkWidget *widget, GdkEventMotion *event, struct GlobalParams *params)
{
gint x,y;
char xstr[25];
GdkModifierType state;

    gdk_window_get_pointer (event->window, &x, &y, &state);

    if (params->pressed) {
	if (params->xpress-x > 20 || params->ypress-y > 20 || params->xpress-x < -20 || params->ypress-y < -20) {
	    mouserotate(widget,(params->xpress-x),(params->ypress-y),params);
	    params->xpress=x;
	    params->ypress=y;
	}
    }

        sprintf(xstr, "X: %5.3f   Y: %5.3f",(((params->xmax2-params->xmin2)*(x-xborder)/(float) params->absxsize)+params->xmin2), (((params->ymax2-params->ymin2)*(params->absysize-(y-yborder))/(float) params->absysize))+params->ymin2);
        gtk_entry_set_text ((GtkEntry *) coord_entry, xstr);
      
    return TRUE;
}


/*************************************************************/
/* This function is called at the end of setupwindow if the  */
/* ok button was pressed, it reinitializes gdpc if necessary */
/* and then continous the animation.                         */
/*************************************************************/
void SetupStartOk(struct GlobalParams *params) 
{
    if (params->absxsize!=params->oldxsize || params->absysize!=params->oldysize) {
	gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area),params->absxsize+2*xborder,params->absysize+2*yborder);
	params->redrawcheck=TRUE;
	gtk_widget_set_usize(drawing_area,params->absxsize+2*xborder,params->absysize+2*yborder);
	XSync(disp,FALSE);
    }

    params->ncolors = allocatecolors (&colors,params->colorset,params->mode) - 2;
    if (params->xcolumn!=params->oldxc || params->ycolumn!=params->oldyc || params->zcolumn!=params->oldzc || params->tcolumn!=params->oldtc ) {
	fseek(fp,SEEK_SET,0);
	params->atEnd=FALSE;
	cleardrawable(window,pixmap,colors,params);
    }
    if (strlen(params->file)>0) {
	fclose(fp);
	fp=fopen(params->file,"r"); 
	    if(fp==NULL) {
		printf("Error opening file: %s\n",params->file);
		gtk_exit (0);
	    }
        fseek(fp,0,0);
	params->numframe=1;
	params->atEnd=FALSE;
	cleardrawable(window,pixmap,colors,params);
    }
    else {
	rotateatoms(pixmap,colors,params);
	params->rotated=TRUE;
    }
    params->setupstop=FALSE;
}


/******************************************************************/
/* This function is called at the end of setwindow if the         */
/*  cancel button was pressed. It simply continous the animation. */
/******************************************************************/
void SetupStartCancel(struct GlobalParams *params) 
{
    params->setupstop=FALSE;
}


/***********************************************************************/
/* This function is called when there has been a change in angle from  */
/* pressing a button.                                                  */
/***********************************************************************/
void drawrotate(GtkWidget *widget, struct GlobalParams *params) 
{
    cleardrawable(widget,pixmap,colors,params);
    rotateatoms(pixmap,colors,params);
    params->rotated=TRUE;
}


/*************************************************************************/
/* This function is the callback for the timeout instruction.            */
/* It checks if a frame is being drawed or pause is pressed or if the    */
/* animation is at the end, else it calls drawatoms which starts drawing */
/* the next frame. When drawatoms is done, if updates the time in the    */
/* timeentry and puts the pixmap onto the screen.                        */
/*************************************************************************/ 
gint drawnext(struct GlobalParams *params) 
{
float atime;
char tstr[20];
GdkImlibImage *ilimage;
gint tmp;
char picname[60];
static gint numinterval=0;

    numinterval++;
    if (numinterval>=(params->interval/MININTERVAL)) {

	if(params->drawcheck && !params->pausecheck && !params->atEnd && !params->setupstop) {
	    params->drawcheck=FALSE;

	    if(!drawatoms(pixmap,colors,drawing_area,fp,&atime,params)) params->atEnd=TRUE;

	    if (!params->atEnd || !params->fxyz) {
		sprintf(tstr,"Time: %5.3f fs",atime);
		gtk_entry_set_text ((GtkEntry *) time_entry, tstr);

		sprintf(tstr,"X: %4.3f - %4.3f",params->xmin2,params->xmax2);
		gtk_entry_set_text((GtkEntry *) maxx_entry, tstr);
		sprintf(tstr,"Y: %4.3f - %4.3f",params->ymin2,params->ymax2);
		gtk_entry_set_text((GtkEntry *) maxy_entry, tstr);
		sprintf(tstr,"Z: %4.3f - %4.3f",params->zmin2,params->zmax2);
		gtk_entry_set_text((GtkEntry *) maxz_entry, tstr); 
		gdk_draw_pixmap(drawing_area->window,drawing_area->style->white_gc,pixmap,0,0,0,0,params->absxsize+2*xborder,params->absysize+2*yborder);
		if (params->dumpname[0]!=(char) NULL) {
		    ilimage=gdk_imlib_create_image_from_drawable(pixmap,NULL,0,0,params->absxsize+2*xborder,params->absysize+2*yborder);
		    if (params->tifjpg) {
			if (params->dumpnum) sprintf(picname,"%s-%d.tif",params->dumpname,params->numframe);
			else sprintf(picname,"%s-%5.3f.tif",params->dumpname,atime);
		    }
		    else {
			if (params->dumpnum) sprintf(picname,"%s-%d.jpg",params->dumpname,params->numframe);
			else sprintf(picname,"%s-%5.3f.jpg",params->dumpname,atime);
		    }
		    tmp=gdk_imlib_save_image(ilimage,picname,NULL);
		}
		params->numframe++;
	    }
	params->drawcheck=TRUE;
	}
    numinterval=0;
    }

    if (params->rotated) {
	gdk_draw_pixmap(drawing_area->window,drawing_area->style->white_gc,pixmap,0,0,0,0,params->absxsize+2*xborder,params->absysize+2*yborder);
	params->rotated=FALSE;
    }

    sprintf(tstr,"X angle: %f",params->xc);
    gtk_entry_set_text((GtkEntry *) xc_entry, tstr); 
    sprintf(tstr,"Y angle: %f",params->yc);
    gtk_entry_set_text((GtkEntry *) yc_entry, tstr); 
    sprintf(tstr,"Z angle: %f",params->zc);
    gtk_entry_set_text((GtkEntry *) zc_entry, tstr); 

return TRUE;
}


/**************************************************************************/
/* This function is called when the redraw button is pressed in the setup */
/* window. It just redraws the frame.                                     */
/**************************************************************************/
void SetupRedraw(struct GlobalParams *params) 
{
    if (params->absxsize!=params->oldxsize || params->absysize!=params->oldysize) {
	gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area),params->absxsize+2*xborder,params->absysize+2*yborder);
	params->redrawcheck=TRUE;
	gtk_widget_set_usize(drawing_area,params->absxsize+2*xborder,params->absysize+2*yborder);
	cleardrawable(window,pixmap,colors,params);
    }

    if (params->erase) cleardrawable(window,pixmap,colors,params);
    params->ncolors = allocatecolors (&colors,params->colorset,params->mode) - 2;
    rotateatoms(pixmap,colors,params);
    params->rotated=TRUE;
    params->oldxc=params->xcolumn;
    params->oldyc=params->ycolumn;
    params->oldzc=params->zcolumn;
    params->oldtc=params->tcolumn;
    params->oldxsize=params->absxsize;
    params->oldysize=params->absysize;
}


/************************************************************************************/
/* The StartEverything function is called by main() after the commandline arguments */
/* or the setupwindow has finished processing of the parameters. This function      */
/* sets up all the buttons, entries, boxes,the timeout and the drawingboard.        */
/************************************************************************************/ 
void StartEverything(struct GlobalParams *params) 
{
GtkWidget	*vbox,*hbox,*vboxleft,*vboxmiddle,*vboxright,*vboxrb,*vboxcoord,*hboxx,*hboxy,*hboxz,*hboxsetup;
GtkWidget	*quit_button,*restart_button,*pause_button,*reseto_button,*setup_button;
GtkWidget	*xplus_button,*yplus_button,*zplus_button,*xminus_button,*yminus_button,*zminus_button;
GtkWidget	*xplus10_button,*yplus10_button,*zplus10_button,*xminus10_button,*yminus10_button,*zminus10_button;
GtkWidget	*xlabel,*ylabel,*zlabel;
char		buf[100];

    params->StartedAlready=TRUE;

/* Initialize imlib. */
    gdk_imlib_init();

/* Set the window title. */
    sprintf(buf,"gdpc "GDPCVER" : %s",params->file);

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), buf);
    gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (window), 5);

/* Create boxes for outlay. */
    vbox = gtk_vbox_new (FALSE, 0);
    hbox = gtk_hbox_new (FALSE, 5);
    vboxleft = gtk_vbox_new (FALSE, 3);
    vboxmiddle = gtk_vbox_new (FALSE, 3);
    vboxright = gtk_vbox_new (FALSE, 3);
    hboxsetup = gtk_hbox_new (FALSE, 3);
    vboxrb = gtk_vbox_new (FALSE, 3);
    vboxcoord = gtk_vbox_new (FALSE, 3);
    hboxx = gtk_hbox_new (FALSE, 3);
    hboxy = gtk_hbox_new (FALSE, 3);
    hboxz = gtk_hbox_new (FALSE, 3);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    xlabel = gtk_label_new(" X ");
    ylabel = gtk_label_new(" Y ");
    zlabel = gtk_label_new(" Z ");

/* Create the drawing area. */

    drawing_area = gtk_drawing_area_new ();
    gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area),params->absxsize+2*xborder,params->absysize+2*yborder);
    gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);

/* Connect the events to their procedures. */
    gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
              (GtkSignalFunc) expose_event, params);

    gtk_signal_connect (GTK_OBJECT (drawing_area), "configure_event",
              (GtkSignalFunc) configure_event, params);

    gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
              (GtkSignalFunc) button_press_event, params);
    gtk_signal_connect (GTK_OBJECT (drawing_area), "button_release_event",
              (GtkSignalFunc) button_release_event, params);

    gtk_signal_connect (GTK_OBJECT (drawing_area), "motion_notify_event",
              (GtkSignalFunc) motion_notify_event, params);

    gtk_widget_set_events (drawing_area, 
                              GDK_EXPOSURE_MASK 
                            | GDK_BUTTON_PRESS_MASK               
                            | GDK_BUTTON_RELEASE_MASK               
                            | GDK_POINTER_MOTION_MASK
                            | GDK_POINTER_MOTION_HINT_MASK);

    
/* Create entries for the  x- and y- coordinates of the cursor and the time. */
    maxx_entry  = gtk_entry_new();
    maxy_entry  = gtk_entry_new();
    maxz_entry  = gtk_entry_new();
    xc_entry    = gtk_entry_new();
    yc_entry    = gtk_entry_new();
    zc_entry    = gtk_entry_new();
    time_entry  = gtk_entry_new();
    coord_entry = gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (vboxcoord), xc_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcoord), yc_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxcoord), zc_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxleft), maxx_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxmiddle), maxy_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxright), maxz_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxleft), time_entry, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxmiddle), coord_entry, FALSE, FALSE, 0);


/* Create buttons and connect them to their procedures. */
/* Create reset orientation button */
    reseto_button = gtk_button_new_with_label ("Reset orientation");
    gtk_box_pack_start (GTK_BOX (vboxright), reseto_button, TRUE, TRUE, 0);
    gtk_signal_connect_object (GTK_OBJECT (reseto_button), "clicked",  
                    GTK_SIGNAL_FUNC (resetob),GTK_OBJECT (window)); 

/* Create restart button. */
    restart_button = gtk_button_new_with_label ("Restart");
    gtk_box_pack_start (GTK_BOX (hboxsetup), restart_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (restart_button), "clicked",
                    GTK_SIGNAL_FUNC (restart),(gpointer) params);

    setup_button = gtk_button_new_with_label ("Setup");
    gtk_box_pack_start (GTK_BOX (hboxsetup), setup_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (setup_button), "clicked",
                    GTK_SIGNAL_FUNC (setup),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (vboxleft), hboxsetup, TRUE, TRUE, 0);

/* Create pause button. */
    pause_button = gtk_button_new_with_label ("Pause");
    gtk_box_pack_start (GTK_BOX (vboxmiddle), pause_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (pause_button), "clicked",
                    GTK_SIGNAL_FUNC (pauseb),(gpointer) params);

/* Create quit button. */
    quit_button = gtk_button_new_with_label ("Quit");
    gtk_box_pack_start (GTK_BOX (vboxright), quit_button, TRUE, TRUE, 0);
    gtk_signal_connect_object (GTK_OBJECT (quit_button), "clicked",
                    GTK_SIGNAL_FUNC (quit),GTK_OBJECT (window));

    xminus10_button = gtk_button_new_with_label ("<<");
    gtk_box_pack_start (GTK_BOX (hboxx), xminus10_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (xminus10_button), "clicked", GTK_SIGNAL_FUNC (xminus10b),(gpointer) params);
    xminus_button = gtk_button_new_with_label ("<");
    gtk_box_pack_start (GTK_BOX (hboxx), xminus_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (xminus_button), "clicked", GTK_SIGNAL_FUNC (xminusb),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (hboxx), xlabel, TRUE, TRUE, 0);
    xplus_button = gtk_button_new_with_label (">");
    gtk_box_pack_start (GTK_BOX (hboxx), xplus_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (xplus_button), "clicked", GTK_SIGNAL_FUNC (xplusb),(gpointer) params);
    xplus10_button = gtk_button_new_with_label (">>");
    gtk_box_pack_start (GTK_BOX (hboxx), xplus10_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (xplus10_button), "clicked", GTK_SIGNAL_FUNC (xplus10b),(gpointer) params);

    yminus10_button = gtk_button_new_with_label ("<<");
    gtk_box_pack_start (GTK_BOX (hboxy), yminus10_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (yminus10_button), "clicked", GTK_SIGNAL_FUNC (yminus10b),(gpointer) params);
    yminus_button = gtk_button_new_with_label ("<");
    gtk_box_pack_start (GTK_BOX (hboxy), yminus_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (yminus_button), "clicked", GTK_SIGNAL_FUNC (yminusb), (gpointer) params);
    gtk_box_pack_start (GTK_BOX (hboxy), ylabel, TRUE, TRUE, 0);
    yplus_button = gtk_button_new_with_label (">");
    gtk_box_pack_start (GTK_BOX (hboxy), yplus_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (yplus_button), "clicked", GTK_SIGNAL_FUNC (yplusb), (gpointer) params);
    yplus10_button = gtk_button_new_with_label (">>");
    gtk_box_pack_start (GTK_BOX (hboxy), yplus10_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (yplus10_button), "clicked", GTK_SIGNAL_FUNC (yplus10b),(gpointer) params);

    zminus10_button = gtk_button_new_with_label ("<<");
    gtk_box_pack_start (GTK_BOX (hboxz), zminus10_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (zminus10_button), "clicked", GTK_SIGNAL_FUNC (zminus10b),(gpointer) params);
    zminus_button = gtk_button_new_with_label ("<");
    gtk_box_pack_start (GTK_BOX (hboxz), zminus_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (zminus_button), "clicked", GTK_SIGNAL_FUNC (zminusb),(gpointer) params);
    gtk_box_pack_start (GTK_BOX (hboxz), zlabel, TRUE, TRUE, 0);
    zplus_button = gtk_button_new_with_label (">");
    gtk_box_pack_start (GTK_BOX (hboxz), zplus_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (zplus_button), "clicked", GTK_SIGNAL_FUNC (zplusb),(gpointer) params);
    zplus10_button = gtk_button_new_with_label (">>");
    gtk_box_pack_start (GTK_BOX (hboxz), zplus10_button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (zplus10_button), "clicked", GTK_SIGNAL_FUNC (zplus10b),(gpointer) params);

    gtk_signal_connect_object (GTK_OBJECT (window), "destroy",
                    GTK_SIGNAL_FUNC (quit),GTK_OBJECT (window));

/* Put the hbox into the vbox. */
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vboxleft, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vboxmiddle, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vboxright, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vboxrb, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vboxcoord, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxrb), hboxx, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxrb), hboxy, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vboxrb), hboxz, FALSE, FALSE, 0);

/* Open the input file, if it fails exit. */
    if (params->file[0]=='_')  fp=stdin; 
    else {
	fp=NULL;
	fp=fopen(params->file,"r"); 
	if(fp==NULL) {
	    printf("Error opening file: %s\n",params->file);
	    gtk_exit (0);
	}
	fseek(fp,0,0);
    }

/* Allocate colors */
    params->ncolors = allocatecolors (&colors,params->colorset,params->mode) - 2;

/* Show all boxes,entries,buttons and pixmaps. */
    gtk_widget_show (vbox);
    gtk_widget_show (hbox);
    gtk_widget_show (vboxleft);
    gtk_widget_show (vboxmiddle);
    gtk_widget_show (vboxright);
    gtk_widget_show (hboxsetup);
    gtk_widget_show (vboxrb);
    gtk_widget_show (hboxx);
    gtk_widget_show (hboxy);
    gtk_widget_show (hboxz);
    gtk_widget_show (vboxcoord);
    gtk_widget_show (maxx_entry);
    gtk_widget_show (maxy_entry);
    gtk_widget_show (maxz_entry);
    gtk_widget_show (xc_entry);
    gtk_widget_show (yc_entry);
    gtk_widget_show (zc_entry);
    gtk_widget_show (time_entry);
    gtk_widget_show (coord_entry);
    gtk_widget_show (restart_button);
    gtk_widget_show (setup_button);
    gtk_widget_show (pause_button);
    gtk_widget_show (quit_button);
    gtk_widget_show (xlabel);
    gtk_widget_show (ylabel);
    gtk_widget_show (zlabel);
    gtk_widget_show (xplus_button);
    gtk_widget_show (yplus_button);
    gtk_widget_show (zplus_button);
    gtk_widget_show (xminus_button);
    gtk_widget_show (yminus_button);
    gtk_widget_show (zminus_button);
    gtk_widget_show (xplus10_button);
    gtk_widget_show (yplus10_button);
    gtk_widget_show (zplus10_button);
    gtk_widget_show (xminus10_button);
    gtk_widget_show (yminus10_button);
    gtk_widget_show (zminus10_button);
    gtk_widget_show (reseto_button);
    gtk_widget_show (drawing_area);
    gtk_widget_show (window);

/* Setup timeout. */
    gtk_timeout_add((guint32) MININTERVAL,(GtkFunction) drawnext, params);
}


/********************************************************************/
/* The main function sets up default parameters and calls either    */
/* handleargs or setupwindow depending on the number of commandline */
/* parameters.                                                      */
/********************************************************************/
int main (int argc, char **argv)
{
struct GlobalParams params;

    params.iangle=0.0; 
    params.jangle=0.0;  
    params.kangle=0.0; 
    params.numframe=1;
    params.pausecheck=FALSE;
    params.drawcheck=TRUE;
    params.atEnd=FALSE;
    params.rotated=FALSE;
    params.setupstop=FALSE;
    params.whitebg=FALSE;
    params.erase=FALSE;
    params.fxyz=FALSE;
    params.pressed=FALSE;
    params.dumpnum=FALSE;
    params.StartedAlready=FALSE;
    params.redrawcheck=FALSE;
    params.usetypes=FALSE;

    disp= XOpenDisplay(NULL);

/* Start gtk initialization. */
    gtk_init (&argc, &argv);

    printf("\n gdpc version "GDPCVER", Copyright (C) 1999 Jonas Frantz\n");
    printf(" gdpc comes with ABSOLUTELY NO WARRANTY; for details\n");
    printf(" check out the documentation.  This is free software, and\n"); 
    printf(" you are welcome to redistribute it under all conditions.\n\n");

   params.absxsize=drawXsize;
   params.absysize=drawYsize;

   params.xmin=65535.0;  
   params.ymin=65535.0;  
   params.zmin=65535.0;
   params.xmax=0.0;
   params.ymax=0.0;
   params.zmax=0.0;

   params.vary=0;
   params.scol=0;
   params.sort=0;
   params.radius=5;
   params.mode=1;
   params.colorset=0;
   params.dumpnum=FALSE;

   params.xcolumn=1;
   params.ycolumn=2;
   params.zcolumn=3;
   params.tcolumn=4;

   params.interval=MININTERVAL;    /* Make sure the drawing routine doesnt
				      get called to often */
   params.dumpname[0]=(char) NULL;

/* Handle arguments passed to the program. */
    if (argc==1) {
	setupwindow(&params);
	}
    else { 
	if(!handleargs(argc,argv,&params)) gtk_exit(0);
	StartEverything(&params);
    }

/* Start gtk. */
    gtk_main ();

    exit(0);
}



