/*  **************************************************************************
 * 
 * --- File: chart.c
 * 
 * --- Purpose: a program to draw data charts.
 * 		If you think that this source is a mess, you're quite right. 
 *              The thing is, I wanted to write code that could be compiled
 *              with both Turbo C++ and gcc/DJGPP.
 *              So consider this program as a programming exercise.
 * 
 * --- Developed with: Pentium PC, Linux 2.0.27, gcc 2.7.2.1
 * 
 * --- Copyright: Guido Gonzato, guido@ibogfs.cineca.it
 * 
 * --- Last updated: December 4, 1997.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * ************************************************************************ */

/*
 * History of changes:
 *
 * Version 1.0: submitted to SimTel.
 * 
 * Version 1.1: a strange error occurs when handling multiples files.
 * 
 * Version 1.2: adapted to be compiled with gcc/DJGPP + bcc2grx (Linux and
 * MS-DOS); submitted to Pluto + Sunsite. The error was caused by the fact 
 * that the identifiers 'y1' and 'y2', previously used instead of y_inf and 
 * y_sup, are already used by functions in math.h.
 * 
 * Version 2.0: ported to X11 using the new grx22.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifdef __GNUC__
#  include <libbcc.h>
#  ifdef __linux__
     extern int khbit(void), getkey(void);
#  else
#    include <pc.h>
#  endif
#  include "grx20.h"
#  include "grfontdv.h"
#  define getch() getkey()
#else
#  include <graphics.h>
#  include <conio.h>
#endif

#include "pcxutil.h"
#include "uxcrt.h"

#define MAX_LEN 256
#define ME      "chart"
#define VERSION "2.0.0"

typedef enum { CROSS, DIAMOND, DOT, SQUARE, TRIANGLE } PLOT_STYLE;

typedef struct {

  double x_inf, x_sup,       /*  range for X axis */
         y_inf, y_sup,       /*  range for Y axis */
         tick1_x, tick2_x,   /*  ticks for X */
         tick1_y, tick2_y,   /*  ticks for Y */
         slope, inter;       /*  regression line */

  char   name[80],
         title[60],
         title_x[60],
         title_y[60],
         remark[80];

  PLOT_STYLE style;

  short  draw_grid,          /* chart options */
         join_data,
         write_regr,
         write_x_val,
         write_y_val,
         draw_tick1_x,
         draw_tick2_x,
         draw_tick1_y,
         draw_tick2_y,
         svga1,              /* command line options */
         svga2,
         saveps,
         savepcx,
         regr,
         nopause;

  FILE   *ini, *dat;         /* data file */

} CHART_OPTIONS;

#ifdef __GNUC__
GrFont *font1, *font2, *font3;
#endif

void initsvga(int);
void ps_save(int, int, int, int, char *);
void find_range(double *, double *, double *, double *);
void set_default(CHART_OPTIONS *);
#ifdef __GNUC__
void setfont(GrTextOption *, GrFont *, int, int, int, int, int);
#endif
void draw_chart(CHART_OPTIONS *);
void init_data(char *, CHART_OPTIONS *);
void usage();

/* ----- */

void initsvga(int mode)
/* Initialize graphics */
{

  int gd, gm, err;

#ifdef __TURBOC__
  initgraph(&gd, &gm, "");
  err = graphresult();
  if (err != grOk) {
    fprintf(stderr, "%s: couldn't initialize graphics\n", ME);
    exit(1);
  }
#else
  switch (mode) {
  
    case 0: set_BGI_mode_whc(&gd, &gm, 640, 480, 16);
            initgraph(&gd, &gm, "");
            break;

    case 1: set_BGI_mode_whc(&gd, &gm, 800, 600, 16);
            initgraph(&gd, &gm, "");
            break;

    case 2: set_BGI_mode_whc(&gd, &gm, 1024, 768, 16);
            initgraph(&gd, &gm, "");

  }
#endif

} /* initsvga() */

/* ----- */

void ps_save(int x1, int y1, int x2, int y2, char *title)
/* This function saves a screen portion as a PostScript file, centered on a
   A4 sheet of paper, 144 dpi. */
{
  
  FILE *f;
  int cx, cy, x, y, b, i;
  int vet[4] = { 8, 4, 2, 1 };
  char hexa[] = "0123456789abcdef";

  cx = 298;  /* screen centre in 1/72 inch */
  cy = 420;
  x = x2 - x1 + 1;
  y = y2 - y1 + 1;
  
  f = fopen(title, "w");
  fprintf(f, "%%!PS-Adobe-2.0 EPSF-2.0\n");
  fprintf(f, "%%%%Creator: %s\n", ME);
  fprintf(f, "%%%%Title: %s\n", title);
  fprintf(f, "%%%%Pages: 1\n");
  fprintf(f, "%%%%BoundingBox: %d %d %d %d\n", cx - x / 4, cy - y / 4,
                                               cx + x / 4, cy + y / 4);
  fprintf(f, "%%%%EndComments\n");
  fprintf(f, "%%%%EndProlog\n");
  fprintf(f, "%%%%Page: 1 1\n");
  fprintf(f, "/origstate save def\n");
  fprintf(f, "20 dict begin\n");
  fprintf(f, "/pix %d string def\n", x / 8);
  fprintf(f, "%d %d translate\n", cx - x / 4, cy - y / 4);
  fprintf(f, "%d %d scale\n", x / 2, y / 2);
  fprintf(f, "%d %d 1\n", x, y);
  fprintf(f, "[ %d 0 0 -%d 0 %d ]\n", x, y, y);
  fprintf(f, "{currentfile pix readhexstring pop}\n");
  fprintf(f, "image\n");

  /* pixels */
  
  for (y = y1; y < y2 + 1; y++)
    for (x = x1; x < x2 + 1; x += 4) {
      
      b = 15; /* all white */
      
      for (i = 0; i < 4; i++)
        if (getpixel(x + i, y) != BLACK)
          b -= vet[i];
      
      fputc(hexa[b], f);

    }
  
  fprintf(f, "\n\n\n");
  fprintf(f, "showpage\n");
  fprintf(f, "end\n");
  fprintf(f, "origstate restore\n");
  fprintf(f, "%%%%Trailer\n");
  
  fclose(f);

} /* ps_save() */

/* ----- */

double tenpow(int exp)
/* Return 10 ** exp */
{

  int i, neg;
  double temp = 1;

  neg = (exp < 0);

  if (neg)
    exp = -exp;

  for (i = 0; i < exp; i++)
    temp *= 10;

  return ( (neg) ? (1 / temp): temp );

} /* tenpow() */

/* ----- */

void find_range(double *v1, double *v2, double *tick1, double *tick2)
/* Find the ticks for a range and change the bounds. */
{

  int     pos_exp, err;
  char    mant_str[30], exp_str[30], temp[MAX_LEN];
  double  modulus, range, expon, mantissa;

  /* The Automatic Scaling Problem. I will solve it this way:
   * 
   * - split range into mantissa * 10 ** exponent
   * - divide mantissa by 1, 0.5, 0.2, or 0.1
   * - tick = divisor for which Int(mantissa / divisor) >= 5
   * - tick = tick * 10 ** exponent
   * - round v1 e v2 with tick.
   */

  range = *v2 - *v1;

  sprintf(temp, "%e", range);  /* print the number as x.xxxxe+yy */

  /* take exponent and mantissa */

  pos_exp = chrpos('e', temp);
  strcpy(exp_str, temp);
  strmid(exp_str, pos_exp + 2, strlen(temp) - pos_exp - 1);
  strcpy(mant_str, temp);
  strmid(mant_str, 0, pos_exp);

  /* get mantissa and exponent as numbers */

  mantissa = atof(mant_str);
  expon = atof(exp_str);

  if (floor(mantissa) >= 5.0) {
    *tick1 = 1.0;
    *tick2 = 0.2; /* the error used to strike here */
  }
  else
  if (floor(mantissa / 0.5) >= 5.0) {
    *tick1 = 0.5;
    *tick2 = 0.1;
  }
  else
  if (floor(mantissa / 0.2) >= 5.0) {
    *tick1 = 0.2;
    *tick2 = 0.05;
  }
  else
  if (floor(mantissa / 0.1) >= 5.0) {
    *tick1 = 0.1;
    *tick2 = 0.02;
  }

  *tick1 *= tenpow((int) (expon));
  *tick2 *= tenpow((int) (expon));

  /* now correct range with tick */

  modulus = *tick1 * (*v1 / *tick1 - (int) (*v1 / *tick1));

  if (modulus != 0)          /* if it is to be rounded  */
    if (*v1 < 0)             /* negative: far from zero */
      *v1 = *v1 - *tick1 - modulus;
    else                     /* positive: near to zero  */
      *v1 = *v1 - modulus;

  modulus = *tick1 * (*v2 / *tick1 - (int) (*v2 / *tick1));

  if (modulus != 0)
    if (*v2 < 0)             /* negative: far from zero */
      *v2 = *v2 - modulus;
    else                     /* positive: near to zero  */
      *v2 = *v2 + *tick1 - modulus;

} /* find_range() */

/* ----- */

void set_default(CHART_OPTIONS *opt)
/* Given range values for X and Y, establish the defaults. */
{

  int ch, ok;
  char temp[80];

  /* find the range and the ticks */

  opt->tick1_x = opt->tick2_x = opt->tick1_y = opt->tick2_y = 0.0;
  
  find_range(&opt->x_inf, &opt->x_sup, &opt->tick1_x, &opt->tick2_x);

  /* the error used to strike here, after parsing two datafiles. */
  
  find_range(&opt->y_inf, &opt->y_sup, &opt->tick1_y, &opt->tick2_y); 

  /* now look for a .cht file corrisponding to the data file */

  strcpy(temp, opt->name);
  if (chrpos('.', opt->name) != -1)
    strmid(temp, 0, chrpos('.', opt->name));

  strcat(temp, ".cht");

  if ( (opt->ini = fopen(temp, "r")) == NULL ) {

  /* not found: create it */

    opt->ini = fopen(temp, "w");

    fprintf(opt->ini,
    "C  # plot style: C)ross, D)iamond, dO)t, S)quare, T)riangle\n");

    fprintf(opt->ini, "R %6.4f %6.4f %6.4f %6.4f %6.4f %6.4f %6.4f %6.4f\n",
 		    opt->x_inf, opt->x_sup, opt->tick1_x, opt->tick2_x, 
 		    opt->y_inf, opt->y_sup, opt->tick1_y, opt->tick2_y);
    
/*  I wish I knew why this doesn't work. 
 *   num2str(opt->x_inf, temp);
 *   fprintf(ini, "R %s ", temp);
 *   num2str(opt->x_sup, temp);
 *   fprintf(ini, "%s ", temp);
 *   num2str(opt->tick1_x, temp);
 *   fprintf(ini, "%s ", temp);
 *   num2str(opt->tick2_x, temp);
 *   fprintf(opt->ini, "%s ", temp);
 *   num2str(opt->y_inf, temp);
 *   fprintf(opt->ini, "%s ", temp);
 *   num2str(opt->y_sup, temp);
 *   fprintf(opt->ini, "%s ", temp);
 *   num2str(opt->tick1_y, temp);
 *   fprintf(opt->ini, "%s ", temp);
 *   num2str(opt->tick2_y, temp);
 *   fprintf(opt->ini, "%s\n", temp);
 */
    
    fprintf(opt->ini, "Y  # draw grid?\n");
    fprintf(opt->ini, "N  # join data points?\n");
    fprintf(opt->ini, "%c  # regression line?\n", 
	    opt->regr ? 'Y': 'N');
    fprintf(opt->ini, "Y  # write X values?\n");
    fprintf(opt->ini, "Y  # write Y values?\n");
    fprintf(opt->ini, "Y  # main tick for X\n");
    fprintf(opt->ini, "Y  # second tick for X\n");
    fprintf(opt->ini, "Y  # main tick for Y\n");
    fprintf(opt->ini, "Y  # second tick for Y\n");
    fprintf(opt->ini, "%s\n", opt->name);
    fprintf(opt->ini, "X axis\n");
    fprintf(opt->ini, "Y axis\n");
    fprintf(opt->ini, "You can insert your remarks here\n");
    fclose(opt->ini);

  } /* if */

  /* now load the configuration */

  opt->ini = fopen(temp, "r");
  fgets(temp, 100, opt->ini);
  ch = temp[0];

  switch (ch) {

    case 'C': opt->style = CROSS;
              break;

    case 'D': opt->style = DIAMOND;
              break;

    case 'O': opt->style = DOT;
              break;

    case 'S': opt->style = SQUARE;
              break;

    case 'T': opt->style = TRIANGLE;

  }

  fgets(temp, 100, opt->ini);
  sscanf(temp, "%c %lf %lf %lf %lf %lf %lf %lf %lf", &ch,
 		&opt->x_inf, &opt->x_sup, &opt->tick1_x, &opt->tick2_x,
 		&opt->y_inf, &opt->y_sup, &opt->tick1_y, &opt->tick2_y); 

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->draw_grid =     (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->join_data =     (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->write_regr =    (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->write_x_val =   (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->write_y_val =   (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->draw_tick1_x =  (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->draw_tick2_x =  (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->draw_tick1_y =  (ch == 'Y') || (ch == 'y');

  fgets(temp, 100, opt->ini);
  ch = temp[0];
  opt->draw_tick2_y =  (ch == 'Y') || (ch == 'y');

  fgets(opt->title, 100, opt->ini);
  fgets(opt->title_x, 100, opt->ini);
  fgets(opt->title_y, 100, opt->ini);
  fgets(opt->remark, 100, opt->ini);

  fclose(opt->ini);

} /* set_default() */

/* ----- */

#ifdef __GNUC__
void setfont(GrTextOption *opt, GrFont *font, int xal, int yal, int direc,
             int fc, int bc)
/* Set font characteristics */
{

  opt->txo_font = font;
  opt->txo_xalign = xal;
  opt->txo_yalign = yal;
  opt->txo_direct = direc;
  opt->txo_fgcolor.v = fc;
  opt->txo_bgcolor.v = bc;

} /* setfont() */
#endif

/* ----- */

void draw_chart(CHART_OPTIONS *opt)
/* Draw the chart */
{

  int i, j, first, ch,
      oldx, oldy,
      x0, y0,            /* screen centre */
      xl, yu,            /* upper left corner */
      xr, yl,            /* lower right corner */
      adjust_x,          /* to adjust panel width */
      adjust_y,
      offset_centre,     /* to move y0 upwards */
      offset_title,      /* space between panel and opt->title */
      offset_title_x,    /* ditto, title X */
      offset_title_y,    /* ditto, title Y */
      offset_rem1,       /* ditto, remark below opt->title X */
      offset_regr,       /* ditto, regression line */
      xx, yy,            /* for the plotting routine */
      xx2, yy2,
      side_x,		 /* chart side */
      side_y,
      tick_len,
      arm;		 /* cross, square etc. arm */

  double range_x,
         range_y,
         kx, ky,
         x, y;

  char   temp[80], temp2[80], *err;

#ifdef __GNUC__
  GrTextOption gr_opt;
  GrFont *font;
#endif

  x0 = getmaxx() / 2;
  y0 = getmaxy() / 2;
  side_x = getmaxx() * 0.7;
  side_y = getmaxy() * 0.7;
  adjust_x = getmaxx() / 50;
  adjust_y = getmaxy() / 50;
  offset_centre = getmaxy() / 15;
  offset_title = getmaxy() / 18;
  offset_title_x = getmaxx() / 20;
  offset_title_y = getmaxy() / 6;
  offset_rem1 = getmaxy() / 7;
  offset_regr = getmaxy() / 9;
  xl = x0 - side_x / 2;
  yu = y0 - side_y / 2 - offset_centre;
  xr = x0 + side_x / 2;
  yl = y0 + side_y / 2 - offset_centre;

  range_x = (opt->x_sup - opt->x_inf);
  range_y = (opt->y_sup - opt->y_inf);
  kx = side_x / range_x;
  ky = side_y / range_y;
  tick_len = 8;

  /* draw retangle and titles */

  setcolor(WHITE);
  rectangle(xl - adjust_x, yu - adjust_y, xr + adjust_x, yl + adjust_y);

#ifdef __TURBOC__
  /* BGI fonts are cheesy... */
  settextstyle(SMALL_FONT, HORIZ_DIR, 8);
  settextjustify(CENTER_TEXT, CENTER_TEXT);
  setcolor(LIGHTGREEN);
  outtextxy(x0, yu - adjust_y - offset_title, opt->title);

  settextstyle(SMALL_FONT, HORIZ_DIR, 6);
  outtextxy(x0, yl + adjust_y + offset_title_x, opt->title_x);

  settextstyle(SMALL_FONT, VERT_DIR, 6);
  outtextxy(xl - offset_title_y, y0 - offset_centre, opt->title_y);

  settextstyle(SMALL_FONT, HORIZ_DIR, 6);
  settextjustify(LEFT_TEXT, CENTER_TEXT);
  outtextxy(0, yl + adjust_y + offset_rem1, opt->remark);

  setcolor(WHITE);
#else
  /* grx fonts are much better */
  memset(&gr_opt, 0, sizeof(gr_opt));
  setfont(&gr_opt, font1, GR_ALIGN_CENTER, GR_ALIGN_CENTER, GR_TEXT_RIGHT,
          LIGHTGREEN, GrBlack());
  GrDrawString(opt->title, strlen(opt->title),
               x0, yu - adjust_y - offset_title / 2, &gr_opt);

  memset(&gr_opt, 0, sizeof(gr_opt));
  setfont(&gr_opt, font2, GR_ALIGN_CENTER, GR_ALIGN_CENTER, GR_TEXT_RIGHT,
          LIGHTGREEN, GrBlack());
  GrDrawString(opt->title_x, strlen(opt->title_x),
               x0, yl + adjust_y + offset_title_x, &gr_opt);

  memset(&gr_opt, 0, sizeof(gr_opt));
  setfont(&gr_opt, font2, GR_ALIGN_CENTER, GR_ALIGN_CENTER, GR_TEXT_UP,
          LIGHTGREEN, GrBlack());
  GrDrawString(opt->title_y, strlen(opt->title_y),
               xl - offset_title_y, y0 - offset_centre, &gr_opt);

  memset(&gr_opt, 0, sizeof(gr_opt));
  setfont(&gr_opt, font2, GR_ALIGN_LEFT, GR_ALIGN_CENTER, GR_TEXT_RIGHT,
          LIGHTGREEN, GrBlack());
  GrDrawString(opt->remark, strlen(opt->remark),
               0, yl + adjust_y + offset_rem1, &gr_opt);
#endif

  /* draw ticks and grid for X */

  /* prepare to write X values */

#ifdef __TURBOC__
  settextstyle(SMALL_FONT, HORIZ_DIR, 5);
  settextjustify(CENTER_TEXT, CENTER_TEXT);
#else
  memset(&gr_opt, 0, sizeof(gr_opt));
  setfont(&gr_opt, font3, GR_ALIGN_CENTER, GR_ALIGN_CENTER, GR_TEXT_RIGHT,
          WHITE, GrBlack());
#endif

  i = 0;
  x = opt->x_inf;
  xx = xl + (int) ((x - opt->x_inf) * kx);

  while (xx < xr + adjust_x) {

    if (opt->draw_grid) {
#ifdef __TURBOC__
      setcolor(LIGHTBLUE);
      setlinestyle(USERBIT_LINE, 0x1010, 1);
      line(xx, yl, xx, yu);
      setcolor(WHITE);
#else
      for (j = yu; j < yl; j += 10)
        putpixel(xx, j, LIGHTBLUE);
#endif
    }

    if (opt->draw_tick1_x) {
      setlinestyle(SOLID_LINE, 0, 1);
      line(xx, yl + adjust_y, xx, yl + adjust_y - tick_len);
      line(xx, yu - adjust_y, xx, yu - adjust_y + tick_len);
    }

    if (opt->write_x_val) {
      num2str(x, temp);
#ifdef __TURBOC__
      outtextxy(xx, yl + 2 * adjust_y, temp);
#else
      GrDrawString(temp, strlen(temp), xx, yl + 2 * adjust_y, &gr_opt);
#endif
    } /* if */

    i++;
    x = opt->x_inf + (i * opt->tick1_x);
    xx = xl + (int) ((x - opt->x_inf) * kx);

  } /* while */

  /* ticks and grid for Y */

  /* prepare to write Y values Y */

#ifdef __TURBOC__
  settextjustify(RIGHT_TEXT, CENTER_TEXT);
#else
  memset(&gr_opt, 0, sizeof(gr_opt));
  setfont(&gr_opt, font3, GR_ALIGN_RIGHT, GR_ALIGN_CENTER, GR_TEXT_RIGHT,
          WHITE, GrBlack());
#endif

  i = 0;
  y = opt->y_inf;
  yy = yl - (int) ((y - opt->y_inf) * ky);

  while (yy > yu - adjust_y) {

    if (opt->draw_grid) {
#ifdef __TURBOC__
      setcolor(LIGHTBLUE);
      setlinestyle(USERBIT_LINE, 0x1010, 1);
      line(xl, yy, xr, yy);
      setcolor(WHITE);
#else
/* setlinestyle() doesn't work well in bcc2grx */
      for (j = xl; j < xr; j += 10)
        putpixel(j, yy, LIGHTBLUE);
#endif
    } /* if */

    if (opt->draw_tick1_y) {
      setlinestyle(SOLID_LINE, 0, 1);
      line(xl - adjust_x, yy, xl - adjust_x + tick_len, yy);
      line(xr + adjust_x, yy, xr + adjust_x - tick_len, yy);
    }

    if (opt->write_y_val) {
      num2str(y, temp);
#ifdef __TURBOC__
      outtextxy(xl - adjust_x - tick_len, yy, temp);
#else
      GrDrawString(temp, strlen(temp), xl - adjust_x - tick_len, yy, &gr_opt);
#endif

    } /* if */

    i++;
    y = opt->y_inf + (i * opt->tick1_y);
    yy = yl - (int) ((y - opt->y_inf) * ky);

  } /* while */

  /* draw secundary ticks */

  i = 1;
  x = opt->x_inf + opt->tick2_x;
  xx = xl + (int) ((x - opt->x_inf) * kx);

  while (xx < xr) {

    if (opt->draw_tick2_x) {
      setlinestyle(SOLID_LINE, 0, 1);
      line(xx, yl + adjust_y, xx, yl + adjust_y - tick_len / 2);
      line(xx, yu - adjust_y, xx, yu - adjust_y + tick_len / 2);
    }

    i++;
    x = opt->x_inf + (i * opt->tick2_x);
    xx = xl + (int)((x - opt->x_inf) * kx);

  } /* while */

  i = 1;
  y = opt->y_inf + opt->tick2_y;
  yy = yl - (int) ((y - opt->y_inf) * ky);

  while (yy > yu) {

    if (opt->draw_tick2_y) {
      setlinestyle(SOLID_LINE, 0, 1);
      line(xl - adjust_x, yy, xl - adjust_x + tick_len / 2, yy);
      line(xr + adjust_x, yy, xr + adjust_x - tick_len / 2, yy);
    }

    i++;
    y = opt->y_inf + (i * opt->tick2_y);
    yy = yl - (int) ((y - opt->y_inf) * ky);

  } /* while */

  /* now plot the data */

  arm = 3;
  first = TRUE;
  opt->dat = fopen(opt->name, "r"); /* no checks */
  i = 0;
  
  while (fscanf(opt->dat, "%s", temp) != EOF) {

    /* ignore non-numeric and empty strings */

    if (i & 1)
      y = strtod(temp, &err);
    else
      x = strtod(temp, &err);
    if ( *err == '\0' ) /* good conversion from string to number */
      i++;
    else
      continue;

    if (i & 1)
    /* it's x: read y */
      continue;
      
    xx = xl + (int) ((x-opt->x_inf) * kx);
    yy = yl - (int) ((y-opt->y_inf) * ky);
    setcolor(YELLOW);

    switch (opt->style) {

      case CROSS:    line(xx - arm, yy - arm, xx + arm, yy + arm);
		     line(xx + arm, yy - arm, xx - arm, yy + arm);
		     break;

      case DIAMOND:  line(xx - arm, yy, xx, yy - arm);
		     line(xx, yy - arm, xx + arm, yy);
		     line(xx + arm, yy, xx, yy + arm);
		     line(xx, yy + arm, xx - arm, yy);
		     break;

      case DOT:      putpixel(xx, yy, YELLOW);
                     break;

      case SQUARE:   rectangle(xx - arm, yy - arm, xx + arm, yy + arm);
		     break;

      case TRIANGLE: line(xx - arm, yy + arm, xx, yy - arm);
		     line(xx, yy - arm, xx + arm, yy + arm);
		     line(xx + arm, yy + arm, xx - arm, yy + arm);
                     break;

    }  /* switch */

    if (first) {
      first = FALSE;
      oldx = xx;
      oldy = yy; /* moveto(xx, yy); */
    }

    if (opt->join_data) {
      setcolor(LIGHTRED);
      line(oldx, oldy, xx, yy);
      oldx = xx;
      oldy = yy; /* moveto(xx, yy); */
    }

  } /* while */

  fclose(opt->dat);

  if (opt->write_regr) { /* draw regression line */

    setviewport(xl, yu, xr, yl, TRUE);  /* to clip the line */
    xx = 0;
    yy = yl - yu - (int) ((opt->slope * opt->x_inf + 
			   opt->inter - opt->y_inf) * ky);
    xx2 = (int) ((opt->x_sup - opt->x_inf) * kx);
    yy2 = yl - yu - (int) ((opt->slope * opt->x_sup + 
			    opt->inter - opt->y_inf) * ky);
    setlinestyle(DOTTED_LINE, 0, 1);
    setcolor(YELLOW);
    line(xx, yy, xx2, yy2);
    setviewport(0, 0, getmaxx(), getmaxy(), TRUE);
    strcpy(temp, "Y = ");
    num2str(opt->slope, temp2);
    strcat(temp, temp2);
    strcat(temp, " X + ");
    num2str(opt->inter, temp2);
    strcat(temp, temp2);
#ifdef __TURBOC__ 
    settextstyle(SMALL_FONT, HORIZ_DIR, 6);
    settextjustify(LEFT_TEXT, CENTER_TEXT);
    outtextxy(0, yl + adjust_y + offset_regr, temp);
#else
    memset(&gr_opt, 0, sizeof(gr_opt));
    setfont(&gr_opt, font3, GR_ALIGN_LEFT, GR_ALIGN_CENTER, GR_TEXT_RIGHT,
            YELLOW, GrBlack());
    GrDrawString(temp, strlen(temp), 0, yl + adjust_y + offset_regr, &gr_opt);
#endif    

  } /* if (opt->write_regr) */

  if (opt->savepcx) {   /* save the chart as a monochrome pcx file */

    strcpy(temp, opt->name);
    if (chrpos('.', opt->name) != -1)
      strmid(temp, 0, chrpos('.', opt->name));

    strcat(temp, ".pcx");
    pcxinit();
    pcxsave(0, 0, getmaxx(), getmaxy(), temp);

  } /* if (opt->savepcx) */
  
  if (opt->saveps) {   /* save the chart as a PostScript file */

    strcpy(temp, opt->name);
    if (chrpos('.', opt->name) != -1)
      strmid(temp, 0, chrpos('.', opt->name));

    strcat(temp, ".ps");
    ps_save(0, 0, getmaxx(), getmaxy(), temp);

  } /* if (opt->saveps) */

  if (!opt->nopause)
    getch();

  cleardevice();

} /* draw_chart() */

/* ----- */

void init_data(char *s, CHART_OPTIONS *opt)
/* Find the range of the data. */
{

  long num_data;
  double x, y, sum_x, sum_y, mean_x, mean_y, sxx, sxy, syy;
  char temp[MAX_LEN], *err;

  /* find the range for X and Y, considering X = dat(i); Y = dat(i+1). */

  strcpy (opt->name, s);
  
  if ( (opt->dat = fopen(opt->name, "r")) == NULL) {
    fprintf(stderr, "Couldn't find file %s, so there.\n", opt->name);
    exit(1);
  }

  num_data = sum_x = sum_y 
           = opt->x_inf = opt->x_sup = opt->y_inf = opt->y_sup = 0;

  while (fscanf(opt->dat, "%s", temp) != EOF) {

    /* ignore non-numeric and empty strings */

    if (num_data & 1)
      y = strtod(temp, &err);
    else
      x = strtod(temp, &err);
    if ( *err == '\0' ) /* good conversion from string to number */
      num_data++;
    else
      continue;

    if (num_data & 1) { /* it's x */
      sum_x += x;
      if (x < opt->x_inf)
        opt->x_inf = x;
      if (x > opt->x_sup)
        opt->x_sup = x;
    }
    else { /* it's y */
      sum_y += y;
      if (y < opt->y_inf)
        opt->y_inf = y;
      if (y > opt->y_sup)
        opt->y_sup = y;
    }
            
  } /* while */

  fclose(opt->dat);
  num_data /= 2;
  
  if (num_data == 0) {
    fprintf(stderr, "%s: warning: invalid data file %s\n", ME, s);
    /* exit(1); */
  }
  
  if (opt->y_inf == opt->y_sup) { /* prevent error */
    opt->y_inf--;
    opt->y_sup++;
  }

  if (opt->x_inf == opt->x_sup) { /* prevent error */
    opt->x_inf--;
    opt->x_sup++;
  }

  opt->dat = fopen(opt->name, "r"); /* reopen to do the regression line */
  mean_x = sum_x / num_data;
  mean_y = sum_y / num_data;

  num_data = sxy = sxx = syy = 0.0;

  while (fscanf(opt->dat, "%s", temp) != EOF) {

    /* ignore non-numeric and empty strings */

    if (num_data & 1)
      y = strtod(temp, &err);
    else
      x = strtod(temp, &err);
    if ( *err == '\0' ) /* good conversion from string to number */
      num_data++;
    else
      continue;

    if (! (num_data & 1) ) { /* y read */
      sxy += x * (y - mean_y);
      sxx += x * (x - mean_x);
      syy += (y - mean_y) * (y - mean_y);
    }
    
  } /* while */

  fclose(opt->dat);
  opt->slope = sxy / sxx;
  opt->inter = mean_y - opt->slope * mean_x;

  set_default(opt);

} /* init_data() */

/* ----- */

void usage()
/* Give the user a hint */
{

  fprintf(stderr, "Usage: %s [-12Snrs] <datafile[.dat]>\n", ME);
  fprintf(stderr, "\t-1 = svga mode (800x600)\n");
  fprintf(stderr, "\t-2 = svga mode (1024x768)\n");
  fprintf(stderr, "\t-S = save a .ps file\n");
  fprintf(stderr, "\t-n = don't pause between charts\n");
  fprintf(stderr, "\t-r = force regression line\n");
  fprintf(stderr, "\t-s = save a .pcx file\n");
  fprintf(stderr, "\t-v = output version number and exit\n");

} /* usage() */

/* ----- */

void main(int argc, char *argv[])
{

  char s[MAX_LEN], ch;
  int i, j;
  CHART_OPTIONS opt;

  if (argc == 1) {
    usage();
    exit(1);
  }

  opt.svga1 = opt.svga2 = opt.regr = opt.saveps 
             = opt.savepcx = opt.nopause = FALSE;

  i = 1;
  strcpy(s, argv[i]);

  /* the following is the code for parsing the switches. I don't use
   *  getopt() because I don't know if Turbo C++ supports it.
   */

  while (s[0] == '-') { /* parse switches */

    for (j = 1; s[j] != '\0'; j++)

      switch(s[j]) {

        case '1': opt.svga1 = TRUE;
                  opt.svga2 = FALSE;
                  break;

	case '2': opt.svga2 = TRUE;
                  opt.svga1 = FALSE;
                  break;

        case 'S': opt.saveps = TRUE;
                  break;
                  
        case 's': opt.savepcx = TRUE;
                  break;

        case 'r': opt.regr = TRUE;
                  break;

        case 'n': opt.nopause = TRUE;
                  break;

        case 'v': printf("%s version %s\n", ME, VERSION);
                  exit(1);
                  break;

        default:  fprintf(stderr, "Unknown switch %c\n", s[j]);
                  usage();
                  exit(1);

      } /* switch */
    
    i++;
    strcpy(s, argv[i]);

  } /* while */

  if (opt.svga1)
    initsvga(1);
  else
  if (opt.svga2)
    initsvga(2);
  else
    initsvga(0);

#ifdef __TURBOC__
  if (registerbgifont(small_font) < 0)
    exit(1);
#else
  font1 = GrLoadFont("helv22b.fnt");
  font2 = GrLoadFont("helv17.fnt");
  font3 = GrLoadFont("helv15.fnt");
#endif

  do {
    init_data(s, &opt);
    draw_chart(&opt);
    i++;
    if (i < argc)
      strcpy(s, argv[i]);
  } while (i < argc);

  restorecrtmode();
  exit(0);

}

/* --- End of file chart.c --- */
