/*  **************************************************************************
 * 
 * --- File: uxcrt.c
 * 
 * --- Purpose: micro-xcrt. This is a small subset of my old xcrt, a set of
 *              routines I wrote for Turbo C++ (extended CRT).
 * 
 * --- Developed with: Pentium PC, Linux 2.0.27, gcc 2.7.2.1
 * 
 * --- Copyright: Guido Gonzato, guido@ibogfs.cineca.it
 * 
 * --- Last updated:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * ************************************************************************ */


#include    <stdio.h>
#include    <ctype.h>
#include    <string.h>
#ifdef __TURBOC__
#  include    <dos.h>
#  include    <conio.h>
#endif
#include    <stdlib.h>
#include    "uxcrt.h"

static int  _months[] = {31,28,31,30,31,30,31,31,30,31,30,31};

/* ======================================================================== */

char *strings(char *s, int ch, int num)
/* Return a string made by num occurrences of character ch */
{

  int i;

  for (i = 0; i < num; i++)
    s[i] = ch;

  s[i] = '\0';

  return(s);

} /* strings() */

/* ----- */

char *toleft(char *s, int n)
/* Write a left-aligned string, that is followed by as many blank characters
   as are needed to reach n */
{

  int l;
  char temp[MAX_LEN];

  l = strlen(s);
  strings(temp,' ',n-l);
  strcat(s, temp);

  return(s);

} /* toleft() */

/* ----- */

/* --- input etc. ---- */

char *strdel(char *s, int pos, int num)
/* Delete num characters from string s, starting from pos. Equivalent to
   Turbo Pascal's Delete(). */
{

  int i;

  for (i = pos; s[i + num] != '\0'; i++)
    s[i] = s[i + num];

  s[i] = '\0';
  return(s);

} /* strdel() */

/* ----- */

char *strmid(char *s, int pos, int num)
/* strmid("ABCDEFG",2,3) = "CDE". Equivalent to Turbo Pascal's Copy(). */
{

  int i, len = strlen(s);

  if (pos >= len) {
    s[0] = '\0';
    return(s);
  }

  for (i = 0; i < num; i++)
    if (s[pos + i] != '\0')
      s[i] = s[pos + i];
    else
      break;

  s[i] = '\0';
  return(s);

} /* strmid() */

/* ----- */

char *scpymid(char *dest, char *source, int pos, int num)
/* Copy specified piece of source into dest. */
{

  int i;

  for (i = 0; i < num; i++)
    dest[i] = source[pos + i];

  dest[i] = '\0';
  return(dest);

} /* scpymid() */

/* ----- */

char *chrcat(char *s, int ch)
/* Append ch to s */
{

  int len;

  len = strlen(s);
  s[len] = ch;
  s[len + 1] = '\0';
  return(s);

} /* chrcat() */

/* ----- */

char *chrins(char *s, int ch, int pos)
/* Insert ch into s at position pos */
{

  int i, len = strlen(s);

  s[len + 1] = '\0';

  for (i = len; i > pos; i--)
    s[i] = s[i - 1];

  s[i] = ch;
  return(s);

} /* chrins() */

/* ----- */

int chrinstr(int ch, char *s)
{

  char *ptr;

  ptr = strchr(s, ch);
  return( (ptr) ? 1: 0);

} /* chrinstr() */

/* ----- */

int chrpos(int ch, char *s)
/* Like Turbo Pascal's Pos() */
{

  char *ptr = strchr(s, ch);

  return ( (ptr) ? (ptr - s): -1 );

} /* chrpos() */

/* ----- */

char *strins(char *dest, char *source, int pos)
/* Insert source into dest at position pos.
   Equivalent to Turbo Pascal's Insert(). */
{

   int i;

   for (i = 0; i < strlen(source); i++)
     chrins(dest, source[i], pos++);

   return(dest);

} /* strins() */

/* ----- */

void comma2num(char *s, double *n, int *err)
/* Convert the number with commas (e.g. "123,456") in string s into n. */
{

  int len, poscomma, i;
  int neg;

  /* if s is empty, n will be 0 */

  neg = FALSE;
  *err = 0;

  if (!strcmp(s,"")) {
    *n = 0;
    *err = 0;
    return;
  }

  len = strlen(s);

  if (s[0] == '-') {
    strmid(s,1,len-1);
    neg = TRUE;
  }

  len = strlen(s);

  /* check position of commas in s */

  for (i = 1; i < len / 4 + 1; i++) {
    poscomma = (len-i)-3*i;
    if (s[poscomma] != ',') {
      *err = poscomma;
      return;
    }
    strdel(s,poscomma,1);
  }

  *n = atof(s);
  if (neg)
    *n = -*n;

} /* comma2num() */

/* ----- */

char *num2comma(double n, char *s, int zero)
/* Turn 123456 into "123,456" */
{

  int i, len;
  int neg;

  if ( zero && (n == 0) ) {
    strcpy(s,"");
    return(s);
  }

  neg = (n < 0);
  if (neg)
    n = -n;      /* to avoid a comma too much */

  sprintf(s,"%.0f", n);
  len = strlen(s);

  for (i = 1; i < (len-1) / 3 + 1; i++)
    chrins(s,',',(len)-3*i);

  if (neg)
    chrins(s,'-',0);

  return(s);

} /* num2comma */

/* ----- */

char *num2str(double n, char *s)
/* From number to string using the shortest possible format */
{

  int i;

  if (n > 1E12)   /* keep exponential format */
    sprintf(s,"%f",n);
  else {
    sprintf(s,"%12.12f",n);
    for (i = strlen(s)-1; s[i] == '0'; i--)
      strdel(s,i,1);
    if (s[i] == '.')
      strdel(s,i,1);
  }

  return(s);

} /* num2str() */

/* ----- */

int leap(int year)
{

  return( ((year % 4) == 0) && ((year % 100) != 0) || ((year % 400) == 0) );

} /* leap() */

/* ----- */

long _date2num(char *d)
{

  int  day, month, year;
  long l;

  day = atof(strmid(d,0,2));
  month = atof(strmid(d,3,2));
  year = atof(strmid(d,6,2));
  l = (long) year*10000 + month*100 + day;
  return(l);

} /* _date2num() */

/* ----- */

int dateless(char *d1, char *d2)
/* Check if date d1 is earlier than d2. Input dates must be correct. */
{

  long int l1, l2;

  l1 = _date2num(d1);
  l2 = _date2num(d2);

  return(l1 < l2);

} /* dateless() */

/* ----- */

int datelesseq(char *d1, char *d2)
{

  long int l1, l2;

  l1 = _date2num(d1);
  l2 = _date2num(d2);

  return(l1 <= l2);

} /* datelesseq() */

/* ----- */

int dateincluded(char *d1, char *d2, char *d3)
/* Check if d1 is included between d2 and d3. */
{

  long l1, l2, l3;

  l1 = _date2num(d1);
  l2 = _date2num(d2);
  l3 = _date2num(d3);

  return( (l2 <= l1) && (l1 <= l3) );

} /* dateincluded() */

/* ----- */

int dayofyear(char *s)
/* Find the order of a day; for instance, February 2 is the 33-d day of
   the year. */
{

  int i, day, month, year, num;

  day = atof(strmid(s,0,2));
  month = atof(strmid(s,3,2));
  year = atof(strmid(s,6,2));

  num = 0;

  for (i = 0; i < month-1; i++)
    num += _months[i];

  num += day;
  if ( leap(year) && (month > 2) )
    num++;

  return(num);

} /* dayofyear() */

/* ----- */

long daydiff(char *d1, char *d2)
/* Return the number of days between d1 and d2 (d1 < d2). */
{

  int i, yr1, yr2, ord1, ord2;
  unsigned diff;
  char y1[3], y2[3];

  y1[0] = d1[6];
  y1[1] = d1[7];
  y2[0] = d2[6];
  y2[1] = d2[7];
  y1[2] = y2[2] = '\0';
  yr1 = atof(y1);
  yr2 = atof(y2);
  ord1 = dayofyear(d1);
  ord2 = dayofyear(d2);
  diff = 0;

  if (yr1 == yr2)
    diff = ord2 - ord1;
  else { /* count the years in between */

    for (i = yr1 + 1; i < yr2; i++)
      if (leap(i))
        diff += 366;
      else
        diff += 365;

    diff += 365; /* first year minus the days gone by */
    diff -= ord1;
    if (leap(yr1))
      diff++;
    diff += ord2;

  }

  return(diff);

} /* daydiff() */

/* ----- */

char *word2exa(UNS n, char *s)
/* Convert from number to a string containing that number in hexadecimal
   form (i.e. 65535 --> "FFFF") */
{

  char exadigits[] = "0123456789ABCDEF";
  UNS l,h;

  /* This is a quick but NOT PORTABLE algorithm */

  h = n / 256;
  l = n % 256;
  s[0] = exadigits[h >> 4];
  s[1] = exadigits[h & 15];
  s[2] = exadigits[l >> 4];
  s[3] = exadigits[l & 15];
  s[4] = '\0';

  return(s);

} /* word2exa() */

/* ----- */

char *long2exa(ULONG n, char *s)
{

  char s1[5], s2[5];

  word2exa(n / 65536, s1);
  word2exa(n % 65536, s2);
  strcpy(s,s1);
  strcat(s,s2);
  return(s);

} /* long2exa() */


/* --- End of file uxcrt.c --- */
