/*  **************************************************************************
 * 
 * --- File: pcxutil.c
 * 
 * --- Purpose: utilities for handling pcx files.
 * 
 * --- Developed with: Pentium PC, Linux 2.0.27, gcc 2.7.2.1
 * 
 * --- Copyright: Guido Gonzato, guido@ibogfs.cineca.it
 * 
 * --- Last updated: October 10, 1997
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * ************************************************************************ */


#include <stdio.h>

#ifdef __GNUC__
#  include <libbcc.h>
#  ifdef __linux__
     extern int khbit(void), getkey(void);
#  else
#    include <pc.h>
#    include "grx20.h"
#  endif
#  define getch() getkey()
#else
#  include <graphics.h>
#  include <conio.h>
#endif

#include "vpcxutil.h"
#include "vscrn.h"

/* ============================================================================================================================================== */

typedef unsigned char BYTE;

static BYTE vet[8] = { 128, 64, 32, 16, 8, 4, 2, 1 };

static int  maxx, maxy, bkcol;

/* ----- */

void pcxinit()
{

  maxx = vgetmaxx()+1;
  maxy = vgetmaxy()+1;
  bkcol = BLACK;

} /* pcxinit() */

/* ----- */

void pcxsave(int x1, int y1, int x2, int y2, char *name)
/* Save a picture as a mono .PCX file. */
{

  FILE     *f;        /* the .pcx file */
  register
  BYTE     b,         /* bytes read from the screen */
           b1,
           i,         /* counter */
           c;
  register
  int      x,y;       /* for loops */
  int      maxbytes,  /* bytes per line */
           numbytes;  /* counter */

  maxbytes = ((x2-x1+1) / 8 + 0.5); /* round */
  if (maxbytes % 2 != 0)  /* if odd */
    maxbytes++;           /* bytes/line should be an even number */
  f = fopen(name,"wb");

  /* write the header */

  fprintf(f,"%c",10);                 /* manif */
  fprintf(f,"%c",3);                  /* hard */
  fprintf(f,"%c",1);                  /* encod */
  fprintf(f,"%c",1);                  /* bitpx */
  fprintf(f,"%c%c",x1,x1/256);        /* x1 */
  fprintf(f,"%c%c",y1,y1/256);        /* y1 */
  fprintf(f,"%c%c",x2,x2/256);        /* x2 */
  fprintf(f,"%c%c",y2,y2/256);        /* y2 */
  fprintf(f,"%c%c",maxx,maxx/256);    /* hres */
  fprintf(f,"%c%c",maxy,maxy/256);    /* vres */

  /* skip palette information */

  for (i = 0; i < 48; i++)
    fprintf(f,"%c",0);

  fprintf(f,"%c",0);                          /* vmode */
  fprintf(f,"%c",1);                          /* nplanes */
  fprintf(f,"%c%c",maxbytes,maxbytes/256);    /* bplin */

  /* skip dummy bytes */

  for (i = 0; i < 60; i++)
    fprintf(f,"%c",0);

  /* now encode the picture */

  for (y = y1; y <= y2; y++) {

    x = x1;
    numbytes = 0;
    do {

      /* read a first byte from the screen */

      c = 1;
      b = 255;
      for (i = 0; i < 8; i++)
        if (vgetpixel(x+i,y) != bkcol)
          b ^= vet[i];
      x += 8;
      numbytes++;

      if (numbytes < maxbytes) {

        do {  /* read more bytes */

          b1 = 255;
          for (i = 0; i < 8; i++)
            if (vgetpixel(x+i,y) != bkcol)
              b1 ^= vet[i];
          x += 8;
          c++;
          numbytes++;

        } while ( (c != 63) && (numbytes != maxbytes) && (b1 == b) );

        if (b1 != b) {
          x -= 8;
          c--;
          numbytes--;
        }

      } /* if numbytes... */

      /* now write either count+byte, or just the byte */

      if ( (c == 1) && (b < 192) )
        fprintf(f,"%c",b);
      else {
        fprintf(f,"%c",c | 192);
        fprintf(f,"%c",b);
      }

    } while (numbytes != maxbytes);

  } /* for y */

  fclose(f);

} /* pcxsave() */

/* ----- */

void pcxload(int x0, int y0, char *name)
/* Load a .pcx file. For the moment, no check for the accuracy of input
   file. */
{

  FILE *f;       /* the .pcx file */
  register
  BYTE c,i,j;    /* counters */
  BYTE
       b;        /* byte read from the file */
  int  x,y,      /* counters */
       x1,y1,    /* picture borders */
       x2,y2,
       maxbytes, /* bytes per line */
       numbytes;

  f = fopen(name,"rb");

  /* read the header */

  fscanf(f,"%c",&b);       /* manif - must be 1 */
  fscanf(f,"%c",&b);       /* hard - 3 for mono .pcx */
  fscanf(f,"%c",&b);       /* encod - must be 1 */
  fscanf(f,"%c",&b);       /* bitpx - must be 1 */
  fscanf(f,"%c",&b);       /* x1 */
  x1 = b;
  fscanf(f,"%c",&b);
  x1 += 256*b;
  fscanf(f,"%c",&b);       /* y1 */
  y1 = b;
  fscanf(f,"%c",&b);
  y1 += 256*b;
  fscanf(f,"%c",&b);       /* x2 */
  x2 = b;
  fscanf(f,"%c",&b);
  x2 += 256*b;
  fscanf(f,"%c",&b);       /* y2 */
  y2 = b;
  fscanf(f,"%c",&b);
  y2 += 256*b;
  fscanf(f,"%c",&b);       /* hres - doesn't matter */
  fscanf(f,"%c",&b);
  fscanf(f,"%c",&b);       /* vres - doesn't matter */
  fscanf(f,"%c",&b);

  /* palette information does not matter */

  for (i = 0; i < 48; i++)
    fscanf(f,"%c",&b);

  fscanf(f,"%c",&b);      /* vmode - must be 0 */
  fscanf(f,"%c",&b);      /* nplanes - must be 1 */
  fscanf(f,"%c",&b);      /* bplin */
  maxbytes = b;
  fscanf(f,"%c",&b);
  maxbytes += 256*b;

  /* skip dummy bytes */

  for (i = 0; i < 60; i++)
    fscanf(f,"%c",&b);

  /* now decode the picture */

  for (y = y1; y <= y2; y++) {

    x = x1;
    numbytes = 0;
    do {

      fscanf(f,"%c",&b);
      if (b > 192) { /* it's a counter */
        c = b & 63;
        fscanf(f,"%c",&b);
      }
      else
        c = 1;

      for (i = 0; i < c; i++) { /* plot the byte */
        for (j = 0; j < 8; j++)
          if (!(b & vet[j]))
            vputpixel(x0+x+j,y0+y);
        x += 8;
        numbytes++;
      } /* for i */

    } while (numbytes != maxbytes);

  } /* for y */

  fclose(f);

} /* pcxload() */


/* --- End of file pcxutil.c --- */
