/*  **************************************************************************
 * 
 * --- File: vscrn.c
 * 
 * --- Purpose: implementation of a virtual screen. The "screen" is an array
 *              of bytes, whose bits are the "pixels".
 * 
 * --- Developed with: Pentium PC, Linux 2.0.27, gcc 2.7.2.1
 * 
 * --- Copyright: Guido Gonzato, guido@ibogfs.cineca.it
 * 
 * --- Last updated: October 10, 1997
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * ************************************************************************ */


#include <stdio.h>
#include <stdlib.h>

#ifdef __TURBOC__
#  define HUGE huge
#else
#  define HUGE
#endif

#include "vscrn.h"

/* modify the following #define to increase the screen size */

#define NPIXELS 2048
#define ME      "vscrn"

/* ----- */

/* types and static variables */

typedef unsigned char BYTE;

/* an N * N pixel virtual screen uses N * N / 8 bytes. */

static BYTE vet[8] = { 128, 64, 32, 16, 8, 4, 2, 1 };
static BYTE HUGE vscr [NPIXELS] [NPIXELS / 8];
static int  maxx, maxy;
static int xl, xr, yt, yb;

/* ----- */

void vcleardevice()
/* Clear the virtual screen */
{

  int x, y;

  for (y = 0; y < NPIXELS; y++)
    for (x = 0; x < NPIXELS / 8; x++)
      vscr[y][x] = V_WHITE;

} /* vcleardevice() */

/* ----- */

int vgetmaxx()
/* Return max x coordinate */
{

  return(NPIXELS - 1);

} /* vgetmaxx() */

/* ----- */

int vgetmaxy()
/* Return max y coordinate */
{

  return(NPIXELS - 1);

} /* vgetmaxy() */

/* ----- */

void vputpixel(int x, int y)
/* Plot a pixel on the virtual screen */
{

  if ( (x < 0) || (x > NPIXELS) || (y < 0) || (y > NPIXELS) )
    return;

  switch (x % 8) {

    case 0: vscr[y][x / 8] |= 128;  /* bit 7 */
            break;

    case 1: vscr[y][x / 8] |= 64;
            break;

    case 2: vscr[y][x / 8] |= 32;
            break;

    case 3: vscr[y][x / 8] |= 16;
            break;

    case 4: vscr[y][x / 8] |= 8;
            break;

    case 5: vscr[y][x / 8] |= 4;
            break;

    case 6: vscr[y][x / 8] |= 2;
            break;

    case 7: vscr[y][x / 8] |= 1;  /* bit 0 */

  } /* switch */

} /* vputpixel() */

/* ----- */

int vgetpixel(int x, int y)
/* Check whether a pixel is on */
{

  if ( (x < 0) || (x > NPIXELS) || (y < 0) || (y > NPIXELS) )
    return(0);

  switch (x % 8) {

    case 0:  return(vscr[y][x / 8] & 128);

    case 1:  return(vscr[y][x / 8] & 64);

    case 2:  return(vscr[y][x / 8] & 32);

    case 3:  return(vscr[y][x / 8] & 16);

    case 4:  return(vscr[y][x / 8] & 8);

    case 5:  return(vscr[y][x / 8] & 4);

    case 6:  return(vscr[y][x / 8] & 2);

    case 7:  return(vscr[y][x / 8] & 1);

  }

} /* vgetpixel() */

/* ----- */

void vline(int x1, int y1, int x2, int y2)
/* Draw a line on the virtual screen */
{

  int loop, steps, temp;
  double dx, dy, x, y;

  /* x1 must be < x2 */

  if (x1 > x2) {
    temp = x1;
    x1 = x2;
    x2 = temp;
  }

  if (y1 > y2) {
    temp = y1;
    y1 = y2;
    y2 = temp;
  }

  dx = x2 - x1;
  dy = y2 - y1;
  steps = ( (abs(dx) > abs(dy)) ? (int) dx: (int) dy);
  steps++;

  dx = dx / steps;
  dy = dy / steps;
  x = x1 + 0.5;
  y = y1 + 0.5;

  for (loop = 0; loop < steps; loop++) {
    vputpixel((int) x, (int) y);
    x = x + dx;
    y = y + dy;
  }

  vputpixel((int) x, (int) y);

} /* vline() */

/* ----- */

void initvpcx()
/* Initialise the pcx subsystem variables */
{

  maxx = maxy = NPIXELS + 1;

} /* initvpcx() */

/* ----- */

void vpcxsave(int x1, int y1, int x2, int y2, char *name)
/* Save a virtual screen as a mono .PCX file. */
{

  FILE     *f;        /* the .pcx file */
  register
  BYTE     b,         /* bytes read from the screen */
           b1, i, c;  /* counters */
  register
  int      x, y;      /* for loops */
  int      maxbytes,  /* bytes per line */
           numbytes;  /* counter */

  maxbytes = ((x2 - x1 + 1) / 8 + 0.5);
  if (maxbytes % 2 != 0)  /* if odd */
    maxbytes++;           /* bytes/line should be an even number */
  f = fopen(name, "wb");

  /* write the header */

  fprintf(f, "%c%c%c%c", 10, 3, 1, 1);    /* manif, hard, encod, bitpix */
  fprintf(f, "%c%c", x1, x1 / 256);       /* x1 */
  fprintf(f, "%c%c", y1, y1 / 256);       /* y1 */
  fprintf(f, "%c%c", x2, x2 / 256);       /* x2 */
  fprintf(f, "%c%c", y2, y2 / 256);       /* y2 */
  fprintf(f, "%c%c", maxx, maxx / 256);   /* hres */
  fprintf(f, "%c%c", maxy, maxy / 256);   /* vres */

  /* skip palette information */

  for (i = 0; i < 48; i++)
    fprintf(f, "%c", 0);

  fprintf(f, "%c%c", 0, 1);               /* vmode, nplanes */
  fprintf(f, "%c%c", maxbytes,
                     maxbytes / 256);     /* bplin */

  /* skip dummy bytes */

  for (i = 0; i < 60; i++)
    fprintf(f, "%c", 0);

  /* now encode the picture */

  for (y = y1; y <= y2; y++) {

    x = x1;
    numbytes = 0;
    do {

      /* read a first byte from the screen */

      c = 1;
      b = 255;
      for (i = 0; i < 8; i++)
        if (vgetpixel(x + i, y) /*!= bkcol*/)
          b ^= vet[i];
      x += 8;
      numbytes++;

      if (numbytes < maxbytes) {

        do {  /* read more bytes */

          b1 = 255;
          for (i = 0; i < 8; i++)
            if (vgetpixel(x + i, y) /*!= bkcol*/)
              b1 ^= vet[i];
          x += 8;
          c++;
          numbytes++;

        } while ( (c != 63) && (numbytes != maxbytes) && (b1 == b) );

        if (b1 != b) {
          x -= 8;
          c--;
          numbytes--;
        }

      } /* if numbytes... */

      /* now write either count+byte, or just the byte */

      if ( (c == 1) && (b < 192) )
        fprintf(f, "%c", b);
      else
        fprintf(f, "%c%c", c | 192, b);

    } while (numbytes != maxbytes);

  } /* for y */

  fclose(f);

} /* vpcxsave() */

/* ----- */

void vpcxload(int x0, int y0, char *name)
/* Load a mono .PCX file. No check for the accuracy of input file. */
{

  FILE *f;         /* the .pcx file */
  register
  BYTE c, i, j;    /* counters */
  BYTE
       b, b1;      /* bytes read from the file */
  int  x, y,       /* counters */
       x1, y1,     /* picture borders */
       x2, y2,
       maxbytes,   /* bytes per line */
       numbytes;

  f = fopen(name, "rb");

  /* read the header */

  fscanf(f, "%c%c%c%c",
           &b, &b, &b, &b);   /* manif(1), hard(3), encod(1), bitpix(1) */
  fscanf(f, "%c%c", &b, &b1); /* x1 */
  x1 = b + 256 * b1;
  fscanf(f, "%c%c", &b, &b1); /* y1 */
  y1 = b + 256 * b1;
  fscanf(f, "%c%c", &b, &b1); /* x2 */
  x2 = b + 256 * b1;
  fscanf(f, "%c%c", &b, &b1); /* y2 */
  y2 = b + 256 * b1;
  fscanf(f, "%c%c%c%c", &b, &b, &b, &b);  /* hres & vres - doesn't matter */

  xl = x1;
  xr = x2;
  yt = y1;
  yb = y2;

  /* skip palette information */

  for (i = 0; i < 48; i++)
    fscanf(f, "%c", &b);

  fscanf(f, "%c%c", &b, &b);  /* vmode(0), nplanes(1) */
  fscanf(f, "%c%c", &b, &b1); /* bplin */
  maxbytes = b + 256 * b1;

  /* skip dummy bytes */

  for (i = 0; i < 60; i++)
    fscanf(f, "%c", &b);

  /* now decode the picture */

  for (y = y1; y <= y2; y++) {

    x = x1;
    numbytes = 0;

    do {

      fscanf(f, "%c", &b);
      if (b > 192) { /* it's a counter */
        c = b & 63;
        fscanf(f, "%c", &b);
      }
      else
        c = 1;

      for (i = 0; i < c; i++) { /* plot the byte */
        for (j = 0; j < 8; j++)
          if (!(b & vet[j]))
            vputpixel(x0 + x + j, y0 + y);
        x += 8;
        numbytes++;
      } /* for i */

    } while (numbytes != maxbytes);

  } /* for y */

  fclose(f);

} /* vpcxload() */

/* ----- */

void vpcxdim(int *x1, int *y1, int *x2, int *y2)
/* Return current .pcx dimensions. */
{

  *x1 = xl;
  *x2 = xr;
  *y1 = yt;
  *y2 = yb;

} /* vpcxdim() */

/* ----- */

/* --- End of file vscrn.c --- */
