#include "expression.h"

RCSTAG( "$Id: expression.cpp,v 1.28 1997/12/23 12:34:48 kde Exp $" );

#include <qpainter.h>
#include <qpdevmet.h>
#include <qregexp.h>

//#define __DEBUG

QDict<double>*   dictConstants;
QDict<funcCall>* dictFunctions;

int
parse( QString expression, KNode*& exp )
{
#ifdef __DEBUG
  debug( "expression now: %s", (const char*)expression );
#endif

  if ( !expression.length() ) return err_expressionEmpty;

  exp = static_cast<KNode*>( NULL );
  KNode* expLeft, * expRight;
  QString sLeft, sRight;
  int nError;

  // remove spaces
  expression = expression.stripWhiteSpace();

  // convert 3x into 3*x
  int n1 = 0;
  while ( ( n1 = expression.find( QRegExp( "[0-9][a-z]" ) ) ) != -1 ) {
    expression.insert( n1 + 1, '*' );
#ifdef __DEBUG
    debug( "inserted '*': %s", (const char*)expression );
#endif
  } // while

  // check brackets
  int* bracketLevel = static_cast<int*>( NULL );
  bool firstLevelFound = FALSE;

  while ( !firstLevelFound ) {
    bracketLevel = new int[ expression.length() ];
    CHECK_PTR( bracketLevel );

    int currentLevel = 0;

    uint x;
    for ( x = 0; x < expression.length(); x++ ) {
      if ( expression[ x ] == '(' ) bracketLevel[ x ] = ++currentLevel;
      else if ( expression[ x ] == ')' ) bracketLevel[ x ] = currentLevel--;
      else bracketLevel[ x ] = currentLevel;

      if ( currentLevel < 0 ) {
	delete[] bracketLevel;
	return err_TooManyClosingBrackets;
      } // if

      if ( bracketLevel[ x ] == 0 ) firstLevelFound = TRUE;
    } // for

    if ( currentLevel != 0 ) {
      delete[] bracketLevel;
      return err_ClosingBracketMissing;
    } // if

    if ( !firstLevelFound ) {
      // we have something that looks like this: (3+4+(5*7)), so we remove
      // the outer brackets
      expression.remove( 0, 1 );
      expression.truncate( expression.length() - 1 );

#ifdef __DEBUG
      debug( "removed brackets: %s", (const char*)expression );
#endif

      delete[] bracketLevel;
    } // if
  } // while

  // search for the expression to be evaluated *last*
  int lastExpression;
  int x = expression.length();

  // search for "[+-]" in the string, excluding parts in brackets
  while ( ( ( lastExpression = expression.findRev( QRegExp( "[+-]" ), x ) )
	    != -1 ) && ( bracketLevel[ lastExpression ] > 0 ) ) {
     x = lastExpression - 1;
#ifdef __DEBUG
    debug( "ignored: %c at %i (bracket level %i)",
	   expression[ lastExpression ], lastExpression,
	   bracketLevel[ lastExpression ] );
#endif
  } // while

  if ( lastExpression == 0 ) {
#ifdef __DEBUG
    debug( "sign at position 0: %c", expression[ 0 ] );
#endif

    // there are only higher level expression to the right, so this is a
    // sign

    // ignore +, create 0-... for - sign
    if ( expression[ lastExpression ] == '-' ) {
      exp = new KNodeMinus();
      CHECK_PTR( exp );
      expLeft = new KNodeValue( 0.0 );
      CHECK_PTR( expLeft );
      exp->left = expLeft;
      sRight = expression.right( expression.length() - 1 );

      nError = parse( sRight, expRight );
      if ( nError != err_Ok ) {
	delete[] bracketLevel;
	delete exp;
	exp = static_cast<KNode*>( NULL );
	return nError;
      } // if
      exp->right = expRight;

      delete[] bracketLevel;
      return err_Ok;
    } // if
    else {
      sRight = expression.right( expression.length() -1 );
      
      nError = parse( sRight, exp );
      if ( nError != err_Ok ) {
	delete[] bracketLevel;
	delete exp;
	exp = static_cast<KNode*>( NULL );
	return nError;
      } // if

      delete[] bracketLevel;
      return err_Ok;
    } // else
  } // if
  
  if ( lastExpression != -1 ) {
#ifdef __DEBUG
    debug( "lastExpression = %i: %c", lastExpression,
	   expression[ lastExpression ] );
#endif

    if ( expression[ lastExpression ] == '+' ) exp = new KNodePlus();
    else exp = new KNodeMinus();
    CHECK_PTR( exp );
  
    sLeft = expression.left( lastExpression );
    nError = parse( sLeft, expLeft );
    if ( nError != err_Ok ) {
      delete[] bracketLevel;
      delete exp;
      exp = static_cast<KNode*>( NULL );
      return nError;
    } // if
    exp->left = expLeft;
    
    sRight = expression.right( expression.length() - lastExpression - 1 );
    nError = parse( sRight, expRight );
    if ( nError != err_Ok ) {
      delete[] bracketLevel;
      delete exp;
      exp = static_cast<KNode*>( NULL );
      return nError;
    } // if
    exp->right = expRight;

    delete[] bracketLevel;
    return err_Ok;
  } // if

  // no + or - found
  x = expression.length();
  while ( ( ( lastExpression = expression.findRev( QRegExp( "[*/]" ), x ) )
	    != -1 ) && ( bracketLevel[ lastExpression ] > 0 ) ) {
    x = lastExpression - 1;
#ifdef __DEBUG
    debug( "ignored: %c at %i (bracket level %i)",
	   expression[ lastExpression ], lastExpression,
	   bracketLevel[ lastExpression ] );
#endif
  } // while
  
  if ( lastExpression != -1 ) {
#ifdef __DEBUG
    debug( "lastExpression = %i: %c", lastExpression,
	   expression[ lastExpression ] );
#endif
    
    if ( expression[ lastExpression ] == '*' ) exp = new KNodeMult();
    else exp = new KNodeDiv();
    CHECK_PTR( exp );
    
    sLeft = expression.left( lastExpression );
    nError = parse( sLeft, expLeft );
    if ( nError != err_Ok ) {
      delete[] bracketLevel;
      delete exp;
      exp = static_cast<KNode*>( NULL );
      return nError;
    } // if
    exp->left = expLeft;
    sRight = expression.right( expression.length() - lastExpression - 1 );
    nError = parse( sRight, expRight );
    if ( nError != err_Ok ) {
      delete[] bracketLevel;
      delete exp;
      exp = static_cast<KNode*>( NULL );
      return nError;
    } // if
    exp->right = expRight;

    delete[] bracketLevel;
    return err_Ok;
  } // if

  
  // no +,-,*,/ found
  x = 0;
  while ( ( ( lastExpression = expression.find( '^', x ) )
	    != -1 ) && ( bracketLevel[ lastExpression ] > 0 ) ) {
    x = lastExpression + 1;
#ifdef __DEBUG
    debug( "ignored: %c at %i (bracket level %i)",
	   expression[ lastExpression ], lastExpression,
	   bracketLevel[ lastExpression ] );
#endif
  } // while
  
  if ( lastExpression != -1 ) {
#ifdef __DEBUG
    debug( "lastExpression = %i: %c", lastExpression,
	   expression[ lastExpression ] );
#endif
    
    exp = new KNodePow();
    CHECK_PTR( exp );
    sLeft = expression.left( lastExpression );
    nError = parse( sLeft, expLeft );
    if ( nError != err_Ok ) {
      delete[] bracketLevel;
      delete exp;
      exp = static_cast<KNode*>( NULL );
      return nError;
    } // if
    exp->left = expLeft;
    sRight = expression.right( expression.length() - lastExpression - 1 );
    nError = parse( sRight, expRight );
    if ( nError != err_Ok ) {
      delete[] bracketLevel;
      delete exp;
      exp = static_cast<KNode*>( NULL );
      return nError;
    } // if
    exp->right = expRight;

    delete[] bracketLevel;
    return err_Ok;
  } // if

  // x?
  if ( expression == "x" ) {
    exp = new KNodeX();
    CHECK_PTR( exp );
    delete[] bracketLevel;
    return err_Ok;
  } // if

  // try to see it as a constant
#ifdef __DEBUG
  debug( "looking up constant %s", (const char*)expression );
#endif
  double* constant = (*dictConstants)[ static_cast<const char*>( expression) ];
  if ( constant ) {
#ifdef __DEBUG
    debug( "  - found! = %f", *constant );
#endif
    exp = new KNodeValue( *constant );
    CHECK_PTR( exp );
    delete[] bracketLevel;
    return err_Ok;
  } // if

  // no constant -> function?
  // function calls need brackets. May be changed in later versions!
  KNodeFuncCall* fc;

  int bracket = expression.find( '(' );
  if ( bracket != -1 ) {
    funcCall* f = (*dictFunctions)[
		   static_cast<const char*>( expression.left( bracket ) ) ];
    if ( f ) {
#ifdef __DEBUG
      debug( "function %s",
	     static_cast<const char*>( expression.left( bracket ) ) );
#endif

      // doesn't work with nested function calls! no fix yet.
      if ( expression.contains( ',' ) == f->numParameters - 1 ) {
#ifdef __DEBUG
	debug( "%i parameters", f->numParameters );
#endif
	fc = new KNodeFuncCall;
	CHECK_PTR( fc );

	int startOfParameter = bracket + 1;
	int endOfParameter;
	int y;
	for( y = 0; y < f->numParameters - 1; y++ ) {
	  endOfParameter = expression.find( ',', startOfParameter );
	  sLeft = expression.mid( startOfParameter,
				  endOfParameter - startOfParameter );
#ifdef __DEBUG
	  debug( "Parameter: %s", static_cast<const char*>( sLeft ) );
#endif

	  nError = parse( sLeft, expLeft );
	  if ( nError != err_Ok ) {
	    delete[] bracketLevel;
	    delete exp;
	    exp = static_cast<KNode*>( NULL );
	    return nError;
	  } // if
	  fc->param[ y ] = expLeft;
	  startOfParameter = endOfParameter + 1;
	} // for

	endOfParameter = expression.findRev( ')' );
	sLeft = expression.mid( startOfParameter,
				endOfParameter - startOfParameter );
#ifdef __DEBUG
	debug( "Parameter: %s", (const char*)sLeft );
#endif

	nError = parse( sLeft, expLeft );
	if ( nError != err_Ok ) {
	  delete[] bracketLevel;
	  delete exp;
	  exp = static_cast<KNode*>( NULL );
	  return nError;
	} // if
	fc->param[ f->numParameters - 1 ] = expLeft;
	fc->function = *f;

	exp = fc;

	delete[] bracketLevel;
	return err_Ok;
      } // if ( contains... )
      else {
	delete[] bracketLevel;
	return err_wrongNumberOfParameters;
      }
    } // if( f )
    else {
      delete[] bracketLevel;
      return err_funcCallUnknown;
    } // else
  } // if

  bool f;
  double d = expression.toDouble( &f );
  delete[] bracketLevel;
  if ( f ) {
    exp = new KNodeValue( d );
    CHECK_PTR( exp );
    return err_Ok;
  } // if
  return err_notANumber;
} // parse



KNode::KNode()
{
  left = right = static_cast<KNode*>( NULL );
}

double
KNode::getValue( double ) const
{
  warning(
  "internal error: KNode::getValue called, please contact Ritzert@T-Online.de"
  );
  return -1;
} // KNode::getValue


KNode::~KNode()
{
  if ( left ) delete left;
  if ( right ) delete right;
}


double
KNodeFuncCall::getValue( double x ) const
{
  switch( function.numParameters ) {
  case 2: {
    return (*function.twoParams)( param[0]->getValue( x ),
				  param[1]->getValue( x ) );
  }
  case 3: {
    return (*function.threeParams)( param[0]->getValue( x ),
				    param[1]->getValue( x ),
				    param[2]->getValue( x ) );
  }
  default: return (*function.oneParam)( param[0]->getValue( x ) );
  } // switch

  return -1;
} // KNodeFuncCall::getValue


KExpression::KExpression()
{
  expression = static_cast<KNode*>( NULL );
} // KExpression constructor ()

KExpression::KExpression( QString exp )
{
  expression = static_cast<KNode*>( NULL );
  parse( exp );
} // KExpression constructor ( QString )


KExpression::~KExpression()
{
  if ( expression ) delete expression;
} // KExpression destructor


int
KExpression::parse( QString exp )
{
  if ( expression ) delete expression;
  return ::parse( exp, expression );
} // KExpression::parse


void
KExpression::plot( QPaintDevice* pd, QPainter* p, QPen pen,
		   double minX, double minY, double maxX, double maxY )
{
#ifdef __DEBUG
  debug( "KExpression::plot called!" );
#endif

  if ( !expression ) return;

  QPaintDeviceMetrics pdm( pd );
  p->setPen( pen );

  double deltaX = ( maxX - minX ) / pdm.width();
  double deltaY = ( maxY - minY ) / pdm.height();

#ifdef __DEBUG
  debug( "deltaX = %f, deltaY = %f", deltaX, deltaY );
#endif

  double x = minX;
  double y = getValue( x );
  int y_screen = pdm.height() - static_cast<int>( ( y - minY ) / deltaY ) - 1;

  p->moveTo( 0, y_screen );

  int x_screen;
  int w = pdm.width();
  int h = pdm.height();

  bool outOfScreen = FALSE;
  bool NaN = FALSE;

#ifdef __DEBUG
  debug( "start painting" );
#endif

  for ( x_screen = 1; x_screen < w; x_screen++ ) {
    x += deltaX;
    y = getValue( x );
    y_screen = h - static_cast<int>( ( y - minY ) / deltaY ) - 1;

    if ( isnan( y ) ) {
      NaN = TRUE;
      outOfScreen = FALSE;
    } // if
    else if  ( ( y_screen > h ) || ( y_screen < 0 ) ) {
      if ( !outOfScreen ) p->lineTo( x_screen, y_screen );
      else p->moveTo( x_screen, y_screen );
      outOfScreen = TRUE;
      NaN = FALSE;
    } // else if
    else {
      if( NaN ) p->moveTo( x_screen, y_screen );
      else p->lineTo( x_screen, y_screen );
      outOfScreen = NaN = FALSE;
    } // else
  } // for

#ifdef __DEBUG
  debug( "end painting" );
#endif
} // KExpression::plot
