// included from expression.cpp

void
count( KNode* t, QDict<double>* dict, QList<KNode>* mult, QList<KNode>* div,
       bool isDiv );


bool
KNodeMult::isConst() const
{
  bool leftConst = left->isConst();
  bool rightConst = right->isConst();
  if( leftConst && rightConst ) return TRUE;

  QDict<double> dict; dict.setAutoDelete( TRUE );
  dict.insert( "x", new double( 1.0 ) );
  if( leftConst && left->getValue( &dict ) == 0.0 ) return TRUE;
  if( rightConst && right->getValue( &dict ) == 0.0 ) return TRUE;

  return FALSE;
} // KNodeMult::isConst


bool
KNodeDiv::isConst() const
{
  bool leftConst = left->isConst();
  bool rightConst = right->isConst();
  if( leftConst && rightConst ) return TRUE;

  QDict<double> dict; dict.setAutoDelete( TRUE );
  dict.insert( "x", new double( 1.0 ) );
  if( leftConst && left->getValue( &dict ) == 0.0 ) return TRUE;
  return FALSE;
} // KNodeDiv::isConst


bool
KNodeFuncCall::isConst() const
{
  bool ret = TRUE;
  int n;
  for( n = 0; n < function->numParameters; n++ )
    ret = ret && param[ n ]->isConst();
  return ret;
} // KNodeFuncCall::isConst


bool
KNodePow::isConst() const
{
  bool leftConst = left->isConst();
  bool rightConst = right->isConst();
  if( leftConst && rightConst ) return TRUE;

  QDict<double> dict; dict.setAutoDelete( TRUE );
  dict.insert( "x", new double( 1.0 ) );
  if ( rightConst && right->getValue( &dict ) == 0 ) return TRUE;
  if ( leftConst && ( left->getValue( &dict ) == 0
		      || left->getValue( &dict ) == 1 ) ) return TRUE;
  return FALSE;
} // KNodePow::isConst


void
KExpression::simplify()
{
  KNode* n;
  if ( ( n = expression->simplify() ) ) {
    delete expression;
    expression = n;
  } // if
} // KExpression::simplify


KNode*
KNode::simplify()
{
  // first simplify our "children"
  KNode* new_left = left ? left->simplify() : static_cast<KNode*>( NULL );
  if ( new_left ) { delete left; left = new_left; }
  KNode* new_right = right ? right->simplify() : static_cast<KNode*>( NULL );
  if ( new_right ) { delete right; right = new_right; }

  // new simplify ourselves
  if( isConst() ) {
    QDict<double> dict; dict.setAutoDelete( TRUE );
    dict.insert( "x", new double( 1.0 ) );
    KNode* ret = new KNodeValue( getValue( &dict ) );
    return ret;
  } // if
  else return static_cast<KNode*>( NULL );
} // KNode::simplify


KNode*
KNodePlus::simplify()
{
  KNode* n = KNode::simplify();
  if( n ) return n;

  QDict<double> dict; dict.setAutoDelete( TRUE );
  dict.insert( "x", new double( 1.0 ) );
  if( left->isConst() && left->getValue( &dict ) == 0.0 )
    return right->copy();
  if( right->isConst() && right->getValue( &dict ) == 0.0 )
    return left->copy();

  return static_cast<KNode*>( NULL );
} // KNodePlus::simplify


KNode*
KNodeMinus::simplify()
{
  KNode* n = KNode::simplify();
  if( n ) return n;

  QDict<double> dict; dict.setAutoDelete( TRUE );
  dict.insert( "x", new double( 1.0 ) );
  if( left->isConst() && left->getValue( &dict ) == 0.0 ) {
    KNodeMult* mult = new KNodeMult;
    mult->left = new KNodeValue( -1.0 );
    mult->right = right->copy();
    return mult;
  } // if
  if( right->isConst() && right->getValue( &dict ) == 0.0 )
    return left->copy();

  return static_cast<KNode*>( NULL );
} // KNodeMinus::simplify


KNode*
KNodeMult::simplify()
{
  KNode* a = KNode::simplify();
  if( a ) return a;

  QDict<double> dict; QList<KNode> mult; QList<KNode> div;
  dict.setAutoDelete( TRUE );
  count( this, &dict, &mult, &div, FALSE );

  double* d = dict[ "%value%" ];
  if ( d && *d == 0.0 ) return new KNodeValue( 0.0 );
  if ( d && *d == 1.0 ) {
    dict.remove( "%value%" );
    d = static_cast<double*>( NULL );
  } // if
  if ( d ) {
    mult.insert( 0, new KNodeValue( *d ) );
    dict.remove( "%value%" );
  } // if

  QDictIterator<double> it( dict );
  while( it.current() ) {
    if( *it.current() == 1.0 ) {
      mult.append( new KNodeVar( it.currentKey() ) );
    } // if
    else if ( *it.current() != 0 ) { // a^0 = 1 => can be ignored
      KNodePow* pow = new KNodePow;
      pow->left = new KNodeVar( it.currentKey() );
      pow->right = new KNodeValue( *it.current() );
      mult.append( pow );
    } // else

    ++it;
  } // while

  if ( mult.count() + div.count() == 0 ) return new KNodeValue( 1.0 );

  if ( div.count() > 0 ) {
    KNodeDiv* ndiv = new KNodeDiv;
    CHECK_PTR( ndiv );
    ndiv->left = new KNodeValue( 1.0 );
    CHECK_PTR( ndiv->left );

    uint n;
    KNode* temp = ndiv;
    for( n = 0; n < div.count() - 1; n++ ) {
      temp->right = new KNodeMult;
      CHECK_PTR( temp->right );
      temp = temp->right;
      temp->left = div.at( n );
    } // for
    temp->right = div.at( div.count() - 1 );

    mult.append( ndiv );
  } // if

  if ( mult.count() == 1 ) return mult.at( 0 );
  uint n;
  KNode* nmult = new KNodeMult;
  nmult->left = mult.at( 0 );
  KNode* temp = nmult;
  for( n = 1; n < mult.count() - 1; n++ ) {
    temp->right = new KNodeMult;
    CHECK_PTR( temp->right );
    temp = temp->right;
    temp->left = mult.at( n );
  } // for
  temp->right = mult.at( mult.count() - 1 );
  return nmult;
} // KNodeMult::simplify


KNode*
KNodeDiv::simplify()
{
  KNode* a = KNode::simplify();
  if( a ) return a;

  QDict<double> dict; QList<KNode> mult; QList<KNode> div;
  dict.setAutoDelete( TRUE );
  count( this, &dict, &mult, &div, FALSE );

  double* d = dict[ "%value%" ];
  if ( d && *d == 0.0 ) return new KNodeValue( 0.0 );
  if ( d && *d == 1.0 ) {
    dict.remove( "%value%" );
    d = static_cast<double*>( NULL );
  } // if
  if ( d ) {
    mult.insert( 0, new KNodeValue( *d ) );
    dict.remove( "%value%" );
  } // if

  QDictIterator<double> it( dict );
  while( it.current() ) {
    if( *it.current() == 1.0 ) {
      mult.append( new KNodeVar( it.currentKey() ) );
    } // if
    else if ( *it.current() != 0 ) { // a^0 = 1 => can be ignored
      KNodePow* pow = new KNodePow;
      pow->left = new KNodeVar( it.currentKey() );
      pow->right = new KNodeValue( *it.current() );
      mult.append( pow );
    } // else

    ++it;
  } // while

  if ( mult.count() + div.count() == 0 )  return new KNodeValue( 1.0 );

  if ( div.count() > 0 ) {
    KNodeDiv* ndiv = new KNodeDiv;
    CHECK_PTR( ndiv );
    ndiv->left = new KNodeValue( 1.0 );
    CHECK_PTR( ndiv->left );

    uint n;
    KNode* temp = ndiv;
    for( n = 0; n < div.count() - 1; n++ ) {
      temp->right = new KNodeMult;
      CHECK_PTR( temp->right );
      temp = temp->right;
      temp->left = div.at( n );
    } // for
    temp->right = div.at( div.count() - 1 );

    mult.append( ndiv );
  } // if

  if ( mult.count() == 1 ) return mult.at( 0 );
  uint n;
  KNode* nmult = new KNodeMult;
  nmult->left = mult.at( 0 );
  KNode* temp = nmult;
  for( n = 1; n < mult.count() - 1; n++ ) {
    temp->right = new KNodeMult;
    CHECK_PTR( temp->right );
    temp = temp->right;
    temp->left = mult.at( n );
  } // for
  temp->right = mult.at( mult.count() - 1 );

  return nmult;
} // KNodeDiv::simplify


void
count( KNode* t, QDict<double>* dict, QList<KNode>* mult, QList<KNode>* div,
       bool isDiv )
{
  QDict<double> x; x.setAutoDelete( TRUE ); x.insert( "x", new double( 1.0 ) );
  if ( t->left->type() == "*" || t->left->type() == "/" )
    count( t->left, dict, mult, div, isDiv );
  else if( t->left->type() == "var" ) {
    QString sVarName = static_cast<KNodeVar*>( t->left )->sVarName;
    double* d = (*dict)[ sVarName ];
    if ( !d ) { d = new double( 0.0 ); dict->insert( sVarName, d ); }
    (*d) += isDiv ? -1 : 1;
  } // else if
  else if( t->left->type() == "value" ) {
    double* d = (*dict)[ "%value%" ];
    if ( !d ) { d = new double( 1.0 ); dict->insert( "%value%", d ); }
    if ( isDiv ) (*d) /= t->left->getValue( &x );
    else (*d) *= t->left->getValue( &x );
  } // else if
  else if( t->left->type() == "^" && t->left->right->type() == "value" &&
	   t->left->left->type() == "var" ) {
      QString sVarName = static_cast<KNodeVar*>( t->left->left )->sVarName;
      double* d = (*dict)[ sVarName ];
      if ( !d ) { d = new double( 0.0 ); dict->insert( sVarName, d ); }
      (*d) += isDiv ? -t->left->right->getValue( &x )
	: t->left->right->getValue( &x );
  } // else if
  else {
    if ( isDiv ) div->append( t->left->copy() );
    else mult->append( t->left->copy() );
  } // else

  if( t->type() == "/" ) isDiv = !isDiv;

  if ( t->right->type() == "*" || t->right->type() == "/" )
    count( t->right, dict, mult, div, isDiv );
  else if( t->right->type() == "var" ) {
    QString sVarName = static_cast<KNodeVar*>( t->right )->sVarName;
    double* d = (*dict)[ sVarName ];
    if ( !d ) { d = new double( 0.0 ); dict->insert( sVarName, d ); }
    (*d) += isDiv ? -1 : 1;
  } // else if
  else if( t->right->type() == "value" ) {
    double* d = (*dict)[ "%value%" ];
    if ( !d ) { d = new double( 1.0 ); dict->insert( "%value%", d ); }
    if ( isDiv ) (*d) /= t->right->getValue( &x );
    else (*d) *= t->right->getValue( &x );
  } // else if
  else if( t->right->type() == "^" && t->right->right->type() == "value" &&
	   t->right->left->type() == "var" ) {
      QString sVarName = static_cast<KNodeVar*>( t->right->left )->sVarName;
      double* d = (*dict)[ sVarName ];
      if ( !d ) { d = new double( 0.0 ); dict->insert( sVarName, d ); }
      (*d) += isDiv ? -t->right->right->getValue( &x )
	: t->right->right->getValue( &x );
  } // else if
  else {
    if ( isDiv ) div->append( t->right->copy() );
    else mult->append( t->right->copy() );
  } // else
} // count


KNode*
KNodePow::simplify()
{
  KNode* a = KNode::simplify();
  if( a ) return a;

  if ( right->type() == "value" && left->type() == "^"
       && left->right->type() == "value" ) {
    QDict<double> x;

    KNodePow* ret = new KNodePow;
    CHECK_PTR( ret );
    ret->left = left->left->copy();
    ret->right = new KNodeValue( left->right->getValue( &x )
				 * right->getValue( &x ) );
    return ret;
  } // if

  return static_cast<KNode*>( NULL );
} // KNodePow::simplify

