/*
    Mplot++ : a math plotter for Unix(R)/Windows(R)/MacOS X(R) -
              - version 0.78     
    Copyright (C)  2002    Ivano Primi ( ivano.primi@tin.it )    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    You can contact the author of this software by paper mail writing to
    the next address

	Ivano Primi
	via Colle Cannetacce 50/A
	C.A.P. 00038 - Valmontone (ROMA)
	Italy                                                          .

    If you prefer the electronic mail you can write to the address

	ivano.primi@tin.it                                             .
*/

#include<cmath>
#include"graphut.h"

float absv(float x)
{
  return ((x>=0) ? x: -x);
}

float xsq(float x, float y, float s)
{
  //I must suppose (x,y) != (0,0)
  double alpha=atan2((double)y,(double)x);
  
  if( (alpha >= -PI/4) && (alpha < PI/4) )
    return s;
  else if( (alpha >= PI/4) && (alpha < 3*PI/4) )
    return s*(cos(alpha)/sin(alpha));
  else if( (alpha >= -3*PI/4) && (alpha < -PI/4) )
    return -s*(cos(alpha)/sin(alpha));
  else
    return -s;
}
      
float ysq(float x, float y, float s)
{
  //I must suppose (x,y) != (0,0)
  double alpha=atan2((double)y,(double)x);
  
  if( (alpha >= -PI/4) && (alpha < PI/4) )
    return s*tan(alpha);
  else if( (alpha >= PI/4) && (alpha < 3*PI/4) )
    return s;
  else if( (alpha >= -3*PI/4) && (alpha < -PI/4) )
    return -s;
  else
    return -s*tan(alpha);
}

static double fogfunction(double d, double fog)
{
  double a=pow(7,2*fog*fog)-1;

  return (0.5-atan(a*d/6)/PI);
}

static unsigned char whatcolor(const struct Param3d* par, const struct point p)
{
  double lambda=par->lambda*PI/180;
  double phi=par->phi*PI/180;
  double d=cos(lambda)*cos(phi)*p.x+sin(lambda)*cos(phi)*p.y+sin(phi)*p.z;
  //  double d=cos(lambda)*cos(phi)*(p.x-par->x0)+sin(lambda)*cos(phi)*(p.y-par->y0)+sin(phi)*(p.z-par->z0);

  return (unsigned char)( (NGRAYS-1)*fogfunction(d,par->fog) );
}

/* Procedura per la rappresentazione in assonometria di punti dello spazio */

struct point Assonometria(const struct Param3d* par, const struct point p)
{
  struct point P;
  double X,Y;
  double lambda=par->lambda*PI/180;
  double phi=par->phi*PI/180;
  double mu=par->mu*PI/180;
   
  if(p.color==127) // this special value is used to signal points you must not
    return p;      // draw
  X=cos(lambda)*(p.y-par->y0)-sin(lambda)*(p.x-par->x0);
  Y=(p.z-par->z0)*cos(phi)-sin(phi)*(cos(lambda)*(p.x-par->x0)+sin(lambda)*(p.y-par->y0));
  P.x=(float)( cos(mu)*X+sin(mu)*Y );
  P.y=(float)( -sin(mu)*X+cos(mu)*Y );
  P.z=0;
  P.color=whatcolor(par,p); //TODO
  return P; 
}

// Look out! The point (x0,y0,z0) of the space is sent in the point (0,0) of
// the plane


float norm(float x,float y)
{
  return (float)sqrt((double)(x*x+y*y));
}

int colorcmp(const void* p1,const void* p2)
{
  const struct element* e1=(struct element*)p1;
  const struct element* e2=(struct element*)p2;

  if(e1->key > e2->key)
    return -1;
  else if (e1->key < e2->key)
    return 1;
  else
    return 0;
}

inline int iszero(float x)
{
  return( ((x < epsilon) && (x > -epsilon)) ? 1:0 );
} 

inline int isvectornull(float v[],unsigned long n)
{
  unsigned long i;
  float norm2=0;

  for(i=0 ; i<n ; i++)
    norm2+=v[i]*v[i];
  return( (norm2 < n*epsilon*epsilon) ? 1:0 );
}

inline int isin01(float t)
{
  return ( ((t>= 0) && (t<= 1)) ? 1:0 );
}

int isout(float x, float y, struct Rect R)
{
  if( absv(x-R.x0) <= R.a  && absv(y-R.y0) <= R.b )
    return 0; //Yes, we are inside
  else
    return 1; // No, we are outside
}

// (xin,yin) is a point inside  R
//     (x,y) is a point outside R
static int where(float x, float y, struct Rect R, float xin, float yin)
{
  double r,l,t,b;
  double a1,a2,a3,a4,a;

  r= (double)(R.x0+R.a-xin); // r >= 0
  t= (double)(R.y0+R.b-yin); // t >= 0
  l= (double)(R.x0-R.a-xin); // l <= 0
  b= (double)(R.y0-R.b-yin); // b <= 0
  a = atan2((double)(y-yin),(double)(x-xin));
  if( iszero(r) && iszero(t) )
    {
      a1= atan2((double)R.b,(double)R.a); // 0 < a1 < PI/2
      a2= PI;
      a3= a1-PI;
      a4= -PI/2;
      if( (a > a3) && (a <= a4+epsilon) )
	return -2;
      else if( (a > a4+epsilon) && (a <= a1) )
	return 1;
      else if( (a > a1) && (a < a2-epsilon) )
	return 2;
      else
	return -1;
    }
  else if( iszero(l) && iszero(t) )
    {
      a1= 0;
      a2= atan2((double)R.b,(double)-R.a); // PI/2 < a2 < PI
      a3= -PI/2;
      a4= a2-PI;
      if( (a >= a3-epsilon) && (a <= a4) )
	return -2;
      else if( (a > a4) && (a <= a1+epsilon) )
	return 1;
      else if( (a > a1+epsilon) && (a <= a2) )
	return 2;
      else
	return -1;
    }
  else if( iszero(l) && iszero(b) )
    {
      a1= atan2((double)R.b,(double)R.a); // 0 < a1 < PI/2
      a2= PI/2;
      a3= a1-PI;
      a4= 0;
      if( (a >= a3) && (a < a4-epsilon) )
	return -2;
      else if( (a >= a4-epsilon) && (a < a1) )
	return 1;
      else if( (a >= a1) && (a <= a2+epsilon) )
	return 2;
      else
	return -1;      
    }
  else if( iszero(r) && iszero(b) )
    {
      a1= PI/2;
      a2= atan2((double)R.b,(double)-R.a); // PI/2 < a2 < PI
      a3= -PI;
      a4= a2-PI;
      if( (a > a3+epsilon) && (a < a4) )
	return -2;
      else if( (a >= a4) && (a < a1-epsilon) )
	return 1;
      else if( (a >= a1-epsilon) && (a < a2) )
	return 2;
      else
	return -1;      
    }
  else
    {
      a1= atan2(t,r); //  0    <= a1 <=  PI/2 
      a2= atan2(t,l); //  PI/2 <= a2 <=  PI
      if( b > -epsilon*epsilon )
	a3= -PI;
      else
	a3= atan2(b,l); // -PI   <= a3 <= -PI/2
      a4= atan2(b,r); // -PI/2 <= a4 <= 0
      if( (a >= a3) && (a < a4) )
	return -2;
      else if( (a >= a4) && (a < a1) )
	return 1;
      else if( (a >= a1) && (a < a2) )
	return 2;
      else
	return -1;
    }
}


// Trova le intersezioni della linea retta r con la prima diagonale
// del rettangolo R; restituisce 0 quando r ha intersezione vuota con
// questa diagonale, 1 quando la interseca in uno ed un sol punto,
// 2 quando r contiene per intero la prima diagonale di R.
// In questi 2 ultimi casi *t contiene un valore di t per il quale si ha che 
// (r.x0 + r.l*t, r.y0+ r.m*t) e` un punto della prima diagonale di R.
// La funzione restituisce -1 quando r degenera in un punto (l=m=0).

static int rcap1d(struct sline r,struct Rect R, float* t)
{
  float det= 2*R.b*r.l+2*R.a*r.m;
  float s;

  if( iszero(r.l) && iszero(r.m) )
    return -1;
  else if( iszero(det) ) // r has the same direction as the 1st diagonal of R
    {
      *t= ( !iszero(r.l) ) ? (R.x0-r.x0)/r.l : (R.y0-r.y0)/r.m;
      if( iszero(r.m*(R.x0-r.x0)-r.l*(R.y0-r.y0)) )
	return 2;
      else
	return 0;
    }
  else
    {
      *t= ( 2*R.b*(R.x0-r.x0-R.a)+2*R.a*(R.y0-r.y0+R.b) )/det;
      s = ( r.l*(R.y0-r.y0+R.b)-r.m*(R.x0-r.x0-R.a) )/det;
      if( isin01(s) )
	return 1;
      else
	return 0;
    }
}


// Trova le intersezioni della linea retta r con la seconda diagonale
// del rettangolo R; restituisce 0 quando r ha intersezione vuota con
// questa diagonale, 1 quando la interseca in uno ed un sol punto,
// 2 quando r contiene per intero la seconda diagonale di R.
// In questi 2 ultimi casi *t contiene un valore di t per il quale si ha che 
// (r.x0 + r.l*t, r.y0+ r.m*t) e` un punto della seconda diagonale di R.
// La funzione restituisce -1 quando r degenera in un punto (l=m=0).

static int rcap2d(struct sline r,struct Rect R, float* t)
{
  float det= -2*R.b*r.l+2*R.a*r.m;
  float s;

  if( iszero(r.l) && iszero(r.m) )
    return -1;
  else if( iszero(det) ) // r has the same direction as the 2nd diagonal of R
    {
      *t= ( !iszero(r.l) ) ? (R.x0-r.x0)/r.l : (R.y0-r.y0)/r.m;
      if( iszero(r.m*(R.x0-r.x0)-r.l*(R.y0-r.y0)) )
	return 2;
      else
	return 0;
    }
  else
    {
      *t= ( -2*R.b*(R.x0-r.x0-R.a)+2*R.a*(R.y0-r.y0-R.b) )/det;
      s= ( r.l*(R.y0-r.y0-R.b)-r.m*(R.x0-r.x0-R.a) )/det;
      if( isin01(s) )
	return 1;
      else
	return 0;
    }
}


// Trova le intersezioni del segmento [P1,P2] con la prima diagonale del
// rettangolo R; restituisce 0 quando [P1,P2] ha intersezione vuota con
// questa diagonale, 1 quando la interseca in uno ed un sol punto,
// 2 quando l' intersezione di [P1,P2] con la prima diagonale di R e` un
// intero segmento. Nel secondo caso, *Q1=*Q2 e` il punto di intersezione di
// [P1,P2] con la prima diagonale di R, nel terzo *Q1 e *Q2 sono i vertici
// del segmento intersezione.

static int segcap1d(struct point P1, struct point P2, struct Rect R, struct point* Q1,
	     struct point* Q2)
{
  struct sline r;
  float t,t1,t2;

  r.x0=P1.x;
  r.y0=P1.y;
  r.l =P2.x-P1.x;
  r.m =P2.y-P1.y;
  switch( rcap1d(r,R,&t) )
    {
    case -1: //P1 == P2
      if( iszero(R.b*(r.x0-R.x0)-R.a*(R.y0-r.y0)) )
	{
	  if( isin01( (r.x0-R.x0+R.a)/(2*R.a) ) )
	    {
	      // Then P1==P2 belongs to the 1st diagonal
	      Q2->x=Q1->x=r.x0;
	      Q2->y=Q1->y=r.y0;
	      return 1;
	    }
	  else
	    return 0;
	}
      else
	return 0;
    case 0:
      return 0; // Empty intersection
    case 1:
      if( isin01(t) ) // [P1,P2] cap 1d == one only point
	{
	  Q2->x=Q1->x=r.x0+r.l*t;
	  Q2->y=Q1->y=r.y0+r.m*t;
	  return 1;
	}
      else
	return 0;
    case 2:
      // La retta [P1,P2] contiene per intero la prima
      // diagonale di R. Innanzitutto calcoliamo i tempi
      // di passaggio di questa retta per gli estremi
      // della diagonale.
      if( iszero(r.l) )
	{
	  t1=(R.y0-r.y0+R.b)/r.m;
	  t2=(R.y0-r.y0-R.b)/r.m;
	}
      else
	{
	  t1=(R.x0-r.x0-R.a)/r.l;
	  t2=(R.x0-r.x0+R.a)/r.l;
	}
      if( isin01(t1) && isin01(t2) )
	{
	  // Gli estremi della diagonale sono contenuti nel
	  // segmento [P1,P2]
	  Q1->x= r.x0+r.l*t1;
	  Q1->y= r.y0+r.m*t1;
	  Q2->x= r.x0+r.l*t2;
	  Q2->y= r.y0+r.m*t2;
	  return 2;
	}
      else if( isin01(t1) ) // Solo uno degli estremi della
	{ // diagonale e` contenuto in [P1,P2]
	  Q1->x= r.x0+r.l*t1;
	  Q1->y= r.y0+r.m*t1;
	  if( t2 < 0 ) // l'altro precede P1
	    {
	      Q2->x= P1.x;
	      Q2->y= P1.y;
	    }
	  else // l'altro segue P2
	    {
	      Q2->x= P2.x;
	      Q2->y= P2.y;
	    }
	  return 2;
	}
      else if( isin01(t2) ) // Solo uno degli estremi della
	{ // diagonale e` contenuto in [P1,P2]
	  Q1->x= r.x0+r.l*t2;
	  Q1->y= r.y0+r.m*t2;
	  if( t1 < 0 ) // l'altro precede P1
	    {
	      Q2->x= P1.x;
	      Q2->y= P1.y;
	    }
	  else // l'altro segue P2
	    {
	      Q2->x= P2.x;
	      Q2->y= P2.y;
	    }
	  return 2;
	}
      else // Gli estremi della diagonale si trovano al di fuori
	{  // di [P1,P2]
	  if( (t1<0 && t2<0) || (t1>1 && t2>1) )
	    // Si trovano entrambi prima di P1 o dopo P2
	    return 0;
	  else if( (t1<0 && t2>1) || (t1>1 && t2<0) )
	    {
	      // P1 e P2 sono compresi tra gli estremi della diagonale
	      Q1->x= P1.x;
	      Q1->y= P1.y;
	      Q2->x= P2.x;
	      Q2->y= P2.y;
	      return 2;
	    }
	}
    } // end switch(...)
}


// Trova le intersezioni del segmento [P1,P2] con la seconda diagonale del
// rettangolo R; restituisce 0 quando [P1,P2] ha intersezione vuota con
// questa diagonale, 1 quando la interseca in uno ed un sol punto,
// 2 quando l' intersezione di [P1,P2] con la seconda diagonale di R e` un
// intero segmento. Nel secondo caso, *Q1=*Q2 e` il punto di intersezione di
// [P1,P2] con la seconda diagonale di R, nel terzo *Q1 e *Q2 sono i vertici
// del segmento intersezione.

static int segcap2d(struct point P1, struct point P2, struct Rect R, struct point* Q1,
	     struct point* Q2)
{
  struct sline r;
  float t,t1,t2;

  r.x0=P1.x;
  r.y0=P1.y;
  r.l =P2.x-P1.x;
  r.m =P2.y-P1.y;
  switch( rcap2d(r,R,&t) )
    {
    case -1: //P1 == P2
      if( iszero(R.b*(r.x0-R.x0)-R.a*(r.y0-R.y0)) )
	{
	  if( isin01( (r.x0-R.x0+R.a)/(2*R.a) ) )
	    {
	      // Then P1==P2 belongs to the 1st diagonal
	      Q2->x=Q1->x=r.x0;
	      Q2->y=Q1->y=r.y0;
	      return 1;
	    }
	  else
	    return 0;
	}
      else
	return 0;
    case 0:
      return 0; // Empty intersection
    case 1:
      if( isin01(t) ) // [P1,P2] cap 2d == one only point
	{
	  Q2->x=Q1->x=r.x0+r.l*t;
	  Q2->y=Q1->y=r.y0+r.m*t;
	  return 1;
	}
      else
	return 0;
    case 2:
      // La retta [P1,P2] contiene per intero la prima
      // diagonale di R. Innanzitutto calcoliamo i tempi
      // di passaggio di questa retta per gli estremi
      // della diagonale.
      if( iszero(r.l) )
	{
	  t1=(R.y0-r.y0-R.b)/r.m;
	  t2=(R.y0-r.y0+R.b)/r.m;
	}
      else
	{
	  t1=(R.x0-r.x0-R.a)/r.l;
	  t2=(R.x0-r.x0+R.a)/r.l;
	}	
      if( isin01(t1) && isin01(t2) )
	{
	  // Gli estremi della diagonale sono contenuti nel
	  // segmento [P1,P2]
	  Q1->x= r.x0+r.l*t1;
	  Q1->y= r.y0+r.m*t1;
	  Q2->x= r.x0+r.l*t2;
	  Q2->y= r.y0+r.m*t2;
	  return 2;
	}
      else if( isin01(t1) ) // Solo uno degli estremi della
	{ // diagonale e` contenuto in [P1,P2]
	  Q1->x= r.x0+r.l*t1;
	  Q1->y= r.y0+r.m*t1;
	  if( t2 < 0 ) // l'altro precede P1
	    {
	      Q2->x= P1.x;
	      Q2->y= P1.y;
	    }
	  else // l'altro segue P2
	    {
	      Q2->x= P2.x;
	      Q2->y= P2.y;
	    }
	  return 2;
	}
      else if( isin01(t2) ) // Solo uno degli estremi della
	{ // diagonale e` contenuto in [P1,P2]
	  Q1->x= r.x0+r.l*t2;
	  Q1->y= r.y0+r.m*t2;
	  if( t1 < 0 ) // l'altro precede P1
	    {
	      Q2->x= P1.x;
	      Q2->y= P1.y;
	    }
	  else // l'altro segue P2
	    {
	      Q2->x= P2.x;
	      Q2->y= P2.y;
	    }
	  return 2;
	}
      else // Gli estremi della diagonale si trovano al di fuori
	{  // di [P1,P2]
	  if( (t1<0 && t2<0) || (t1>1 && t2>1) )
	    // Si trovano entrambi prima di P1 o dopo P2
	    return 0;
	  else if( (t1<0 && t2>1) || (t1>1 && t2<0) )
	    {
	      // P1 e P2 sono compresi tra gli estremi della diagonale
	      Q1->x= P1.x;
	      Q1->y= P1.y;
	      Q2->x= P2.x;
	      Q2->y= P2.y;
	      return 2;
	    }
	}
    } // end switch(...)
}


// The 2 following functions return the y-coordinate (x-coordinate)
// of the intersection of the segment (x1, y1, x2, y2) with the line
// {x=c} ({y=c}).

static float x_intersection(float x1, float y1, float x2, float y2, float c)
{
  //Here I can suppose x2 != x1

  return( y1+(y2-y1)*((c-x1)/(x2-x1)) );
}

static float y_intersection(float x1, float y1, float x2, float y2, float c)
{
  //Here I can suppose y2 != y1

  return( x1+(x2-x1)*((c-y1)/(y2-y1)) );
}

int 
intersect(struct point P1,struct point P2, struct Rect R, struct point* Q1, struct point* Q2)
{
  struct sline r;
  int is1out, is2out;
  float aux,auy;

  r.x0=P1.x;
  r.y0=P1.y;
  r.l =P2.x-P1.x;
  r.m =P2.y-P1.y;
  if( iszero(r.l) && iszero(r.m) ) // P1 = P2
    {
      if( (absv(r.x0 - R.x0) <= R.a) && (absv(r.y0 - R.y0) <= R.b) )
	{
	  Q2->x=Q1->x=r.x0;
	  Q2->y=Q1->y=r.y0;
	  return 1;
	}
      else
	return 0;
    }
  else // P1 != P2
    {
      is1out= isout(P1.x, P1.y, R);
      is2out= isout(P2.x, P2.y, R);
      if( !is1out && !is2out )
	{
	  Q1->x= P1.x;
	  Q1->y= P1.y;
	  Q2->x= P2.x;
	  Q2->y= P2.y;
	  return 1;
	}
      else if ( !is1out )
	{
	  Q1->x= P1.x;
	  Q1->y= P1.y;
	  switch( where(P2.x,P2.y,R,P1.x,P1.y) )
	    {
	    case -1:
	      aux=R.x0-R.a;
	      auy=x_intersection(P1.x, P1.y, P2.x, P2.y, aux);
	      break;
	    case  1:
	      aux=R.x0+R.a;
	      auy=x_intersection(P1.x, P1.y, P2.x, P2.y, aux);
	      break;
	    case -2:
	      auy=R.y0-R.b;
	      aux=y_intersection(P1.x, P1.y, P2.x, P2.y, auy);
	      break;
	    case  2:
	      auy=R.y0+R.b;
	      aux=y_intersection(P1.x, P1.y, P2.x, P2.y, auy);
	    }
	  Q2->x=aux;
	  Q2->y=auy;
	  return 1;
	}
      else if ( !is2out )
	{
	  Q2->x= P2.x;
	  Q2->y= P2.y;
	  switch( where(P1.x,P1.y,R,P2.x,P2.y) )
	    {
	    case -1:
	      aux=R.x0-R.a;
	      auy=x_intersection(P1.x, P1.y, P2.x, P2.y, aux);
	      break;
	    case  1:
	      aux=R.x0+R.a;
	      auy=x_intersection(P1.x, P1.y, P2.x, P2.y, aux);
	      break;
	    case -2:
	      auy=R.y0-R.b;
	      aux=y_intersection(P1.x, P1.y, P2.x, P2.y, auy);
	      break;
	    case  2:
	      auy=R.y0+R.b;
	      aux=y_intersection(P1.x, P1.y, P2.x, P2.y, auy);
	    }
	  Q1->x=aux;
	  Q1->y=auy;
	  return 1;
	}
      else // P1 and P2 are both out of the rectangle
	{
	  if( segcap1d(P1,P2,R,Q1,Q2) != 0)
	    {
	      // Q2 e` un punto del rettangolo !!!
	      switch( where(P1.x,P1.y,R,Q2->x,Q2->y) )
		{
		case -1:
		  aux=R.x0-R.a;
		  auy=x_intersection(P1.x, P1.y, Q2->x, Q2->y, aux);
		  break;
		case  1:
		  aux=R.x0+R.a;
		  auy=x_intersection(P1.x, P1.y, Q2->x, Q2->y, aux);
		  break;
		case -2:
		  auy=R.y0-R.b;
		  aux=y_intersection(P1.x, P1.y, Q2->x, Q2->y, auy);
		  break;
		case  2:
		  auy=R.y0+R.b;
		  aux=y_intersection(P1.x, P1.y, Q2->x, Q2->y, auy);
		}
	      Q1->x=aux;
	      Q1->y=auy;
	      switch( where(P2.x,P2.y,R,Q2->x,Q2->y) )
		{
		case -1:
		  aux=R.x0-R.a;
		  auy=x_intersection(P2.x, P2.y, Q2->x, Q2->y, aux);
		  break;
		case  1:
		  aux=R.x0+R.a;
		  auy=x_intersection(P2.x, P2.y, Q2->x, Q2->y, aux);
		  break;
		case -2:
		  auy=R.y0-R.b;
		  aux=y_intersection(P2.x, P2.y, Q2->x, Q2->y, auy);
		  break;
		case  2:
		  auy=R.y0+R.b;
		  aux=y_intersection(P2.x, P2.y, Q2->x, Q2->y, auy);
		}
	      Q2->x=aux;
	      Q2->y=auy;
	      return 1;
	    }
	  else if(segcap2d(P1,P2,R,Q1,Q2) != 0)
	    {
	      // Q2 e` un punto del rettangolo !!!
	      switch( where(P1.x,P1.y,R,Q2->x,Q2->y) )
		{
		case -1:
		  aux=R.x0-R.a;
		  auy=x_intersection(P1.x, P1.y, Q2->x, Q2->y, aux);
		  break;
		case  1:
		  aux=R.x0+R.a;
		  auy=x_intersection(P1.x, P1.y, Q2->x, Q2->y, aux);
		  break;
		case -2:
		  auy=R.y0-R.b;
		  aux=y_intersection(P1.x, P1.y, Q2->x, Q2->y, auy);
		  break;
		case  2:
		  auy=R.y0+R.b;
		  aux=y_intersection(P1.x, P1.y, Q2->x, Q2->y, auy);
		}
	      Q1->x=aux;
	      Q1->y=auy;
	      switch( where(P2.x,P2.y,R,Q2->x,Q2->y) )
		{
		case -1:
		  aux=R.x0-R.a;
		  auy=x_intersection(P2.x, P2.y, Q2->x, Q2->y, aux);
		  break;
		case  1:
		  aux=R.x0+R.a;
		  auy=x_intersection(P2.x, P2.y, Q2->x, Q2->y, aux);
		  break;
		case -2:
		  auy=R.y0-R.b;
		  aux=y_intersection(P2.x, P2.y, Q2->x, Q2->y, auy);
		  break;
		case  2:
		  auy=R.y0+R.b;
		  aux=y_intersection(P2.x, P2.y, Q2->x, Q2->y, auy);
		}
	      Q2->x=aux;
	      Q2->y=auy;
	      return 1;
	    }
	  else
	    return 0;
	} // end inner else 
    } // end outer else
}
