/* -------------------------- gnuInterface class --------------------------

   This class contains all the code necessary to issue commands to gnuplot
   through a named pipe. It does not check for error messages as a result
   of a bad command.

   The plotting command is reasonably smart. All you have to do is set the
   options you want and call doPlot().

   It is intended that all commands be passed through gnuInterface so one
   should only have to look at this header file to see all commands that
   are available. This class may pass the commands on and call equivalent
   commands from other classes if necessary. 

   This file is part of Xgfe: X Windows GUI front end to Gnuplot
   Copyright (C) 1998 David Ishee

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   ------------------------------------------------------------------------*/

#ifndef gnuInterface_included
#define gnuInterface_included

#include <string>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include <stdio.h>
#include "gnuPlotFile.h"
#include "gnuPlotFunction.h"
#include "gnuMultiFile.h"
#include "gnuMultiFunc.h"
#include "gnuCurveFit.h"

class gnuInterface
{
public:
  gnuInterface();
  /* Description:
       Constructor function */
  
  FILE* openGnuplot();
  /* Incoming arguments:
       none
     Outgoing arguments:
       FILE*: standard I/O stream returned by popen. If the return value
              is NULL, then an error opening the pipe occurred.
              See "man popen" for details
      Description:
        Opens a write only named pipe to gnuplot.*/
  
  void doPlot();
  /* Incoming arguments:
       none
     Outgoing arguments:
       none
     Description:
       Issues all necessary commands for a plot. This does all the work of
       setting labels, variables, etc. */
  
  void setPlotFileFlag(int flag);
  /* Incoming arguments:
       int flag: 0 for false or 1 for true
     Outgoing arguments:
       none
     Description:
       Sets a flag to control whether a datafile will be plotted.
       If set to 1, make sure filename is set! */

  int getPlotFileFlag();
  /* Description:
       Gets flag */

  void setPlotMultipleFileFlag(int flag);
  /* Incoming arguments:
       int flag: 0 for false or 1 for true
     Outgoing arguments:
       none
     Description:
       Sets a flag to control whether multiple datafiles will be plotted.
       If set to 1, make sure multiple filenames are set! */

  int getPlotMultipleFileFlag();
  /* Description:
       Gets flag */
  
  void setPlotFuncFlag(int flag);
  /* Incoming arguments:
       int flag: 0 for false or 1 for true
     Outgoing arguments:
       none
     Description:
       Sets a flag to control whether a function will be plotted.
       If set to 1, make sure the function is set! */

  int getPlotFuncFlag();
  /* Description:
       Gets flag */

  void setPlotMultipleFuncFlag(int flag);
  /* Incoming arguments:
       int flag: 0 for false or 1 for true
     Outgoing arguments:
       none
     Description:
       Sets a flag to control whether multiple functions will be plotted.
       If set to 1, make sure the functions are set! */

  int getPlotMultipleFuncFlag();
  /* Description:
       Gets flag */
  
  void setPlotFilename(string filename);
  /* Incoming arguments:
       string filename: Path and name of file to plot
     Outgoing arguments:
       none
     Description:
       Sets the datafile name to plot */

  string getPlotFilename();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string filename: Path and name of file to plot
     Description:
       Gets the single datafile name to plot */
  
  void setPlotFunction(string function);
  /* Incoming arguments:
       string function: Function or comma seperated multiple functions
     Outgoing arguments:
       none
     Description:
       Sets the function or functions to plot*/

  string getPlotFunction();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string function: Function to plot
     Description:
       Gets the single function to plot*/
  
  void setGnuFileSave(string file);
  /* Incoming arguments:
       string file: Filename to save plot to
     Outgoing arguments:
       none
     Description:
       Sets the filename to save a gnuplot plot to */
  
  void setGnuFileLoad(string file);
  /* Incoming arguments:
       string file: Filename to load plot from
     Outgoing arguments:
       none
     Description:
       Sets the filename to load a gnuplot plot from */
  
  void setFilePlotType(string type);
  /* Incoming arguments:
       string type: plot or splot
     Outgoing arguments:
       none
     Description:
       This sets the type of plot for the data file. Use plot for 2D plots,
       and splot for 3D plots */

  string getFilePlotType();
  /* Description:
       Gets type of plot for the single data file. */
  
  void setFileStyleType(string type);
  /* Incoming arguments:
       string type: style of plotting. 
     Outgoing arguments:
       none
     Description:
       Sets the style of plotting the datafile */

  string getFileStyleType();
  /* Description:
       Gets style type of plot for the single data file. */

  void setFileDataSetStart(string start);
  /*  Description: 
       Sets the start of the data set for the index command */

  string getFileDataSetStart();
  /*  Description: 
       Gets the start of the data set for the index command */

  void setFileDataSetEnd(string end);
  /*  Description: 
       Sets the end of the data set for the index command */

  string getFileDataSetEnd();
  /*  Description: 
       Gets the end of the data set for the index command */

  void setFileDataSetIncrement(string inc);
  /*  Description: 
       Sets the increment value of the data set for the index command */

  string getFileDataSetIncrement();
  /*  Description: 
       Gets the increment value of the data set for the index command */

  void setFileSampPointInc(string pinc);
  /*  Description: 
       Sets the point increment value of the periodic sampling 
       for the every command */

  string getFileSampPointInc();
  /*  Description: 
       Gets the point increment value of the periodic sampling 
       for the every command */

  void setFileSampLineInc(string linc);
  /*  Description: 
       Sets the line increment value of the periodic sampling 
       for the every command */

  string getFileSampLineInc();
  /*  Description: 
       Gets the line increment value of the periodic sampling 
       for the every command */

  void setFileSampStartPoint(string startp);
  /*  Description: 
       Sets the point start value of the periodic sampling 
       for the every command */

  string getFileSampStartPoint();
  /*  Description: 
       Gets the point start value of the periodic sampling 
       for the every command */

  void setFileSampStartLine(string startl);
  /*  Description: 
       Sets the line start value of the periodic sampling 
       for the every command */

  string getFileSampStartLine();
  /*  Description: 
       Gets the line start value of the periodic sampling 
       for the every command */

  void setFileSampEndPoint(string endp);
  /*  Description: 
       Sets the point end value of the periodic sampling 
       for the every command */

  string getFileSampEndPoint();
  /*  Description: 
       Gets the point end value of the periodic sampling 
       for the every command */

  void setFileSampEndLine(string endl);
  /*  Description: 
       Sets the line end value of the periodic sampling 
       for the every command */

  string getFileSampEndLine();
  /*  Description: 
       Gets the line end value of the periodic sampling 
       for the every command */

  void setFileSmoothType(string type);
  /*  Description: 
       Sets the smoothing type for the smooth command */

  string getFileSmoothType();
  /*  Description: 
       Gets the smoothing type for the smooth command */

  void setFileXColumn(string column);
  /* Incoming arguments:
       string column: column to use for x variable
     Outgoing arguments:
       none
     Description: 
       Sets the column to use for the x variable */

  string getFileXColumn();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: x column that is currently specified
     Description: 
       Gets the column to use for the x variable */

  void setFileYColumn(string column);
  /* Incoming arguments:
       string column: column to use for y variable 
     Outgoing arguments:
       none
     Description: 
       Sets the column to use for the y variable */

  string getFileYColumn();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: y column that is currently specified
     Description: 
       Gets the column to use for the y variable */

  void setFileZColumn(string column);
  /* Incoming arguments:
       string column: column to use for z variable 
     Outgoing arguments:
       none
     Description: 
       Sets the column to use for the z variable */

  string getFileZColumn();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: z column that is currently specified
     Description: 
       Gets the column to use for the z variable */

  void setFileFormatString(string format);
  /* Incoming arguments:
       string format: scanf format string including quotes
     Outgoing arguments:
       none
     Description: 
       Sets the scanf format string to read the data file with */

  string getFileFormatString();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current format string
     Description: 
       Gets the current scanf format string */

  void setRawFileFormatString(string format);
  /* Incoming arguments:
       string format: format for plotting file
     Outgoing arguments:
       none
     Description: 
       Sets the format string to whatever is specified. If this is set, it 
       will be used instead of building one from the columns and format 
       string specifications. This allows complex formats to be specified. */  

  string getRawFileFormatString();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current raw file format
     Description: 
       Gets the current format string */

  void setFileFilter(string filter);
  /* Incoming arguments:
       string filter: filter command
     Outgoing arguments:
       none
     Description:
       Sets filter command for the single file */

  string getFileFilter();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: filter command
     Description:
       Gets filter command for the single file */

  void setFileFilterQuoteChar(string quote);
  /* Incoming arguments:
       char quote: quoting char
     Outgoing arguments:
       none
     Description:
       Sets quoting character for the single file filter cmd */

  string getFileFilterQuoteChar();
  /* Incoming arguments:
       none
     Outgoing arguments:
       none
     Description:
       Gets quoting character for the single file filter cmd */
  
  void setFuncPlotType(string type);
  /* Incoming arguments:
       string type: plot or splot              
     Outgoing arguments:
       none
     Description:
       This sets the type of plot for the function. Use plot for 2D plots,
       and splot for 3D plots */

  string getFuncPlotType();
  /* Description:
       Gets plot type */
  
  void setFuncStyleType(string type);
  /* Incoming arguments:
       string type: style of plotting. With Gnuplot 3.5 the following types
       are supported: points, lines, linespoints, impulses, dots, steps,
       errorbars, and boxes. Others can be specified if Gnuplot will
       understand it. 
     Outgoing arguments:
       none
     Description:
       Sets the style of plotting the function */

  string getFuncStyleType();
  /* Description:
       Gets style type */
  
  void setXVariableName(string name);
  /* Incoming arguments:
       string name: X variable name
     Outgoing arguments:
       none
     Description:
       Sets the X variable name */

  string getXVariableName();
  
  void setXRangeStart(string range);
  /* Incoming arguments:
       string range: Starting range for the X axis
     Outgoing arguments:
       none
     Description:
       Sets the starting range for the X axis */
  
  string getXRangeStart();

  void setXRangeEnd(string range);
  /* Incoming arguments:
       string range: Ending range for the X axis
     Outgoing arguments:
       none
     Description:
       Sets the ending range for the X axis */

  string getXRangeEnd();
  
  void setYVariableName(string name);
  /* Incoming arguments:
       string name: Y variable name
     Outgoing arguments:
       none
     Description:
       Sets the Y variable name */   
    
  string getYVariableName();  

  void setYRangeStart(string range);
  /* Incoming arguments:
       string range: Starting range for the Y axis       
     Outgoing arguments:
       none
     Description:
       Sets the starting range for the Y axis */
  
  string getYRangeStart();

  void setYRangeEnd(string range);
  /* Incoming arguments:
       string range: Ending range for the Y axis       
     Outgoing arguments:
       none
     Description:
       Sets the ending range for the Y axis */       

  string getYRangeEnd();
  
  void setZRangeStart(string range);
  /* Incoming arguments:
       string range: Starting range for the Z axis       
     Outgoing arguments:
       none
     Description:
       Sets the starting range for the Z axis */       

  string getZRangeStart();
  
  void setZRangeEnd(string range);
  /* Incoming arguments:
       string range: Ending range for the Z axis       
     Outgoing arguments:
       none
     Description:
       Sets the ending range for the Z axis */      

  string getZRangeEnd();
  
  void setXlabel(string label);
  /* Incoming arguments:
       string label: Label for the X axis
     Outgoing arguments:
       none
     Description:
       Sets the label for the X axis */

  string getXlabel();
  
  void setXOffset_X(string offset);
  /* Incoming arguments:
       string offset: X offset value for X axis
     Outgoing arguments:
       none
     Description:
       Sets X offset value for X axis */

  string getXOffset_X();
  
  void setXOffset_Y(string offset);
  /* Incoming arguments:
       string offset: Y offset value for X axis       
     Outgoing arguments:
       none
     Description:
       Sets Y offset value for X axis */       

  string getXOffset_Y();
  
  void setYlabel(string label);
  /* Incoming arguments:
       string label: Label for the Y axis
     Outgoing arguments:
       none
     Description:
       Sets the label for the Y axis */       

  string getYlabel();
  
  void setYOffset_X(string offset);
  /* Incoming arguments:
       string offset: X offset value for Y axis       
     Outgoing arguments:
       none
     Description:
       Sets X offset value for Y axis */

  string getYOffset_X();
  
  void setYOffset_Y(string offset);
  /* Incoming arguments:
       string offset: Y offset value for Y axis              
     Outgoing arguments:
       none
     Description:
       Sets Y offset value for Y axis */       

  string getYOffset_Y();
  
  void setZlabel(string label);
  /* Incoming arguments:
       string label: Label for the Z axis       
     Outgoing arguments:
       none
     Description:
       Sets the label for the Z axis */       

  string getZlabel();
  
  void setZOffset_X(string offset);
  /* Incoming arguments:
       string offset: X offset value for Z axis              
     Outgoing arguments:
       none
     Description:
       Sets X offset value for Z axis */       

  string getZOffset_X();
  
  void setZOffset_Y(string offset);
  /* Incoming arguments:
       string offset: X offset value for Z axis              
     Outgoing arguments:
       none
     Description:
       Sets Y offset value for Z axis */              
  
  string getZOffset_Y();

  void setTitle(string intitle);
  /* Incoming arguments:
       string intitle: Title of plot
     Outgoing arguments:
       none
     Description:
       Sets title of plot */

  string getTitle();
  
  void setTitleOffset_X(string offset);
  /* Incoming arguments:
       string offset: X offset value for the title              
     Outgoing arguments:
       none
     Description:
       Sets X offset value for the title */

  string getTitleOffset_X();
  
  void setTitleOffset_Y(string offset);
  /* Incoming arguments:
       string offset: Y offset value for the title                     
     Outgoing arguments:
       none
     Description:
       Sets Y offset value for the title */       

  string getTitleOffset_Y();

  void setTerminal(string terminal);
  /* Description:
       Sets the terminal name */

  string getTerminal();
  /* Description:
       Gets the terminal name */
  
  void setTermLateXEmtexFont(string font);
  /* Description:
       Sets the latex/emtex terminal font with any options needed */
  
  string getTermLateXEmtexFont();
  /* Description:
       Returns the current value of the terminal font */

  void setTermLateXEmtexFontSize(string size);
  /* Description:
       Sets the latex/emtex terminal size */

  string getTermLateXEmtexFontSize();
  /* Description:
       Gets the latex/emtex terminal size */

  void setTermPBMFontSize(string size);
  /* Description:
       Sets the PBM terminal font size */

  string getTermPBMFontSize();
  /* Description:
       Gets the PBM terminal font size */

  void setTermPBMColormode(string color);
  /* Description:
       Sets the PBM terminal colormode */

  string getTermPBMColormode();
  /* Description:
       Gets the PBM terminal colormode */

  void setTermPBMhSize(string size);
  /* Description:
       Sets the PBM terminal horizontal size (pixels) */

  string getTermPBMhSize();
  /* Description:
       Gets the PBM terminal horizontal size (pixels) */

  void setTermPBMvSize(string size);
  /* Description:
       Sets the PBM terminal vertical size (pixels) */

  string getTermPBMvSize();
  /* Description:
       Gets the PBM terminal vertical size (pixels) */

  void setTermPSmode(string mode);
  /* Description:
       Sets the postscript terminal mode */

  string getTermPSmode();
  /* Description:
       Gets the postscript terminal mode */

  void setTermPScolor(string color);
  /* Description:
       Sets the postscript terminal color */

  string getTermPScolor();
  /* Description:
       Gets the postscript terminal color */

  void setTermPSdashed(string type);
  /* Description:
       Sets the postscript terminal line-type */

  string getTermPSdashed();
  /* Description:
       Gets the postscript terminal line-type */

  void setTermPSfont(string font);
  /* Description:
       Sets the postscript terminal font */

  string getTermPSfont();
  /* Description:
       Gets the postscript terminal font */

  void setTermPSfontSize(string size);
  /* Description:
       Sets the postscript terminal font size */

  string getTermPSfontSize();
  /* Description:
       Gets the postscript terminal font size */

  void setTermPSenhanced(string inenhanced);
  /* Description:
       Sets the postscript terminal enhanced */

  string getTermPSenhanced();
  /* Description:
       Gets the postscript terminal enhanced */

  void setTermPShSize(string size);
  /* Description:
       Sets the postscript terminal horizontal size (inches) */

  string getTermPShSize();
  /* Description:
       Gets the postscript terminal horizontal size (inches) */

  void setTermPSvSize(string size);
  /* Description:
       Sets the postscript terminal vertical size (inches) */

  string getTermPSvSize();
  /* Description:
       Gets the postscript terminal vertical size (inches) */
  
  void setHorizSize(string size);
  /* Incoming arguments:
       string size: Horizontal size of plot
     Outgoing arguments:
       none
     Description:
       Sets horizontal size of plot. Depending on the terminal type,
       the units could be pixels, inches, or scaling factor */

  string getHorizSize();
  /* Description:
       Gets horizontal size of plot. */
  
  void setVertSize(string size);
  /* Incoming arguments:
       string size: Vertical size of plot
     Outgoing arguments:
       none
     Description:
       Sets vertical size of plot. Depending on the terminal type,
       the units could be pixels, inches, or scaling factor */

  string getVertSize();
  /* Description:
       Gets vertical size of plot. */
  
  void setOutput(string output);
  /* Incoming arguments:
       string output: Output filename for plot
     Outgoing arguments:
       none
     Description:
       Sets the output filename for the plot */

  string getOutput();
  
  void doSave();
  /* Incoming arguments:
       none
     Outgoing arguments:
       none
     Description:
       Uses the "save" command of gnuplot to the file specified by the output
       variable */
  
  void doLoad();
  /* Incoming arguments:
       none
     Outgoing arguments:
       none
     Description:
       Uses the "load" command of gnuplot to the file specified by the output
       variable */       
  
  void closeGnuplot();
  /* Incoming arguments:
       none
     Outgoing arguments:
       none
     Description:
       Closes the named pipe to gnuplot */
  
  void doRawCommand(string command);
  /* Incoming arguments:
       string command: command to send to gnuplot
     Outgoing arguments:
       none
     Description:
       Issues a raw gnuplot command. */

  void setReplotFlag(int flag);
  /* Incoming arguments:
       int flag: 0 for false or 1 for true
     Outgoing arguments:
       none
     Description:
       Sets a flag to control whether a function will use the replot command. */

  void setLegendFlag(string flag);
  /* Incoming arguments:
       string flag: key for a legend, nokey for no legend
     Outgoing arguments:
       none
     Description:
       Specifies if a legend will be used */

  string getLegendFlag();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: value of legend flag
     Description:
       Returns current value of legend flag */

  void setLegendPositionLeft(int position);
  /* Incoming arguments:
       int position: 1 or 0 : 1 = true, 0 = false
     Outgoing arguments:
       none
     Description:
       Sets value of legend position if it is on the left or not*/
  
  int getLegendPositionLeft();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: value of current legend position if it is left
     Description:
       Gets current value of legend position if it is left */

  void setLegendPositionRight(int position);
  /* Incoming arguments:
       int position: 1 or 0 : 1 = true, 0 = false
     Outgoing arguments:
       none
     Description:
       Sets value of legend position if it is on the right or not*/
  
  int getLegendPositionRight();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: value of current legend position if it is right
     Description:
       Gets current value of legend position if it is right */

  void setLegendPositionTop(int position);
  /* Incoming arguments:
       int position: 1 or 0 : 1 = true, 0 = false
     Outgoing arguments:
       none
     Description:
       Sets value of legend position if it is on the top or not*/
  
  int getLegendPositionTop();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: value of current legend position if it is top
     Description:
       Gets current value of legend position if it is top */

  void setLegendPositionBottom(int position);
  /* Incoming arguments:
       int position: 1 or 0 : 1 = true, 0 = false
     Outgoing arguments:
       none
     Description:
       Sets value of legend position if it is on the bottom or not*/
  
  int getLegendPositionBottom();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: value of current legend position if it is bottom
     Description:
       Gets current value of legend position if it is bottom */

  void setLegendPositionOutside(int position);
  /* Incoming arguments:
       int position: 1 or 0 : 1 = true, 0 = false
     Outgoing arguments:
       none
     Description:
       Sets value of legend position if it is on the outside or not*/
  
  int getLegendPositionOutside();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: value of current legend position if it is outside
     Description:
       Gets current value of legend position if it is outside */

  void setLegendPositionBelow(int position);
  /* Incoming arguments:
       int position: 1 or 0 : 1 = true, 0 = false
     Outgoing arguments:
       none
     Description:
       Sets value of legend position if it is below or not*/
  
  int getLegendPositionBelow();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: value of current legend position if it is below
     Description:
       Gets current value of legend position if it is below */

  void setLegendPositionXVal(string val);
  /* Incoming arguments:
       string val: X position of legend
     Outgoing arguments:
       none
     Description:
       Sets current value of X coordinate for legend position */

  string getLegendPositionXVal();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of X coordinate
     Description:
       Gets current value of X coordinate for legend position */

  void setLegendPositionYVal(string val);
  /* Incoming arguments:
       string val: Y position of legend
     Outgoing arguments:
       none
     Description:
       Sets current value of Y coordinate for legend position */

  string getLegendPositionYVal();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of Y coordinate
     Description:
       Gets current value of Y coordinate for legend position */

  void setLegendPositionZVal(string val);
  /* Incoming arguments:
       string val: Z position of legend
     Outgoing arguments:
       none
     Description:
       Sets current value of Z coordinate for legend position */

  string getLegendPositionZVal();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of Z coordinate
     Description:
       Gets current value of Z coordinate for legend position */

  void setLegendTextJustify(string justify);
  /* Incoming arguments:
       string justify: Justification of legend text
     Outgoing arguments:
       none
     Description:
       Sets current value of justification for legend text */

  string getLegendTextJustify();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of text justification
     Description:
       Gets current value of justification for legend text */

  void setLegendReverse(string reverse);
  /* Incoming arguments:
       string reverse: reverse/noreverse for legend sample and text
     Outgoing arguments:
       none
     Description:
       Sets current value of reverse for legend text and samples */

  string getLegendReverse();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of reverse
     Description:
       Gets current value of reverse for legend text and samples */

  void setLegendBox(string box);
  /* Incoming arguments:
       string box: box/nobox value to use a box or not
     Outgoing arguments:
       none
     Description:
       Sets current value of box for legend  */

  string getLegendBox();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of box
     Description:
       Gets current value of box for legend  */

  void setLegendLinetype(string type);
  /* Incoming arguments:
       string type: linetype for legend box
     Outgoing arguments:
       none
     Description:
       Sets current value of linetype for box around legend  */
  
  string getLegendLinetype();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of linetype
     Description:
       Gets current value of linetype for box around legend  */

  void setLegendSampleLength(string length);
  /* Incoming arguments:
       string length: length of line sample in legend
     Outgoing arguments:
       none
     Description:
       Sets current value of line sample length for legend  */

  string getLegendSampleLength();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of sample length
     Description:
       Gets current value of line sample length for legend  */

  void setLegendSpacing(string spacing);
  /* Incoming arguments:
       string spacing: vertical spacing of lines in legend
     Outgoing arguments:
       none
     Description:
       Sets current value of vertical spacing between lines for legend */

  string getLegendSpacing();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of legend spacing
     Description:
       Gets current value of vertical spacing between lines for legend */

  void setLegendWidthIncrement(string width);
  /* Incoming arguments:
       string width: extra width for box around legend
     Outgoing arguments:
       none
     Description:
       Sets current value of extra spacing for box around legend */

  string getLegendWidthIncrement();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of extra spacing
     Description:
       Gets current value of extra spacing for box around legend */

  void setLegendTitle(string title);
  /* Incoming arguments:
       string title: title of legend
     Outgoing arguments:
       none
     Description:
       Sets current value of title for legend */

  string getLegendTitle();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: current value of legend title
     Description:
       Gets current value of title for legend */

  void setXticsOnFlag(int flag);
  int getXticsOnFlag();
  void setXticsLocation(string location);
  string getXticsLocation();
  void setXticsMirror(string mirror);
  string getXticsMirror();
  void setXticsRotation(string rotation);
  string getXticsRotation();
  void setXticsPositionType(string type);
  /* Description:
       Sets position type
       Position = SIE for start/inc/edit type or LABELS = labels/pos type */
  string getXticsPositionType();
  void setXticsStartPos(string pos);
  string getXticsStartPos();
  void setXticsIncPos(string pos);
  string getXticsIncPos();
  void setXticsEndPos(string pos);
  string getXticsEndPos();
  void setXticsLabelsPos(string labels);
  string getXticsLabelsPos();
  
  void setYticsOnFlag(int flag);
  int getYticsOnFlag();
  void setYticsLocation(string location);
  string getYticsLocation();
  void setYticsMirror(string mirror);
  string getYticsMirror();
  void setYticsRotation(string rotation);
  string getYticsRotation();
  void setYticsPositionType(string type);
  /* Description:
       Sets position type
       Position = SIE for start/inc/edit type or LABELS = labels/pos type */
  string getYticsPositionType();
  void setYticsStartPos(string pos);
  string getYticsStartPos();
  void setYticsIncPos(string pos);
  string getYticsIncPos();
  void setYticsEndPos(string pos);
  string getYticsEndPos();
  void setYticsLabelsPos(string labels);
  string getYticsLabelsPos();

  void setZticsOnFlag(int flag);
  int getZticsOnFlag();
  void setZticsMirror(string mirror);
  string getZticsMirror();
  void setZticsRotation(string rotation);
  string getZticsRotation();
  void setZticsPositionType(string type);
  /* Description:
       Sets position type
       Position = SIE for start/inc/edit type or LABELS = labels/pos type */
  string getZticsPositionType();
  void setZticsStartPos(string pos);
  string getZticsStartPos();
  void setZticsIncPos(string pos);
  string getZticsIncPos();
  void setZticsEndPos(string pos);
  string getZticsEndPos();
  void setZticsLabelsPos(string labels);
  string getZticsLabelsPos();

  void setX2ticsOnFlag(int flag);
  int getX2ticsOnFlag();
  void setX2ticsLocation(string location);
  string getX2ticsLocation();
  void setX2ticsMirror(string mirror);
  string getX2ticsMirror();
  void setX2ticsRotation(string rotation);
  string getX2ticsRotation();
  void setX2ticsPositionType(string type);
  /* Description:
       Sets position type
       Position = SIE for start/inc/edit type or LABELS = labels/pos type */
  string getX2ticsPositionType();
  void setX2ticsStartPos(string pos);
  string getX2ticsStartPos();
  void setX2ticsIncPos(string pos);
  string getX2ticsIncPos();
  void setX2ticsEndPos(string pos);
  string getX2ticsEndPos();
  void setX2ticsLabelsPos(string labels);
  string getX2ticsLabelsPos();

  void setY2ticsOnFlag(int flag);
  int getY2ticsOnFlag();
  void setY2ticsLocation(string location);
  string getY2ticsLocation();
  void setY2ticsMirror(string mirror);
  string getY2ticsMirror();
  void setY2ticsRotation(string rotation);
  string getY2ticsRotation();
  void setY2ticsPositionType(string type);
  /* Description:
       Sets position type
       Position = SIE for start/inc/edit type or LABELS = labels/pos type */
  string getY2ticsPositionType();
  void setY2ticsStartPos(string pos);
  string getY2ticsStartPos();
  void setY2ticsIncPos(string pos);
  string getY2ticsIncPos();
  void setY2ticsEndPos(string pos);
  string getY2ticsEndPos();
  void setY2ticsLabelsPos(string labels);
  string getY2ticsLabelsPos();
  

  void insertMultiFileNew(string filename);
  /* Incoming arguments:
       string filename: new filename
     Outgoing arguments:
       none
     Description:
       Inserts a new file into multiple file list */

  void removeMultiFile(string filename);
  /* Incoming arguments:
       string filename: file to remove
     Outgoing arguments:
       none
     Description:
       Removes file from multiple file list */

  string getMultiFileFirstFilename();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: filename for the first file
     Description:
       Returns filename for the first file in the multiple file list 
       Returns END if empty list */

  string getMultiFileNextFilename();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: filename for the next file
     Description:
       Returns filename for the next file in the multiple file list 
       Returns END if at the end of the list */

  void setMultiFileDataSetStart(string filename, string start);
  /*  Description: 
       Sets the start of the data set for the index command */

  string getMultiFileDataSetStart(string filename);
  /*  Description: 
       Gets the start of the data set for the index command */

  void setMultiFileDataSetEnd(string filename,string end);
  /*  Description: 
       Sets the end of the data set for the index command */

  string getMultiFileDataSetEnd(string filename);
  /*  Description: 
       Gets the end of the data set for the index command */

  void setMultiFileDataSetIncrement(string filename,string inc);
  /*  Description: 
       Sets the increment value of the data set for the index command */

  string getMultiFileDataSetIncrement(string filename);
  /*  Description: 
       Gets the increment value of the data set for the index command */

  void setMultiFileSampPointInc(string filename,string pinc);
  /*  Description: 
       Sets the point increment value of the periodic sampling 
       for the every command */

  string getMultiFileSampPointInc(string filename);
  /*  Description: 
       Gets the point increment value of the periodic sampling 
       for the every command */

  void setMultiFileSampLineInc(string filename,string linc);
  /*  Description: 
       Sets the line increment value of the periodic sampling 
       for the every command */

  string getMultiFileSampLineInc(string filename);
  /*  Description: 
       Gets the line increment value of the periodic sampling 
       for the every command */

  void setMultiFileSampStartPoint(string filename,string startp);
  /*  Description: 
       Sets the point start value of the periodic sampling 
       for the every command */

  string getMultiFileSampStartPoint(string filename);
  /*  Description: 
       Gets the point start value of the periodic sampling 
       for the every command */

  void setMultiFileSampStartLine(string filename,string startl);
  /*  Description: 
       Sets the line start value of the periodic sampling 
       for the every command */

  string getMultiFileSampStartLine(string filename);
  /*  Description: 
       Gets the line start value of the periodic sampling 
       for the every command */

  void setMultiFileSampEndPoint(string filename,string endp);
  /*  Description: 
       Sets the point end value of the periodic sampling 
       for the every command */

  string getMultiFileSampEndPoint(string filename);
  /*  Description: 
       Gets the point end value of the periodic sampling 
       for the every command */

  void setMultiFileSampEndLine(string filename,string endl);
  /*  Description: 
       Sets the line end value of the periodic sampling 
       for the every command */

  string getMultiFileSampEndLine(string filename);
  /*  Description: 
       Gets the line end value of the periodic sampling 
       for the every command */

  void setMultiFileSmoothType(string filename,string type);
  /*  Description: 
       Sets the smoothing type for the smooth command */

  string getMultiFileSmoothType(string filename);
  /*  Description: 
       Gets the smoothing type for the smooth command */

  void insertMultiFileXColumnOption(string filename, string xcol);
  /* Incoming arguments:
       string filename: file to operate on
       string xcol: X column to plot
     Outgoing arguments:
       none
     Description:
       Sets x column for a given file in the multiple file list */

  string getMultiFileXColumnOption(string filename);
  /* Incoming arguments:
       string filename: file to operate on
     Outgoing arguments:
       string: x column for given file
     Description:
       Return x column for a given file in the multiple file list */

  void insertMultiFileYColumnOption(string filename, string ycol);
  /* Incoming arguments:
       string filename: file to operate on       
       string ycol: Y column to plot
     Outgoing arguments:
       none       
     Description:
       Sets y column for a given file in the multiple file list */

  string getMultiFileYColumnOption(string filename);
  /* Incoming arguments:
       string filename: file to operate on       
     Outgoing arguments:
       string: y column for a given file
     Description:
       Return y column for a given file in the multiple file list */

  void insertMultiFileZColumnOption(string filename, string zcol);
  /* Incoming arguments:
       string filename: file to operate on       
       string zcol: Z column to plot
     Outgoing arguments:
       none       
     Description:
       Sets z column for a given file in the multiple file list */

  string getMultiFileZColumnOption(string filename);
  /* Incoming arguments:
       string filename: file to operate on       
     Outgoing arguments:
       string: z column for the given file
     Description:
       Return z column for a given file in the multiple file list */

  void insertMultiFileFormatOption(string filename, string format);
  /* Incoming arguments:
       string filename: file to operate on       
       string format: format for plotting
     Outgoing arguments:
       none       
     Description:
       Set format for the given file in the multiple file list */

  string getMultiFileFormatOption(string filename);
  /* Incoming arguments:
       string filename: file to operate on       
     Outgoing arguments:
       string: format for given file
     Description:
       Returns format for the given file in the multiple file list */

  void insertMultiFileRawFormatOption(string filename, string format);
  /* Incoming arguments:
       string filename: file to operate on       
       string format: raw format string for plotting
     Outgoing arguments:
       none       
     Description:
       Set raw format for the given file in the multiple file list */

  string getMultiFileRawFormatOption(string filename);
  /* Incoming arguments:
       string filename: file to operate on       
     Outgoing arguments:
       string: raw format for given file
     Description:
       Returns raw format for the given file in the multiple file list */

  void setMultiFileStyleOption(string filename, string style);
  /* Incoming arguments:
       string filename: file to operate on       
       string format: style string for plotting
     Outgoing arguments:
       none       
     Description:
       Set style for the given file in the multiple file list */

  string getMultiFileStyleOption(string filename);
  /* Incoming arguments:
       string filename: file to operate on       
     Outgoing arguments:
       string: style for given file
     Description:
       Returns style for the given file in the multiple file list */

  void insertMultiFuncNew(string function);
  /* Incoming arguments:
       string: function to insert
     Outgoing arguments:
       none
     Description:
       Inserts a new function into list */

  void removeMultiFunc(string function);
  /* Incoming arguments:
       string: function to remove
     Outgoing arguments:
       none
     Description:
       Removes a function from list */

  void setMultiFuncStyleOption(string function, string style);
  /* Incoming arguments:
       function: function to operate on
       style: plotting style to use
     Outgoing arguments:
       none
     Description:
       Sets the plotting style for the specified function */

  string getMultiFuncStyleOption(string function);
  /* Incoming arguments:
       function: function to operate on
     Outgoing arguments:
       string: plotting style
     Description:
       Gets the plotting style for the specified function */

  string getMultiFuncFirstFunction();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: First function in list
               Returns END if list is empty
     Description:
       Gets the first function in the list */

  string getMultiFuncNextFunction();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: Next function in list
               Returns END if the end of the list is reached
     Description:
       Gets the next function in the list */

  string getMultiFuncFirstPlotCmd();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: First function's plotting command in list
               Returns END if list is empty
     Description:
       Gets the first function's plotting command in the list */

  string getMultiFuncNextPlotCmd();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: Next function's plotting command in list
               Returns END if the end of the list is reached
     Description:
       Gets the next function's plotting command in the list */  

  void setFileLegendTitle(string title);
  /* Incoming arguments:
       title: title for legend
     Outgoing arguments:
       none
     Description:
       Sets title to be used in legend for the single file */

  void setMultiFileLegendTitle(string filename, string title);
  /* Incoming arguments:
       filename: filename to operate on
       title: title for legend
     Outgoing arguments:
       none
     Description:
       Sets title to be used in legend for the given file */

  string getFileLegendTitle();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: title for current file
     Description:
       Gets title to be used in legend for the single file */

  string getMultiFileLegendTitle(string filename);
  /* Incoming arguments:
       filename: filename to operate on
     Outgoing arguments:
       string: title for given file
     Description:
       Gets title to be used in legend for the given file */

  void setFuncLegendTitle(string title);
  /* Incoming arguments:
       title: title for legend
     Outgoing arguments:
       none
     Description:
       Sets title to be used in legend for the single function */

  void setMultiFuncLegendTitle(string function, string title);
  /* Incoming arguments:
       function: function to operate on
       title: title for legend
     Outgoing arguments:
       none
     Description:
       Sets title to be used in legend for the given function */

  string getFuncLegendTitle();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: title for legend
     Description:
       Gets title to be used in legend for the single function */

  string getMultiFuncLegendTitle(string function);
  /* Incoming arguments:
       function: function to operate on
     Outgoing arguments:
       string: title for given function
     Description:
       Gets title to be used in legend for the given function */

  void setMultiFileFilter(string filename, string filter);
  /* Incoming arguments:
       string filename: filename to operate on
       string filter: filter command
     Outgoing arguments:
       none
     Description:
       Sets filter command for the file */

  string getMultiFileFilter(string filename);
  /* Incoming arguments:
       string: filename to operate on
     Outgoing arguments:
       string: filter command
     Description:
       Gets filter command for the file */

  void setMultiFileFilterQuoteChar(string filename, string quote);
  /* Incoming arguments:
       string filename: filename to operate on
       string quote: quoting char
     Outgoing arguments:
       none
     Description:
       Sets quoting character for the file's filter cmd */

  string getMultiFileFilterQuoteChar(string filename);
  /* Incoming arguments:
       string filename: filename to operate on
     Outgoing arguments:
       string: quoting character
     Description:
       Gets quoting character for the file's filter cmd */

  void setLogScaleBase(int base);

  int getLogScaleBase();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: base of log scale
     Description:
       Gets the base for log scale */

  void setLogScaleXAxis(int xAxis);

  int getLogScaleXAxis();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: flag for x axis 
     Description:
       Gets the flag to determine if x axis is to be plotted log scale */

  void setLogScaleYAxis(int yAxis);

  int getLogScaleYAxis();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: flag for y axis 
     Description:
       Gets the flag to determine if y axis is to be plotted log scale */

  void setLogScaleZAxis(int zAxis);

  int getLogScaleZAxis();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: flag for z axis 
     Description:
       Gets the flag to determine if z axis is to be plotted log scale */

  void setLogScaleX2Axis(int x2Axis);

  int getLogScaleX2Axis();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: flag for x2 axis 
     Description:
       Gets the flag to determine if x2 axis is to be plotted log scale */

  void setLogScaleY2Axis(int y2Axis);

  int getLogScaleY2Axis();
  /* Incoming arguments:
       none
     Outgoing arguments:
       int: flag for y2 axis 
     Description:
       Gets the flag to determine if y2 axis is to be plotted log scale */

  void setBarSizeOption(string size);
  /* Incoming arguments:
       string: size of bars
     Outgoing arguments:
       none
     Description:
       Sets the size of bars for bar styles of plotting */

  string getBarSizeOption();
  /* Incoming arguments:
       none
     Outgoing arguments:
       string: bar size
     Description:
       Gets the size of bars for bar styles of plotting */

  void setCurveFitFunctionName(string name);
  /* Description:
       Sets function name (left side of equal sign) to fit to */

  string getCurveFitFunctionName();
  /* Description:
       Gets function name to fit to */

  void setCurveFitFunctionValue(string function);
  /* Description:
       Sets function value (right side of equal sign) to fit to */

  string getCurveFitFunctionValue();
  /* Description:
       Gets function value (right side of equal sign) to fit to  */

  void setCurveFitDataFile(string file);
  /* Description:
       Sets data file for plotting */

  string getCurveFitDataFile();
  /* Description:
       Gets data file for plotting  */

  void setCurveFitVarXRangeName(string range);
  /* Description:
       Sets x variable name for xrange */

  string getCurveFitVarXRangeName();
  /* Description:
       Gets x variable name for xrange  */

  void setCurveFitVarXRangeMin(string min);
  /* Description:
       Sets x variable min value */

  string getCurveFitVarXRangeMin();
  /* Description:
       Gets x variable min value */

  void setCurveFitVarXRangeMax(string max);
  /* Description:
       Sets x variable max value */

  string getCurveFitVarXRangeMax();
  /* Description:
       Gets x variable max value */

  void setCurveFitVarYRangeName(string range);
  /* Description:
       Sets y variable name for yrange */

  string getCurveFitVarYRangeName();
  /* Description:
       Gets y variable name for yrange */

  void setCurveFitVarYRangeMin(string min);
  /* Description:
       Sets y variable min value */

  string getCurveFitVarYRangeMin();
  /* Description:
       Gets y variable min value */

  void setCurveFitVarYRangeMax(string max);
  /* Description:
       Sets y variable max value */

  string getCurveFitVarYRangeMax();
  /* Description:
       Gets y variable max value */

  void setCurveFitParamFile(string file);
  /* Description:
       Sets parameter file for plotting parameters */

  string getCurveFitParamFile();
  /* Description:
       Gets parameter file for plotting parameters */

  void setCurveFitParamFileFlag(int flag);
  /* Description:
       Sets flag for selecting parameter file vs comma seperated 
       value parameters */

  int getCurveFitParamFileFlag();
  /* Description:
       Gets flag for selecting parameter file vs comma seperated 
       value parameters */

  void setCurveFitParamCSLFlag(int flag);
  /* Description:
       Sets flag for selecting parameter file vs comma seperated 
       value parameters */ 

  int getCurveFitParamCSLFlag();
  /* Description:
       Gets flag for selecting parameter file vs comma seperated 
       value parameters */

  void setCurveFitParamCSL(string list);
  /* Description:
       Sets comma seperated list of plotting parameters */

  string getCurveFitParamCSL();
  /* Description:
       Gets comma seperated list of plotting parameters */

  void setCurveFitFitLimit(string limit);
  /* Description:
       Sets fit limit value for convergence */

  string getCurveFitFitLimit();
  /* Description:
       Gets fit limit value for convergence */

  void setCurveFitFitMaxIter(string iter);
  /* Description:
       Sets maximum number of iterations for non-convergence case */

  string getCurveFitFitMaxIter();
  /* Description:
       Gets maximum number of iterations for non-convergence case */

  void setCurveFitDataSetStart(string start);
  /* Description:
       Sets starting data set of the data file modifiers */

  string getCurveFitDataSetStart();
  /* Description:
       Gets starting data set of the data file modifiers */

  void setCurveFitDataSetEnd(string end);
  /* Description:
       Sets ending data set of the data file modifiers */

  string getCurveFitDataSetEnd();
  /* Description:
       Gets ending data set of the data file modifiers */

  void setCurveFitDataSetInc(string inc);
  /* Description:
       Sets data set increment of the data file modifiers */

  string getCurveFitDataSetInc();
  /* Description:
       Gets data set increment of the data file modifiers */

  void setCurveFitPointInc(string inc);
  /* Description:
       Sets the data point increment of the data file modifiers */

  string getCurveFitPointInc();
  /* Description:
       Gets the data point increment of the data file modifiers */

  void setCurveFitLineInc(string inc);
  /* Description:
       Sets the increment value for lines in data file (modifier) */

  string getCurveFitLineInc();
  /* Description:
       Gets the increment value for lines in data file (modifier) */

  void setCurveFitStartPoint(string start);
  /* Description:
       Sets the starting point in data file (modifier) */

  string getCurveFitStartPoint();
  /* Description:
       Gets the starting point in data file (modifier) */

  void setCurveFitStartLine(string start);
  /* Description:
       Sets the starting line in data file (modifier) */

  string getCurveFitStartLine();
  /* Description:
       Gets the starting line in data file (modifier) */

  void setCurveFitEndPoint(string end);
  /* Description:
       Sets the end point in data file (modifier) */

  string getCurveFitEndPoint();
  /* Description:
       Gets the end point in data file (modifier) */

  void setCurveFitEndLine(string end);
  /* Description:
       Sets the ending line in data file (modifier) */

  string getCurveFitEndLine();
  /* Description:
       Gets the ending line in data file (modifier) */

  void setCurveFitXColumn(string col);
  /* Description:
       Sets the x column from data file to plot (modifier) */

  string getCurveFitXColumn();
  /* Description:
       Gets the x column from data file to plot (modifier) */

  void setCurveFitYColumn(string col);
  /* Description:
       Sets the y column from data file to plot (modifier)  */

  string getCurveFitYColumn();
  /* Description:
       Gets the y column from data file to plot (modifier) */

  void setCurveFitZColumn(string col);
  /* Description:
       Sets the z column from data file to plot (modifier) */

  string getCurveFitZColumn();
  /* Description:
       Gets the z column from data file to plot (modifier) */

  void setCurveFitFormat(string informat);
  /* Description:
       Sets scanf format string for plotting from data file (modifier)
       (format should be surrounded with double quotes) */

  string getCurveFitFormat();
  /* Description:
       Gets format string for plotting from data file (modifier)
       (format will be surrounded with double quotes) */

  void setCurveFitRawFormat(string format);
  /* Description:
       Sets format string for plotting from data file (modifier)
       (no automatic quoting, must be supplied by user) */

  string getCurveFitRawFormat();
  /* Description:
       Sets format string for plotting from data file (modifier)
       (no automatic quoting, must be supplied by user) */

  void doCurveFit();
  /* Description:
       Issue curve fitting command */

  void setBoxWidth(string width);
  /* Description:
       Sets box width for boxes plotting styles */

  string getBoxWidth();
  /* Description:
       Gets box width for boxes plotting styles */

  void savePlotData(string filename);
  /* Description:
       Saves plotting variables individually to the given filename  
       as a specially formatted file */

  void loadPlotData(string filename);
  /* Description:
       Loads plotting variables from the given filename that
       corresponds to the special format */

private:
  string gnuFileSave;           // filename to save plot to
  string gnuFileLoad;           // filename to load plot from

  gnuPlotFile* plotFileOb;      // plot file object
  gnuMultiFile* multiFile;      // multiple file object
  gnuPlotFunction* plotFunctionOb; // plot function object
  gnuMultiFunc* multiFunc;      // multiple function object

  string filePlotType;          // plot or splot for datafile
  string funcPlotType;          // plot or splot for the function
  string funcStyleType;         // style to use for the function plot

  string XVariableName;         // variable name for x axis
  string XRangeStart;           // starting value for x axis
  string XRangeEnd;             // ending value for x axis

  string YVariableName;         // variable name for y axis
  string YRangeStart;           // starting value for y axis
  string YRangeEnd;             // ending value for y axis

  string ZRangeStart;           // starting value for z axis
  string ZRangeEnd;             // ending value for z axis

  string XLabel;                // x label
  string XOffset_X;             // x label offset in x direction
  string XOffset_Y;             // x label offset in y direction

  string YLabel;                // y label
  string YOffset_X;             // y label offset in x direction
  string YOffset_Y;             // y label offset in y direction

  string ZLabel;                // z label
  string ZOffset_X;             // z label offset in x direction
  string ZOffset_Y;             // z label offset in y direction

  string title;                 // title of plot
  string titleOffset_X;         // title offset in x direction
  string titleOffset_Y;         // title offset in y direction

  string terminal;              // terminal type

  string termLatexEmtexFont;    // latex/Emtex terminal font name
  string termLatexEmtexFontSize; // latex/Emtex terminal size

  string termPBMfontSize;       // PBM terminal font size
  string termPBMcolormode;      // PBM terminal colormode
  string termPBMhSize;          // PBM terminal horiz size (pixels)
  string termPBMvSize;          // PBM terminal vertical size (pixels)

  string termPSmode;            // postscript terminal mode
  string termPScolor;           // postscript terminal color
  string termPSdashed;          // postscript terminal line type
  string termPSfont;            // postscript terminal font
  string termPSfontSize;        // postscript terminal font size
  string termPSenhanced;        // postscript terminal enhanced/noenhanced
  string termPShSize;           // postscript terminal horiz size
  string termPSvSize;           // postscript terminal vert size

  string output;                // output filename

  string hSize;                 // horizontal size of plot
  string vSize;                 // vertical size of plot

  string legendFlag;            // key/nokey to specify if legend is present
  int legendPositionLeft;       // position of legend
  int legendPositionRight;      // position of legend
  int legendPositionTop;        // position of legend
  int legendPositionBottom;     // position of legend
  int legendPositionOutside;    // position of legend
  int legendPositionBelow;      // position of legend
  string legendPositionXVal;    // position of legend x coordinate
  string legendPositionYVal;    // position of legend y coordinate
  string legendPositionZVal;    // position of legend z coordinate
  string legendTextJustify;     // justification of legend text
  string legendReverse;         // reverse/noreverse of legend
  string legendBox;             // box/nobox of legend
  string legendLinetype;        // linetype of legend box
  string legendSampleLength;    // length of line sample for legend
  string legendSpacing;         // vertical spacing of elements in legend
  string legendWidthIncrement;  // extra spacing around elements for box
  string legendTitle;           // title of legend

  int xticsOnFlag;              // X tics options
  string xticsLocation;
  string xticsMirror;
  string xticsRotation;
  string xticsPositionType;
  string xticsStartPos;
  string xticsIncPos;
  string xticsEndPos;
  string xticsLabelsPos;

  int yticsOnFlag;              // Y tics options
  string yticsLocation;
  string yticsMirror;
  string yticsRotation;
  string yticsPositionType;
  string yticsStartPos;
  string yticsIncPos;
  string yticsEndPos;
  string yticsLabelsPos;

  int zticsOnFlag;              // Z tics options
  string zticsMirror;
  string zticsRotation;
  string zticsPositionType;
  string zticsStartPos;
  string zticsIncPos;
  string zticsEndPos;
  string zticsLabelsPos;

  int x2ticsOnFlag;            // X2 tics options
  string x2ticsLocation;
  string x2ticsMirror;
  string x2ticsRotation;
  string x2ticsPositionType;
  string x2ticsStartPos;
  string x2ticsIncPos;
  string x2ticsEndPos;
  string x2ticsLabelsPos;

  int y2ticsOnFlag;            // Y2 tics options
  string y2ticsLocation;
  string y2ticsMirror;
  string y2ticsRotation;
  string y2ticsPositionType;
  string y2ticsStartPos;
  string y2ticsIncPos;
  string y2ticsEndPos;
  string y2ticsLabelsPos;

  int plotFileFlag;             // flag to specify plotting file or not
  int plotMultipleFileFlag;     // flag to specify plotting multiple files
  int plotFuncFlag;             // flag to specify plotting function or not
  int plotMultipleFuncFlag;     // flag to specify plotting multiple functions
  int replotFlag;               // flag to specify replot command

  int logScaleBase;             // log scale base
  int logScaleXAxisFlag;        // flag to plot log scale on X axis
  int logScaleYAxisFlag;        // flag to plot log scale on Y axis
  int logScaleZAxisFlag;        // flag to plot log scale on Z axis
  int logScaleX2AxisFlag;       // flag to plot log scale on X2 axis
  int logScaleY2AxisFlag;       // flag to plot log scale on Y2 axis

  string barSize;               // size of bar in bar styles of plotting

  string boxWidth;              // with of boxes in boxes type of plotting

  FILE* gnuCommand;             // named pipe to gnuplot

  gnuCurveFit* curveFitOb;      // curve fitting object

  string getValue(ifstream& in);
  
};

#endif // gnuInterface_included

