/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"
#include "python/python_command.h"
#include "math.h"
#define NAME_LEN 100

static gdouble function 				(GtkPlot *plot, 
							 GtkPlotData *data, 
							 gdouble x, 
							 gboolean *error);

static void sg_dataset_refresh_python			(SGdataset *dataset);
static void sg_dataset_get_grid				(SGdataset *dataset);
static void get_neighbors				(GtkPlotData *data, 
							 gint index,
              						 gint *ne, gint *nn, 
							 gint *nw, gint *ns);

SGdataset *
sg_dataset_new (gint type)
{
  gint i;
  SGdataset *dataset;

  dataset = g_new(SGdataset, 1);

  switch(type){
    case SG_DATA_VBARS:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_bar_new(GTK_ORIENTATION_VERTICAL)); 
        break;
    case SG_DATA_HBARS:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_bar_new(GTK_ORIENTATION_HORIZONTAL)); 
        break;
    case SG_DATA_VBOXES:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_box_new(GTK_ORIENTATION_VERTICAL)); 
        break;
    case SG_DATA_HBOXES:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_box_new(GTK_ORIENTATION_HORIZONTAL)); 
        break;
    case SG_DATA_FLUX:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_flux_new()); 
        break;
    case SG_DATA_CONTOUR:
    case SG_DATA_CSURFACE:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_csurface_new()); 
        break;
    case SG_DATA_DMAP:
    case SG_DATA_SURFACE:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_surface_new()); 
        break;
    case SG_DATA_COLORS:
    case SG_DATA_BUBBLES:
    case SG_DATA_LPOINTS:
    case SG_DATA_FUNCTION:
    case SG_DATA_PYTHON:
    default:
        dataset->real_data = GTK_PLOT_DATA(gtk_plot_data_new()); 
        break;
  }

  dataset->type = type;

  gtk_plot_data_set_link(dataset->real_data, dataset);

  dataset->worksheet = NULL;

  dataset->x = -1;
  dataset->y = -1;
  dataset->z = -1;
  dataset->a = -1;
  dataset->dx = -1;
  dataset->dy = -1;
  dataset->dz = -1;
  dataset->da = -1;
  dataset->l = -1;

  dataset->exp = NULL;

  for (i=0; i<9; i++)
    dataset->p_exp[i] = NULL;

  return dataset;
}

SGdataset *
sg_dataset_new_with_worksheet  (gint type,
                                SGworksheet *worksheet,
				gint col_x,
				gint col_y,
				gint col_z,
				gint col_a,
				gint col_ex,
				gint col_ey,
				gint col_ez,
				gint col_ea,
				gint col_l)
{
  SGdataset *dataset;

  dataset = sg_dataset_new(type);
  sg_dataset_set_worksheet(dataset, worksheet, 
                           col_x, col_y, col_z, col_a,
                           col_ex, col_ey, col_ez, col_ea, col_l);

  return dataset;
}

SGdataset *
sg_dataset_new_function  (gchar *exp)
{
  SGdataset *dataset;

  dataset = sg_dataset_new(SG_DATA_FUNCTION);
  sg_dataset_set_exp(dataset, exp);

  return dataset;
}

SGdataset *
sg_dataset_new_python  (gint type, gchar *p_exp[9])
{ gint i;
  SGdataset *dataset;

  dataset = sg_dataset_new(type);

  if (p_exp)
     for (i=0;i<9;i++){
       if(p_exp[i])
         dataset->p_exp[i] = g_strdup(p_exp[i]);
     }

  return dataset;
}


void
sg_dataset_set_worksheet  (SGdataset *dataset,
			   SGworksheet *worksheet,
		   	   gint col_x,
			   gint col_y,
			   gint col_z,
			   gint col_a,
			   gint col_ex,
			   gint col_ey,
			   gint col_ez,
			   gint col_ea,
			   gint col_l)
{
  dataset->worksheet = worksheet;

  dataset->x = col_x;
  dataset->y = col_y;
  dataset->z = col_z;
  dataset->a = col_a;
  dataset->dx = col_ex;
  dataset->dy = col_ey;
  dataset->dz = col_ez;
  dataset->da = col_ea;
  dataset->l = col_l;

  sg_dataset_refresh_name(dataset);
  sg_dataset_refresh(dataset);
}


void
sg_dataset_add_point(SGdataset *dataset,
                     gdouble *x, gdouble *y, gdouble *z, gdouble *a, 
                     gdouble *ex, gdouble *ey, gdouble *ez, gdouble *ea,
                     gchar *label)
{
  gint num_points;

  num_points = dataset->real_data->num_points + 1;

  if(dataset->x != -1){
    dataset->real_data->x = g_renew(gdouble,dataset->real_data->x,num_points);
    dataset->real_data->x[num_points-1] = *x; 
  }

  if(dataset->y != -1){
    dataset->real_data->y = g_renew(gdouble,dataset->real_data->y,num_points);
    dataset->real_data->y[num_points-1] = *y; 
  }

  if(dataset->z != -1){
    dataset->real_data->z = g_renew(gdouble,dataset->real_data->z,num_points);
    dataset->real_data->z[num_points-1] = *z; 
  }

  if(dataset->a != -1){
    dataset->real_data->a = g_renew(gdouble,dataset->real_data->a,num_points);
    dataset->real_data->a[num_points-1] = *a; 
  }

  if(dataset->dx != -1){
    dataset->real_data->dx = g_renew(gdouble,dataset->real_data->dx,num_points);
    dataset->real_data->dx[num_points-1] = *ex; 
  }

  if(dataset->dy != -1){
    dataset->real_data->dy = g_renew(gdouble,dataset->real_data->dy,num_points);
    dataset->real_data->dy[num_points-1] = *ey; 
  }

  if(dataset->dz != -1){
    dataset->real_data->dz = g_renew(gdouble,dataset->real_data->dz,num_points);
    dataset->real_data->dz[num_points-1] = *ez; 
  }

  if(dataset->da != -1){
    dataset->real_data->da = g_renew(gdouble,dataset->real_data->da,num_points);
    dataset->real_data->da[num_points-1] = *ea; 
  }

  if(dataset->l != -1){
    dataset->real_data->labels = g_renew(gchar *,dataset->real_data->labels,
                                                    num_points);
    dataset->real_data->labels[num_points-1] = NULL;
    if(label) 
      dataset->real_data->labels[num_points-1] = g_strdup(label);
  }

  dataset->real_data->num_points++;
}

void
sg_dataset_refresh(SGdataset *dataset)
{
  SGworksheet *worksheet;
  GtkSheet *sheet;
  gdouble *x = NULL, *y = NULL, *z = NULL, *a = NULL;
  gdouble *ex = NULL, *ey = NULL, *ez = NULL, *ea = NULL;
  gchar **labels = NULL;
  gint row, col_x, col_y, col_z, col_a, col_ex, col_ey, col_ez, col_ea, col_l;
  gchar *text[9];
  gdouble val[9];
  gint numrows = 0;
  gint begin;
  gint end;
  gint n;
  gboolean error;

  if(dataset->type == SG_DATA_FUNCTION ||
     dataset->type == SG_DATA_PYTHON) return;

  if (dataset->p_exp[0]) 
  { 
    sg_dataset_refresh_python(dataset);
    return;
  }
  if(!dataset->worksheet) return;


  sg_dataset_free_points(dataset);

  worksheet = dataset->worksheet;
  sheet = GTK_SHEET(worksheet->sheet);

  col_x = dataset->x;
  col_y = dataset->y;
  col_z = dataset->z;
  col_a = dataset->a;
  col_ex = dataset->dx;
  col_ey = dataset->dy;
  col_ez = dataset->dz;
  col_ea = dataset->da;
  col_l = dataset->l;

  begin = MAX(0, worksheet->begin);
  end = sheet->maxallocrow;
 
  if(worksheet->end >= 0)
    end = MIN(end, worksheet->end);

  if(worksheet->begin <= sheet->maxallocrow){ 
    n = end - begin + 1;
    x = g_new(gdouble,n);
    y = g_new(gdouble,n);
    if(col_z >= 0)
      z = g_new(gdouble,n);
    if(col_a >= 0)
      a = g_new(gdouble,n);
    if(col_ex >= 0)
      ex = g_new(gdouble,n);
    if(col_ey >= 0)
      ey = g_new(gdouble,n);
    if(col_ez >= 0)
      ez = g_new(gdouble,n);
    if(col_ea >= 0)
      ea = g_new(gdouble,n);
    if(col_l >= 0)
      labels = g_new(gchar *,n);

    for(row = begin; row <= end; row++){
      gboolean error;

      text[0] = sg_worksheet_cell_get_text(worksheet, row, col_x);      
      text[1] = sg_worksheet_cell_get_text(worksheet, row, col_y);      

      val[0] = sg_worksheet_cell_get_double(worksheet, row, col_x,&error);      
      val[1] = sg_worksheet_cell_get_double(worksheet, row, col_y,&error);      

      if(col_z >= 0)
          val[2] = sg_worksheet_cell_get_double(worksheet, row, col_z,&error);      
      if(col_a >= 0)
          val[3] = sg_worksheet_cell_get_double(worksheet, row, col_a,&error);      
      if(col_ex >= 0)
          val[4] = sg_worksheet_cell_get_double(worksheet, row, col_ex,&error);      
      if(col_ey >= 0)
          val[5] = sg_worksheet_cell_get_double(worksheet, row, col_ey,&error);      
      if(col_ez >= 0)
          val[6] = sg_worksheet_cell_get_double(worksheet, row, col_ez,&error);      
      if(col_ea >= 0)
          val[7] = sg_worksheet_cell_get_double(worksheet, row, col_ea,&error);      
      if(col_l >= 0)
          text[8] = sg_worksheet_cell_get_text(worksheet, row, col_l);      

      if(text[0] && strlen(text[0]) > 0 &&
         text[1] && strlen(text[1]) > 0){

            x[numrows] = val[0];      
            y[numrows] = val[1];      

            if(col_z >= 0)
               z[numrows] = val[2];      

            if(col_a >= 0)
               a[numrows] = val[3];      

            if(col_ex >= 0)
               ex[numrows] = val[4];      

            if(col_ey >= 0)
               ey[numrows] = val[5];      

            if(col_ez >= 0)
               ez[numrows] = val[6];      

            if(col_ea >= 0)
               ea[numrows] = val[7];      

            if(col_l >= 0)
               if(text[8] != NULL)
                 labels[numrows] = g_strdup(text[8]);      
               else
                 labels[numrows] = NULL;

            numrows++;
      }
    }

    gtk_plot_data_set_points(dataset->real_data, x, y, ex, ey, numrows); 
    gtk_plot_data_set_z(dataset->real_data, z);
    gtk_plot_data_set_a(dataset->real_data, a);
    gtk_plot_data_set_dz(dataset->real_data, ez);
    gtk_plot_data_set_da(dataset->real_data, ea);
    gtk_plot_data_set_labels(dataset->real_data, labels);  

  }
 
  if(dataset->type == SG_DATA_SURFACE ||
     dataset->type == SG_DATA_CONTOUR ||
     dataset->type == SG_DATA_CSURFACE)
        sg_dataset_get_grid(dataset); 
}


static void
sg_dataset_refresh_python(SGdataset *dataset)
{
  gdouble *x[8];
  gchar **labels = NULL;
  gchar *p_exp[9];
  gchar name[NAME_LEN];
  gint i,num,len[4];

  sg_dataset_free_points(dataset);

  len[0]=-1;
  for (i=0;i<8;i++)
    {  
       if (dataset->p_exp[i] && strlen(dataset->p_exp[i]))
          x[i]=sg_eval_expr_double(dataset->p_exp[i], &len[i], len[0]);
       else 
          x[i]=NULL;
       if (!len[0]) return;
    }

  if (dataset->p_exp[i] && strlen(dataset->p_exp[i]))   
    labels=sg_eval_expr_string(dataset->p_exp[i], &num);


  gtk_plot_data_set_points(dataset->real_data, x[0], x[1], x[4], x[5], len[0]);
  gtk_plot_data_set_labels(dataset->real_data, labels);
  gtk_plot_data_set_z(dataset->real_data, x[2]);
  gtk_plot_data_set_dz(dataset->real_data, x[6]);
  gtk_plot_data_set_a(dataset->real_data, x[3]);
  gtk_plot_data_set_da(dataset->real_data, x[7]);
  dataset->worksheet=NULL;
}



void
sg_dataset_destroy(SGdataset *dataset)
{
  sg_dataset_free_points(dataset);

  gtk_widget_destroy(GTK_WIDGET(dataset->real_data));

  g_free(dataset);
}

void
sg_dataset_free_points(SGdataset* dataset)
{
  if(dataset->real_data->x) g_free(dataset->real_data->x);
  if(dataset->real_data->y) g_free(dataset->real_data->y);
  if(dataset->real_data->z) g_free(dataset->real_data->z);
  if(dataset->real_data->dx) g_free(dataset->real_data->dx);
  if(dataset->real_data->dy) g_free(dataset->real_data->dy);
  if(dataset->real_data->dz) g_free(dataset->real_data->dz);
  if(dataset->real_data->labels) {
      gint i;
      for (i = 0; i < dataset->real_data->num_points; i++){
            g_free(dataset->real_data->labels[i]);
      }
      g_free(dataset->real_data->labels);
  }

  dataset->real_data->x = NULL;
  dataset->real_data->y = NULL;
  dataset->real_data->z = NULL;
  dataset->real_data->dx = NULL;
  dataset->real_data->dy = NULL;
  dataset->real_data->dz = NULL;
  dataset->real_data->labels = NULL;
}


void
sg_dataset_set_name(SGdataset *dataset, gchar *name)
{
  gtk_plot_data_set_name(dataset->real_data, name);
}
                     
void
sg_dataset_set_exp(SGdataset *dataset, gchar *exp)
{
  if(dataset->exp){
    g_free(dataset->exp);
    dataset->exp = NULL;
  }

  dataset->real_data->is_function = TRUE;
  if(exp){
     dataset->exp = g_strdup(exp);
     dataset->real_data->function = (GtkPlotFunc)function;
  } else {
     dataset->real_data->function = NULL;
  }

}

void
sg_dataset_set_python(SGdataset *dataset, gchar *p_exp[9])
{ 
  gint i;

  for (i=0;i<9;i++)
  {  
    if(dataset->p_exp[i])
      { 
        g_free(dataset->p_exp[i]);
        dataset->p_exp[i] = NULL;
      }

     if(p_exp[i])
        dataset->p_exp[i] = g_strdup(p_exp[i]);
  }
  sg_dataset_refresh_python(dataset);
}


static gdouble
function (GtkPlot *plot, GtkPlotData *data, gdouble x, gboolean *error)
{
  SGdataset *dataset;
  gdouble val;

  dataset = (SGdataset *)gtk_plot_data_get_link(data);
  *error=sg_eval_func(dataset->exp, x, &val);

  *error = FALSE;

  return val;
}

static void
get_neighbors(GtkPlotData *data, gint index,
              gint *ne, gint *nn, gint *nw, gint *ns)
{
  gdouble x, y;
  gdouble d, dx, dy;
  gdouble de, ds, dw, dn;
  gdouble angle;
  gint i;

  x = data->x[index];
  y = data->y[index];

  *ne = *ns = *nw = *nn = -1;
  for (i = 0; i < data->num_points; i++){

    if(i != index){
      dx = data->x[i] - x;
      dy = data->y[i] - y;
      d = dx * dx + dy * dy;
  
      angle = 0.0; 
  
      if(dx != 0.0)
         angle = atan2(dy, dx);
      else if(d != 0.0)
         angle = asin(dy / sqrt(d));
  
      angle = angle / 3.141592654 * 180.0; 
      if(angle < 0.0) angle = 360.0 + angle;

      if(angle > 315.0 || angle <= 45.0)
         if(*ne == -1 || d < de){
          *ne = i;
            de = d;
         }
      if(angle > 45.0 && angle <= 135.0)
         if(*nn == -1 || d < dn){
            *nn = i;
            dn = d;
         }
      if(angle > 135.0 && angle <= 225.0)
         if(*nw == -1 || d < dw){
            *nw = i;
            dw = d;
         }
      if(angle > 225.0 && angle <= 315.0)
         if(*ns == -1 || d < ds){
            *ns = i;
            ds = d;
         }
    }
  }  
/*
  printf("NN: %d %d %d %d\n",*ne,*nn,*nw,*ns);
*/
}

static void
sg_dataset_get_grid(SGdataset *dataset)
{
  GtkPlotData *data;
  gint nx, ny, ntot;
  gint i, j;
  gint ne, nw, ns, nn;
  gint *index;
  gdouble *aux, *aux2;
  gint last, first;

  data = GTK_PLOT_DATA(dataset->real_data);

  ntot = data->num_points;

  index = g_new(gint, ntot + 1);
  aux = g_new(gdouble,ntot + 1);

  for (i = 0; i < ntot; i++){
    get_neighbors(data, i, &ne, &nn, &nw, &ns);
    if(nw == -1 && ns == -1) {
        last = i;
        break;
    }
  }

/*
  printf("ORIGIN %d %f %f\n",last,data->x[last],data->y[last]);
*/

  ny = 1;
  nx = 1;

  index[0] = last;
  first = last;

  for (i = 1; i < ntot; i++){
    get_neighbors(data, last, &ne, &nn, &nw, &ns);
    if(ne == -1){
        nx = 1;
        ny ++;
        get_neighbors(data, first, &ne, &nn, &nw, &ns);
        first = nn; 
        last = nn;
    } else {
        last = ne;
        nx++;
    }
    index[i] = last;
/*
    printf("%d %d\n",i, index[i]);
    printf("%d %f %f\n",last,data->x[last],data->y[last]);
*/
  }   

/*
  printf("NX NY: %d %d\n",nx, ny);
*/

  for(i = 0; i < ntot; i++){
    aux[i] = data->x[index[i]];
  }
  aux2 = data->x;
  data->x = aux; 
  aux = aux2;
  for(i = 0; i < ntot; i++){
    aux[i] = data->y[index[i]];
  }
  aux2 = data->y;
  data->y = aux; 
  aux = aux2;
  for(i = 0; i < ntot; i++){
    aux[i] = data->z[index[i]];
  }
  aux2 = data->z;
  data->z = aux; 
  aux = aux2;

  if(data->dx){
    for(i = 0; i < ntot; i++){
      aux[i] = data->dx[index[i]];
    }
    aux2 = data->dx;
    data->dx = aux; 
    aux = aux2;
  }
  if(data->dy){
    for(i = 0; i < ntot; i++){
      aux[i] = data->dy[index[i]];
    }
    aux2 = data->dy;
    data->dy = aux; 
    aux = aux2;
  }
  if(data->dz){
    for(i = 0; i < ntot; i++){
      aux[i] = data->dz[index[i]];
    }
    aux2 = data->dz;
    data->dz = aux; 
    aux = aux2;
  }
  if(data->a){
    for(i = 0; i < ntot; i++){
      aux[i] = data->a[index[i]];
    }
    aux2 = data->a;
    data->a = aux; 
    aux = aux2;
  }
  if(data->da){
    for(i = 0; i < ntot; i++){
      aux[i] = data->da[index[i]];
    }
    aux2 = data->da;
    data->da = aux; 
    aux = aux2;
  }

  GTK_PLOT_SURFACE(data)->nx = nx;
  GTK_PLOT_SURFACE(data)->ny = ny;
  g_free(aux);
  g_free(index);
}   

void
sg_dataset_refresh_name(SGdataset *dataset)
{
  gint i;
  gint data_type;
  gint dataset_columns[9];
  gchar name[100];
  gchar column_text[9][100];
  gchar *column_labels[][10] = {
  {"", "", "", "", "", "", "", "", ""},
  {"", "", "", "", "", "", "", "", ""},
  {"X", "Y", "Z", "", "Xerr", "Yerr", "Zerr", "", "Lbls"},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "DX", "DY", "DZ", "Amp", ""},
  {"X", "Y", "Z", "Size", "", "", "", "Amp", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "", "", "", "", ""},};
  gchar *names[] = {
                     "python",
                     "function",
                     "scatter",
                     "vbars",
                     "hbars",
                     "vboxes",
                     "hboxes",
                     "bubbles",
                     "vectors",
                     "color",
                     "density",
                     "contour",
                     "contour",
                     "surface" };

  data_type = dataset->type;

  if(!dataset->worksheet) return;

  if(data_type == SG_DATA_FUNCTION || data_type == SG_DATA_PYTHON) 
     return;

  dataset_columns[0] = dataset->x;
  dataset_columns[1] = dataset->y;
  dataset_columns[2] = dataset->z;
  dataset_columns[3] = dataset->a;
  dataset_columns[4] = dataset->dx;
  dataset_columns[5] = dataset->dy;
  dataset_columns[6] = dataset->dz;
  dataset_columns[7] = dataset->da;
  dataset_columns[8] = dataset->l;

  for(i = 0; i < 9; i++){
    gchar *column_name;
    if(dataset_columns[i] != -1){
      column_name = GTK_SHEET(dataset->worksheet->sheet)->column[dataset_columns[i]].name;
      if(i == 0)
        g_snprintf(column_text[i], 100, "%s(%s)", column_name,
                                         column_labels[data_type][i]);
      else
        g_snprintf(column_text[i], 100, ",%s(%s)", column_name,
                                         column_labels[data_type][i]);
    }
    else
      sprintf(column_text[i],"");
  }
  g_snprintf(name, 100, "%s:%s->%s%s%s%s%s%s%s%s%s",
                                       names[data_type],
                                       dataset->worksheet->name,
                                       column_text[0],
                                       column_text[1],
                                       column_text[2],
                                       column_text[3],
                                       column_text[4],
                                       column_text[5],
                                       column_text[6],
                                       column_text[7],
                                       column_text[8]);

  sg_dataset_set_name(dataset, name);
}
