/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "../pixmaps/sg_small.xpm"
#include "../pixmaps/python_small.xpm"

#include "main.h"

enum{
  IMPORT_WORKSHEET,
  EXPORT_WORKSHEET,
  IMPORT_PLOT,
  EXPORT_PLOT,
  OPEN_PROJECT,
  IMPORT_PROJECT,
  EXPORT_PROJECT,
  IMPORT_SCRIPT,
  IMPORT_IMAGE,
};

static gchar *titles[] = {
                       "Import Worksheet",
                       "Export Worksheet",
                       "Import Plot",
                       "Export Plot",
                       "Import",
                       "Read Project",
                       "Save Project",
                       "Import Script",
                       "Import Image"};

static gchar *project_export_formats[] = {
                       "SciGraphica XML (*.sg)",
                       NULL,};
static gchar *project_export_filters[] = {
                       "*.sg",
                       NULL,};
static gchar *project_open_formats[] = {
                       "SciGraphica XML (*.sg)",
                       NULL,};
static gchar *project_open_filters[] = {
                       "*.sg",
                       NULL,};
static gchar *project_import_formats[] = {
                       "SciGraphica XML (*.sg)",
                       "SciGraphica Plot (*.sgp)",
                       "SciGraphica Worksheet (*.sgw)",
                       "Python Script (*.py)",
                       NULL,};
static gchar *project_import_filters[] = {
                       "*.sg",
                       "*.sgp",
                       "*.sgw",
                       "*.py",
                       NULL,};
static gchar *worksheet_export_formats[] = {
                       "SciGraphica Worksheet (*.sgw)",
                       "ASCII",
                       "HTML",
                       "TeX",
                       NULL,};
static gchar *worksheet_export_filters[] = {
                       "*.sgw",
                       "*.dat",
                       "*.htm*",
                       "*.tex",
                       NULL,};
static gchar *worksheet_import_formats[] = {
                       "ASCII",
                       "SciGraphica Worksheet XML (*.sgw)",
                       NULL,};
static gchar *worksheet_import_filters[] = {
                       "*.dat",
                       "*.sgw",
                       NULL,};
static gchar *plot_export_formats[] = {
                       "PostScript (*.ps)",
                       "Encapsulated PostScript (*.eps)",
                       "Scigraphica Plot (*.sgp)",
                       NULL,};
static gchar *plot_export_filters[] = {
                       "*.ps",
                       "*.eps",
                       "*.sgp",
                       NULL,};
static gchar *plot_import_formats[] = {
                       "Scigraphica Plot XML (*.sgp)",
                       NULL,};
static gchar *plot_import_filters[] = {
                       "*.sgp",
                       NULL,};
static gchar *script_import_formats[] = {
                       "Python script (*.py)",
                       NULL,};
static gchar *script_import_filters[] = {
                       "*.py",
                       NULL,};
#ifdef WITH_GDK_IMLIB
static gchar *image_import_formats[] = {
/*
                       "All Images",
*/
                       "GIF (*.gif)",
                       "JPEG (*.jpg, *.jpeg)",
                       "PNG (*.png)",
                       "TIFF (*.tif, *.tiff)",
                       "Pixmap XPM (*.xpm)",
                       NULL,};
/*
static gchar *image_import_filters[] = {
                       "*.gif, *.jpg, *.jpeg, *.png, *.tif, *.tiff, *.xpm",
                       "*.gif",
                       "*.jpg, *.jpeg",
                       "*.png",
                       "*.tif, *.tiff",
                       "*.xpm",
                       NULL,};
*/
static gchar *image_import_filters[] = {
                       "*.gif",
                       "*.jp*g",
                       "*.png",
                       "*.ti*f",
                       "*.xpm",
                       NULL,};
#else
static gchar *image_import_formats[] = {
                       "Pixmap XPM (*.xpm)",
                       NULL,};
static gchar *image_import_filters[] = {
                       "*.xpm",
                       NULL,};
#endif

#ifdef WITH_GDK_IMLIB
#include <gdk_imlib.h>

static GdkImlibImage *image;
#endif

static gchar **labels;
static gchar **filters;
static gchar *path = NULL;
static GtkWidget *format_combo;
static GdkPixmap *return_pixmap;
static gint action;
static gint return_value;
static gboolean dialog_killed = FALSE;

static gboolean select_file		();
static gint cancel_action 		(GtkWidget *widget, gpointer data);
static gint ok_clicked 			(GtkWidget *widget, gpointer data);
static void new_format			(GtkWidget *widget, gpointer data);
static void set_path			(gchar **path, gchar *new_path);
static gboolean	check_file_read		(gchar *path);

static gboolean
mw_destroy(GtkWidget *widget)
{
/*  sg_dialog_kill(widget);
*/

  /* This is needed to get out of gtk_main */
  gtk_main_quit ();

  return FALSE;
}

gboolean
sg_project_import()
{
  action = IMPORT_PROJECT;
  path = last_project_path;
  labels = project_import_formats;
  filters = project_import_filters;

  return_value = select_file();
  
  return(return_value);
}

gboolean
sg_project_open()
{
  action = OPEN_PROJECT;
  path = last_project_path;
  labels = project_open_formats;
  filters = project_open_filters;

  return_value = select_file();
  
  return(return_value);
}

gboolean
sg_project_export()
{
  return_value = TRUE;

  action = EXPORT_PROJECT;
  path = last_project_path;
  labels = project_export_formats;
  filters = project_export_filters;
  if (!project_name_changed)
      return_value = select_file();
  else
      ok_clicked(NULL,NULL);

  return(return_value);
}

gboolean
sg_project_export_as()
{
  action = EXPORT_PROJECT;
  path = last_project_path;
  labels = project_export_formats;
  filters = project_export_filters;

  return_value = select_file();
  sg_project_set_title();

  return(return_value);
}

gboolean
sg_worksheet_import()
{
  action = IMPORT_WORKSHEET;
  path = last_worksheet_path;
  labels = worksheet_import_formats;
  filters = worksheet_import_filters;

  return_value = select_file();

  return(return_value);
}


gboolean
sg_worksheet_export()
{
  action = EXPORT_WORKSHEET;
  path = last_worksheet_path;
  labels = worksheet_export_formats;
  filters = worksheet_export_filters;

  return_value = select_file();

  return(return_value);
}

gboolean
sg_plot_export()
{
  action = EXPORT_PLOT;
  path = last_plot_path;
  labels = plot_export_formats;
  filters = plot_export_filters;

  return_value = select_file();

  return(return_value);
}

gboolean
sg_plot_import()
{
  action = IMPORT_PLOT;
  path = last_plot_path;
  labels = plot_import_formats;
  filters = plot_import_filters;

  return_value = select_file();

  return(return_value);
}

gboolean
sg_script_import()
{
  action = IMPORT_SCRIPT;
  path = last_project_path;
  labels = script_import_formats;
  filters = script_import_filters;

  return_value = select_file();

  return(return_value);
}

GdkPixmap *
sg_image_import()
{
  action = IMPORT_IMAGE;
  path = last_project_path;
  labels = image_import_formats;
  filters = image_import_filters;
  return_pixmap = NULL;

  return_value = select_file();
  return(return_pixmap);
}



static gboolean 
select_file()
{
  GtkWidget *filesel;
  GtkWidget *label;
  GtkWidget *bbox;
  gint type;
  gchar *aux_path;

  filesel = gtk_icon_file_selection_new (titles[action]);

  type = gtk_file_list_add_type
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         (const gchar **)sg_small_xpm);
  gtk_file_list_add_type_filter
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         type,
                         "*.sg");
  gtk_file_list_add_type_filter
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         type,
                         "*.sgp");
  gtk_file_list_add_type_filter
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         type,
                         "*.sgw");
  type = gtk_file_list_add_type
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         (const gchar **)python_small_xpm);
  gtk_file_list_add_type_filter
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         type,
                         "*.py");
  gtk_file_list_add_type_filter
                        (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         type,
                         "*.pyc");


/*  sg_dialog_new(filesel);
*/

  /* Set window as modal */
  gtk_window_set_modal (GTK_WINDOW(filesel),TRUE);

  gtk_table_resize(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area), 3, 4);

  label = gtk_label_new("File Format:        ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area),
                            label,
                            0, 1, 2, 3);
  gtk_widget_show(label);

  format_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area),
                            format_combo,
                            1, 3, 2, 3);
  gtk_widget_show(format_combo);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(format_combo)->entry), FALSE);

  if(path[strlen(path) - 1] != G_DIR_SEPARATOR_S[0])
    aux_path = g_strconcat(path, G_DIR_SEPARATOR_S, NULL);
  else
    aux_path = g_strdup(path);

  gtk_icon_file_selection_open_dir(GTK_ICON_FILESEL(filesel), aux_path);
  g_free(aux_path);

  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_CENTER);

  /* connect destroy signal to exit main loop */
  gtk_signal_connect (GTK_OBJECT (filesel), "destroy",
                      GTK_SIGNAL_FUNC (mw_destroy), NULL);


#ifdef WITH_GNOME
  gtk_widget_destroy(GTK_BIN(GTK_ICON_FILESEL(filesel)->ok_button)->child);
  gtk_widget_destroy(GTK_BIN(GTK_ICON_FILESEL(filesel)->cancel_button)->child);

  bbox=gtk_hbox_new(FALSE, 0);
  label = gtk_label_new("OK");
  gtk_box_pack_start(GTK_BOX(bbox),gnome_stock_pixmap_widget_new(filesel, GNOME_STOCK_BUTTON_OK),FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(bbox),label,FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(GTK_ICON_FILESEL(filesel)->ok_button),bbox);
  gtk_widget_show_all(GTK_ICON_FILESEL(filesel)->ok_button);

  bbox=gtk_hbox_new(FALSE, 0);
  label = gtk_label_new("Cancel");
  gtk_box_pack_start(GTK_BOX(bbox),gnome_stock_pixmap_widget_new(filesel, "Button_Cancel"),FALSE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(bbox),label,FALSE, TRUE, 0);
  gtk_container_add(GTK_CONTAINER(GTK_ICON_FILESEL(filesel)->cancel_button),bbox);
  gtk_widget_show_all(GTK_ICON_FILESEL(filesel)->cancel_button);
#endif

  /* Connect the ok_button function */
  gtk_signal_connect (GTK_OBJECT (GTK_ICON_FILESEL(filesel)->ok_button),
                      "clicked",
                      (GtkSignalFunc) ok_clicked, filesel);
  gtk_signal_connect (GTK_OBJECT (GTK_ICON_FILESEL(filesel)->file_entry),
                      "activate",
                      (GtkSignalFunc) ok_clicked, filesel);


  /* Connect the cancel_button to destroy the widget */
  gtk_signal_connect (GTK_OBJECT (GTK_ICON_FILESEL(filesel)->cancel_button),
                      "clicked",
                      (GtkSignalFunc) cancel_action, filesel);

  /* Connect the cancel_button to destroy the widget */
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(format_combo)->entry),
                      "changed",
                      (GtkSignalFunc) new_format, filesel);

  sg_combo_set_items(GTK_COMBO(format_combo), labels);

  gtk_widget_show(filesel);

  /* Wait until window gets destroyed */
  gtk_main();

  return return_value;
}

static void
new_format(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gint file_format;
  gchar *filter = NULL;

  child = GTK_LIST(GTK_COMBO(format_combo)->list)->selection->data;
  file_format = gtk_list_child_position(GTK_LIST(GTK_COMBO(format_combo)->list), child);

  filter = filters[file_format];

  gtk_entry_set_text(GTK_ENTRY(GTK_ICON_FILESEL(data)->filter_entry), filter); 
  gtk_file_list_set_filter(GTK_FILE_LIST(GTK_ICON_FILESEL(data)->file_list), filter);
}


static gint 
cancel_action (GtkWidget *widget, gpointer data) {
  gtk_widget_destroy(GTK_WIDGET(data));
  return FALSE; 
}

static gint
ok_clicked (GtkWidget *widget, gpointer data)
{
  GtkSheetRange range;
  GtkIconFileSel *filesel;
  GtkWidget *child = NULL;
  gchar *file_path, *file_name, *full_path, *entry_name;
  gint file_format;
  gint veto = TRUE;
  gboolean file_exists = FALSE;
  gchar message[255];
  GList *sheets;
  GdkBitmap *mask = NULL;

  if (data){
      filesel = GTK_ICON_FILESEL(data);
      file_name = gtk_file_list_get_filename(GTK_FILE_LIST(filesel->file_list));
      file_path = gtk_file_list_get_path(GTK_FILE_LIST(filesel->file_list));
      entry_name = gtk_entry_get_text(GTK_ENTRY(filesel->file_entry));
      if(entry_name)
          file_name = entry_name;
      if (!file_name || !strlen(file_name))
      {  sg_message_dialog ("No file selected!", 0);
         return TRUE;
      }
  }
  else{
      switch(action){
      case EXPORT_PROJECT:
          file_name=g_strdup(last_project_filename);
          file_path=g_strdup(last_project_path);
          break;
/* These two cases _cannot_ happen yet, but is put here for future reference */
      case EXPORT_WORKSHEET:
          file_name=g_strdup(last_worksheet_filename);
          file_path=g_strdup(last_worksheet_path);
          break;
      case EXPORT_PLOT:
          file_name=g_strdup(last_plot_filename);
          file_path=g_strdup(last_plot_path);
          break;
      }
  }

  full_path = g_strdup(file_path);
  full_path = g_strconcat(full_path, file_name, NULL);

  if (data){
      child = GTK_LIST(GTK_COMBO(format_combo)->list)->selection->data;
      file_format = gtk_list_child_position(GTK_LIST(GTK_COMBO(format_combo)->list), child);
  }

  file_exists = check_file_read(full_path);
  if(data && file_exists &&
     (action == EXPORT_WORKSHEET || action == EXPORT_PLOT ||
      action == EXPORT_PROJECT)){
        g_snprintf(message, 255,"File Exists. Overwrite %s?", file_name);
        veto = sg_accept_dialog(message, 1);
        if(veto == NO_CLICKED || veto == CANCEL_CLICKED){
            return TRUE;
        }
  }

  if(data && (action == EXPORT_WORKSHEET || action == EXPORT_PLOT ||
      action == EXPORT_PROJECT) && !sg_check_file_writeable(full_path)){
        g_snprintf(message, 255,"Cannot write to file:\n`%s'", file_name);
        sg_message_dialog(message, 0);
        return TRUE;

  }

  if(!file_exists &&
     (action == IMPORT_WORKSHEET || action == IMPORT_PLOT ||
      action == IMPORT_PROJECT || action == IMPORT_SCRIPT ||
      action == IMPORT_IMAGE || action == OPEN_PROJECT)){
        g_snprintf(message, 255,"Unable to open file:\n`%s'", full_path);
        sg_message_dialog(message, 0);
        return TRUE;
  }


  switch(action){
    case IMPORT_WORKSHEET:
      set_path(&last_worksheet_path, file_path);
      set_path(&last_worksheet_filename, file_name);
      switch(file_format){
        case 1:
            return_value = sg_project_file_import_xml(full_path, active_worksheet, NULL);
            sg_worksheet_update_exp_all(active_worksheet);
            break;
        case 0:
/*          return_value = sg_worksheet_file_import_ascii(active_worksheet, full_path);*/
            gtk_widget_hide(GTK_WIDGET(data));
            sg_import_dialog (full_path,file_name);
            break;
      }

      if(rename_worksheets) {
        sg_worksheet_rename(active_worksheet, file_name); 
        worksheet_name_changed=TRUE;
      }

      gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist1), 
                                    active_worksheet->icon);
      gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist1), NULL);

      break;
    case EXPORT_WORKSHEET:
      set_path(&last_worksheet_path, file_path);
      set_path(&last_worksheet_filename, file_name);
      switch(file_format){
        case 0:
           range.row0 = range.col0 = 0;
           range.coli = GTK_SHEET(active_worksheet->sheet)->maxcol;
           range.rowi = GTK_SHEET(active_worksheet->sheet)->maxrow;
/*
           full_path = g_strconcat(full_path, ".sgw", NULL);
*/
           return_value = sg_worksheet_file_export_xml(active_worksheet, full_path, &range, NULL);
           break;
        case 1:
           return_value = sg_worksheet_file_export_ascii(active_worksheet, full_path, NULL);
           break;
        case 2:
           return_value = sg_worksheet_file_export_html(active_worksheet, full_path, NULL);
           break;
        case 3:
           return_value = sg_worksheet_file_export_tex(active_worksheet, full_path, NULL);
           break;
      }
      break;
    case EXPORT_PLOT:
      set_path(&last_plot_path, file_path);
      set_path(&last_plot_filename, file_name);
      switch(file_format){
        case 2:
           return_value = sg_plot_file_export_xml(active_plot, full_path, NULL);
           break; 
        case 1:
/*           return_value =
*/
            gtk_plot_canvas_export_ps(GTK_PLOT_CANVAS(active_plot->real_canvas),
                                      full_path, 
                                      active_plot->orientation,
                                      TRUE,
                                      active_plot->page_size);
           break;
        case 0:
        default:
/*           return_value =
*/
            gtk_plot_canvas_export_ps(GTK_PLOT_CANVAS(active_plot->real_canvas),
                                      full_path, 
                                      active_plot->orientation,
                                      FALSE,
                                      active_plot->page_size);
           break ;
      }
      break;
    case IMPORT_PLOT:
      set_path(&last_plot_path, file_path);
      set_path(&last_plot_filename, file_name);
      switch(file_format){
        case 0:
          return_value = sg_project_file_import_xml(full_path, NULL, active_plot);
          break;
      }

      if(rename_plots) {
        sg_plot_rename(active_plot, file_name); 
        plot_name_changed=TRUE;
      }

      gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist2), 
                                    active_plot->icon);
      gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist2), NULL);
      break;
    case EXPORT_PROJECT:
      set_path(&last_project_path, file_path);
      set_path(&last_project_filename, file_name);
      return_value = sg_project_file_export_xml(full_path);
      project_name_changed = TRUE;
      sg_project_changed(FALSE);
      break;
    case OPEN_PROJECT:
      if(project_changed){
        g_snprintf(message, 80, "Save project \"%s\" ?", last_project_filename);
        veto = sg_accept_dialog(message, 0);
        switch (veto){
          case YES_CLICKED:
             return_value = sg_project_export();
             break;
          case NO_CLICKED:
             return_value = TRUE;
             break;
          case CANCEL_CLICKED:
             return_value = TRUE;
             if (data) gtk_widget_destroy(GTK_WIDGET(data));
             return TRUE;
        }
      }

      sg_project_close();
      sg_project_new();

      set_path(&last_project_path, file_path);
      set_path(&last_project_filename, file_name);

      switch(file_format){
        case 0:
        default:
          return_value = sg_project_file_import_xml(full_path, NULL, NULL);
          sheets=worksheets;
          while (sheets && sheets->data)
          {   
              sg_worksheet_update_exp_all((SGworksheet *)sheets->data);
              sheets=sheets->next;
          }
          project_name_changed=TRUE;
          break;
      }
      sg_project_changed(FALSE);
      break;
   case IMPORT_PROJECT:
      set_path(&last_project_path, file_path);
      set_path(&last_project_filename, file_name);
      switch(file_format){
        case 0:
          return_value = sg_project_file_import_xml(full_path, NULL, NULL);
          sheets=worksheets;
          while (sheets && sheets->data)
          {   
              sg_worksheet_update_exp_all((SGworksheet *)sheets->data);
              sheets=sheets->next;
          }
          break;
        case 1:
          active_plot = sg_project_new_plot();
          return_value = sg_project_file_import_xml(full_path, NULL, active_plot);
          break;
        case 2:
          active_worksheet = sg_project_new_worksheet();
          return_value = sg_project_file_import_xml(full_path, active_worksheet, NULL);
          sg_worksheet_update_exp_all(active_worksheet);
          break;
        case 3:
          sg_eval_script(full_path);
          break;
      }
      break;
    case IMPORT_SCRIPT:
      set_path(&last_project_path, file_path);
      sg_eval_script(full_path);
      break;
    case IMPORT_IMAGE:
      set_path(&last_project_path, file_path);
#ifdef WITH_GDK_IMLIB
      image = gdk_imlib_load_image(full_path);
      if(image){
        gdk_imlib_render(image, image->rgb_width, image->rgb_height);
        return_pixmap = gdk_imlib_copy_image(image);
        gdk_imlib_destroy_image(image);
      }
#else
      return_pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, 
                                             gdk_colormap_get_system(),
                                             &mask, NULL, full_path);
      if(mask) gdk_bitmap_unref(mask);
#endif

      break;
  }

  g_free(full_path);
  if (data) gtk_widget_destroy(GTK_WIDGET(data));

  return TRUE;
}

static void
set_path(gchar **path, gchar *new_path)
{
  if(*path)
    g_free(*path);

  *path = g_strdup(new_path);
}

static gboolean 
check_file_read(gchar *path)
{ struct stat fstatus;
  if(stat(path,&fstatus)<0)
      return FALSE;
  else if (S_ISREG(fstatus.st_mode)) return TRUE;

  return FALSE;
}

gboolean
sg_check_file_writeable(gchar *path)
{ struct stat fstatus;
  FILE *fp;
  if ((fp = fopen(path, "a")) == NULL || stat(path,&fstatus)<0 || !S_ISREG(fstatus.st_mode)){
       if (fp) fclose(fp);
       return FALSE;
  }
  if (fp) fclose(fp);
  return TRUE;
}

