/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "main.h"

static GtkWidget *thick_spin; 
static GtkWidget *length_spin; 
static GtkWidget *color_combo; 
static GtkWidget *bottom_scale; 
static GtkWidget *top_scale; 
static GtkWidget *left_scale; 
static GtkWidget *right_scale; 
static GtkWidget *xfactor; 
static GtkWidget *yfactor; 
static GtkWidget *zfactor; 
static GtkPlot *plot;
static SGlayer *the_layer;

void
sg_frame_dialog_update_plot(GtkWidget *widget)
{
  GtkColorCombo *combo;
  GdkColor color;
  gchar *color_name;
  gfloat line_width;
  gint ticks_length;
  SGplot *parent;

  line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(thick_spin)); 
  ticks_length = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(length_spin)); 
  combo = GTK_COLOR_COMBO(color_combo);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(color_combo->style->colormap, &color);

  plot->left->ticks_length = ticks_length;
  plot->right->ticks_length = ticks_length;
  plot->top->ticks_length = ticks_length;
  plot->bottom->ticks_length = ticks_length;

  plot->left->ticks_width = line_width;
  plot->right->ticks_width = line_width;
  plot->top->ticks_width = line_width;
  plot->bottom->ticks_width = line_width;

  if(the_layer->type != SG_PLOT_3D){
    plot->left_align = GTK_RANGE(left_scale)->adjustment->value;
    plot->right_align = GTK_RANGE(right_scale)->adjustment->value;
    plot->top_align = GTK_RANGE(top_scale)->adjustment->value;
    plot->bottom_align = GTK_RANGE(bottom_scale)->adjustment->value;
  } else {
    gtk_plot3d_frame_set_attributes(GTK_PLOT3D(plot), 0, line_width, &color);
    GTK_PLOT3D(plot)->xfactor = GTK_RANGE(xfactor)->adjustment->value;
    GTK_PLOT3D(plot)->yfactor = GTK_RANGE(yfactor)->adjustment->value;
    GTK_PLOT3D(plot)->zfactor = GTK_RANGE(zfactor)->adjustment->value;
  }  

  gtk_plot_axis_set_attributes(plot, 0, line_width, &color);
  gtk_plot_axis_set_attributes(plot, 1, line_width, &color);
  gtk_plot_axis_set_attributes(plot, 2, line_width, &color);
  gtk_plot_axis_set_attributes(plot, 3, line_width, &color);
  plot->left->labels_attr.fg = color;
  plot->right->labels_attr.fg = color;
  plot->top->labels_attr.fg = color;
  plot->bottom->labels_attr.fg = color;

  parent = (SGplot *)the_layer->parent;
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(parent->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(parent->real_canvas));
}

static void
init_dialog()
{
  GdkColor line_color;
  gfloat line_width;
  GtkPlotLineStyle line_style;

  if(the_layer->type != SG_PLOT_3D)
    gtk_plot_axis_get_attributes(plot, 0, &line_width, &line_color);
  else 
    gtk_plot3d_frame_get_attributes(GTK_PLOT3D(plot), &line_style, &line_width, &line_color);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(thick_spin), line_width); 
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(length_spin), plot->top->ticks_length); 

  sg_color_combo_init(GTK_COLOR_COMBO(color_combo), line_color);

  if(the_layer->type != SG_PLOT_3D){
    gtk_adjustment_set_value(GTK_RANGE(left_scale)->adjustment, 
                             plot->left_align);
    gtk_adjustment_set_value(GTK_RANGE(right_scale)->adjustment, 
                             plot->right_align);
    gtk_adjustment_set_value(GTK_RANGE(top_scale)->adjustment, 
                             plot->top_align);
    gtk_adjustment_set_value(GTK_RANGE(bottom_scale)->adjustment, 
                             plot->bottom_align);
  } else {
    gtk_adjustment_set_value(GTK_RANGE(xfactor)->adjustment, 
                             GTK_PLOT3D(plot)->xfactor);
    gtk_adjustment_set_value(GTK_RANGE(yfactor)->adjustment, 
                             GTK_PLOT3D(plot)->yfactor);
    gtk_adjustment_set_value(GTK_RANGE(zfactor)->adjustment, 
                             GTK_PLOT3D(plot)->zfactor);
  }

}

GtkWidget *
sg_frame_dialog_new (SGlayer *layer)
{
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *table;
  GtkWidget *label;
  GtkRequisition req;
  GtkAdjustment *adjust;

  the_layer = layer;
  plot = GTK_PLOT(layer->real_plot);

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);

  /* Poperties */

  frame = gtk_frame_new("Properties");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);

  table=gtk_table_new(3, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new("Thickness(pts) ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new("Ticks length ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  label = gtk_label_new("Color ");
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

  adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  thick_spin = gtk_spin_button_new(adjust, 0, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(thick_spin), TRUE);

  adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 20., 1., 1., 0.);
  length_spin = gtk_spin_button_new(adjust, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(length_spin), TRUE);

  color_combo = gtk_color_combo_new();

  gtk_widget_size_request(thick_spin, &req);
  req.width /= 2;
  gtk_widget_set_usize(thick_spin, req.width, req.height);
  gtk_widget_set_usize(length_spin, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), thick_spin, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), length_spin, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), color_combo, 1, 2, 2, 3);


  /* Alignments */
  if(layer->type != SG_PLOT_3D){

    frame = gtk_frame_new("Axis position");
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);

    table=gtk_table_new(7, 3, FALSE);  
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table), 2);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_container_add(GTK_CONTAINER(frame), table);
  
    label = gtk_label_new("left");
    gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);
    label = gtk_label_new("right");
    gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
    gtk_table_attach_defaults(GTK_TABLE(table),label, 2, 3, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table),
                              gtk_label_new("Left axis"), 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table),
                              gtk_label_new("Right axis"), 0, 1, 2, 3);
  
    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    left_scale = gtk_hscale_new(adjust);
    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    right_scale = gtk_hscale_new(adjust);
  
    gtk_table_attach_defaults(GTK_TABLE(table), left_scale, 1, 3, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table), right_scale, 1, 3, 2, 3);
  
    gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 3, 3, 4);
  
    label = gtk_label_new("bottom");
    gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 4, 5);
    label = gtk_label_new("top");
    gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
    gtk_table_attach_defaults(GTK_TABLE(table),label, 2, 3, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE(table),
                              gtk_label_new("Bottom axis"), 0, 1, 5, 6);
    gtk_table_attach_defaults(GTK_TABLE(table),
                              gtk_label_new("Top axis"), 0, 1, 6, 7);
  
    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    bottom_scale = gtk_hscale_new(adjust);
    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    top_scale = gtk_hscale_new(adjust);
  
    gtk_table_attach_defaults(GTK_TABLE(table), bottom_scale, 1, 3, 5, 6);
    gtk_table_attach_defaults(GTK_TABLE(table), top_scale, 1, 3, 6, 7);
  
  } else {

    frame = gtk_frame_new("Scale factors");
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);

    table=gtk_table_new(3, 3, FALSE);  
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table), 2);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_container_add(GTK_CONTAINER(frame), table);
  
    label = gtk_label_new("X");
    gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    label = gtk_label_new("Y");
    gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
    gtk_table_attach_defaults(GTK_TABLE(table),label, 0, 1, 1, 2);
    label = gtk_label_new("Z");
    gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
    gtk_table_attach_defaults(GTK_TABLE(table),label, 0, 1, 2, 3);

    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    xfactor = gtk_hscale_new(adjust);
    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    yfactor = gtk_hscale_new(adjust);
    adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .01, .01, 0.);
    zfactor = gtk_hscale_new(adjust);
  
    gtk_table_attach_defaults(GTK_TABLE(table), xfactor, 1, 3, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table), yfactor, 1, 3, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table), zfactor, 1, 3, 2, 3);
  }

  /* connect signals */

  init_dialog();
  return main_box;
}
