/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __SG_MENU_H__
#define __SG_MENU_H__

#ifndef GNOME_APP_HELPER_H

#define GNOME_PAD 8
#define GNOMEUIINFO_KEY_UIDATA          "uidata"
#define GNOMEUIINFO_KEY_UIBDATA         "uibdata"

/* These values identify the type of pixmap used in an item */
typedef enum {
        GNOME_APP_PIXMAP_NONE,          /* No pixmap specified */
        GNOME_APP_PIXMAP_STOCK,         /* Use a stock pixmap (GnomeStock) */
        GNOME_APP_PIXMAP_DATA,          /* Use a pixmap from inline xpm data */
        GNOME_APP_PIXMAP_FILENAME       /* Use a pixmap from the specified
                                           filename */
} GnomeUIPixmapType;


/* These values identify the item type that a particular GnomeUIInfo structure
 * specifies */
typedef enum {
        GNOME_APP_UI_ENDOFINFO,         /* No more items, use it at the end of
                                           an array */
        GNOME_APP_UI_ITEM,              /* Normal item, or radio item if it is
                                           inside a radioitems group */
        GNOME_APP_UI_TOGGLEITEM,        /* Toggle (check box) item */
        GNOME_APP_UI_RADIOITEMS,        /* Radio item group */
        GNOME_APP_UI_SUBTREE,           /* Item that defines a
                                           subtree/submenu */
        GNOME_APP_UI_SEPARATOR,         /* Separator line (menus) or blank
                                           space (toolbars) */
        GNOME_APP_UI_HELP,              /* Create a list of help topics,
                                           used in the Help menu */
        GNOME_APP_UI_BUILDER_DATA,      /* Specifies the builder data for the
                                           following entries, see code for
                                           further info */
        GNOME_APP_UI_ITEM_CONFIGURABLE, /* A configurable menu item. */
        /* one should be careful when using
         * gnome_app_create_*_[custom|interp|with_data]() functions with
         * GnomeUIInfo arrays containing GNOME_APP_UI_BUILDER_DATA items since
         * their GnomeUIBuilderData structures completely override the ones
         * generated or supplied by the above functions. */
        GNOME_APP_UI_SUBTREE_STOCK      /* Item that defines a
                                           subtree/submenu, same as GNOME_APP_UI_SUBTREE,
                                           but the texts should be looked up in the
                                           gnome-libs catalog
                                        */
} GnomeUIInfoType;


typedef struct {
        GnomeUIInfoType type;           	/* Type of item */
        gchar *label;                   /* String to use in the label */
        gchar *hint;                    /* For toolbar items, the tooltip. For
                                           menu items, the status bar message */        gpointer moreinfo;              /* For an item, toggleitem, or
                                           radioitem, this is a pointer to the
                                           function to call when the item is
                                           activated. For a subtree, a pointer
                                           to another array of GnomeUIInfo
                                           structures. For a radioitem lead
                                           entry, a pointer to an array of
                                           GnomeUIInfo structures for the radio
                                           item group. For a help item,
                                           specifies the help node to load
                                           (i.e. the application's identifier)
                                           or NULL for the main program's name.
                                           For builder data, points to the
                                           GnomeUIBuilderData structure 
                                           the following items */

        gpointer user_data;             /* Data pointer to pass to callbacks */
        gpointer unused_data;           /* Reserved for future expansion,
                                           should be NULL */
        GnomeUIPixmapType pixmap_type;  	/* Type of pixmap for the item */
        gconstpointer pixmap_info;      /* Pointer to the pixmap information:
                                         *
                                         * For SG_PIXMAP_STOCK, a
                                         * pointer to the stock icon name.
                                         *
                                         * For SG_PIXMAP_DATA, a
                                         * pointer to the inline xpm data.
                                         *
                                         * For SG_PIXMAP_FILENAME, a
                                         * pointer to the filename string.
                                         */
        guint accelerator_key;          /* Accelerator key, or 0 for none */
        GdkModifierType ac_mods;        /* Mask of modifier keys for the
                                           accelerator */

        GtkWidget *widget;              /* Filled in by gnome_app_create*, you
                                           can use this to tweak the widgets
                                           once they have been created */
} GnomeUIInfo;

/* Types useful to language bindings */

typedef struct _GnomeUIBuilderData GnomeUIBuilderData;

typedef void (* GnomeUISignalConnectFunc) (GnomeUIInfo        *uiinfo,
                                        gchar           *signal_name,
                                        GnomeUIBuilderData *uibdata);

struct _GnomeUIBuilderData {
        GnomeUISignalConnectFunc connect_func;  /* Function that connects to the item's signals */
        gpointer data;                          /* User data pointer */
        gboolean is_interp;                     /* Should use gtk_signal_connect_interp or normal gtk_signal_connect? */
        GtkCallbackMarshal relay_func;          /* Marshaller function for language bindings */
        GtkDestroyNotify destroy_func;          /* Destroy notification function for language bindings */
};

/* Used to terminate an array of GnomeUIInfo structures */
#define GNOMEUIINFO_END                 { GNOME_APP_UI_ENDOFINFO, NULL, NULL, NULL, NULL, NULL,         \
                                          (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert a separator line (on a menu) or a blank space (on a toolbar) */
#define GNOMEUIINFO_SEPARATOR           { GNOME_APP_UI_SEPARATOR, NULL, NULL, NULL, NULL, NULL,         \
                                          (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert an item with an inline xpm icon */
#define GNOMEUIINFO_ITEM(label, tooltip, callback, xpm_data) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
                GNOME_APP_PIXMAP_DATA, xpm_data, 0, (GdkModifierType) 0, NULL}

/* Insert an item with a stock icon */
#define GNOMEUIINFO_ITEM_STOCK(label, tooltip, callback, stock_id) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
                GNOME_APP_PIXMAP_STOCK, stock_id, 0, (GdkModifierType) 0, NULL }
/* Insert an item with no icon */
#define GNOMEUIINFO_ITEM_NONE(label, tooltip, callback) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
                GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert an item with an inline xpm icon and a user data pointer */
#define GNOMEUIINFO_ITEM_DATA(label, tooltip, callback, user_data, xpm_data) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, user_data, NULL, \
                GNOME_APP_PIXMAP_DATA, xpm_data, 0, (GdkModifierType) 0, NULL }

/* Insert a toggle item (check box) with an inline xpm icon */
#define GNOMEUIINFO_TOGGLEITEM(label, tooltip, callback, xpm_data) \
        { GNOME_APP_UI_TOGGLEITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
                GNOME_APP_PIXMAP_DATA, xpm_data, 0, (GdkModifierType) 0, NULL }

/* Insert a toggle item (check box) with an inline xpm icon and a user data pointer */
#define GNOMEUIINFO_TOGGLEITEM_DATA(label, tooltip, callback, user_data, xpm_data)              \
                                        { GNOME_APP_UI_TOGGLEITEM, label, tooltip, (gpointer)callback,  \
                                          user_data, NULL, GNOME_APP_PIXMAP_DATA, xpm_data,     \
                                          0, (GdkModifierType) 0, NULL }

/* Insert all the help topics based on the application's id */
#define GNOMEUIINFO_HELP(app_name) \
        { GNOME_APP_UI_HELP, NULL, NULL, app_name, NULL, NULL, \
                (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert a subtree (submenu) */
#define GNOMEUIINFO_SUBTREE(label, tree) \
        { GNOME_APP_UI_SUBTREE, label, NULL, tree, NULL, NULL, \
                (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert a subtree with a hint */
#define GNOMEUIINFO_SUBTREE_HINT(label, hint, tree) \
        { GNOME_APP_UI_SUBTREE, label, hint, tree, NULL, NULL, \
                (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert a subtree (submenu) with a stock icon */
#define GNOMEUIINFO_SUBTREE_STOCK(label, tree, stock_id) \
        { GNOME_APP_UI_SUBTREE, label, NULL, tree, NULL, NULL, \
                GNOME_APP_PIXMAP_STOCK, stock_id, 0, (GdkModifierType) 0, NULL }
/* Insert a list of radio items */
#define GNOMEUIINFO_RADIOLIST(list) \
        { GNOME_APP_UI_RADIOITEMS, NULL, NULL, list, NULL, NULL, \
                (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0, NULL }

/* Insert a radio item with an inline xpm icon */
#define GNOMEUIINFO_RADIOITEM(label, tooltip, callback, xpm_data) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
                GNOME_APP_PIXMAP_DATA, xpm_data, 0, (GdkModifierType) 0, NULL }

/* Insert a radio item with an inline xpm icon and a user data pointer */
#define GNOMEUIINFO_RADIOITEM_DATA(label, tooltip, callback, user_data, xpm_data)               \
                                        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback,        \
                                          user_data, NULL, GNOME_APP_PIXMAP_DATA, xpm_data,     \
                                          0, (GdkModifierType) 0, NULL }


#endif /* GNOME_APP_HELPER_H */


void
sg_ui_create_menus 			(GtkWidget *container, 
					 GnomeUIInfo *uiinfo);
void
sg_ui_create_menus_custom 		(GtkWidget *app,
                               		 GnomeUIInfo *uiinfo,
                               		 GnomeUIBuilderData *uibdata);

void
sg_ui_fill_menu_custom 			(GtkMenuShell       *menu_shell,
                        		 GnomeUIInfo        *uiinfo,
                        		 GnomeUIBuilderData *uibdata,
                        		 GtkAccelGroup      *accel_group,
                        		 gboolean            uline_accels,
                        		 gint                pos);

void
sg_ui_create_toolbar 			(GtkWidget *container, 
					 GnomeUIInfo *uiinfo);
void
sg_ui_create_toolbar_custom 		(GtkWidget *app,
                                	 GnomeUIInfo *uiinfo,
                                 	 GnomeUIBuilderData *uibdata);

void
sg_ui_fill_toolbar_custom 		(GtkToolbar *toolbar,
                                         GnomeUIInfo *uiinfo,
                               		 GnomeUIBuilderData *uibdata,
                               		 GtkAccelGroup *accel_group);

#endif /* __SG_MENU_H__ */

