/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include "main.h"
#include "../pixmaps/plot_icon2.xpm"
#define EVENT_METHOD(i, x) GTK_WIDGET_CLASS(GTK_OBJECT(i)->klass)->x

const gdouble unit_pt[]={ 1.0, 2.83, 28.35, 72.0, (72.0/SG_DPI)};

GdkPixmap *plot_icon_pixmap;
GdkBitmap *plot_icon_mask;


static void	sg_plot_unmap				(GtkWidget *window, 
							 GdkEvent *event);
static void	sg_plot_destroy				(GtkWidget *window); 
static gint	sg_plot_delete				(GtkWidget *window); 
static void     reorder_buttons				(SGplot *plot);
static void     button_toggled				(GtkWidget *widget, 
							 gpointer data);
static gint     button_clicked				(GtkWidget *widget, 
						         GdkEventButton *event,
							 gpointer data);

static gint 	edit_text				(GtkWidget *widget, 
							 GdkEventButton *event);
static GtkWidget *open_text_dialog			(GtkPlotText *text); 
static gint 	apply_dialog_text			(GtkWidget *dialog);
static gint 	click_on_item				(GtkPlotCanvas *canvas,
							 GdkEvent *event,
		 					 gpointer item_data);
static void 	activate_layer				(GtkPlotCanvas *canvas);
static gboolean focus_in				(GtkWidget *widget,
							 GdkEventFocus *event);
static gboolean resize_item				(GtkPlotCanvas *canvas,
							 GtkPlotCanvasChild *item,
							 gdouble width, 
							 gdouble height, 
							 gpointer data);
static gboolean move_item				(GtkPlotCanvas *canvas,
							 GtkPlotCanvasChild *item,
							 gdouble x, gdouble y, 
							 gpointer data);

static gboolean key_press				(GtkWidget *widget, 
							 GdkEventKey *key);
static void update_ruler_expose_x			(GtkWidget *scroll, 
							 gpointer data);
static void update_ruler_expose_y			(GtkWidget *scroll, 
							 gpointer data);
static void sg_plot_motion				(GtkWidget *widget, 
							 GdkEventMotion *event,
							 gpointer data);
static void canvas_changed				(GtkPlotCanvas *canvas,
							 gpointer data);

gint sg_char_dialog(gchar *font_prefix);

SGplot *
sg_plot_new(gchar *name)
{
   SGplot *plot;

   plot = g_new(SGplot, 1);

   plot->tool = SG_TOOL_ARROW;
   plot->orientation = GTK_PLOT_PORTRAIT;
   plot->page_size = GTK_PLOT_LETTER;
   plot->scale = .65;
   plot->page_width = GTK_PLOT_LETTER_W;
   plot->page_height = GTK_PLOT_LETTER_H;
   plot->page_units = SG_UNIT_IN;

   plot->hruler = NULL;
   plot->vruler = NULL;

   plot->real_canvas = gtk_plot_canvas_new(GTK_PLOT_LETTER_W,
                                           GTK_PLOT_LETTER_H,
                                           plot->scale);

   plot->layers = NULL;
   plot->nlayers = 0;
	
   plot->name = g_strdup(name);

   plot->window = NULL;
   plot->sw = NULL;
   plot->is_mapped = FALSE;
   plot->toolbox = NULL;

   plot->x = 20;
   plot->y = 20;
   plot->width = 500;
   plot->height = 500;

   sg_plot_rescale(plot, .65);

 /* Signals */
   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "changed",
                      (GtkSignalFunc)canvas_changed, NULL);

   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "select_item",
                      (GtkSignalFunc)click_on_item, NULL);

   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "button_press_event",
                      (GtkSignalFunc)sg_plot_menu_show, NULL);

   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "button_press_event",
                      (GtkSignalFunc)edit_text, NULL);

   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "key_press_event",
                      (GtkSignalFunc)key_press, NULL);

   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "move_item",
                      (GtkSignalFunc)move_item, NULL);

   gtk_signal_connect(GTK_OBJECT(plot->real_canvas), "resize_item",
                      (GtkSignalFunc)resize_item, NULL);

   return plot;
}   

SGplot *
sg_plot_new_with_layer(gint plot_type, gchar *name)
{
   SGplot *plot;
   SGlayer *default_layer;

   plot = sg_plot_new(name);

   if(plot_type != SG_PLOT_POLAR)
     default_layer = sg_layer_new(plot_type, .65, .45);
   else
     default_layer = sg_layer_new(plot_type, .75, .45);

   sg_plot_add_layer(plot, default_layer, .175, .15);

   return plot;
}


void
sg_plot_add_layer(SGplot *plot, SGlayer *layer, gdouble x, gdouble y)
{
   gint nlayers;

   layer->parent = plot;

   nlayers = plot->nlayers;

   gtk_fixed_put(GTK_FIXED(plot->real_canvas), 
                 layer->button,
                 nlayers * 24, 0);
   gtk_widget_show(layer->button);
   sg_layer_button_set_label(layer, nlayers + 1);

   gtk_plot_canvas_add_plot(GTK_PLOT_CANVAS(plot->real_canvas), 
                            GTK_PLOT(layer->real_plot), x, y);

   gtk_widget_show(GTK_WIDGET(layer->real_plot));
   plot->layers = g_list_append(plot->layers, layer);

   sg_plot_set_active_layer(plot, layer);
   plot->nlayers++;

   gtk_signal_connect(GTK_OBJECT(layer->button), "toggled",
                      GTK_SIGNAL_FUNC(button_toggled), layer);
   gtk_signal_connect(GTK_OBJECT(layer->button), "button_press_event",
                      GTK_SIGNAL_FUNC(button_clicked), layer);

   sg_layer_control_refresh(plot->name);
}

void
sg_plot_move_layer(SGplot *plot, SGlayer *layer, gdouble x, gdouble y)
{
   gint nlayers;

   layer->parent = plot;

   gtk_plot_move(GTK_PLOT(layer->real_plot), x, y);

   gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
   gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

void
sg_plot_resize_layer(SGplot *plot, SGlayer *layer, gdouble w, gdouble h)
{
   gtk_plot_resize(GTK_PLOT(layer->real_plot), w, h);

   gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
   gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));
}

void
sg_plot_remove_layer(SGplot *plot, SGlayer *layer)
{
  GList *layers = NULL;

  if(plot->nlayers == 0) return;

  sg_layer_clear(layer);

  gtk_container_remove(GTK_CONTAINER(plot->real_canvas), layer->real_plot); 
  gtk_container_remove(GTK_CONTAINER(plot->real_canvas), layer->button); 

  layers = plot->layers;
  while(layers) {
     if(layers->data == layer){
         plot->layers = g_list_remove_link(plot->layers, layers);
         g_list_free_1(layers);
         plot->nlayers--;
         break;
     }
     layers = layers->next;
  }

  if(layer) g_free(layer);

  reorder_buttons(plot);

  if(plot->nlayers > 0) 
      activate_layer(GTK_PLOT_CANVAS(plot->real_canvas)); 

  sg_layer_control_refresh(plot->name);
}

static void
reorder_buttons(SGplot *plot)
{
  GList *layers = NULL;
  SGlayer *layer;
  gint nlayers = 0;

  layers = plot->layers;

  while(layers) {
     layer = (SGlayer *)layers->data;

     gtk_fixed_move(GTK_FIXED(plot->real_canvas), 
                   layer->button,
                   nlayers * 24, 0);
     gtk_widget_show(layer->button);
     sg_layer_button_set_label(layer, nlayers + 1);

     nlayers++;
     layers = layers->next;
  }

}


static void
button_toggled(GtkWidget *widget, gpointer data)
{
  SGlayer *layer;

  layer = (SGlayer *)data;
  sg_plot_set_active_layer((SGplot *)layer->parent, layer);
}

static gint
button_clicked(GtkWidget* widget, GdkEventButton *event, gpointer data)
{
  SGlayer *layer;
  GList *llink;
  GdkModifierType mods;
  gchar path[1000];

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;
  if(event->type != GDK_2BUTTON_PRESS) return FALSE;

  layer = (SGlayer *)data;
  llink = g_list_find(active_plot->layers, layer);
  sprintf(path, "%s:%d:dataset", active_plot->name,
          g_list_position(active_plot->layers, llink)+1);
  sg_layer_control(path);

  gtk_grab_remove(widget);
  return FALSE;
}
   
void
sg_plot_set_active_layer(SGplot *plot, SGlayer *layer)
{
  SGlayer *child = NULL;
  GList *list;
  active_plot = plot;

  plot->active_layer = layer;

  list = plot->layers;
  while(list){
    child = (SGlayer *)list->data;
    GTK_BUTTON(child->button)->button_down = FALSE;
    GTK_TOGGLE_BUTTON(child->button)->active = FALSE;
    gtk_widget_set_state(child->button, GTK_STATE_NORMAL);
    gtk_widget_draw(child->button, NULL);
    list = list->next;
  }

  if(!layer) return;

  gtk_plot_canvas_set_active_plot(GTK_PLOT_CANVAS(plot->real_canvas),
                                  GTK_PLOT(layer->real_plot));

  GTK_BUTTON(layer->button)->button_down = TRUE;
  GTK_TOGGLE_BUTTON(layer->button)->active = TRUE;
  gtk_widget_set_state(layer->button, GTK_STATE_ACTIVE);
  gtk_widget_draw(child->button, NULL);
}


void
sg_plot_open(SGplot *plot)
{
 GtkWidget *table, *hbox, *hbox2, *vbox, *top_hruler, *left_vruler;
 GtkWidget *frame,*sbar_vert, *sbar_horiz;
 GtkStyle *style;
 GtkWidget *toolbar;
 gboolean changed = project_changed;
#ifndef WITH_GNOME
  GtkWidget *hb;
#endif

 

 if(plot->is_mapped) {
      gdk_window_raise(plot->window->window);
      return;
 }

 vbox = gtk_vbox_new(FALSE, 0);
 plot->canvas_box = gtk_vbox_new(TRUE, 0);
 gtk_box_set_spacing(GTK_BOX(vbox), 5);
 hbox = gtk_hbox_new(FALSE, 0);
 hbox2 = gtk_hbox_new(FALSE, 0);
 table = gtk_table_new(4, 3, FALSE);


#ifdef WITH_GNOME
 plot->window = gnome_app_new (PACKAGE,plot->name);
 gnome_app_set_contents(GNOME_APP(plot->window),vbox);
#else
 plot->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
 gtk_container_add(GTK_CONTAINER(plot->window), vbox);
#endif



 toolbar = sg_plot_build_toolbar(plot);

 gtk_window_set_title(GTK_WINDOW(plot->window), plot->name);
 gtk_window_set_policy(GTK_WINDOW(plot->window), TRUE, TRUE, FALSE);

 gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
 plot->sw=gtk_viewport_new( NULL,NULL);
 sbar_vert=gtk_vscrollbar_new(gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw)));
 sbar_horiz=gtk_hscrollbar_new(gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw)));
 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(sbar_vert), 3, 4, 1, 2,
                  0,
                  GTK_FILL|GTK_SHRINK|GTK_EXPAND, 0, 0);
 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(sbar_horiz), 2, 3, 2, 3,
                  GTK_FILL|GTK_SHRINK|GTK_EXPAND,
                  0, 0, 0);



/* gtk_signal_connect(GTK_OBJECT(plot->window),"unmap_event",
                    (GtkSignalFunc) sg_plot_unmap, NULL);
*/

 gtk_signal_connect(GTK_OBJECT(plot->window),"destroy",
                    (GtkSignalFunc) sg_plot_destroy, NULL);


 gtk_signal_connect(GTK_OBJECT(plot->window),"delete_event",
                    (GtkSignalFunc) sg_plot_delete, NULL);


 gtk_signal_connect(GTK_OBJECT(plot->window),"focus_in_event",
                    (GtkSignalFunc) focus_in, NULL);


 plot->toolbox = sg_toolbox_new();
 gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(plot->toolbox), FALSE, FALSE, 0);
 gtk_widget_show_all(plot->toolbox);

 gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);

 gtk_container_add(GTK_CONTAINER(plot->sw),hbox2);
 gtk_box_pack_start(GTK_BOX(hbox2), GTK_WIDGET(plot->canvas_box), TRUE, FALSE, 0);
 gtk_box_pack_start(GTK_BOX(plot->canvas_box), GTK_WIDGET(plot->real_canvas), TRUE, FALSE, 0);
 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(plot->sw), 2, 3, 1, 2,
                  GTK_FILL|GTK_SHRINK|GTK_EXPAND,
                  GTK_FILL|GTK_SHRINK|GTK_EXPAND, 0, 0);

/*-------------------------------------------------------------*/
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw))),
                    "changed",
                    (GtkSignalFunc)update_ruler_expose_x,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw))),
                    "value_changed",
                    (GtkSignalFunc)update_ruler_expose_x,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw))),
                    "changed",
                    (GtkSignalFunc)update_ruler_expose_y,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw))),
                    "value_changed",
                    (GtkSignalFunc)update_ruler_expose_y,(gpointer)plot);

/*---------------------------------------------------------------*/

 plot->hruler = top_hruler = gtk_hruler_new();
 gtk_ruler_set_range(GTK_RULER(top_hruler), 0,
                     plot->page_width, 0, plot->page_width);

 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(top_hruler), 2,3,0,1,
                  GTK_SHRINK|GTK_FILL|GTK_EXPAND,0,0,0);
 gtk_widget_show(top_hruler);


 plot->vruler = left_vruler = gtk_vruler_new();
 gtk_ruler_set_range(GTK_RULER(left_vruler), 0,
                     plot->page_height, 0, plot->page_height);


 gtk_table_attach(GTK_TABLE(table), left_vruler,1,2,1,2,
                  0,GTK_SHRINK|GTK_FILL|GTK_EXPAND,0,0);
 gtk_widget_show(left_vruler);
 
 style=gtk_style_copy(GTK_WIDGET(plot->window)->style);
/* style->font=gdk_font_load("-*-*-medium-r-*-*-10-*-*-*-*-*-*");*/

 if (style && style->font){
  gtk_widget_set_style(top_hruler,style);
  gtk_widget_set_style(left_vruler,style);
 }

 gtk_signal_connect(GTK_OBJECT(plot->hruler),
                    "map",
                    (GtkSignalFunc)update_ruler_expose_x,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(plot->vruler),
                    "map",
                    (GtkSignalFunc)update_ruler_expose_y,(gpointer)plot);

 gtk_widget_show (left_vruler);

 gtk_signal_connect(GTK_OBJECT(plot->real_canvas),"motion_notify_event",
                    GTK_SIGNAL_FUNC(sg_plot_motion), plot);

/*-------------------------------------------------------------*/

 frame = gtk_frame_new(NULL);
 gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
 gtk_box_pack_end(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
 gtk_widget_show(frame);

 plot->label = gtk_label_new("");
 gtk_misc_set_alignment(GTK_MISC(plot->label), 0., .5);
 gtk_container_add(GTK_CONTAINER(frame), plot->label);
 gtk_widget_show(plot->label);

/*-------------------------------------------------------------*/
 gtk_plot_canvas_set_size(GTK_PLOT_CANVAS(plot->real_canvas),
                          GTK_PLOT_CANVAS(plot->real_canvas)->width,
                          GTK_PLOT_CANVAS(plot->real_canvas)->height);

 gtk_widget_ensure_style(plot->real_canvas);
 gtk_widget_realize(plot->window);
 plot_icon_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                gdk_colormap_get_system(),
                                                &plot_icon_mask, NULL, plot_icon2_xpm);

 gdk_window_set_icon(plot->window->window, NULL, plot_icon_pixmap, plot_icon_mask);
/*
 gtk_widget_show(vbox);
 gtk_widget_show(hbox);
 gtk_widget_show(hbox2);
 gtk_widget_show(table);
 gtk_widget_show(sbar_vert);
 gtk_widget_show(sbar_horiz);
 gtk_widget_show(plot->sw);
 gtk_widget_show(plot->canvas_box);
 gtk_widget_show(plot->real_canvas);
*/

 if(plot->width > 0 && plot->height > 0){
          gtk_widget_set_uposition(plot->window,
                                   plot->x, plot->y);
          gtk_widget_set_usize(GTK_WIDGET(plot->window),
                               plot->width,
                               plot->height);
 }else{
          gtk_widget_set_usize(GTK_WIDGET(plot->window),
                               500,
                               500);
 }
 gtk_widget_show_all(plot->window); 
 gtk_widget_ref(plot->real_canvas);
 plot->is_mapped = TRUE;

 gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->real_canvas));
 gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot->real_canvas));

 sg_plot_toolbox_init(plot);

/*
 sg_project_changed(changed);
*/
}

void
sg_plot_hide(SGplot *plot)
{
  gtk_container_remove(GTK_CONTAINER(plot->canvas_box), plot->real_canvas);
  if(GTK_IS_WIDGET(plot->sw)) gtk_widget_destroy(plot->sw);
  if(GTK_OBJECT(plot->real_canvas)->ref_count > 1) 
                     gtk_widget_unref(plot->real_canvas);

  plot->is_mapped = FALSE;
  gtk_widget_destroy(plot->window);
  plot->window = NULL;
  plot->sw = NULL;
  plot->hruler = NULL;
  plot->vruler = NULL;
}

gint
sg_plot_rename(SGplot *plot, gchar *name)
{
  if(strcmp(plot->name, name) == 0) return FALSE;

  if(plot->name){
     g_free(plot->name);
     plot->name = NULL;
  }
 
  plot->name = g_strdup(name);

  if(plot->window) 
     gtk_window_set_title(GTK_WINDOW(plot->window),name);

  if(gui_iconlist){
    if(plot->icon->label){
       g_free(plot->icon->label);
       plot->icon->label = NULL;
    }
 
    plot->icon->label = g_strdup(name);

    if(plot->icon->entry_label){
       g_free(plot->icon->entry_label);
       plot->icon->entry_label = NULL;
    }
 
    plot->icon->entry_label = g_strdup(name);

    gtk_entry_set_text(GTK_ENTRY(plot->icon->entry), name);
  }

  return TRUE;
}


static void
sg_plot_unmap(GtkWidget *window, GdkEvent *event)
{
  GList *w;
  SGplot *plot = NULL;
  GtkWidget *data;

  w = plots;
  while(w){
    plot = (SGplot *)w->data;
    data = plot->window;
    if(data && data == window) break;
    w = w->next;
  }

  gdk_window_get_position(plot->window->window, 
                          &plot->x,
                          &plot->y);
  gdk_window_get_size(plot->window->window, 
                          &plot->width,
                          &plot->height);

  sg_plot_hide(plot);

}

static void
sg_plot_destroy(GtkWidget *window)
{
  GList *w;
  SGplot *plot = NULL;
  GtkWidget *data;

  sg_layer_control_destroy();

  w = plots;
  while(w){
    plot = (SGplot *)w->data;
    data = plot->window;
    if(data && data == window) break;
    w = w->next;
  }

  if(!plot || !plot->window) return;
  if(!GTK_WIDGET_MAPPED(window) && plot->is_mapped){
            sg_plot_hide(plot);
  }

  if(!gui_iconlist){
    sg_project_remove_plot(plot);
  }
}

static gint
sg_plot_delete(GtkWidget *window)
{
  GList *w;
  SGplot *plot = NULL;
  GtkWidget *data;

  sg_layer_control_destroy();

  w = plots;
  while(w){
    plot = (SGplot *)w->data;
    data = plot->window;
    if(data && data == window) break;
    w = w->next;
  }

  gdk_window_get_position(plot->window->window, 
                          &plot->x,
                          &plot->y);
  gdk_window_get_size(plot->window->window, 
                      &plot->width,
                      &plot->height);

  sg_plot_hide(plot);

  return FALSE;
}

SGplot *
sg_plot_get_from_canvas(GtkWidget *canvas)
{
  GList *w;
  SGplot *plot = NULL;
  GtkWidget *data;

  w = plots;
  while(w){
    plot = (SGplot *)w->data;
    data = plot->real_canvas;
    if(data && data == canvas) break;
    w = w->next;
  }
 
  if(w)
   return w->data;

  return NULL;
}

void
sg_plot_rescale(SGplot *plot, gdouble scale)
{
  if(scale < .15) return;

  plot->scale = scale;
  gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(plot->real_canvas),
                                    plot->scale);

  if(plot->hruler) update_ruler_expose_x(NULL, plot);
  if(plot->vruler) update_ruler_expose_y(NULL, plot);
}

void
sg_plot_fit_page(SGplot *plot)
{
  gdouble width, height;
  gdouble w_width, w_height;
  GtkBin *bin;
  gdouble scale,scalex,scaley;

  w_width=(gdouble)plot->sw->allocation.width;
  w_height=(gdouble)plot->sw->allocation.height;
  width = (gdouble)GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_width;
  height = (gdouble)GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_height;

  scalex=w_width/width;
  scaley=w_height/height;

  if(scalex<scaley)   sg_plot_rescale(plot, scalex*plot->scale);
  else   sg_plot_rescale(plot, scaley*plot->scale);

}

void
sg_plot_fit_page_h(SGplot *plot)
{
  gdouble width;
  gdouble w_width;
  GtkBin *bin;
  gdouble scalex;

  w_width=(gdouble)plot->sw->allocation.width;
  width = (gdouble)GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_width;

  scalex=w_width/width;

  sg_plot_rescale(plot, scalex*plot->scale);

}

void
sg_plot_fit_page_v(SGplot *plot)
{
  gdouble height;
  gdouble w_height;
  GtkBin *bin;
  gdouble scaley;

  w_height=(gdouble)plot->sw->allocation.height;
  height = (gdouble)GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_height;

  scaley=w_height/height;

  sg_plot_rescale(plot, scaley*plot->scale);

}

void
sg_plot_set_size(SGplot *plot, gint page_size, 
                 gint width, gint height, gint orientation)
{
  gint real_width, real_height;

  if(orientation == GTK_PLOT_PORTRAIT){
    real_width = width;
    real_height = height;
  }else{
    real_width = height;
    real_height = width;
  }

  plot->page_size = page_size;
  plot->page_width = real_width;
  plot->page_height = real_height;
  plot->orientation = orientation;

  GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_width = real_width;
  GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_height = real_height;

  gtk_plot_canvas_set_size(GTK_PLOT_CANVAS(plot->real_canvas), 
                           real_width, real_height);
} 

void
sg_plot_clear(SGplot *plot)
{
  GList *list;

  list = plot->layers;
  while(list){
    SGlayer *layer;
 
    layer = (SGlayer *)list->data; 
    sg_plot_remove_layer(plot, layer);

    list = plot->layers;
  }  
  
  plot->layers = NULL;  
}

void
sg_plot_remove(SGplot *plot)
{
  GList *list;

  if(plot->is_mapped)
     sg_plot_hide(plot);
 
  sg_plot_clear(plot); 

  gtk_widget_destroy(plot->real_canvas);

  g_free(plot->name);
  g_free(plot);
}

/* Events */

gint
dialog_quit (GtkWidget *widget)
{
  sg_dialog_kill(widget);

  gtk_main_quit();

  return FALSE;
}

static gint
click_on_item(GtkPlotCanvas *canvas, GdkEvent *event, 
              gpointer item_data)
{
  GtkPlotCanvasChild *item = (GtkPlotCanvasChild *) item_data;
  GtkPlotCanvasChild *child = NULL;
  GList *llink;
  gchar path[255];
  gboolean double_click;
  gboolean return_value = TRUE;

  double_click = (event->type == GDK_2BUTTON_PRESS);

  activate_layer(canvas);

  llink = g_list_find(active_plot->layers, active_plot->active_layer);

  if(active_plot->tool == SG_TOOL_ZOOM ||
     active_plot->tool == SG_TOOL_LINE ||
     active_plot->tool == SG_TOOL_LINE_ARROW ||
     active_plot->tool == SG_TOOL_RECTANGLE ||
     active_plot->tool == SG_TOOL_ELLIPSE)
    return TRUE;

  switch(item->type){
    case GTK_PLOT_CANVAS_PLOT:
       sprintf(path, "%s:%d", active_plot->name,
               g_list_position(active_plot->layers, llink)+1);
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
            sg_layer_control(path);
            return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_TEXT:
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
         GtkPlotCanvasChild *child;
         GtkPlotText *real_text;
         GtkWidget *dialog;

         child = (GtkPlotCanvasChild *)item->data;
         real_text = (GtkPlotText *)child->data;

         dialog = open_text_dialog(real_text);
         gtk_signal_connect_object (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->apply_button), 
                             "clicked",
                             GTK_SIGNAL_FUNC (apply_dialog_text), 
                             GTK_OBJECT(dialog));
         gtk_main();

         gtk_plot_canvas_paint(GTK_PLOT_CANVAS(canvas));
         gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(canvas));
         return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_TITLE:
       sprintf(path, "%s:%d:axis:title", active_plot->name,
               g_list_position(active_plot->layers, llink)+1);
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
            sg_layer_control(path);
            return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_DATA:
       sprintf(path, "%s:%d:dataset:style", active_plot->name,
               g_list_position(active_plot->layers, llink)+1);
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
            sg_layer_control(path);
            return_value = FALSE;
       }
       if(active_plot->tool == SG_TOOL_POINTER){
            sprintf(path, "(X,Y) = (%f,%f)", 
                    GTK_PLOT_CANVAS(canvas)->active_x,
                    GTK_PLOT_CANVAS(canvas)->active_y);
            gtk_label_set_text(GTK_LABEL(active_plot->label), path);
            return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_AXIS:
       sprintf(path, "%s:%d:axis", active_plot->name,
               g_list_position(active_plot->layers, llink)+1);
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
            sg_layer_control(path);
            return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_LEGENDS:
       sprintf(path, "%s:%d:legends", active_plot->name,
               g_list_position(active_plot->layers, llink)+1);
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
            sg_layer_control(path);
            return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_LINE:
       child = (GtkPlotCanvasChild *)item->data;
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
         sg_line_dialog(canvas, (GtkPlotCanvasLine *)child->data);
         return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_RECTANGLE:
       child = (GtkPlotCanvasChild *)item->data;
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
         sg_rectangle_dialog(canvas, (GtkPlotCanvasRectangle *)child->data);
         return_value = FALSE;
       }
       break;
    case GTK_PLOT_CANVAS_ELLIPSE:
       child = (GtkPlotCanvasChild *)item->data;
       if(double_click && active_plot->tool == SG_TOOL_ARROW){
         sg_ellipse_dialog(canvas, (GtkPlotCanvasEllipse *)child->data);
         return_value = FALSE;
       }
       break;
    default:
  }

  return return_value;
}

static gint 
edit_text(GtkWidget *widget, GdkEventButton *event)
{
  GtkPlotCanvasChild *child;
  GdkModifierType mods;
  GtkPlotText text;
  GtkWidget *dialog;
  GtkPlotText *real_text;
  gint x, y;

  if(active_plot->tool != SG_TOOL_TEXT) return FALSE;

  gdk_window_get_pointer(widget->window, &x, &y, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;
  activate_layer(GTK_PLOT_CANVAS(widget));

  text.text = g_strdup("Enter text here");
  gdk_color_black(gdk_colormap_get_system(), &text.fg);
  gdk_color_white(gdk_colormap_get_system(), &text.bg);
  text.angle = 0;
  text.height = 16;
  text.transparent = TRUE;
  text.border = 0;
  text.border_width = 0;
  text.border_space = 2;
  text.shadow_width = 3;
  text.justification = GTK_JUSTIFY_LEFT;
  text.font = g_strdup("Helvetica");

  gtk_plot_canvas_get_position(GTK_PLOT_CANVAS(widget), x, y,
                               &text.x, &text.y);

  child = gtk_plot_canvas_put_text(GTK_PLOT_CANVAS(active_plot->real_canvas),
                                   text.x, text.y,
                                   text.font, text.height,
                                   text.angle,
                                   &text.fg, &text.bg,
                                   text.transparent,
                                   text.justification,
                                   text.text);

  real_text = (GtkPlotText *)child->data; 
  real_text->border_space = 2;
  real_text->shadow_width = 3;

  dialog = open_text_dialog(real_text);
  gtk_signal_connect_object (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->apply_button), 
                      "clicked",
                      GTK_SIGNAL_FUNC (apply_dialog_text), 
                      GTK_OBJECT(dialog));
  gtk_main();

  if(!real_text->text || strlen(real_text->text) == 0){
    gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(active_plot->real_canvas), child);
  } 
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));

  return TRUE;
}

static gint
apply_dialog_text(GtkWidget *dialog)
{
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static GtkWidget *
open_text_dialog(GtkPlotText *text)
{
  GtkWidget *dialog;

  dialog = sg_text_dialog_new(text);

  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

  gtk_widget_show(dialog);
  sg_dialog_new(dialog);

  gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                     GTK_SIGNAL_FUNC(dialog_quit), NULL);

  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->ok_button), 
                      "clicked",
                      GTK_SIGNAL_FUNC (gtk_widget_destroy), 
                      GTK_OBJECT(dialog));

  gtk_signal_connect_object (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->cancel_button),
                             "clicked",
                             GTK_SIGNAL_FUNC (gtk_widget_destroy), 
                             GTK_OBJECT(dialog));

  return dialog;
}

static gboolean
focus_in(GtkWidget *widget, GdkEventFocus *event)
{
  GList *list;
  SGplot *plot;

  list = plots;
  while(list)
   {
     plot = (SGplot *)list->data;

     if(plot->window == widget){
        active_plot = plot;
        break;
     }

     list = list->next;
   }

  return FALSE;
}

static void
activate_layer(GtkPlotCanvas *canvas)
{
  GList *list;
  SGlayer *layer;

  if(!canvas) return;

  list = active_plot->layers;
  while(list)
   {
     layer = (SGlayer *)list->data;

     if(GTK_PLOT(layer->real_plot) == canvas->active_plot){
        sg_plot_set_active_layer(active_plot, layer);
        break;
     }

     list = list->next;
   }

}

static gboolean
resize_item(GtkPlotCanvas *canvas, GtkPlotCanvasChild *item,
            gdouble new_width, gdouble new_height, gpointer data)
{
  return TRUE;
}

static gboolean
move_item(GtkPlotCanvas *canvas, GtkPlotCanvasChild *item,
          gdouble x, gdouble y, gpointer data)
{
  switch(item->type){
    case GTK_PLOT_CANVAS_PLOT:
        return TRUE;
        break;
    case GTK_PLOT_CANVAS_LEGENDS:
        return TRUE;
        break;
    case GTK_PLOT_CANVAS_TITLE:
        return TRUE;
        break;
    case GTK_PLOT_CANVAS_TEXT:
        return TRUE;
        break;
    default:
        return TRUE;
        break;
  }
     
  gtk_plot_canvas_paint(canvas);
  gtk_plot_canvas_refresh(canvas);
  return FALSE;
}


static gboolean
key_press(GtkWidget *widget, GdkEventKey *key)
{
  GtkPlotCanvas *canvas = GTK_PLOT_CANVAS(active_plot->real_canvas);
  GtkPlotCanvasChild *child = NULL;
  GtkPlotAxis *axis = NULL;
  gint i = 0;
  gint the_axis = -1;


  switch(key->keyval){
    case GDK_Escape:
      if(canvas->action != GTK_PLOT_CANVAS_ACTION_INACTIVE)
         gtk_plot_canvas_cancel_action(canvas);
      break;
    case GDK_Delete:
      if(canvas->state == GTK_STATE_SELECTED)
        switch(canvas->active_item.type){
         case GTK_PLOT_CANVAS_PLOT:
          gtk_plot_canvas_cancel_action(canvas);
          if(canvas->num_plots  > 1 && sg_accept_dialog("Remove layer?", 1) == YES_CLICKED)
             sg_plot_remove_layer(active_plot, active_plot->active_layer);
          return TRUE;
         case GTK_PLOT_CANVAS_LEGENDS:
          gtk_plot_canvas_cancel_action(canvas);
          gtk_plot_hide_legends(GTK_PLOT(canvas->active_plot));
          gtk_plot_canvas_paint(canvas);
          gtk_plot_canvas_refresh(canvas);
          return TRUE;
          break;
         case GTK_PLOT_CANVAS_TITLE:
          for(i = 0; i < 4; i++){
             axis = gtk_plot_get_axis(GTK_PLOT(canvas->active_plot), i);
             if(axis == canvas->active_item.data){
                the_axis = i;
                break;
             }
          }

          gtk_plot_canvas_cancel_action(canvas);
          if(the_axis != -1)
              gtk_plot_axis_hide_title(GTK_PLOT(canvas->active_plot), the_axis);
          
          gtk_plot_canvas_paint(canvas);
          gtk_plot_canvas_refresh(canvas);
          return TRUE;
         case GTK_PLOT_CANVAS_AXIS:
          return TRUE;
         default:
          child = (GtkPlotCanvasChild *)canvas->active_item.data;
          gtk_plot_canvas_cancel_action(canvas);
          gtk_plot_canvas_remove_child(canvas, child); 
          gtk_plot_canvas_paint(canvas);
          gtk_plot_canvas_refresh(canvas);
          return TRUE;
       }
     break;
    default:
     break;
  }

  return FALSE;
}

static void
update_ruler_expose_x(GtkWidget *scroll, gpointer data)
{
    GtkAdjustment *adj;
    SGplot *plot;
    gdouble start, end, fac, fac2,p_size;
    gint width,s_width,height;
    GtkAllocation allocation;

    if (!data) return;

    plot=(SGplot *)data;

    allocation = plot->real_canvas->allocation;

    /* width of vscrollbar*/
    s_width=plot->sw->allocation.width;
          
    p_size=GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_width;
    adj=gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw));

    /* Conversion factor from pixels to real physical unit */
    fac=plot->page_width/unit_pt[plot->page_units];

    if (p_size<s_width)
    {  
        start=-((s_width-p_size)/2)/p_size*fac;
        end=((s_width-p_size)/2+p_size)/p_size*fac;
        allocation.x = (s_width - p_size)/2;
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    else
    { 
        start=adj->value/adj->upper*fac;
        end=(adj->value+adj->page_size)/adj->upper*fac;
        allocation.x = 0;
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    gtk_ruler_set_range(GTK_RULER(plot->hruler), start,
                        end, start, end);

}

static void
update_ruler_expose_y(GtkWidget *scroll, gpointer data)
{
    GtkAdjustment *adj;
    SGplot *plot;
    gdouble start, end, fac, p_size;
    gint height,s_height;
    GtkAllocation allocation;

    if (!data) return;

    plot=(SGplot *)data;

    allocation = plot->real_canvas->allocation;

    /* height of hscrollbar*/
    s_height=plot->sw->allocation.height;

    p_size=GTK_PLOT_CANVAS(plot->real_canvas)->pixmap_height;
    adj=gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw));

    /* Conversion factor from pixels to real physical unit */
    fac=plot->page_height/unit_pt[plot->page_units];

    if (p_size<s_height)
    {  
        start=-((s_height-p_size)/2)/p_size*fac;
        end=((s_height-p_size)/2+p_size)/p_size*fac;
        allocation.y = (s_height - p_size)/2;
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    else
    { 
        start=adj->value/adj->upper*fac;
        end=(adj->value+adj->page_size)/adj->upper*fac;
        allocation.y = 0;
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    gtk_ruler_set_range(GTK_RULER(plot->vruler), start,
                        end, start, end);
}

static void
sg_plot_motion(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
  SGplot *plot;
 
  plot = (SGplot *) data;

  if(plot->is_mapped){
    gtk_widget_event(plot->hruler, (GdkEvent *)event);   
    gtk_widget_event(plot->vruler,(GdkEvent *)event);   
  } 
}

static void
canvas_changed (GtkPlotCanvas *canvas, gpointer data)
{
  sg_project_changed(TRUE);
}

void
sg_plot_put_pixmap(SGplot *plot, GdkPixmap *pixmap)
{
  GtkPlotCanvas *canvas;
  GtkPlotCanvasChild *child;
  gint width, height;

  canvas = GTK_PLOT_CANVAS(plot->real_canvas);

  gdk_window_get_size(pixmap, &width, &height);

  child = gtk_plot_canvas_put_pixmap(canvas,
                                     pixmap,
                                     0., 0.);

  gtk_plot_canvas_refresh(canvas);
}
