/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <string.h>
#include <gtkextra/gtkextra.h>
#include "../pixmaps/sheet_icon2.xpm"
#include "../pixmaps/plot_icon2.xpm"
#include "main.h"

static void remove_worksheet			(SGworksheet *worksheet);
static void remove_plot				(SGplot *plot);
static gint rename_worksheet			(GtkIconList *iconlist, 
						 GtkIconListItem *item);
static gint rename_plot				(GtkIconList *iconlist, 
						 GtkIconListItem *item);

void
sg_project_new()
{
 sg_project_init();

 if(gui_iconlist){
   gtk_signal_connect(GTK_OBJECT(iconlist2), "deactivate_icon",
                      (GtkSignalFunc)rename_plot, iconlist2);

   gtk_signal_connect(GTK_OBJECT(iconlist1), "deactivate_icon",
                      (GtkSignalFunc)rename_worksheet, iconlist1);
 }
}

void
sg_project_init()
{
 last_function = num_functions = 0;
 last_plot = num_plots = 0;
 last_worksheet = num_worksheets = 0;
 active_icon = NULL;
 active_worksheet = NULL;
 active_plot = NULL;
 active_layer = NULL;
 active_dataset = NULL;
 arrange_rows = 2;
 arrange_cols = 1;
 arrange_top = .15;
 arrange_bottom = .15;
 arrange_left = .15;
 arrange_right = .15;
 arrange_vgap = .15;
 arrange_hgap = .15;

 project_changed = FALSE;

 sg_clipboard_init();
 sg_dialogs_init();
 plots = NULL;
 worksheets = NULL;
 functions = NULL;
 expressions = NULL;

 if(last_project_filename){
    g_free(last_project_filename);
    last_project_filename = g_strdup("untitled.sg");
 }

 sg_project_set_title();
}

void
sg_project_set_title()
{
  gchar title[1000];
  gchar icon_title[1000];

  if(project_changed){
    g_snprintf(title, 1000, "SciGraphica: %s (changed)", last_project_filename);
    g_snprintf(icon_title, 1000, "%s (changed)", last_project_filename);
  } else {
    g_snprintf(title, 1000, "SciGraphica: %s", last_project_filename);
    g_snprintf(icon_title, 1000, "%s", last_project_filename);
  }

  if(!gui_iconlist) return;

  gtk_window_set_title(GTK_WINDOW(main_window), title);
  gdk_window_set_icon_name(GTK_WIDGET(main_window)->window, icon_title);
}

void
sg_project_changed(gboolean changed)
{
  if(project_changed != changed){
    project_changed = changed;
    sg_project_set_title();
  }
}
  
SGworksheet *
sg_project_new_worksheet()
{
 GtkIconList *iconlist;
 SGworksheet *worksheet;
 gchar *name;
 gchar label[20];

 sg_project_changed(TRUE);

 num_worksheets++;
 last_worksheet++;
 sprintf(label,"Data%d",last_worksheet);
 name = g_strdup(label);

 worksheet = sg_worksheet_new(name, 20, 5);
 worksheets = g_list_append(worksheets, worksheet);

 active_worksheet = worksheet;

 if(!gui_iconlist){
    g_free(name);
    return worksheet;
 }

 iconlist = GTK_ICON_LIST(iconlist1);

 worksheet->iconlist = iconlist;
 worksheet->icon = gtk_icon_list_add_from_data(iconlist,
                                               sheet_icon2_xpm,
                                               name,
                                               worksheet);

 gtk_icon_list_set_active_icon(iconlist, worksheet->icon);

 gtk_entry_select_region(GTK_ENTRY(worksheet->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(worksheet->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 0);

 gtk_signal_connect(GTK_OBJECT(worksheet->sheet), "button_press_event",
                    (GtkSignalFunc)sg_worksheet_menu_show, NULL);
 g_free(name);

 return worksheet;
}

SGplot *
sg_project_new_plot()
{
 GtkIconList *iconlist;
 SGplot *plot;
 gchar *name;
 gchar label[20];

 sg_project_changed(TRUE);

 num_plots++;
 last_plot++;
 sprintf(label,"Plot%d",last_plot);
 name = g_strdup(label);


 plot = sg_plot_new(name);
 plots = g_list_append(plots, plot);

 active_plot = plot;

 if(!gui_iconlist){
    g_free(name);
    return plot;
 }

 iconlist = GTK_ICON_LIST(iconlist2);

 plot->iconlist = iconlist;
 plot->icon = gtk_icon_list_add_from_data(iconlist, plot_icon2_xpm, name, plot);

 gtk_icon_list_set_active_icon(iconlist, plot->icon);
 gtk_entry_select_region(GTK_ENTRY(plot->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(plot->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 1);

 sg_layer_control_refresh(label);
 g_free(name);

 return plot;
}


SGplot *
sg_project_new_plot_with_layer(gint plot_type)
{
 SGplot *plot;
 GtkIconList *iconlist;
 gchar *name;
 gchar label[20];

 sg_project_changed(TRUE);

 num_plots++;
 last_plot++;
 sprintf(label,"Plot%d",last_plot);
 name = g_strdup(label);


 plot = sg_plot_new_with_layer(plot_type, name);
 plots = g_list_append(plots, plot);

 active_plot = plot;

 if(!gui_iconlist){
    g_free(name);
    return plot;
 }
 
 iconlist = GTK_ICON_LIST(iconlist2);

 plot->iconlist = iconlist;
 plot->icon = gtk_icon_list_add_from_data(iconlist, plot_icon2_xpm, name, plot);

 gtk_icon_list_set_active_icon(iconlist, plot->icon);
 gtk_entry_select_region(GTK_ENTRY(plot->icon->entry), 0, strlen(name));
 sg_entry_cancel_space(GTK_ENTRY(plot->icon->entry));

 gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 1);

 sg_layer_control_refresh(label);

 g_free(name);

 return plot;
}

void
sg_project_add_function(SGdataset *function)
{
  sg_project_changed(TRUE);

  functions = g_list_append(functions, function);
  num_functions++;
  last_function++;
}

void
sg_project_add_expression(SGdataset *exp)
{
  sg_project_changed(TRUE);

  expressions = g_list_append(expressions, exp);
  num_expressions++;
  last_expression++;
}


void
sg_project_remove_worksheet(SGworksheet *worksheet)
{
  remove_worksheet(worksheet);    

  if(gui_worksheet && !worksheets) gui_worksheet = FALSE;

  if(!gui_iconlist && !gui_plot && !gui_worksheet){
    main_quit();
  }
}

void
sg_project_remove_plot(SGplot *plot)
{
  remove_plot(plot);    

  if(gui_plot && !plots) gui_plot = FALSE;

  if(!gui_iconlist && !gui_plot && !gui_worksheet){
    main_quit();
  }
}

void
sg_project_close()
{
  GList *list;

  if(gui_iconlist){
#ifndef WITH_GNOME
    gtk_signal_disconnect_by_data(GTK_OBJECT(iconlist1), iconlist1);
    gtk_signal_disconnect_by_data(GTK_OBJECT(iconlist2), iconlist2);
#endif
    gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist1), NULL);
    gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist2), NULL);
  }

  list = plots;
  while(list){
    remove_plot((SGplot *)list->data);
    list = plots;
  }

  list = worksheets;
  while(list){
    remove_worksheet((SGworksheet *)list->data);
    list = worksheets;
  }

  list = functions;
  while(list){
    functions = g_list_remove_link(functions, list);
    g_list_free_1(list);
    list = functions;
  }

  list = expressions;
  while(list){
    expressions = g_list_remove_link(expressions, list);
    g_list_free_1(list);
    list = functions;
  }

  sg_layer_control_destroy();
  sg_clipboard_clear();
  sg_dialogs_destroy();
}

static void
remove_worksheet(SGworksheet *worksheet)
{
  GList *list;
  SGdataset *dataset = NULL;
  SGlayer *layer = NULL;
  SGplot *plot = NULL;
  gboolean found = FALSE;

  if(worksheet == NULL) return;

  list = plots;
  while(list){
    GList *layers;
    GList *layer_data;

    plot = (SGplot *)list->data;
    layers = plot->layers;

    while(layers){
      layer = (SGlayer *)layers->data;

      layer_data = layer->datasets;
      while(layer_data){
        dataset = (SGdataset *)layer_data->data;

        if(dataset->worksheet == worksheet){
            found = TRUE;
        }

        layer_data = layer_data->next;
      }

      layers = layers->next;
    }

    list = list->next;
  }

  if(found){
     gint veto;
     veto = sg_accept_dialog("The worksheet is connected to datasets\nAre you sure you want to remove it?", 0);
     if(veto != YES_CLICKED) return;
  }


  list = g_list_find(worksheets, worksheet);

  if(!list) return;

  sg_project_changed(TRUE);

  if(gui_iconlist)
    gtk_icon_list_remove(worksheet->iconlist, worksheet->icon);

  sg_worksheet_remove(worksheet);

  worksheets = g_list_remove_link(worksheets, list);
  g_list_free_1(list);

  num_worksheets--;

  if(num_worksheets == 0) worksheets = NULL;
}

static void
remove_plot(SGplot *plot)
{
  GList *list;

  if(plot == NULL) return;

  list = g_list_find(plots, plot);

  if(!list) return;

  sg_project_changed(TRUE);

  if(gui_iconlist)
    gtk_icon_list_remove(plot->iconlist, plot->icon);

  sg_plot_remove(plot);

  plots = g_list_remove_link(plots, list);
  g_list_free_1(list);

  num_plots--;

  if(num_plots == 0) plots = NULL;

  sg_layer_control_refresh(NULL);
}

static gint
rename_worksheet(GtkIconList *iconlist, GtkIconListItem *item)
{
  gchar *name;
  SGworksheet *worksheet;

  if(!gui_iconlist) return;

  worksheet = (SGworksheet *)gtk_icon_list_get_link(item);
  name = gtk_entry_get_text(GTK_ENTRY(item->entry));

  return (sg_project_rename_worksheet(worksheet, name));
}


gint
sg_project_rename_worksheet(SGworksheet *worksheet, gchar *name)
{
  GList *list;
  gint n;
  gint veto;
  gboolean accept = FALSE;
  gboolean free_name = FALSE;

  while(!name || strlen(name) == 0){
      if(free_name) g_free(name);
      free_name = FALSE;
      name = sg_text_dialog("The worksheet must have a name.\nEnter a name:", 0);
      free_name = TRUE;
  }

  list = worksheets;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGworksheet *data;

     data = (SGworksheet *)list->data;

     if(data != worksheet && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
           name = sg_text_dialog("Another worksheet has the same name.\nEnter a new name:", 0);
        }
        free_name = TRUE;
        new_name = TRUE;
        break; 
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }

  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+' && name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
      }
  }

  veto = sg_worksheet_rename(worksheet, name);

  if(veto){
    sg_project_refresh_datasets(worksheet);
    sg_project_changed(TRUE);
  }

  if(free_name) g_free(name);
  return TRUE;
}

static gint
rename_plot(GtkIconList *iconlist, GtkIconListItem *item)
{
  gchar *name;
  SGplot *plot;

  if(!gui_iconlist) return;

  plot = (SGplot *)gtk_icon_list_get_link(item);
  name = gtk_entry_get_text(GTK_ENTRY(item->entry));

  return(sg_project_rename_plot(plot, name));
}

gint
sg_project_rename_plot(SGplot *plot, gchar *name)
{
  GList *list;
  gint n;
  gint veto;
  gboolean accept = FALSE;
  gboolean free_name = FALSE;

  while(!name || strlen(name) == 0){
    if(free_name) g_free(name);
    free_name = FALSE;
    name = sg_text_dialog("The plot must have a name.\nEnter a name:", 0);
    free_name = TRUE;
  }

  list = plots;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGplot *data;

     data = (SGplot *)list->data;

     if(data != plot && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
          name = sg_text_dialog("Another plot has the same name.\nEnter a new name:", 0);
        }
        free_name = TRUE;
        new_name = TRUE;
        break;
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }

  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+' && name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
      }
  }

  veto = sg_plot_rename(plot, name);

  if(veto) sg_project_changed(TRUE);

  if(free_name) g_free(name);
  return TRUE;
}

void
sg_project_open_worksheet(GtkWidget *widget, 
                          GtkIconListItem *item, GdkEvent *event,
                          gpointer data)
{
  GtkIconList *iconlist;
  SGworksheet *worksheet;
  GdkModifierType mods;
  iconlist = GTK_ICON_LIST(widget);

  sg_layer_control_destroy();

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  worksheet = (SGworksheet *)gtk_icon_list_get_link(item);

  if(!event) return;
  if((mods & GDK_BUTTON1_MASK) && event->type == GDK_2BUTTON_PRESS)
                sg_worksheet_open(worksheet);

  if(mods & GDK_BUTTON3_MASK)
                sg_worksheet_popup_show(item,(GdkEventButton *)event);

  active_worksheet = worksheet;

  return;

}

void
sg_project_open_plot(GtkWidget *widget, 
                     GtkIconListItem *item,
                     GdkEvent *event,
                     gpointer data)
{
  GtkIconList *iconlist;
  SGplot *plot;
  GdkModifierType mods;
  iconlist = GTK_ICON_LIST(widget);

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  plot = (SGplot *)gtk_icon_list_get_link(item);
  if(!event) return;
  if((mods & GDK_BUTTON1_MASK) && event->type == GDK_2BUTTON_PRESS){
                sg_plot_open(plot);
  }

  if(mods & GDK_BUTTON3_MASK)
                sg_plot_popup_show(item,(GdkEventButton *)event);

  active_plot = plot;

  return;

}

SGworksheet *
sg_project_get_worksheet (gchar *name)
{
  SGworksheet *worksheet;
  GList *list;

  if(!name || !strcmp(name,"current")) return active_worksheet;

  list = worksheets;
  while(list){
    worksheet = (SGworksheet *) list->data;

    if(strcmp(worksheet->name, name) == 0) return worksheet;
    list = list->next;
  }

  return NULL;
} 

SGplot *
sg_project_get_plot (gchar *name)
{
  SGplot *plot;
  GList *list;

  if(!name) return active_plot;

  list = plots;
  while(list){
    plot = (SGplot *) list->data;

    if(strcmp(plot->name, name) == 0) return plot;
    list = list->next;
  }

  return NULL;
}

void
sg_project_refresh_datasets(SGworksheet *worksheet)
{
  GList *list;

  list = plots;
  while(list){
    SGplot *plot;
    GList *aux_layers;

    plot = (SGplot *)list->data;

    aux_layers = plot->layers;
    while(aux_layers){
      SGlayer *layer;
      GList *aux_datasets;
      
      layer = (SGlayer *)aux_layers->data;

      aux_datasets = layer->datasets;
      while(aux_datasets){
        SGdataset *data;

        data = (SGdataset *)aux_datasets->data;
 
        if(data->worksheet == worksheet)
                      sg_dataset_refresh_name(data);
        aux_datasets = aux_datasets->next;
      }
      aux_layers = aux_layers->next;
    }
    list = list->next;
  }
}
