/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>

#include "main.h"

static void write_header_html			(SGworksheetfile *file); 
static void write_footer_html			(SGworksheetfile *file); 
static void write_col_title_html		(SGworksheetfile *file, 
						 gint col); 
static void write_row_title_html		(SGworksheetfile *file, 
						 gint row); 
static void write_cell_html			(SGworksheetfile *file,  
						 gint row, gint col); 
static void new_row_html			(SGworksheetfile *file, 
                                                 gint row); 
static void justify_html			(SGworksheetfile *file, 
						 GtkJustification just);

gboolean
sg_worksheet_file_export_html (SGworksheet *worksheet, 
                               gchar *filename,
                               GtkSheetRange *range)
{  
  SGworksheetfile *file;
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet->sheet);
  file = sg_worksheet_file_new(worksheet, filename);

  if ((file->file = fopen(file->filename, "w")) == NULL){
       g_warning("ERROR: Cannot write to file: %s", filename);
       g_free(file);
       return FALSE;
  }

  file->write_header = write_header_html; 
  file->write_footer = write_footer_html; 
  file->write_col_title = write_col_title_html; 
  file->write_row_title = write_row_title_html; 
  file->write_cell = write_cell_html; 
  file->new_row = new_row_html; 

  file->titlecolor = g_strdup("\"BBBBBB\"");
  file->cellcolor = g_strdup("\"FFFFFF\"");

  sg_worksheet_file_export(file, range);

  fclose(file->file);

  g_free(file->filename);
  g_free(file->titlecolor);
  g_free(file->cellcolor);
  g_free(file);

  return TRUE;
}

static void
write_header_html(SGworksheetfile *file)
{
  fprintf(file->file,"<BODY>\n");
  fprintf(file->file,"<HTML>\n");

  fprintf(file->file, "<FONT=\"lucida,helvetica\">\n"); 
  fprintf(file->file, "<TABLE WIDTH=80%% BORDER=1 CELLSPACING=0 CELLPADDING=0> \n");

  fprintf(file->file, "<TR>\n"); 
  fprintf(file->file,"<TD BGCOLOR=%s> <BR> </TD>\n", file->titlecolor);
}

static void
write_footer_html(SGworksheetfile *file)
{
  fprintf(file->file, "</TR>\n\n");
  fprintf(file->file, "</TABLE>\n\n");
  fprintf(file->file,"</BODY>\n");
  fprintf(file->file,"</HTML>\n");
}

static void
new_row_html(SGworksheetfile *file, gint row)
{

  fprintf(file->file, "</TR>\n"); 
  fprintf(file->file, "<TR>\n"); 

}

static void
write_col_title_html(SGworksheetfile *file, gint col)
{
  gchar *title;

  title = GTK_SHEET(file->worksheet->sheet)->column[col].name;

  fprintf(file->file, "<TD BGCOLOR=%s ", file->titlecolor);
  justify_html(file, GTK_JUSTIFY_CENTER);

  if(title && strlen(title) > 0)
    fprintf(file->file, ">\n<B> %s </B> </TD>\n", title);
  else
    fprintf(file->file, ">\n <BR> </TD>\n");

}

static void
write_row_title_html(SGworksheetfile *file, gint row)
{
  gchar *title;

  title = GTK_SHEET(file->worksheet->sheet)->row[row].name;

  fprintf(file->file, "<TD BGCOLOR=%s ", file->titlecolor);
  justify_html(file, GTK_JUSTIFY_CENTER);

  if(title && strlen(title) > 0)
    fprintf(file->file, ">\n<B> %s </B> </TD>\n", title);
  else
    fprintf(file->file, ">\n <BR> </TD>\n");

}

static void
write_cell_html(SGworksheetfile *file, gint row, gint col)
{
  gchar *text;
  GtkSheetCellAttr attr;
  gchar *xml_text = NULL;

  text = sg_worksheet_cell_get_text(file->worksheet, row, col);
  gtk_sheet_get_attributes(GTK_SHEET(file->worksheet->sheet),
                           row, col, &attr);

  fprintf(file->file, "<TD BGCOLOR=%s ", file->cellcolor);
  justify_html(file, attr.justification);

  if(text && strlen(text) > 0){
    xml_text = xml_get_string(text);
    fprintf(file->file, ">\n %s </TD>\n", xml_text);
    g_free(xml_text);
  }
  else
    fprintf(file->file, ">\n <BR> </TD>\n");

}

static void
justify_html(SGworksheetfile *file, GtkJustification just)
{
  switch (just) {
    case GTK_JUSTIFY_LEFT:
      fprintf(file->file, "ALIGN=LEFT");
      break;
    case GTK_JUSTIFY_RIGHT:
      fprintf(file->file, "ALIGN=RIGHT");
      break;
    case GTK_JUSTIFY_CENTER:
      fprintf(file->file, "ALIGN=CENTER");
      break;
    case GTK_JUSTIFY_FILL:
    default:
      break;
  }
}
