/* calc.c
 *
 * guts of the calculator */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <stdint.h>

#include "calc.h"
#include "data.h"
#include "stack.h"
#include "queue.h"
#include "llist.h"

#define COMMAND_BUFFER_SIZE 256
#define TOKEN_LIST_SIZE 128

stack *s;

char savepath[255];
char *tokens[TOKEN_LIST_SIZE];

void init_calc(void)
{
	char *p;
	token tok;

	datalist_init();

	/* Create a stack 0 */
	s = get_stack(0);

	opt.deg = FALSE;
	opt.num = FALSE;
	flags.error = FALSE;
	flags.syntax = FALSE;
	opt.fix = 10;

#ifdef SYS_LINUX
	p = getenv("HOME");
#endif
#ifdef SYS_DOS
	p = NULL;
#endif
	if(p && *p != '\0') { 
		strncpy(savepath, p, 245);
		savepath[245] = '\0';
		if(*(strend(savepath) - 1) != '/')
			strcat(savepath, "/");
	} else
		savepath[0] = '\0';

#ifdef SYS_LINUX
	strcat(savepath, ".acalcsav");
#endif
#ifdef SYS_DOS
	strcat(savepath, "acalcsav");
#endif

	tok.typ = TTYP_STR;
	tok.dat.str = strdup(savepath);
	ac_push(&tok);
	func_dup();
	func_loadregs();
	func_load();

	return;
}

void end_calc(void)
{
	token tok;
	tok.typ = TTYP_STR;
	tok.dat.str = strdup(savepath);
	ac_push(&tok);
	func_save();

	datalist_free();

	return;
}

int exec_string(char *str)
{
	char buffer[COMMAND_BUFFER_SIZE];
	char **tokenlist;
	char **tlp;
	token tok;
	int i;
	int funcret;
	queue inq;

	llist_node *n;

	inq = queue_new(sizeof(token));

	tokenlist = parse_command(str, buffer);
	for(tlp = tokenlist; *tlp; tlp++) {
		tok = det_type(*tlp);
		if(tok.typ == TTYP_NONE) {
			sprintf(error_buf, "Syntax Error: Unknown token %s", *tlp);
			/* Clear queue */
			for(n = inq.h; n; n = n->next) {
				free_tok((token *)(n->data));
			}
			queue_free(&inq);
			return -1;
		}
		enqueue(&inq, &tok);
	}

	/* Line was parsed successfully */
	
	while(inq.h) {
		dequeue(&inq, &tok);
		switch(tok.typ) {
			case TTYP_DEC:
			case TTYP_HEX:
			case TTYP_FLT:
			case TTYP_STR:
				ac_push(&tok);
				break;
			case TTYP_FUNC:
				if(s->size < funclist[tok.dat.lnum].args) {
					strcpy(error_buf, "Not enough arguments.");
					return -2;
				}
				for(i = 0, n = s->h; n && i < funclist[tok.dat.lnum].args; 
								i++, n = n->next) {
					if(!( ((token *)(n->data))->typ & funclist[tok.dat.lnum].argtype)) {
						strcpy(error_buf, "Invalid argument.");
						return -2;
					}
				}
				funcret = (funclist[tok.dat.lnum].fn)();
				if(funcret == -1) return -2; /* Function encountered an error; bail. */
				break;
			default:
				break;
		}
	}	

	queue_free(&inq);
	
	return 0;
}

char **parse_command(char *command, char *buffer)
/* A token consists of either a string, delimited but including quotes,
 * or of a string delimited by whitespace. */
{
	char **tlp, *bp;
	strcpy(buffer, command);
	tlp = tokens; bp = buffer;

	while(1) {
		while(*bp && index(" \t\n\r", *bp)) {
			*bp = '\0';
			bp++;
		}
		if(!(*bp)) break;
		*(tlp++) = bp;
		if(*bp == '\"') { /* Starts a string; search to closing quote. */
			bp++;
			while(*bp && *bp != '\"') {
				bp++;
			}
			if(*bp) bp++;
		} else {
			while(*bp && !index(" \t\n\r", *bp)) {
				bp++;
			}
		}
		
	}	

	*tlp = NULL; 

	return tokens;
}

token det_type(char *t)
/* Rules for determining what type of token this is:
 *
 * 1- if it starts with a quotation mark:
 *   a- convert everything except first and last quotation marks to a string
 *
 * 2- if it starts with a number or it's a two-or more character token thats
 *		 starts with a dash:
 *   a- if there is a decimal point in it, it's assumed to be a flt
 *   b- if the only characters are numbers 0-9 and E, -, parse as a flt
 *   c- if the only characters are numbers 0-9, parse as a decimal integer
 *   d- otherwise, parse it as a hex integer
 *
 * 3- if it starts with any other character:
 *   a- try to match it with an IMMED shortcut -- OBSOLETE
 *   b- try to parse as a function (case-insensitive) 
 *   c- if it starts with an 'x' or 'X', parse the remainder as a hex integer 
 *   d- parse the whole thing as a hex integer */
{
	token tok;
	long long intval;
	long double fltval;
	
	tok.typ = TTYP_NONE;  /* Assume the worst */

	if(t[0] == '\"') { /* 1 */
		char tstr[1024];
		int i;
		tok.typ = TTYP_STR;
		for(i = 0; i < 1023 && t[i + 1] && t[i + 1] != '\"'; i++)
			tstr[i] = t[i + 1];
		tstr[i] = '\0';
		
		tok.dat.str = strdup(tstr);
		return tok;
	}		

	if((t[0] >= '0' && t[0] <= '9') || (t[0] == '.') ||
	        ((t[0] == '-') && (strlen(t) > 1))) { /* 2 */
		char neg = FALSE;
		if(t[0] == '-') {
			neg = TRUE;
			t++;
		}
		
		if(strstr(t, ".")) { /* 2a */
			if(string_to_float(t, &fltval))  /* Syntax error */
				return tok;
			tok.typ = TTYP_FLT;
			tok.dat.dnum = fltval;
			if(neg) tok.dat.dnum = -tok.dat.dnum;
			return tok;
		} else if (strlen(t) == strspn(t, "0123456789eE-") &&
						(strstr(t, "e") || strstr(t, "E"))) { /* 2b */
			if(string_to_float(t, &fltval))  /* Syntax error */
				return tok;
			tok.typ = TTYP_FLT;
			tok.dat.dnum = fltval;
			if(neg) tok.dat.dnum = -tok.dat.dnum;
			return tok;
		} else if (strlen(t) == strspn(t, "0123456789")) { /* 2c */
			if(string_to_int(t, &intval))
				return tok;
			tok.typ = TTYP_DEC;
			tok.dat.lnum = intval;
			if(neg) tok.dat.lnum = -tok.dat.lnum;
			return tok;
		} else if (strlen(t) == strspn(t, "0123456789ABCDEFabcdef")) { /* 2d */
			if(hexstring_to_int(t, &intval))
				return tok;
			tok.typ = TTYP_HEX;
			tok.dat.lnum = intval;
			if(neg) tok.dat.lnum = -tok.dat.lnum;
			return tok;
		} else /* Syntax error */
			return tok;
	} else { /* 3 */
		int i;

		for(i = 0; funclist[i].str; i++) { /* 3b */
			if(!strcasecmp(t, funclist[i].str)) {
				tok.typ = TTYP_FUNC;
				tok.dat.lnum = i;
				return tok;
			}
		}

		if((t[0] == 'x' || t[0] == 'X') && 
						(strlen(t) - 1 == strspn(t + 1, 
										"0123456789ABCDEFabcdef"))) { /* 3c */
			if(t[1] == '\0') /* single 'x'; syntax error */
				return tok;
			if(hexstring_to_int(t + 1, &intval))
				return tok;
			tok.typ = TTYP_HEX;
			tok.dat.lnum = intval;
			return tok;
		} else if (strlen(t) == 
						strspn(t, "0123456789ABCDEFabcdef")) { /* 3d */
			if(hexstring_to_int(t, &intval))
				return tok;
			tok.typ = TTYP_HEX;
			tok.dat.lnum = intval;
			return tok;
		} else 
			return tok;
	}
	
	return tok;  
}

int string_to_int(char *s, long long *v)
{
	long long a;

	a = strtoll(s, NULL, 10);
	*v = a;
	return 0;
}

int hexstring_to_int(char *s, long long *v)
{
	long long a;
	int i;
	char buf[20];

	strncpy(buf, s, 19);
	buf[19] = '\0';
	for(i = 0; buf[i]; i++)
		buf[i] = toupper(buf[i]);

	a = strtoull(buf, NULL, 16);
	*v = a;
	return 0;
}

int string_to_float(char *s, long double *v)
{
	long double a;

	a = strtod(s, NULL);
	*v = a;
	return 0;
}								

char *strend(char *str)
{
	char *sp;

	for(sp = str; *sp; sp++);

	return sp;
}

char ttsbuf[256];
char fbuf[10];
char *tok_to_str(token tok)
{
	switch(tok.typ) {
		case TTYP_DEC:
			sprintf(ttsbuf, "%lld", tok.dat.lnum);
			return ttsbuf;
		case TTYP_HEX:
			sprintf(ttsbuf, "X%llX", tok.dat.lnum);
			return ttsbuf;
		case TTYP_FLT:
			sprintf(fbuf, "%%.%dLG", opt.fix + 1);
			sprintf(ttsbuf, fbuf, tok.dat.dnum);
			return ttsbuf;
		case TTYP_STR:
			sprintf(ttsbuf, "\"%s\"", tok.dat.str);
			return ttsbuf;
		default:
			break;
	}

	sprintf(ttsbuf, "Weird token (typ = %d, lnum = %lld, dnum = %Le)",
					tok.typ, tok.dat.lnum, tok.dat.dnum);
	return ttsbuf;
}

char *tok_to_savstr(token tok)
{
	/* Format: tok_type value 
	 * tok_type = 2-digit hex value
	 * value = hexadecimal integer representation if type != TTYP_STR
	 * value = "string" if type == TTYP_STR */
	char typ;
	int len;
	int i;
	char vbuf[128];

	switch(tok.typ) {
		case TTYP_DEC:
		case TTYP_HEX:
		case TTYP_FLT:
		case TTYP_STR:
			typ = tok.typ;
			break;
		default:
			typ = 0;
	}

	sprintf(ttsbuf, "%.2hhX ", typ);

	if(tok.typ & TTYP_NUM) {
		if(tok.typ & TTYP_INT) {
			len = sizeof(ac_int_t);
			memcpy(vbuf, &(tok.dat.lnum), len);
		} else {
			len = sizeof(ac_flt_t);
			memcpy(vbuf, &(tok.dat.dnum), len);
		}
		
		for(i = 0; i < len; i++) {
			sprintf(ttsbuf + 3 + i * 2, "%.2hhX", *(uint8_t *)(vbuf + i));
		}
		
	} else if(tok.typ == TTYP_STR) {
		sprintf(ttsbuf + 3, "\"%s\"", tok.dat.str);
	}

	return ttsbuf;
}

token savstr_to_tok(char *s)
{
	token tok;
	
	char vbuf[128];
	tok_type typ;
	char *nbuf;

	int len, i;
	
	/* First 2 characters are a hex representation of the type */
	vbuf[0] = s[0];
	vbuf[1] = s[1];
	vbuf[2] = '\0';
	
	typ = strtol(vbuf, NULL, 16);
	tok.typ = typ;

	if(typ & TTYP_NUM) {
		/* Assumes that sizes are multiples of 4 bytes */
		if(typ & TTYP_INT) {
			len = sizeof(ac_int_t);
			nbuf = (char *)(&(tok.dat.lnum));
		} else {
			len = sizeof(ac_flt_t);
			nbuf = (char *)(&(tok.dat.dnum));
		}
		
		for(i = 0; i < len && s[i * 2 + 3] && s[i * 2 + 4]; i++, nbuf++) {
			vbuf[0] = s[i * 2 + 3];
			vbuf[1] = s[i * 2 + 4];
			*(uint8_t *)nbuf = strtol(vbuf, NULL, 16);
		}

		if(i != len) {
			/* Bad token string */
			return tok;
		}
		
	} else if(tok.typ == TTYP_STR) {
		strcpy(vbuf, s + 4);
		vbuf[strlen(vbuf) - 1] = '\0';
		tok.dat.str = strdup(vbuf);
	}

	return tok;
}
