/* iface.c
 *
 * interface for acalc */

#include <curses.h>
#include <string.h>
#include "calc.h"
#include "get_str.h"

/* Since pdcurses doesn't define these, we have to do them here. */
#ifndef mvvline
#define mvvline(y, x, ch, n) move(y, x); vline(ch, n)
#endif
#ifndef mvhline
#define mvhline(y, x, ch, n) move(y, x); hline(ch, n)
#endif
#ifndef mvgetnstr
#define mvgetnstr(y, x, str, n) move(y, x); wgetnstr(stdscr, str, n)
#endif


void interface_init(int *argc, char ***argv);
/* Perform setup for the interface (make windows, do curses init, etc.) */

void interface_end(void);
/* Does anything needed to end the interface stuff */

void interface_loop(void);
/* Main loop for the calculator; takes care of drawing the screen, getting
 * input, and passing it off to appropriate functions */

void draw_screen(void);
/* Draws the screen, along with anything in the stack and indicators */

void do_help(void);
/* Display a list of all available functions and associated shortcuts */

void get_command(void);
/* Get a keystroke from the keyboard, execute it if it's a one-key command,
 * or continue to retrieve keystrokes otherwise and return the completed
 * line.  Puts its result in command[]; returns a null string if the command
 * was already executed. */
char command[256];

void interface_init(int *argc, char ***argv)
{
	initscr();
	cbreak();
	nonl();
	noecho();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	return;
}

void interface_end(void)
{
	endwin();
	return;
}

void interface_loop(void)
{
	int exec_return;
	
	flags.quit = FALSE;

	do {
		draw_screen();
		flags.syntax = FALSE;
		flags.error = FALSE;
		get_command();
		if(strlen(command) == 0) continue; /* Nothing was entered, or processing was
											done in get_command */
		exec_return = exec_string(command);
		if(exec_return == -1)  /* Syntax error */
			flags.syntax = TRUE; /* Will be displayed as an indicator on display */
		else if(exec_return == -2) /* Calculation error */
			flags.error = TRUE; /* Ditto */
	} while (!flags.quit);
	return;
}

void draw_screen(void)
{
	int i;
	char *tok, display[256];
	llist_node *n;
	
	erase();
	mvaddch(1, 0, opt.deg ? 'D' : 'R');
	mvaddch(0, 0, flags.syntax ? 'S' : (flags.error ? 'E' : ' '));
	mvvline(0, 1, ACS_VLINE, LINES);
	mvhline(1, 2, ACS_HLINE, COLS - 2);
	mvhline(LINES - 2, 2, ACS_HLINE, COLS - 2);
	if(flags.error || flags.syntax) 
		mvaddnstr(0, 2, error_buf, COLS - 2);

	for(i = 1, n = s->h; n && LINES - i - 2 >= 2; n = n->next, i++) {
		tok = tok_to_str(*(token *)(n->data));
		if(opt.num)
			sprintf(display, "%d: %s", i, tok);
		else
			strcpy(display, tok);
		
		mvaddnstr(LINES - i - 2, 2, display, COLS - 2);
	}

	
	refresh();
	return;
}

void get_command(void)
{
	move(LINES - 1, 2);
	refresh();

	move(LINES - 1, 2);
	get_str(command, 255, COLS - 2);
	return;	
}

void do_help(void)
{
	int i, j;
	char pbuf[80];
	int buf[10], *bp;
	erase();
	move(0,0);
	for(i = 0; funclist[i].str; i++) {
		strcpy(pbuf, funclist[i].str);
		for(j = 0, bp = buf; keys[j].key; j++) 
			if(!strcmp(funclist[i].str, keys[j].cmd)) 
				*(bp++) = keys[j].key;
		*bp = 0;
		if(buf[0]) {
			strcat(pbuf, " (");
			for(j = 0; buf[j]; j++) {
				if(j) strcat(pbuf, ", ");
/* Because the pdcurses keyname function is rather braindead, we
	use the following function.  It may need more changes, depending
	on the key mappings chosen */
#ifdef SYS_LINUX
				strcat(pbuf, keyname(buf[j]));
#endif
#ifdef SYS_DOS
				if(buf[j] > 127) 
					/* Chars > 127 are assumed to be function keys, and are
						 therefore passed to the keyname function */
					strcat(pbuf, keyname(buf[j]));
				else
					strcat(pbuf, unctrl(buf[j]));
#endif
			}
			strcat(pbuf, ")");
		}		

		mvaddnstr(i % (LINES - 1), 0, pbuf, COLS);
		if((i % (LINES - 1)) == LINES - 2) {
			mvaddnstr((i % (LINES - 1)) + 1, 0, "Press a key...", COLS);
			getch();
			clear();
		}
	}

	if(i % (LINES - 1)) {
		mvaddnstr(i % (LINES - 1), 0, "Press a key...", COLS);
		getch();
	}
	return;
}
