/* llist.c */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "error.h"
#include "llist.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

bool llist_remove_data_main( llist *l, void *data,
				bool search_entire, bool shallow );

llist llist_new( int db )
{
	llist l;

	l.h = NULL;
	l.t = NULL;
	l.db = db;
	l.size = 0;
	
	return l;
}

void llist_free( llist *l )
{
	llist_node *n, *m;

	for(n = l->h; n; n = m) {
		m = n->next;
		
		/* Should always be true */
		if(n && n->data) free(n->data);
		if(n) free(n);
	}

	l->h = NULL;
	l->t = NULL;
	l->size = 0;
}

void llist_free_fn( llist *l, void (* free_fn)(void *ptr) )
{
	llist_node *n, *m;

	for(n = l->h; n; n = m) {
		m = n->next;

		if(free_fn && n && n->data) free_fn(n->data);
		if(n) free(n);
	}

	l->h = NULL;
	l->t = NULL;
	l->size = 0;
}

void llist_shallow_free( llist *l )
{
	llist_node *n, *m;

	for(n = l->h; n; n = m) {
		m = n->next;
		free(n);
	}

	l->h = NULL;
	l->t = NULL;
	l->size = 0;
}

void *llist_prepend( llist *l, void *data )
{
	return (llist_prepend_n(l, data))->data;
}

void *llist_prepend_p( llist *l, void *data )
{
	return (llist_prepend_pn(l, data))->data;
}

llist_node *llist_prepend_n( llist *l, void *data )
{
	void *d;

	aotf_assertk(d = malloc(l->db),
					"Couldn't allocate memory for new llist node");
	
	memcpy(d, data, l->db);

	return llist_prepend_pn(l, d);
}
	
llist_node *llist_prepend_pn( llist *l, void *data )
{
	llist_node *node;

	aotf_assertk(node = malloc(sizeof(llist_node)),
					"Couldn't allocate memory for new llist node");
	
	/* Set values of new node */
	node->data = data;
	node->next = l->h;
	node->prev = NULL;

	if(node->next)
		node->next->prev = node;

	/* Place in list */
	l->h = node;
	if(l->t == NULL)
		l->t = node;
		
	l->size++;

	return node;
}

void *llist_append( llist *l, void *data )
{
	return (llist_append_n(l, data))->data;
}

void *llist_append_p( llist *l, void *data )
{
	return (llist_append_pn(l, data))->data;
}

llist_node *llist_append_n( llist *l, void *data )
{
	void *d;

	aotf_assertk(d = malloc(l->db),
					"Couldn't allocate memory for new llist node");
	memcpy(d, data, l->db);

	return llist_append_pn(l, d);
}
	
llist_node *llist_append_pn( llist *l, void *data )
{
	llist_node *node;

	aotf_assertk(node = malloc(sizeof(llist_node)),
					"Couldn't allocate memory for new llist node");

	/* Set values of new node */
	node->next = NULL;
	node->prev = l->t;
	node->data = data;

	if(node->prev)
		node->prev->next = node;

	/* Place in list */
	l->t = node;
	if(l->h == NULL)
		l->h = node;
		
	l->size++;

	return node;
}

void *llist_insert( llist *l, void *data,
				int (*compare) (void *, void *))
{
	return (llist_insert_n(l, data, compare))->data;
}
				
void *llist_insert_p( llist *l, void *data,
				int (*compare) (void *, void *))
{
	return (llist_insert_pn(l, data, compare))->data;
}

llist_node *llist_insert_n( llist *l, void *data,
				int (*compare) (void *, void *))
{
	void *d;
	aotf_assertk(d = malloc(l->db),
					"Couldn't allocate memory for new llist node");
	memcpy(d, data, l->db);

	return llist_insert_pn(l, d, compare);
}

llist_node *llist_insert_pn( llist *l, void *data, 
					int (*compare) (void *, void* ))
{
	llist_node *node, *n;
	
	/* Find the node to insert before */
	for ( n = l->h; n; n = n->next )
		if ( compare(data, n->data) <= 0 )
			break;
	
	if ( node == l->h )
		return llist_prepend_pn(l, data);
	if ( !node )
		return llist_append_pn(l, data);
		
	/* Not a special case... do it ourselves */
	aotf_assertk(node = malloc(sizeof(llist_node)),
					"Couldn't allocate memory for new llist node");

	/* Set values of new node */
	node->data = data;
	node->next = n;
	node->prev = n->prev;
	node->prev->next = node;
	node->next->prev = node;
	
	l->size++;

	return node;
}

void llist_remove( llist *l, llist_node *n )
{
	if (n->data)
		free(n->data);

	llist_shallow_remove(l, n);

	return;
}	

void llist_shallow_remove( llist *l, llist_node *n )
{
	if(l->h == n) {
		l->h = n->next;	
	}

	if(l->t == n) {
		l->t = n->prev;
	}
	
	if(n->prev) {
		n->prev->next = n->next;
	}

	if(n->next) {
		n->next->prev = n->prev;
	}
	
	free(n);

	l->size--;

	return;
}

bool llist_remove_data( llist *l, void *data, bool search_entire )
{
	return llist_remove_data_main(l, data, search_entire, FALSE);
}

bool llist_shallow_remove_data( llist *l, void *data, bool search_entire )
{
	return llist_remove_data_main(l, data, search_entire, TRUE);
}

bool llist_remove_data_main( llist *l, void *data,
				bool search_entire, bool shallow )
{
	llist_node *n, *m;
	bool foundit = FALSE;
	
	for(n = l->h; n; n = m) {
		m = n->next;

		if(n->data == data) {
			if(!shallow) {
				free(n->data);
			}
			llist_shallow_remove(l, n);
			foundit = TRUE;
			if(!search_entire)
				break;
		}
	}

	return foundit;
}

llist_node *llist_find( llist *l, void *data )
{
	llist_node *n;
    
	for ( n = l->h; n; n = n->next )
		if ( n->data == data )
			return n;
    
	return NULL;
}

llist_node *llist_find_idx( llist *l, int n )
{

	llist_node *node;
	int i;

	for(node = l->h, i = 0; node && i < n; node = node->next, i++);

	/* Will return NULL if the list is too short */
	return node;
}

int llist_size( llist *l ) 
{
	llist_node *n;
	int count = 0;

	for (n = l->h; n; n = n->next) {
		count++;
	}
	
	if ( count != l->size )
		fprintf(stderr, "List count does not match size! "
						"count: %d, size: %d\n", count, l->size);

	return count;
}

void *llist_arrayify( llist *l, int *s )
{
	int size = llist_size(l);
	char *data = (void *)malloc(l->db * size);
	llist_node *n;
	int i;

	for (i=0, n = l->h; n; i++, n = n->next) {
		memcpy(data + i * l->db, n->data, l->db);
	}

	*s = size;
	return data;
}

void *llist_arrayify_pointers( llist *l, int *s )
{
	int size = llist_size(l);
	char *data = (void *)malloc(size * sizeof(void *));
	llist_node *n;
	int i;

	for (i=0, n = l->h; n; i++, n = n->next) {
		memcpy(data + i * sizeof(void *), &n->data, sizeof(void *));
	}

	*s = size;
	return data;
}

void llist_foreach( llist *l, void (*fn)( void *data ) )
{
	llist_node *n, *m;
	
	for ( n = l->h; n; n = m )
 	{
		m = n->next;
		fn(n->data);
	}
}
