/*---------------------------------------------------------------------------
  Name:		datentry.c
  Version:	0.7
  Internal:	10
  Date:		27 may 97
  Author:	Marco Rivellino

  Description:	define the date_entry data type functions

---------------------------------------------------------------------------*/
/*
    Copyright (C) 1997  Marco Rivellino & Fabio Menegoni

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"
#include "datentry.h"
#include "dates.h"

#include <stdio.h>
#include <string.h>




/*
 *	copy date src to date dest
 *
 */
void copy_date_entry(struct date_entry *dest, struct date_entry *src)
{
	dest->day = src->day;
	dest->month = src->month;
	dest->year = src->year;
	dest->type = src->type;
	strcpy(dest->comment, src->comment);
}




/*
 *	Read stored date entry from "infile" and store it in "stored".
 *
 *	if no errors occurred	=>	stored->state = DA_E_DE_VALID
 *					return TRUE
 *
 *	if good EOF		=>	return FALSE
 *
 *	if invalid date		=>	stored->state = DA_E_DE_INVALID
 *					return TRUE
 *
 *	if infile format errors	=>	set *format_error to the correct value (see "datentry.h" file)
 *					*format_error_line = last line read
 *					return FALSE
 *
 *	the file format is:
 *		[comment]
 *		day[/month[/year]]
 *		date_type
 *		look_distance
 *		date_comment
 *	where:
 *	day/month/year  form a valid date (see  valid_date_entry() )
 *	date_type = one of {"RECU", "DATE"}
 *
 */
int read_date_entry(struct date_entry *stored, FILE *infile, char **format_error_line, int *format_error)
{
	static char cbuffer[CBUFSIZE];			/* disk read buffer */
	char *buftemp;

	*format_error = DA_E_DF_OK;			/* no format errors */
	*format_error_line = NULL;
	while (fgets(cbuffer, CBUFSIZE, infile) && !feof(infile)) {		/* skip empty lines + comments between dates */
		buftemp = skip_spaces(cbuffer);
		if (!is_comment_char(buftemp) && *buftemp != '\n')
			break;
	}

	if (feof(infile))				/* read + set stored date */
		return FALSE;
	stored->month = 0;
	stored->year = 0;
	if (sscanf(cbuffer, " %d / %d / %d", &(stored->day), &(stored->month), &(stored->year)) == 0 ) {
		*format_error = DA_E_DF_NODAY;
		*format_error_line = cbuffer;
		return FALSE;
	}

	fgets(cbuffer, CBUFSIZE, infile);		/* read + set stored date type */
	if (feof(infile)) {
		*format_error = DA_E_DF_BADEOF;
		*format_error_line = cbuffer;
		return FALSE;
	}
	if (!STRNCASECMP(cbuffer, "DATE", 4))
		stored->type = DE_DATE;
	else
		if (!STRNCASECMP(cbuffer, "RECU", 4))
			stored->type = DE_RECU;
		else {
			*format_error = DA_E_DF_NOTYPE;
			*format_error_line = cbuffer;
			return FALSE;
		}

	fgets(cbuffer, CBUFSIZE, infile);		/* read + set stored date look distance */
	if (feof(infile)) {
		*format_error = DA_E_DF_BADEOF;
		*format_error_line = cbuffer;
		return FALSE;
	}
	if (sscanf(cbuffer, " %d", &(stored->look_distance)) == 0 ) {
		*format_error = DA_E_DF_NOLOOKDIST;
		*format_error_line = cbuffer;
		return FALSE;
	}

	fgets(cbuffer, CBUFSIZE, infile);		/* read + set stored date comment */
	if (feof(infile)) {
		*format_error = DA_E_DF_BADEOF;
		*format_error_line = cbuffer;
		return FALSE;
	}
	strcpy(stored->comment, strip_lf(skip_spaces(cbuffer)));

	if (valid_date_entry(stored))
		stored->state = DA_E_DE_VALID;
	else
		stored->state = DA_E_DE_INVALID;

	return TRUE;
}




/*
 *	Write date entry "de" to file "outfile"
 *	if I/O errors	=>	return FALSE
 *			else	return TRUE
 *
 */
int write_date_entry(struct date_entry *de, FILE *outfile)
{
	fprintf(outfile, "%d/%d/%d\n"
			 "%s\n"
			 "%d\n"
			 "%s\n\n",
		de->day,
		de->month,
		de->year,
		(de->type == DE_DATE ? "DATE" : "RECU"),
		de->look_distance,
		de->comment);
	return (ferror(outfile) ? FALSE : TRUE);
}




/*
 *	if date entry "de"'s day/month/year fields form a valid date
 *		=>	return TRUE
 *		else	return FALSE
 *
 *	A valid date has:
 *		1 <= day <= number of days of the date's month
 *		0 <= month <= 12
 *		0 <= year <= 32766	(TCC bug...)
 *		-1 <= look distance <= MAX_LOOK_DISTANCE
 *
 */
int valid_date_entry(struct date_entry *de)
{
			/* gen feb mar apr may jun jul aug sep oct nov dec */
	int month_len[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	int leap_year_month_len[] = {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

	if (de->year < 0 || de->year > 32766)
		return FALSE;
	if (de->month < 0 || de->month > 12)
		return FALSE;

	if (de->year && !de->month)
		return FALSE;

	if (de->day < 1)
		return FALSE;

	if (de->year)
		if (!leap_year(de->year)) {
			if (de->day > month_len[de->month-1])
				return FALSE;
		} else {
			if (de->day > leap_year_month_len[de->month-1])
				return FALSE;
		}
	else
		if (de->month) {
			if (de->day > leap_year_month_len[de->month-1])
				return FALSE;
		} else
			if (de->day > 31)
				return FALSE;

	if (de->look_distance < -1 || de->look_distance > MAX_LOOK_DISTANCE)
		return FALSE;

	return TRUE;
}




/*
 *	Compare date entries de1 and de2.
 *	Return "de2 - de1" i.e.:
 *		if de1 < de2	=>	return N>0
 *		if de1 == de2	=>	return 0
 *		if de1 > de2	=>	return N<0
 *
 *	these are the compare criteria (listed in order of importance):
 *
 *		state:			DA_E_DE_INVALID < DA_E_DE_VALID
 *
 *		type:			DE_DATE < DE_RECU
 *
 *		day, month, year:	usual gregorian calendar order
 *
 *		look_distance:		usual natural numbers order
 *
 */
int cmp_date_entry(struct date_entry *de1, struct date_entry *de2)
{
	int d;

	if (de1->state == DA_E_DE_INVALID && de2->state == DA_E_DE_INVALID)
		return 0;
	if (de1->state == DA_E_DE_INVALID)
		return 1;
	if (de2->state == DA_E_DE_INVALID)
		return -1;

	if (de1->type == DE_DATE && de2->type == DE_RECU)
		return 1;
	if (de1->type == DE_RECU && de2->type == DE_DATE)
		return -1;

	d=day_distance(de1, de2);
	if (d != 0)
		return d;
	else
		return (de2->look_distance - de1->look_distance);
}




/*
 *	Return signed distance in days between dates of
 *	date entry de1 and de2 (distance = de2 - de1)
 *
 */
int day_distance(struct date_entry *de1, struct date_entry *de2)
{
			/* gen feb mar apr may jun jul aug sep oct nov dec */
	int month_len[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	int daynum1 = 0, daynum2 = 0;		/* number of days of the 2 dates from the beginning of the 2 years */
	int yeardaynum = 0;			/* number of days between the two years */
	int i, sign;
	struct date_entry *tmp;

	if (de2->year < de1->year) {
		tmp = de1;
		de1 = de2;
		de2 = tmp;
		sign = -1;
	} else
		sign = 1;

	for (i = 1; i <= (de1->month) - 1; i++)
		daynum1 += month_len[i-1];
	if (leap_year(de1->year) && daynum1 >= 31 + 28)
		daynum1++;			/* if we are in a leap year add 1 day to feb */
	daynum1 += de1->day;

	for (i = 1; i <= (de2->month) - 1; i++)
		daynum2 += month_len[i-1];
	if (leap_year(de2->year) && daynum2 >= 31 + 28)
		daynum2++;
	daynum2 += de2->day;

	yeardaynum = 365 * (de2->year - de1->year);
	for (i = de1->year; i <= de2->year - 1; i++)
		if (leap_year(i))
			yeardaynum++;

	return (sign * (yeardaynum + daynum2 - daynum1));
}




/*
 *	Return  non-zero if year is a leap year
 *		0 otherwise
 *
 */
int leap_year(int year)
{
	return (year <= 1752 ? (year % 4 == 0) : (year % 4 == 0 && year % 100 != 0) || year % 400 == 0);
}
