/*---------------------------------------------------------------------------
  Name:		datentry.h
  Version:	0.7
  Internal:	8
  Date:		27 may 97
  Author:	Marco Rivellino

  Description:	declare the date_entry data type interface

---------------------------------------------------------------------------*/
/*
    Copyright (C) 1997  Marco Rivellino & Fabio Menegoni

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef DA_DATENTRY_H
#define DA_DATENTRY_H


#include <stdio.h>


/* date_entry type defs */
#define DE_RECU		0
#define DE_DATE		1

/* date_entry comment field size */
#define COMMENTSIZE	128

/* date_entry state defs */
#define DA_E_DE_VALID		0
#define DA_E_DE_INVALID		1

/* data file format errors */
#define DA_E_DF_OK		0
#define DA_E_DF_BADEOF		1
#define DA_E_DF_NODAY		2
#define DA_E_DF_NOTYPE		4
#define DA_E_DF_NOLOOKDIST	8


struct date_entry {
	int day;			/* 1-31 */
	int month;			/* 0-12 */
	int year;			/* 0-32767 */
	int type;			/*  DE_RECU / DE_DATE  */
	int look_distance;		/* -1 - MAX_LOOK_DISTANCE */
	char comment[COMMENTSIZE];
	int state;			/*  DA_E_DE_VALID / DA_E_DE_INVALID  */
};




/* --------------------------- Functions ----------------------------------- */




/*
 *	copy date src to date dest
 *
 */
extern	void copy_date_entry(struct date_entry *dest, struct date_entry *src);




/*
 *	Read stored date entry from "infile" and store it in "stored".
 *
 *	if no errors occurred	=>	stored->state = DA_E_DE_VALID
 *					return TRUE
 *
 *	if good EOF		=>	return FALSE
 *
 *	if invalid date		=>	stored->state = DA_E_DE_INVALID
 *					return TRUE
 *
 *	if infile format errors	=>	set *format_error to the correct value (see "datentry.h" file)
 *					*format_error_line = last line read
 *					return FALSE
 *
 *	the file format is:
 *		[comment]
 *		day[/month[/year]]
 *		date_type
 *		look_distance
 *		date_comment
 *	where:
 *	day/month/year  form a valid date (see  valid_date_entry() )
 *	date_type = one of {"RECU", "DATE"}
 *
 */
extern	int read_date_entry(struct date_entry *stored, FILE *infile, char **format_error_line, int *format_error);




/*
 *	Write date entry "de" to file "outfile"
 *	if I/O errors	=>	return FALSE
 *			else	return TRUE
 *
 */
extern	int write_date_entry(struct date_entry *de, FILE *outfile);




/*
 *	if date entry "de"'s day/month/year fields form a valid date
 *		=>	return TRUE
 *		else	return FALSE
 *
 *	A valid date has:
 *		1 <= day <= number of days of the date's month
 *		0 <= month <= 12
 *		0 <= year <= 32767
 *		-1 <= look distance <= MAX_LOOK_DISTANCE
 *
 */
extern	int valid_date_entry(struct date_entry *de);




/*
 *	Compare date entries de1 and de2.
 *	Return "de2 - de1" i.e.:
 *		if de1 < de2	=>	return N>0
 *		if de1 == de2	=>	return 0
 *		if de1 > de2	=>	return N<0
 *
 *	these are the compare criteria (listed in order of importance):
 *
 *		state:			DA_E_DE_INVALID < DA_E_DE_VALID
 *
 *		type:			DE_DATE < DE_RECU
 *
 *		day, month, year:	usual gregorian calendar order
 *
 *		look_distance:		usual natural numbers order
 *
 */
extern	int cmp_date_entry(struct date_entry *de1, struct date_entry *de2);




/*
 *	Return signed distance in days between dates of
 *	date entry de1 and de2 (distance = de2 - de1)
 *
 */
extern	int day_distance(struct date_entry *de1, struct date_entry *de2);




/*
 *	Return  non-zero if year is a leap year
 *		0 otherwise
 *
 */
extern	int leap_year(int year);


#endif
