/* 
 *  Edit dates.lst - with ncurses interface
 *
 *  AUTHOR : Fabio Menegoni 
 *  VERSION: 0.0.4
 *
 */

/*
	Copyright (C) 1997  Marco Rivellino & Fabio Menegoni

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
*/


#include "config.h"
#include "datentry.h"
#include "dates.h"
#include "edit.h"

#include <form.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	edit_data_file(char *filename, int edit_mode)
|   Descript. :	permette di editare il file filename; edit mode e'
|		EDIT_ALL:	tutte le entry del file
|		EDIT_RECU:	solo quelle in cui il campo type == DE_RECU
|		EDIT_DATE:	solo quelle in cui il campo type == DE_DATE	
|		EDIT_INVALID:	solo quelle marcate non valide
|   Return Val:	DA_E_OK:	tutto ok
|		DA_E_OPEN_W:	errore apertura file filename in scrittura
|		DA_E_CURSES:	errore nella gestione dell'interfaccia curses
|		DA_E_FORMAT:	errore di formato nel file filename
|		DA_E_WRITE:	errore in scrittura
+--------------------------------------------------------------------------*/

int
edit_data_file(char *fname, int emode)
{
	FILE 			*istream, *ostream;
	FIELD			*f[MAXFIELD];
	FORM			*form;
	WINDOW			*formw; 
	struct elem		*e=NULL;
	int			rc=DA_E_OK, c, finished = FALSE;
	int			j;
	char			tmp[TMPSIZE];
	
	
	if ((start_curses()) != DA_E_OK) {
		fprintf(stderr, "initscr() error: aborting...\n\n");	
		return DA_E_CURSES;
	}
	if ((istream=fopen(fname,"r"))==NULL) {
		sprintf(tmp, "File %s not found. Creating a new file.", fname);	
		requester(tmp, "OK", NULL);
		de_2_form_row(f, default_de(), 0);
		f[FIELD_PER_ROW] = NULL;
	}
	else {
		rc = form_readl(istream, f, &e, emode);
		fclose(istream);
		switch(rc) {
			case DA_E_NO_ROOM:
				sprintf(tmp, "Form row count exceed max. allowed (%d)", MAXROW);
				requester(tmp, "OK", NULL);
				break;
			case DA_E_FORMAT:
				requester("File format error. Exiting...", "OK", NULL);
				end_curses();
				return rc;
				break;
			case DA_E_FILE_EMPTY:
				requester("File empty. Starting with a default entry", "OK", NULL);
				de_2_form_row(f, default_de(), 0);
				f[FIELD_PER_ROW] = NULL;
				break;
			case DA_E_NO_REQ_TYPE:
				sprintf(tmp,"No entries of requested type. Exiting...");
				requester(tmp, "OK", NULL);
				end_curses();
				return rc;
				break;
		}
	}

	if (signal (SIGINT, sig_catch) == SIG_ERR)
		requester("signal(SIGINT) error", "OK", NULL);
	if (signal (SIGQUIT, sig_catch) == SIG_ERR)
		requester("signal(SIGQUIT) error", "OK", NULL);
	if (signal (SIGTERM, sig_catch) == SIG_ERR)
		requester("signal(SIGTERM) error", "OK", NULL);

	mk_stdscr(DA_HEADER, DA_FOOTER);

	form = new_form(f); 
	form_row_hl(form, 1);
	
	display_form(form, DA_FORM_HEADER);
	formw = form_win(form);
	
	while (!finished) 
	if (form_driver(form, c=form_virtualize(formw)) != E_OK)
		switch (c) {
			case KEY_UP:
				form_row_hl(form, 0);
				j=((field_index(current_field(form))/FIELD_PER_ROW)*FIELD_PER_ROW)-FIELD_PER_ROW;
				if (j<0)
					j=field_count(form)-FIELD_PER_ROW;
				else
					j=j%field_count(form);
				set_current_field(form, f[j]);
				form_row_hl(form, 1);
				break;
			case KEY_DOWN:
				form_row_hl(form, 0);
				j=((field_index(current_field(form))/FIELD_PER_ROW)*FIELD_PER_ROW)+FIELD_PER_ROW;
				j=j%field_count(form);
				set_current_field(form, f[j]);
				form_row_hl(form, 1);
				break;
			case KEY_NPAGE:
				form_row_hl(form, 0);
				form_driver(form, REQ_NEXT_PAGE);
				form_row_hl(form, 1);
				break;
			case KEY_PPAGE:
				form_row_hl(form, 0);
				form_driver(form, REQ_PREV_PAGE);
				form_row_hl(form, 1);
				break;
			case KEY_HOME:
				form_row_hl(form, 0);
				form_driver(form, REQ_FIRST_FIELD);
				form_row_hl(form, 1);
				break;
			case KEY_END:
				form_row_hl(form, 0);
				form_driver(form, REQ_LAST_FIELD);
				form_row_hl(form, 1);
				break;
			case KEY_F(1):
				help_win();
				break;
			case CTRL('W'):
				if (!form_validate(form, &j)) {
					requester("Wrong entry detected", "OK", NULL);
					set_current_field(form, f[j*FIELD_PER_ROW]);
				}
				else {
					if (requester("Save and quit?", "OK", "CANCEL") == DA_E_REQ_OK) {
						strcpy(tmp, fname);
						strcat(tmp, "~");
						if (!rename(fname, tmp) || requester("Cannot create backup file", "CONTINUE", "CANCEL") == DA_E_REQ_OK) {
							if ((ostream=fopen(fname,"w"))==NULL) 
								requester("Error opening output file.", "OK", NULL);
							else {
								form_row_de_sync(form);
								if (form_writel(ostream, form, &e) != DA_E_OK) 
									requester("Error writing on output file.", "OK", NULL);	
								else
									finished = TRUE;
							}
						}
					}
				}
				break;
			case CTRL('X'):
				if (form_changed(form)) {
					if (requester("Quit without saving?", "OK", "CANCEL")==DA_E_REQ_OK)
						finished=TRUE;
				}
				else
					finished=TRUE;
				break;
			case CTRL('A'):
				if (row_add(form)==DA_E_NO_ROOM)
					requester("Not enaugh space", "OK", NULL);
				break;
			case CTRL('I'):
				if(row_insert(form)==DA_E_NO_ROOM)
					requester("Not enaugh space", "OK", NULL);
				break;
			case CTRL('D'):
				row_delete(form);
				break;
			case CTRL('U'):
				row_undo(form);
				break;
			default:
				beep();
				break;
		}
	
	erase_form(form);
	free_form(form);
	end_curses();
		
	return DA_E_OK;
}
/* end of edit_data_file() */



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	
|   Descript. :	
|   Return Val:	
+--------------------------------------------------------------------------*/

int
form_row_hl(FORM *form, int on)
{
	int			j, hl, k;
	FIELD			**f;
	
	j = ((field_index(current_field(form))/FIELD_PER_ROW)*FIELD_PER_ROW);
	f = form_fields(form);
	switch (on) {
		case 0: 
			hl = A_BOLD;
			break;
		case 1:
			hl = A_REVERSE;
			break;
		default:
			hl = A_NORMAL;
			break;
	}
	for (k = j; k < (j+FIELD_PER_ROW); k++) {
		set_field_fore(f[k], hl);
		set_field_back(f[k], hl);
	}
	
	return DA_E_OK;		
}

/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int start_curses(void)
|   Descript. :	inizializza l'interfaccia curses
|   Return Val:	DA_E_OK:	tutto ok
|		DA_E_CURSES:	errore nella gestione dell'interfaccia curses	
+--------------------------------------------------------------------------*/

int
start_curses()
{
	if (initscr() == NULL)
		return DA_E_CURSES;
	cbreak(); 
	noecho();
	if (start_color() == ERR)
		return DA_E_CURSES;
	if ((has_colors())) {
		init_pair(BG_COLOR, COLOR_WHITE, COLOR_BLACK);
		init_pair(TITLE_COLOR, COLOR_WHITE, COLOR_RED);
		init_pair(BOTTOM_LINE_COLOR, COLOR_CYAN, COLOR_BLUE);
		init_pair(ACTIVE_COLOR, COLOR_WHITE, COLOR_BLUE);
		init_pair(INACTIVE_COLOR, COLOR_WHITE, COLOR_BLACK);
	}
	else {
		/* Monochrome management routines here! */
	}
	return DA_E_OK;
}
/* End of start_curses() */




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int end_curses()
|   Descript. :	chiamata per tornare in modo terminale
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
end_curses()
{
	clear();
	refresh();
	endwin();
	return DA_E_OK;
}
/* end of end_curses() */




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_readl(FILE *istream, FIELD **f, struct elem **e, int emode)
|   Descript. :	legge struct date_entry dal file istream del tipo emode e 
|		li mette in f. Tutti i campi del file in ingresso sono presenti
|		nella lista puntata da e.
|   Return Val:	DA_E_OK		tutto ok
|		DA_E_FORMAT	errore di formato nel file di ingresso
|		DA_E_NO_ROOM	raggiunto il numero massimo di fields
|		DA_E_FILE_EMPTY il file e' vuoto
|		DA_E_NO_REQ_TYPE non esistono entry col campo type richiesto
+--------------------------------------------------------------------------*/

int 
form_readl(FILE *istream, FIELD **f, struct elem **e, int emode)
{
	struct date_entry 	*detmp=NULL;
	struct elem		*etmp=NULL;
	int 			j=0, err=0, to_form, another;
	char 			*errline;
	
	detmp=(struct date_entry *)malloc(sizeof(struct date_entry));
	while ((another=read_date_entry(detmp, istream, &errline, &err)) && j<MAXROW) {
		j++;
		etmp=list_ord_cons(etmp, detmp);
		detmp=(struct date_entry *)malloc(sizeof(struct date_entry));
	}
	free(detmp);
	
	*e=etmp;
	
	if (err!=DA_E_DF_OK) {
		list_free(*e);
		return DA_E_FORMAT;
	}
	if (j==0) 
		return DA_E_FILE_EMPTY;
		 
	j=0;
	while (etmp && (j<MAXROW)) {
		detmp=etmp->de;
		to_form=( 
			(emode==EDIT_ALL) || 
			((emode==EDIT_RECU)    && (detmp->type==DE_RECU) && (detmp->state==DA_E_DE_VALID)) || 
			((emode==EDIT_DATE)    && (detmp->type==DE_DATE) && (detmp->state==DA_E_DE_VALID)) || 
			((emode==EDIT_INVALID) && (detmp->state==DA_E_DE_INVALID)) 
		);
		if (to_form) {
			de_2_form_row(f, detmp, j);
			j++;
		}
		etmp=etmp->next;
	}
	
	f[j*FIELD_PER_ROW]=NULL;
	if (j==0) {
		list_free(*e);
		return DA_E_NO_REQ_TYPE;
	}
	if (j==MAXROW)
		return DA_E_NO_ROOM;
	
	return DA_E_OK;
}
/* End of form_readl() */




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_writel(FILE *ostream, FORM *form, struct elem **e)
|   Descript. :	salva le informazioni contenute in e, integrandole con quelle
|		eventualmente aggiunte/cambiate del form, secondo il formato
|		specificato dai doc: data\ntype\nlook_distance\ncomment\n
|   Return Val:	DA_E_OK
|		DA_E_WRITE
|		DA_E_SYSTEM_ERROR
+--------------------------------------------------------------------------*/

int
form_writel(FILE *ostream, FORM *form, struct elem **e)
{
	int			row = 0, maxrow = 0, rc=DA_E_OK;
	struct date_entry	*detmp;
	struct elem		*etmp=*e;
	
	maxrow=(field_count(form)/FIELD_PER_ROW);
	while (row<maxrow) {
		if (!list_find(etmp, (detmp=form_row_de(form, row))))
			etmp=list_ord_cons(etmp, detmp);
		row++;
	}


	/* write info header */
	fprintf(ostream, 
		"#\n"
		"# DATES data file\n"
		"#\n"
		"# this text file is in the format:\n"
		"#\n"
		"#\t[comment]\n"
		"#\tday[/month[/year]]\n"
		"#\tdate_type\n"
		"#\tlook_distance\n"
		"#\tdate_comment\n"
		"#\n"
		"#\n"
	);
	
	while (etmp && (rc==DA_E_OK)) {
		detmp=etmp->de;
		if (strcmp(detmp->comment, DA_DELETE_TAG)) 
			if (!write_date_entry(detmp, ostream))
				rc=DA_E_WRITE;
		etmp=etmp->next;
	}
	*e=etmp;
	
	return rc;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	struct date_entry *form_row_2_de(FORM *form, int row)
|   Descript. :	ritorna l'indirizzo della struct date_entry relativa 
|		alla riga row del form, aggiornata con i dati della riga.
|   Return Val:	
+--------------------------------------------------------------------------*/

struct date_entry 
*form_row_2_de(FORM *form, int row)
{
	int			cur = (row * FIELD_PER_ROW);
	FIELD			**f;
	struct date_entry	*detmp;
	char 			*stmp;
	
	f = form_fields(form);
	detmp = (struct date_entry *)field_userptr(f[cur]);
	
	detmp->day   = atoi(field_buffer( (form_fields(form))[cur], 0));
	detmp->month = atoi(field_buffer( (form_fields(form))[cur+2], 0));
	detmp->year  = atoi(field_buffer( (form_fields(form))[cur+4], 0));
	
	stmp = field_buffer( form_fields(form)[cur+5], 0 );
	if (stmp[0]=='R')
		detmp->type=DE_RECU;
	else if (stmp[0]=='D')
		detmp->type=DE_DATE;
	
	detmp->look_distance = atoi(field_buffer( (form_fields(form))[cur+6],0 ));
	
	stmp = field_buffer( form_fields(form)[cur+7], 0 );
	strcpy(detmp->comment, stmp);
	
	return detmp;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	struct date_entry *form_row_de(FORM *form, int row)
|   Descript. :	ritorna l'indirizzo della struct date_entry relativa 
|		alla riga row del form.
|   Return Val:	
+--------------------------------------------------------------------------*/

struct date_entry 
*form_row_de(FORM *form, int row)
{
	int			cur = (row * FIELD_PER_ROW);
	FIELD			**f;
	struct date_entry	*detmp;
	
	f = form_fields(form);
	detmp = (struct date_entry *)field_userptr(f[cur]);
	
	return detmp;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	de_2_form_row(FIELD **f, struct date_entry *de, int row)
|   Descript. :	riempie la riga ``row'' del form  
|		con le informazioni contenute in ``de''.
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
de_2_form_row(FIELD **f, struct date_entry *de, int row)
{
	char		day[3], month[3], year[5], type[7], look_d[4];
	char		**types_vet;
	int		cur = row * FIELD_PER_ROW, col=1;
	
	types_vet = (char **)malloc(3*(sizeof(char *)));
	types_vet[0] = (char *)malloc(strlen("Recurr")+1);
	strcpy(types_vet[0],"Recurr");
	types_vet[1] = (char *)malloc(strlen("Date")+1);
	strcpy(types_vet[1],"Date");
	types_vet[2] = NULL;
	
	sprintf(day, "%02d", de->day);
	
	if (de->month)
		sprintf(month, "%02d", de->month);
	else
		month[0]='\0';
	
	if (de->year)
		sprintf(year, "%04d", de->year);
	else
		year[0]='\0';
		
	if (de->type == DE_RECU)
		sprintf(type, "Recurr");
	else if (de->type ==DE_DATE)
		sprintf(type, "Date");
	else
		sprintf(type, "?");
		
	sprintf(look_d, "%d", de->look_distance);
	
	row = row % (LINES - ROW_FROM_BOTTOM);
	f[cur] = make_field(row, col, 1, 2);
	set_field_opts(f[cur], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT | O_BLANK | O_AUTOSKIP | O_STATIC );
	set_field_type(f[cur], TYPE_INTEGER, 2, 1, 31);
	set_field_buffer(f[cur], 0, day); 
	set_field_userptr(f[cur], (char *)de);
	
	if (row == 0)
		set_new_page(f[cur], TRUE);
	cur++; col+=2;
	
	f[cur] = make_label(row, col, "/");
	cur++; col+=1;
	
	f[cur] = make_field(row, col, 1, 2);
	set_field_opts(f[cur], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT | O_BLANK | O_AUTOSKIP | O_NULLOK | O_STATIC ); 
	set_field_type(f[cur], TYPE_INTEGER, 2, 0, 12);	
	set_field_buffer(f[cur], 0, month); 
	cur++; col+=2;
	
	f[cur] = make_label(row, col, "/");
	cur++; col+=1;
	
	f[cur] = make_field(row, col, 1, 4);
	set_field_opts(f[cur], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT | O_BLANK | O_AUTOSKIP | O_NULLOK | O_STATIC );
	set_field_type(f[cur], TYPE_INTEGER, 4, 0, 9999);	
	set_field_buffer(f[cur], 0, year); 
	cur++; col+=6;
		
	f[cur] = make_field(row, col, 1, 6);
	set_field_opts(f[cur], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT | O_BLANK | O_AUTOSKIP | O_STATIC );
	set_field_type(f[cur], TYPE_ENUM, types_vet, FALSE, FALSE);	
	set_field_buffer(f[cur], 0, type); 
	cur++; col+=8;

	f[cur] = make_field(row, col, 1, 3);
	set_field_opts(f[cur], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT | O_BLANK | O_AUTOSKIP | O_STATIC );
	set_field_type(f[cur], TYPE_INTEGER, 0, -1, MAX_LOOK_DISTANCE);	
	set_field_buffer(f[cur], 0, look_d); 
	cur++; col+=5;
		
	f[cur] = make_field(row,col,1,COLS-col-2); 
	set_field_opts(f[cur], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT );
	set_max_field(f[cur], COMMENTSIZE-1);
	set_field_buffer(f[cur], 0, strdup(de->comment));
	
	return DA_E_OK;
} 
/* end of de_2_form_row() */



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int display_form(FORM *f, char *title)
|   Descript. :		
|   Return Val:	DA_E_OK
|		DA_E_SYSTEM_ERROR
|		quelli restituiti da form_post()
+--------------------------------------------------------------------------*/

int
display_form(FORM *f, char *title)
{
	WINDOW		*w;
	int		rc, j;
	char		stmp[COLS-1];
	
	sprintf(stmp, "%s", title);
	for (j=strlen(stmp); j<COLS-2; j++)
		stmp[j]=' ';
	stmp[j]='\0';
	
	if ((w = newwin(LINES-2, COLS, 1, 0)) != NULL) {
		box(w, ACS_VLINE, ACS_HLINE);
		wmove(w, 1, 1);
		wattrset(w, COLOR_PAIR(ACTIVE_COLOR) | A_BOLD | A_ALTCHARSET);
		wprintw(w, "%s", stmp);
		set_form_win(f, w);
		set_form_sub(f, derwin(w, LINES-ROW_FROM_BOTTOM, COLS-2, 2, 1));
		nodelay(w, FALSE);
		keypad(w, TRUE);
	}
	else
		return DA_E_SYSTEM_ERROR;
	
	if ((rc = post_form(f)) != E_OK) 
		return rc;
	wrefresh(w);
	return DA_E_OK;
}
/* end of display_form() */

 

/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int erase_form(FORM *f)
|   Descript. :		
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
erase_form(FORM *f)
{
	WINDOW		*w = form_win(f);
	WINDOW		*s = form_sub(f);

	unpost_form(f);
	werase(w);
	wrefresh(w);
	delwin(s);
	delwin(w);
	return 0;
}
/* end of erase_form() */



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	FIELD *make_label(int frow, int fcol, char *label)
|   Descript. :	crea un un nuovo campo non modificabile
|   Return Val:	puntatore a FIELD.
+--------------------------------------------------------------------------*/

FIELD *make_label(int frow, int fcol, char *label)
{
	FIELD		*f = new_field(1, strlen(label), frow, fcol, 0, 0);

	if (f) {
		set_field_buffer(f, 0, label);
		set_field_opts(f, field_opts(f) & ~O_ACTIVE);
	}
	return f;
}
/* end of make_label() */



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	FIELD *make_field(int frow, int fcol, int rows, int cols)
|   Descript. :	crea un nuovo campo e ne inizializza gli attributi
|   Return Val:	puntatore a FIELD
+--------------------------------------------------------------------------*/

FIELD *make_field(int frow, int fcol, int rows, int cols)
{
	FIELD		*f = new_field(rows, cols, frow, fcol, 0, 0);

	if (f)
		set_field_back(f, A_UNDERLINE);
	return f;
}
/* end of make_field() */




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_validate(FORM *form, int *wrong_row)
|   Descript. :	controlla la validita' dei campi del form
|   Return Val:	TRUE tutti i campi sono validi
|		FALSE la riga wrong_row contiene un dato invalido.
+--------------------------------------------------------------------------*/

int
form_validate(FORM *form, int *wrong_row)
{
	int			row, maxrow, cur;
	FIELD			**f;
	struct date_entry	*detmp;
	char 			*stmp;
	
	f = form_fields(form);
	detmp = (struct date_entry *)malloc(sizeof(struct date_entry));
	maxrow=(field_count(form)/FIELD_PER_ROW);
	
	while (row<maxrow) {
		cur=row*FIELD_PER_ROW;
		
		detmp->day   = atoi(field_buffer( (form_fields(form))[cur], 0));
		detmp->month = atoi(field_buffer( (form_fields(form))[cur+2], 0));
		detmp->year  = atoi(field_buffer( (form_fields(form))[cur+4], 0));
	
		stmp = field_buffer( form_fields(form)[cur+5], 0 );
		if (stmp[0]=='R')
			detmp->type=DE_RECU;
		else if (stmp[0]=='D')
			detmp->type=DE_DATE;
	
		detmp->look_distance = atoi(field_buffer( (form_fields(form))[cur+6],0 ));
	
		stmp = field_buffer( form_fields(form)[cur+7], 0 );
		strcpy(detmp->comment, stmp);
		
		if (!valid_date_entry(detmp)) {
			*wrong_row = row;
			return FALSE;
		}
		else
			row++;
	}

	return TRUE;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_row_de_sync(FORM *form)
|   Descript. :	sincronizza le righe del form con le relative struct date_entry
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
form_row_de_sync(FORM *form)
{
	int			row=0, maxrow=0;
	
	maxrow=(field_count(form)/FIELD_PER_ROW);
	for (row=0; row<maxrow; row++)
		form_row_2_de(form, row);

	return DA_E_OK;
}


/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_changed(FORM *form, int *wrong_row)
|   Descript. :	controlla se c'e' identita' tra il contenuto attuale
|		dei campi del form e il contenuto del file
|   Return Val:	TRUE e' stato trovato qualche cambiamento.
|		FALSE nessun cambiamento.
+--------------------------------------------------------------------------*/

int
form_changed(FORM *form)
{
	int			row=0, maxrow=0, cur;
	struct date_entry	*detmp;
	int			day, month, year, look_d, type;
	char			*stmp;
	
	maxrow=(field_count(form)/FIELD_PER_ROW);
	while (row<maxrow) {
		detmp=form_row_de(form, row);
		cur   = row*FIELD_PER_ROW;
		
		day   = atoi(field_buffer( (form_fields(form))[cur + 0], 0));
		month = atoi(field_buffer( (form_fields(form))[cur + 2], 0));
		year  = atoi(field_buffer( (form_fields(form))[cur + 4], 0));

		stmp = field_buffer( form_fields(form)[cur + 5], 0 );
		if (stmp[0]=='R')
			type=DE_RECU;
		else if (stmp[0]=='D')
			type=DE_DATE;
	
		look_d = atoi(field_buffer( (form_fields(form))[cur + 6],0 ));
	
		stmp = field_buffer( form_fields(form)[cur + 7], 0 );

		if (	(day    != detmp->day) ||
			(month  != detmp->month) ||
			(year   != detmp->year) ||
			(type   != detmp->type) ||
			(look_d != detmp->look_distance) ||
			(strcmp(stmp, detmp->comment)) )
			return TRUE;
		else
			row++;
		
	}

	return FALSE;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int mk_stdscr(char *header, char *footer)
|   Descript. :		
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
mk_stdscr(char *header, char *footer)
{	
	int j;
	move(0, 0);
	attrset(COLOR_PAIR(TITLE_COLOR) | A_BOLD | A_ALTCHARSET);
	printw("%c%c%c %s", 178, 177, 176, header);
	for (j=strlen(header)+4; j<COLS-4; j++)
		printw(" ");
	printw(" %c%c%c", 176, 177, 178);
	attrset(COLOR_PAIR(BOTTOM_LINE_COLOR) | A_BOLD | A_ALTCHARSET);
	move(LINES-1, 0);
	printw("%s",footer);
	for (j=strlen(footer); j<COLS; j++)
		printw(" ");
	refresh();
	return DA_E_OK;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_virtualize(WINDOW *w)
|   Descript. :	prende un carattere dalla window e ritorna il
|		carattere da fornire come argomento a form_driver.
|   Return Val:	
+--------------------------------------------------------------------------*/

int
form_virtualize(WINDOW *w)
{
	static int	mode = REQ_INS_MODE;
	int		c;
	
	c=wgetch(w);
	switch(c) {
		case '\n':
			return(REQ_RIGHT_FIELD);
		case KEY_LEFT:
			return(REQ_LEFT_CHAR);
		case KEY_RIGHT:
			return(REQ_RIGHT_CHAR);
		case CTRL('Y'):
			return(REQ_DEL_LINE);
		case KEY_DC:
			return(REQ_DEL_CHAR);
		case 127:			/* delete */
			return(REQ_DEL_PREV);
		case KEY_IC:
			if (mode == REQ_INS_MODE)
				return(mode = REQ_OVL_MODE);
			else
				return(mode = REQ_INS_MODE);
		default:
			return c;
	}
}
/* end of form_virtualize() */



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int requester(char *msg, char *ok, char *cancel)
|   Descript. :	fa apparire un requester con il messaggio msg
|		e fa scegliere tra ok e cancel.
|   Return Val:	DA_E_OK
|		DA_E_REQ_NO_ROOM
|		DA_E_CURSES
|		DA_E_REQ_OK
|		DA_E_REQ_CANCEL
+--------------------------------------------------------------------------*/

int
requester(char *msg, char *ok, char *cancel)
{
	int		wxs, wys, wxdim, wydim;
	int		minwxdim = 30, ok_can_xdim = 8, reqscrlines = 6;
	int		rc, c;
	WINDOW		*w, *subw;
	FIELD		*f[4];
	FORM		*form;
	
	if (strlen(msg)>(COLS-2))
		return DA_E_REQ_NO_ROOM;
	if ((strlen(msg)+4)>minwxdim)
		wxdim = strlen(msg) + 4;
	else
		wxdim = minwxdim;
	wydim = reqscrlines;
	wxs = (int)((COLS  - wxdim)/2);
	wys = (int)((LINES - wydim)/2);
	
	if ((w = newwin(wydim, wxdim, wys, wxs))==NULL)
		return DA_E_CURSES;
	subw = derwin(w, wydim-2, wxdim-2, 1, 1);
	box(w, ACS_VLINE, ACS_HLINE);
	nodelay(w, FALSE);
	keypad(w, TRUE);	
	
	f[0] = make_label(1, 1, msg);
	set_field_just(f[0], JUSTIFY_CENTER);
	if (cancel) {
		f[1] = make_field(3, 1, 1, ok_can_xdim);
		set_field_opts(f[1], O_VISIBLE | O_ACTIVE | O_PUBLIC);
		set_field_buffer(f[1], 0, ok);
		set_field_just(f[1], JUSTIFY_CENTER);
		f[2] = make_field(3, wxdim - 1 -1 - ok_can_xdim, 1, ok_can_xdim);
		set_field_opts(f[2], O_VISIBLE | O_ACTIVE | O_PUBLIC);
		set_field_buffer(f[2], 0, cancel);
		set_field_just(f[2], JUSTIFY_CENTER);
		f[3] = NULL;
	}
	else {
		f[1] = make_field(3, (wxdim-ok_can_xdim)/2, 1, ok_can_xdim);
		set_field_opts(f[1], O_VISIBLE | O_ACTIVE | O_PUBLIC );
		set_field_buffer(f[1], 0, ok);
		set_field_just(f[1], JUSTIFY_CENTER);
		f[2] = NULL;
	}
	form = new_form(f);
	
	set_form_win(form, w);
	set_form_sub(form, subw);
	post_form(form);
	
	while ((c=wgetch(w))!='\n')
		switch(c) {
			case KEY_LEFT:
				form_driver(form, REQ_PREV_FIELD);
				break;
			case KEY_RIGHT:
				form_driver(form, REQ_NEXT_FIELD);
				break;
			default:
				beep();
		}
	if (f[1] == current_field(form))
		rc=DA_E_REQ_OK;
	else
		rc=DA_E_REQ_CANCEL;
	
	unpost_form(form);
	werase(w);
	wrefresh(w);
	delwin(subw);
	delwin(w);
	free_form(form);

	return rc;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int row_delete(FORM *form)
|   Descript. :	marca deleted la riga corrente del form
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
row_delete(FORM *form)
{
	FIELD			**f, *ftmp[1];
	int			cur;
	
	form_row_hl(form, 0);
	cur = (field_index(current_field(form)) / FIELD_PER_ROW)*FIELD_PER_ROW;
	f = form_fields(form);
	unpost_form(form);
	ftmp[0]=NULL;
	set_form_fields(form, ftmp);
	
	set_field_buffer(f[cur+FIELD_PER_ROW-1], 0, DA_DELETE_TAG);
	field_opts_off(f[cur+FIELD_PER_ROW-1], O_EDIT);
	
	set_form_fields(form, f);
	post_form(form);
	set_current_field(form, f[(cur+FIELD_PER_ROW)%field_count(form)]);
	form_row_hl(form, 1);
	
	return DA_E_OK;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int row_undo(FORM *form)
|   Descript. :	copia la struct associata alla riga corrente sulla riga 
|		corrente
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
row_undo(FORM *form)
{
	FIELD			**f, *ftmp[1];
	int			cur, row;
	struct date_entry	*detmp;
	char			day[3], month[3], year[5], type[7], look_d[4];
	
	
	row = (field_index(current_field(form))) / FIELD_PER_ROW;
	cur = row*FIELD_PER_ROW;
	f = form_fields(form);
	detmp=form_row_de(form, row);

	sprintf(day, "%02d", detmp->day);
	if (detmp->month)
		sprintf(month, "%02d", detmp->month);
	else
		month[0]='\0';
	if (detmp->year)
		sprintf(year, "%04d", detmp->year);
	else
		year[0]='\0';
	if (detmp->type == DE_RECU)
		sprintf(type, "Recurr");
	else if (detmp->type ==DE_DATE)
		sprintf(type, "Date");
	else
		sprintf(type, "?");
	sprintf(look_d, "%d", detmp->look_distance);
	
	unpost_form(form);
	ftmp[0]=NULL;
	set_form_fields(form, ftmp);
	
	set_field_buffer(f[cur+0], 0, day);
	set_field_buffer(f[cur+2], 0, month);
	set_field_buffer(f[cur+4], 0, year);
	set_field_buffer(f[cur+5], 0, type);
	set_field_buffer(f[cur+6], 0, look_d);
	set_field_buffer(f[cur+7], 0, strdup(detmp->comment));	
	set_field_opts(f[cur+7], O_VISIBLE | O_ACTIVE | O_PUBLIC | O_EDIT );
	
	set_form_fields(form, f);
	post_form(form);
	set_current_field(form, f[cur]);
	
	return DA_E_OK;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int row_insert(FORM *form)
|   Descript. :	inserisce una riga nel form, nella posizione attualmente 
|		occupata dal cursore
|   Return Val:	DA_E_OK
|		DA_E_NO_ROOM
+--------------------------------------------------------------------------*/

int
row_insert(FORM *form)
{
	FIELD			**f, *ftmp[1];
	int			currentrow, rowtmp, lastrow;
	int 			cfield;
	struct date_entry	*detmp;
	
	form_row_hl(form, 0);
	cfield=(field_index(current_field(form)));
	lastrow = (field_count(form) / FIELD_PER_ROW);
	if ((lastrow+1) > MAXROW)
		return DA_E_NO_ROOM;
	currentrow = (cfield / FIELD_PER_ROW);
	f=form_fields(form);
	
	unpost_form(form);
	ftmp[0]=NULL;
	set_form_fields(form, ftmp);

	f[(lastrow*FIELD_PER_ROW+FIELD_PER_ROW)] = NULL; 
	for (rowtmp = lastrow; rowtmp > currentrow; rowtmp--)
		row_move(f, rowtmp, rowtmp-1);
	detmp = default_de();
	de_2_form_row(f, detmp, currentrow); 
	form_into_pages(f);

	set_form_fields(form, f);
	post_form(form);
	set_current_field(form, f[cfield]);
	form_row_hl(form, 1);
	
	return DA_E_OK;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	struct date_entry *default_de()
|   Descript. :	ritorna un puntatore ad una struct date_entry, contentente
|		i valori di default.
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

struct date_entry
*default_de()
{
	struct date_entry		*detmp;
	
	detmp = (struct date_entry *)malloc(sizeof(struct date_entry));
	detmp->day = 1;
	detmp->month = 0;
	detmp->year  = 0;
	detmp->type  = DE_DATE;
	detmp->look_distance = DEFAULT_LOOK_DISTANCE;
	detmp->state = DA_E_DE_VALID;
	detmp->comment[0] = '\0';
	
	return detmp;
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int row_add(FORM *form)
|   Descript. :	aggiunge una riga alla fine del form 
|   Return Val:	DA_E_OK
|		DA_E_NO_ROOM
+--------------------------------------------------------------------------*/

int
row_add(FORM *form)
{
	FIELD			**f, *ftmp[1];
	int			lastrow;
	struct date_entry	*detmp;
	
	form_row_hl(form, 0);
	lastrow = (field_count(form) / FIELD_PER_ROW);
	if ((lastrow+1) > MAXROW)
		return DA_E_NO_ROOM;
	f = form_fields(form);
	unpost_form(form);
	ftmp[0]=NULL;
	set_form_fields(form, ftmp);
	
	f[lastrow*FIELD_PER_ROW+FIELD_PER_ROW] = NULL;

	detmp = default_de();
	de_2_form_row(f, detmp, lastrow);
	form_into_pages(f);

	set_form_fields(form, f);
	post_form(form);
	set_current_field(form, f[lastrow*FIELD_PER_ROW]);
	form_row_hl(form, 1);
	
	return DA_E_OK;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int row_move(FIELD **f, int dest, int src)
|   Descript. :	muove la riga src del form nella riga dest, e mette a NULL
|		tutti i campi di src (liberando la memoria).
|   Return Val:	DA_E_OK
|		DA_E_BAD_ARGUMENT: f[src] e' NULL oppure src non e' valido
+--------------------------------------------------------------------------*/

int
row_move(FIELD **f, int dest, int src)
{
	int			cursrc  = src * FIELD_PER_ROW;
	int			curdest = dest * FIELD_PER_ROW;
	int			oldcol, dummy, j;
	
	dest = dest % (LINES - ROW_FROM_BOTTOM);
	if ( (cursrc<0) || (cursrc>MAXFIELD) )
		return DA_E_BAD_ARGUMENT;
	for (j=0; j<FIELD_PER_ROW; j++) {
		if (f[cursrc+j]) {
			field_info(f[cursrc+j], &dummy, &dummy, &dummy, &oldcol, &dummy, &dummy);
			f[curdest+j] = dup_field(f[cursrc + j], dest, oldcol);
			free_field(f[cursrc +j]);
			f[cursrc +j] = NULL;
		}
		else
			return DA_E_BAD_ARGUMENT;
	}
	return DA_E_OK;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int form_into_pages(FIELD **f)
|   Descript. :	page the form
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
form_into_pages(FIELD **f)
{
	int			j;
	
	for (j=0; f[j]; j++) 
		if ( (((j/FIELD_PER_ROW)%(LINES-ROW_FROM_BOTTOM)) == 0) && (j%FIELD_PER_ROW == 0))
			set_new_page(f[j], TRUE);
		else
			set_new_page(f[j], FALSE);
	
	return DA_E_OK;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int help_win()
|   Descript. :	visualizza una finestra di aiuto
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int
help_win()
{
	WINDOW		*w, *subw;
	int		dimx=40, dimy=20; 
	int		x=((COLS-dimx)/2), y=((LINES-dimy)/2);
	int		j;
	
	w=newwin(dimy, dimx, y, x);
	box(w,ACS_VLINE,ACS_HLINE);
	subw=derwin(w, dimy-3, dimx-2, 2, 1);
	wmove(w,1,1);
	wattrset(w, COLOR_PAIR(ACTIVE_COLOR) | A_BOLD | A_ALTCHARSET);
	wprintw(w,"%s", DA_HELP_HEADER);
	for (j=strlen(DA_HELP_HEADER); j<dimx-2; j++)
		wprintw(w," ");
	waddstr(subw, "------------------------------\n");
	waddstr(subw, "Avalaible commands:\n");
	waddstr(subw, "------------------------------\n");
	waddstr(subw, "ENTER  : Next field\n");
	waddstr(subw, "BS     : Previous field\n");
	waddstr(subw, "CTRL-X : Quit without saving\n");
	waddstr(subw, "CTRL-W : Save changes and exit\n");
	waddstr(subw, "CTRL-D : Delete selected row\n");
	waddstr(subw, "CTRL-Y : Delete current field\n");	
	waddstr(subw, "CTRL-I : Insert new row\n");
	waddstr(subw, "INSERT : Toggle insert mode\n");
	waddstr(subw, "CTRL-A : Add new row\n");
	waddstr(subw, "CTRL-U : Undo changes\n");
	waddstr(subw, "PG-DOWN: Next page\n");
	waddstr(subw, "PG-UP  : Previous page\n");
	waddstr(subw, "HOME   : First field in current page\n"); 
	waddstr(subw, "END    : Last  field in current page\n"); 
	
	wrefresh(w);
	nodelay(w, FALSE);
	keypad(w, TRUE);
	wgetch(w);
	werase(w);
	wrefresh(w);
	delwin(subw);
	delwin(w);
	
	return DA_E_OK; 
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	funzione di servizio. buona per il debug.
|   Descript. :	
|   Return Val:	
+--------------------------------------------------------------------------*/

char *
curs_error(int e)
{
	char		err[50];
	
	switch(e) {
		case E_OK:
			strcpy(err,"E_OK");
			break;
		case E_SYSTEM_ERROR:
			strcpy(err,"E_SYSTEM_ERROR");
			break;
		case E_BAD_ARGUMENT:
			strcpy(err,"E_BAD_ARGUMENT");
			break;
		case E_POSTED:
			strcpy(err,"E_POSTED");
			break;
		case E_BAD_STATE:
			strcpy(err,"E_BAD_STATE");
			break;
		case E_NO_ROOM:
			strcpy(err,"E_NO_ROOM");
			break;
		case E_NOT_POSTED:
			strcpy(err,"E_SYSTEM_ERROR");
			break;
		case E_NOT_CONNECTED:
			strcpy(err,"E_NOT_POSTED");
			break;
		default:
			strcpy(err, "UNCLASSIFIED ERROR");
			break;	
	}

	return strdup(err);
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	struct elem *list_cons(struct elem *e, struct date_entry *de)
|   Descript. :	aggiunge una struct date_entry * alla lista puntata da e.
|   Return Val:	la nuova lista.
+--------------------------------------------------------------------------*/

struct elem *
list_cons(struct elem *e, struct date_entry *de)
{
	struct elem		*etmp;
	
	etmp=(struct elem *)malloc(sizeof(struct elem));
	etmp->de=de;
	etmp->next=e;
	return etmp;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	struct elem *list_ord_cons(struct elem *e, struct date_entry *de)
|   Descript. :	aggiunge, con l'ordinamento specificato dalla funzione 
|		cmp_date_entry(), una struct date_entry alla lista puntata 
|		da e. 
|   Return Val:	la nuova lista.
+--------------------------------------------------------------------------*/

struct elem *
list_ord_cons(struct elem *e, struct date_entry *de)
{
	if (e==NULL)
		return list_cons(e, de);
	else {
		struct elem *etmp1, *etmp2;
		etmp1=etmp2=list_cons(e, NULL);
		while (etmp1->next != NULL)
			if (cmp_date_entry(etmp1->next->de, de)>0)
				etmp1=etmp1->next;
			else
				break;
		etmp1->next=list_cons(etmp1->next, de);
		etmp1=etmp2->next;
		free(etmp2);
		return etmp1;
	}
}



/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	int list_free(struct elem *e) 
|   Descript. :	libera la memoria dalla lista puntata da e
|   Return Val:	DA_E_OK
+--------------------------------------------------------------------------*/

int 
list_free(struct elem *e)
{
	struct elem 		*etmp;
	
	while (e) {
		etmp=e->next;
		free(e);
		e=etmp;
	}
	
	return DA_E_OK;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	
|   Descript. :	
|   Return Val:	
+--------------------------------------------------------------------------*/

int
list_find(struct elem *e, struct date_entry *de)
{
	while (e) 
		if (e->de == de)
			return TRUE;
		else 
			e=e->next;
		
	return FALSE;
}




/*---------------------------------------------------------------------------
|   Facility  :	
|   Function  :	
|   Descript. :	
|   Return Val:	
+--------------------------------------------------------------------------*/

void
sig_catch(int signo)
{
	end_curses();
	exit(1);
}