//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "Alarm.h"

#include "Klook.h"
#include "AlarmDialog.h"
#include "Appointment.h"
#include "docking.h"

//
// default constructor
//
Alarm::Alarm(Appointment *parent, Klook* p, const char *n)
	: QObject((QObject*)p,n)
{
	alarmTime = new QDateTime();
	appointment = parent;
	mainW = p;
	dueOption = 0;
	hasFired = false;
	repeating = false;
	alarmBox = 0;
}

//
// constructor for a given date and hour and minute
//
Alarm::Alarm(Appointment *parent, QDate& d, int h, int m, int due, Klook *p , const char *n)
	: QObject((QObject*)p,n)
{
	alarmTime = new QDateTime();
	appointment = parent;
	mainW = p;
	setDate(d);
	setTime(h,m,due);
	dueOption = due;
	hasFired = false;
	repeating = false;
	alarmBox = 0;
}

//
// remove the alarm
//
Alarm::~Alarm()
{
	delete alarmTime;
	if(alarmBox != (AlarmDialog*)0)
	{
		disconnect(alarmBox);
		delete alarmBox;
	}
}

//
// set the date portion of the alarm
//
void
Alarm::setDate(QDate& d)
{
	alarmTime->setDate(d);
}

//
// Start the timer running
//
void
Alarm::goTimer()
{
int secsToGo;
QDateTime now = QDateTime::currentDateTime();

	secsToGo = now.secsTo(*alarmTime);
	if(secsToGo > 0)
	{
		beginTimer(secsToGo);
	}
	else if(!hasFired)
	{
		timerMessage();
	}
}

//
// actually start the timer
//
void
Alarm::beginTimer(int s)
{
	QTimer::singleShot(s * 1000, this, SLOT(timerMessage()));
}

//
// set the time portion of the alarm
//
void
Alarm::setTime(int h, int m, int due)
{
int hour,min,day,month,year,dmon;
int diffHour,diffMin,diffDay;

	diffHour = diffMin = diffDay = 0;

	day = alarmTime->date().day();
	month = alarmTime->date().month();
	year = alarmTime->date().year();
	dmon = alarmTime->date().daysInMonth();

	hour = h;
	min = m;
	
	switch(due)
	{
	case 0: diffMin = 15;	break;
	case 1: diffMin = 30;	break;
	case 2: diffHour = 1;	break;
	case 3: diffHour = 2;	break;
	case 4: diffHour = 3;	break;
	case 5: diffDay = 1;	break;
	case 6: diffDay = 2;	break;
	case 7: diffDay = 3;	break;
	case 8: diffDay = 7;	break;
	}
	
	min -= diffMin;	
	if(min < 0)
	{
		min += 60;
		hour--;
	}

	hour -= diffHour;
	if(hour < 0)
	{
		hour += 24;
		day--;
	}

	day -= diffDay;
	if(day < 0)
	{
		month--;
		if(month == 0)
		{
			month = 12;
			year--;
		}
		QDate tdat(year,month,1);
		dmon = tdat.daysInMonth();
		day += dmon;
	}

	QTime t(hour,min,0);
	alarmTime->setTime(t);
	QDate d(year,month,day);
	alarmTime->setDate(d);
	dueOption = due;
}

//
// return the due option
//
int
Alarm::getDue()
{
	return dueOption;
}

//
// save the alarm
//
void
Alarm::save(QDataStream& fil, bool hasChanged)
{
	if(hasChanged)
	{
		fil << *alarmTime;
		fil << dueOption;
		fil << (int)hasFired;
	}
}

//
// load the alarm
//
void
Alarm::load(QDataStream& fil, int /*version*/)
{
int hf;

	fil >> *alarmTime;
	fil >> dueOption;
	fil >> hf;
	hasFired = (bool)hf;
}

//
// display a message when the timer expires
//
void
Alarm::timerMessage()
{
	if(alarmBox == (AlarmDialog*)0)
	{
		hasFired = true;
		alarmBox = new AlarmDialog(appointment,mainW);
		alarmBox->setGeometry(0,60,200,200);
		alarmBox->blink();
		connect(alarmBox,SIGNAL(fired()),this,SLOT(killAlarmBox()));
		connect(alarmBox,SIGNAL(repeat()),this,SLOT(reIssueAlarmBox()));
		mainW->markAsChanged();
	}
	else
	{
		alarmBox->blink();
	}
	mainW->getDock().setAlarm(this);
}

//
// destroy the alarm box creates by timerMessage
//
void
Alarm::killAlarmBox()
{
	mainW->getDock().resetAlarm(this);
	alarmBox->hide();
	repeating = false;
}

//
// re-issue an alarm
//
void
Alarm::reIssueAlarmBox()
{
	killAlarmBox();
	repeating = true;
	beginTimer();
}

//
//
//
QString&
Alarm::getSubject()
{
	return appointment->getSubject();
}

//
// Return a reference to the alarm box
//
AlarmDialog*
Alarm::getAlarmBox()
{
	return alarmBox;
}

//
// return the status of the alarm
//
bool
Alarm::alreadyFired()
{
	return hasFired;
}

//
// is the alarm currently in a repeat loop
//
bool
Alarm::isRepeating()
{
	return repeating;
}
