//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "Appointment.h"

#include "Alarm.h"
#include "Repeat.h"
#include "Day.h"
#include "Calendar.h"
#include "Export.h"
#include "TimeSlots.h"

//
// create a new appointment with just a start time
// typically, will be called when creating a new appointment
//
Appointment::Appointment(Klook* w, Day* d, int stim)
{
	subject = new QString("");
	text = new QString("");
	if(stim == -1)
	{
		stim = 0;
	}
	setStartTime(stim);
	setFinishTime(stim);
	Sensitivity = 0;
	valid = false;
	alarmed = false;
	mainW = w;
	day = d;
	repeat = (Repeat*)0;
}

//
// create a new appointment
//
Appointment::Appointment(Klook* w, Day* d, QString& sub, int stim , int ftim , QString& txt=0)
{
	subject = new QString(sub);
	text = new QString(txt);
	setStartTime(stim);
	setFinishTime(ftim);
	setText(txt);
	Sensitivity = 0;
	valid = false;
	alarmed = false;
	mainW = w;
	day = d;
	repeat = (Repeat*)0;
}

//
// tidy up
//
Appointment::~Appointment()
{
	delete subject;
	delete text;
	if(alarmed)
	{
		clearAlarm();
	}
	if(repeat != (Repeat*)0)
	{
		day->getCalendar()->removeRepeat(repeat);
		delete repeat;
	}
}

//
// set the subject text
//
void
Appointment::setSubject( QString& sub )
{
	subject->resize(0);
	subject->append(sub);
}

//
// return a reference to the subject text
//
QString&
Appointment::getSubject()
{
	return *subject;
}

//
// set the descriptive text
//
void
Appointment::setText( QString& app )
{
	text->resize(0);
	text->append(app);
}

//
// return a reference to the descriptive text
//
QString&
Appointment::getText()
{
	return *text;
}

//
// set the start time (a value between 0 and TimeSlots::totalSlots)
//
void
Appointment::setStartTime( int t )
{
	startTime = t;
}

//
// set the finish time (a value between 0 and TimeSlots::totalSlots)
//
void
Appointment::setFinishTime( int t )
{
	finishTime = t;
}

//
// return the start time
//
int
Appointment::getStartTime()
{
	return startTime;
}

//
// return the finish time
//
int
Appointment::getFinishTime()
{
	return finishTime;
}

//
// save this appointment to a file
//
void
Appointment::save(QDataStream& fil, bool hasChanged)
{
int len;
char *add;

	if(hasChanged)
	{
		fil << startTime;
		fil << finishTime;
		len = subject->length();
		add = subject->data();
		fil << len;
		if(len)
		{
			fil.writeRawBytes(add,len);
		}
		len = text->length();
		add = text->data();
		fil << len;
		if(len)
		{
			fil.writeRawBytes(add,len);
		}
		int a = alarmed;
		fil << a;
		fil << Sensitivity;

	}

	if(alarmed)
	{
		alarm->save(fil,hasChanged);
	}

	if(hasChanged)
	{
		if(repeat != (Repeat*)0)
		{
			fil << 1;
			repeat->save(fil,hasChanged);
		}
		else
		{
			fil << 0;
		}
	}
}

//
// load an appointment from a file
//
void
Appointment::load(QDataStream& fil, int version, Klook* p, bool isme)
{
int len;

	fil >> startTime;
	fil >> finishTime;
	if(version < 13)
	{
		startTime *= 2;
		finishTime *= 2;
	}
	fil >> len;
	subject->resize(len);
	if(len)
	{
		fil.readRawBytes(subject->data(),len);
	}
	subject->truncate(len);
	fil >> len;
	text->resize(len);
	if(len)
	{
		fil.readRawBytes(text->data(),len);
	}
	text->truncate(len);
	int a;
	fil >> a;
	alarmed = (bool)a;
	if(version > 1)
	{
		fil >> Sensitivity;
	}
	else
	{
		Sensitivity = public_access;
	}

	makeValid(p);

	if(alarmed)
	{
		alarm = new Alarm(this,mainW);
		alarm->load(fil,version);
		if(isme)
		{
			alarm->goTimer();
		}
	}

	if(version > 7)
	{
	int rep;

		fil >> rep;
		if(rep == 1)
		{
			repeat = new Repeat(day,this);
			repeat->load(fil,version);
			day->getCalendar()->addRepeat(repeat);
		}
	}
	else
	{
		repeat = (Repeat*)0;
	}
}

//
// return the status of an appointment
// in order to use the EditAppointment class an instance of an appointment 
// must exist. It is created as 'new' (or not valid). When the user exits
// from the EditAppointment class the instance of the appointment can be
// marked as value (see next method). If the user quits from EditAppointment 
// then the instance of the appointment should be deleted
//
bool
Appointment::isNew()
{
	return !valid;
}

//
// make an appointment valid
//
void
Appointment::makeValid(Klook* w)
{
	valid = true;
	mainW = w;
}

//
// Set an alarm
//
void
Appointment::setAlarm(int due, QDate& day)
{
int h,m;

	h = startTime / TimeSlots::hourUnits;
	m = (startTime % TimeSlots::hourUnits) * TimeSlots::minuteUnits;

	if(!alarmed)
	{
		alarm = new Alarm(this,day,h,m,due,mainW);
	}
	alarmed = true;
	alarm->goTimer();
}

//
// Clear the alarm
//
void
Appointment::clearAlarm()
{
	delete alarm;
	alarmed = false;
}

//
// return the status of any alarm
// 
bool
Appointment::isAlarmed()
{
	return alarmed;
}

//
// return a reference to the alarm
//
Alarm&
Appointment::getAlarm()
{
	return *alarm;
}

//
// return the status of an alarm (if it exists)
//
int
Appointment::getAlarmStatus()
{
int rs = none;

	if(alarmed)
	{
		rs = armed;
		if(alarm->alreadyFired())
		{
			rs = fired;
		}
		if(alarm->isRepeating())
		{
			rs = repeating;
		}
	}
	return rs;
}

//
// set the sensitivity level
//
void
Appointment::setSensitivity(int s)
{
	Sensitivity = s;
}

//
// return the sensitivity level
//
int
Appointment::getSensitivity()
{
	return Sensitivity;
}

void
Appointment::setRepeat(Repeat* r)
{
	repeat = r;
}

Repeat*
Appointment::getRepeat()
{
	return repeat;
}

bool
Appointment::isRepeating()
{
	return (repeat != (Repeat*)0) ? true : false;
}

void
Appointment::clearRepeat()
{
	if(repeat != (Repeat*)0)
	{
		day->getCalendar()->removeRepeat(repeat);
		delete repeat;
	}
	repeat = (Repeat*)0;
}

Day*
Appointment::getDay()
{
	return day;
}


//
// search the subject and text for a string
//
bool
Appointment::findString(QString& str)
{
	return ((subject->find(str.data(),0,false) != -1) || (text->find(str.data(),0,false) != -1)) ? true : false;
}


//
// export the appointment
//
void
Appointment::export(QTextStream& fil, Export& expdata)
{
int len;
char *add;
QString d;
TimeSlots ts;

	fil << expdata.delimiter();
	fil << "A";	// mark it as an appointment
	fil << expdata.delimiter();

	fil << ts.whichTime(startTime);
	fil << expdata.delimiter();

	fil << ts.whichTime(finishTime);
	fil << expdata.delimiter();

	len = subject->length();
	add = subject->data();
	if(len)
	{
		fil.writeRawBytes(add,len);
	}
	else
	{
		fil << " ";
	}
	fil << expdata.delimiter();

	len = text->length();
	add = text->data();
	if(len)
	{
		fil.writeRawBytes(add,len);
	}
	else
	{
		fil << " ";
	}
	fil << expdata.delimiter();

	d.sprintf("%d",Sensitivity);
	fil << d;
//	fil << expdata.delimiter();

//	d.sprintf("%d",(int)alarmed);
//	fil << d;
}


