//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "Kweekview.h"
#include "Kmonthview.h"
#include "Kyearplan.h"
#include "Day.h"
#include "KalDatePicker.h"

#include "Calendar.h"
#include "Kdayview.h"
#include "Search.h"
#include "Export.h"


//
// Order days
//
int
DayList::compareItems( GCI item1, GCI item2 )
{
Day *d1,*d2;

	d1 = (Day*)item1;
	d2 = (Day*)item2;

	return compareItems(d1,d2);
}

int
DayList::compareItems( Day* d1 , Day *d2 )
{
QDate i1;
QDate i2;
int rs;

	i1 = d1->getDate();
	i2 = d2->getDate();

	if(i1 > i2)
	{
		rs = 1;
	}
	else if(i1 < i2)
	{
		rs = -1;
	}
	else
	{
		rs = 0;	// I hope we never get here though
	}

	return rs;
}




//
// create a new calendar with a current date of today
//
Calendar::Calendar()
{
	currDate = QDate::currentDate();
	days.setAutoDelete(true);
	repeats.setAutoDelete(false);
}

//
// create a new calendar with a specified current date
//
Calendar::Calendar(QDate& date)
{
	currDate = date;
	days.setAutoDelete(true);
	repeats.setAutoDelete(false);
}

//
// remove all days from this calendar
//
Calendar::~Calendar()
{
	remove();
}

//
// set the current day
//
void
Calendar::setDay(int d)
{
	currDate.setYMD(currDate.year(),currDate.month(),d);
}

//
// set the current month
//
void
Calendar::setMonth(int m)
{
	currDate.setYMD(currDate.year(),m,currDate.day());
}

//
// set the current year
//
void
Calendar::setYear(int y)
{
	currDate.setYMD(y,currDate.month(),currDate.day());
}

//
// return a refernce to the current date object
//
QDate&
Calendar::getDate()
{
	return currDate;
}

//
// set the current date to the date passed
//
void
Calendar::setDate(QDate& date)
{
	currDate = date;
}

//
// set the current date based upon the day,month and year passed
//
void
Calendar::setDate(int y, int m, int d)
{
	setYear(y);
	setMonth(m);
	setDay(d);
}

//
// return the day of the week of the first day on the current month
// (0=sunday,6=saturday)
//
int
Calendar::firstDay()
{
	QDate *t = new QDate(currDate);
	t->setYMD(currDate.year(),currDate.month(),1);
	int d =  t->dayOfWeek();
	if(d == 7)
	{
		d = 0;
	}
	return d;
}

//
// save the instances of days associated with this calendar
//
void
Calendar::save(QDataStream& fil, bool hasChanged)
{
	Day *curr;
	QListIterator<Day> it(days);
	if(hasChanged)
	{
		int recs = it.count();
		fil << recs;
	}
	for(; it.current(); ++it)
	{
		curr = it.current();
		curr->save(fil,hasChanged);
	}
}

//
// load days into this calendar
//
void
Calendar::load(QDataStream& fil, int version, Klook* p, bool me)
{
Day *read;
int recs;

	fil >> recs;
	for( int i = 0; i < recs; i++)
	{
		read = new Day(this);
		read->load(fil,version,p,me);
		days.inSort(read);
	}
}

//
// remove all days from this calendar
//
void
Calendar::remove()
{
	days.clear();
}

//
// view appointment for the specified date into the single day viewer
//
void
Calendar::viewDate(UserAccess* u, Kdayview* v)
{
Day *dcur;
Repeat *rcur;

	// cycle through day list for today
	QListIterator<Day> itd(days);
	for(; itd.current(); ++itd)
	{
		dcur = itd.current();
		if(dcur->getDate() == currDate)
		{
			dcur->listDay(v,u);
			break;
		}
	}

	// cycle through repeating items for matches
	QListIterator<Repeat> itr(repeats);
	for(; itr.current(); ++itr)
	{
		rcur = itr.current();
		if(rcur->getDay()->getDate() != currDate)
		{
			if(rcur->testDate(currDate))
			{
				if(rcur->isAppointment())
				{
					v->displayItem(rcur->getAppointment());
				}
				if(rcur->isDayEvent())
				{
					v->displayItem(rcur->getDayEvent());
				}
			}
		}
	}
}

//
// view appointment for the specified date and the next six days
// into the week viewer
//
void
Calendar::viewDate(UserAccess* u, Kweekview* v)
{
QDate cd;
int w;
Day *dcur;
Repeat *rcur;

	cd = currDate;

	QListIterator<Day> itd(days);

	v->baseDay(currDate.dayOfWeek());

	for(w = 1; w < 8; w++)
	{
		for(; itd.current(); ++itd)
		{
			dcur = itd.current();
			if(dcur->getDate() == cd)
			{
				dcur->listDay(v,u,w);
			}
		}

		// cycle through repeating items for matches
		QListIterator<Repeat> itr(repeats);
		for(; itr.current(); ++itr)
		{
			rcur = itr.current();
			if(rcur->getDay()->getDate() != cd)
			{
				if(rcur->testDate(cd))
				{
					if(rcur->isAppointment())
					{
						v->displayItem(w,rcur->getAppointment());
					}
				}
			}
		}

		// now reset the lists and increment the day
		itd.toFirst();
		itr.toFirst();
		cd = cd.addDays(1);
	}
}

//
// view appointment for the month of the spoecified date
// into the month viewer
//
void
Calendar::viewDate(UserAccess* u, Kmonthview* v)
{
int w;
Day *dcur;
Repeat *rcur;

	QDate cd(currDate.year(),currDate.month(),1);

	v->setDays(currDate.daysInMonth(),cd.dayOfWeek());
	v->setCurrentDay(currDate.day());

	QListIterator<Day> itd(days);

	for(w = 1; w <= currDate.daysInMonth(); w++)
	{
		for(; itd.current(); ++itd)
		{
			dcur = itd.current();
			if(dcur->getDate() == cd)
			{
				dcur->listDay(v,u,w);
			}
		}

		// cycle through repeating items for matches
		QListIterator<Repeat> itr(repeats);
		for(; itr.current(); ++itr)
		{
			rcur = itr.current();
			if(rcur->getDay()->getDate() != cd)
			{
				if(rcur->testDate(cd))
				{
					if(rcur->isAppointment())
					{
						v->displayItem(w - 1,rcur->getAppointment());
					}
					if(rcur->isDayEvent())
					{
						v->displayItem(w - 1,rcur->getDayEvent());
					}
				}
			}
		}

		// now reset the lists and increment the day
		itd.toFirst();
		itr.toFirst();
		cd = cd.addDays(1);
	}
}


//
// view appointment for the year of the specified date
//
void
Calendar::viewDate(UserAccess* u, Kyearplan* v)
{
Day *dcur;
Repeat *rcur;
QDate cd;

	v->setYear(currDate.year());
	v->setCurrentDay(currDate.month(),currDate.day());

	QListIterator<Day> itd(days);

	for(int m = 1; m <= 12; m++)
	{
		cd.setYMD(currDate.year(),m,1);

		for(int w = 1; w <= currDate.daysInMonth(); w++)
		{
			for(; itd.current(); ++itd)
			{
				dcur = itd.current();
				if(dcur->getDate() == cd)
				{
					dcur->listDay(v,u,m,w);
				}
			}

			// cycle through repeating items for matches
			QListIterator<Repeat> itr(repeats);
			for(; itr.current(); ++itr)
			{
				rcur = itr.current();
				if(rcur->getDay()->getDate() != cd)
				{
					if(rcur->testDate(cd))
					{
						if(rcur->isAppointment())
						{
							v->addAppointment(m,w);
						}
						if(rcur->isDayEvent())
						{
							v->addDayEvent(m,w,rcur->getDayEvent()->isHoliday());
						}
					}
				}
			}

			// now reset the lists and increment the day
			itd.toFirst();
//			itr.toFirst();
			cd = cd.addDays(1);
		}	// end of month per year loop
	}		// end of year loop
	v->repaint();
}


//
// find an appointment
// firstly find an instance of a day matching the current date of this calendar
// class. If one is not found then create it, and then create an appointment
// on that day.
// if a day class is found, then find (or create) an appointment
// newly created appointments are not marked valid until saved (see the
// Appointment class for more detail)
//
Appointment *
Calendar::findAppointment(int ind, Klook* w)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(curr->getDate() == currDate)
		{
			return curr->findAppointment(ind,w);
		}
	}

	curr = new Day(this,currDate);
	days.inSort(curr);
	return curr->createAppointment(w);
}

//
// remove a specific appointment from a day in the calendar
//
void
Calendar::removeAppointment(Appointment* a)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(curr->getDate() == currDate)
		{
			curr->removeAppointment(a);
			break;
		}
	}
}

Day*
Calendar::findDay()
{
	return findDay(currDate);
}

//
// find a specific instance of a day in the calendar
// return a null pointer if the day was not found
//
Day*
Calendar::findDay(QDate& which)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(curr->getDate() == which)
		{
			return curr;
		}
	}
	return (Day*)0;
}

//
// clean all items before the currently selected date
// except repeating items that run into the future.
//
void
Calendar::cleanCalendar()
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current();)
	{
		curr = it.current();
		if(curr->getDate() < currDate)
		{
			if(curr->remove(true))
			{
				days.remove(curr);
			}
			else
			{
				++it;
			}
		}
		else
		{
			++it;
		}
	}
}

//
// add a new day
//
Day*
Calendar::addDay(QString& txt)
{
	Day *day = new Day(this,currDate,txt);
	days.inSort(day);
	return day;
}

//
// add a new day
//
Day*
Calendar::addDay(QDate &d)
{
	Day *day = new Day(this,d);
	days.inSort(day);
	return day;
}

//
// mark days with appointments in the calendar viewer
//
void
Calendar::markCalendar(KalDatePicker* pick, int m, int y)
{
int w,c,e;
bool d,h;
Day *curr;
QDate workDate;
QDate cd;
Repeat *rcur;

	if(m == 0)
	{
		m = currDate.month();
	}
	if(y == 0)
	{
		y = currDate.year();
	}

	workDate.setYMD(y,m,1);

	for(w = 1; w <= workDate.daysInMonth(); w++)
	{
		pick->setDayEvent(w,0,false,0,false);
		cd.setYMD(workDate.year(),workDate.month(),w);
		curr = findDay(cd);
		if(curr != (Day*)0)
		{
			c = curr->countAppointments();
			e = curr->countEvents();
			d = !curr->getDayNote().isEmpty();
			h = curr->findHoliday();
		}
		else
		{
			c = 0;
			d = false;
			e = 0;
			h = false;
		}

		// cycle through repeating items for matches
		QListIterator<Repeat> itr(repeats);
		for(; itr.current(); ++itr)
		{
			rcur = itr.current();
			if(rcur->getDay()->getDate() != cd)
			{
				if(rcur->testDate(cd))
				{
					if(rcur->isAppointment())
					{
						c++;
					}
					if(rcur->isDayEvent())
					{
						e++;
						if(rcur->getDayEvent()->isHoliday())
						{
							h = true;
						}
					}
				}
			}
		}	// end of repeat for loop
		pick->setDayEvent(w,c,d,e,h);
	}
	pick->update();
}


DayEvent *
Calendar::findEvent(int ind)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(curr->getDate() == currDate)
		{
			return curr->findEvent(ind);
		}
	}

	curr = new Day(this,currDate);
	days.inSort(curr);
	return curr->createEvent();
}


void
Calendar::removeEvent(DayEvent* e)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(curr->getDate() == currDate)
		{
			curr->removeEvent(e);
			break;
		}
	}
}


void
Calendar::addRepeat(Repeat* r)
{
	repeats.append(r);
}

void
Calendar::removeRepeat(Repeat* r)
{
	repeats.remove(r);
}

//
// check for a conflict between an appointment and repeating items
//
bool
Calendar::checkForConflict(int ast, int aft, QDate& day)
{
Repeat *rcur;
int tst = 0;
int tft = 0;

	// cycle through repeating items for matches
	QListIterator<Repeat> itr(repeats);
	for(; itr.current(); ++itr)
	{
		rcur = itr.current();
		if(rcur->getDay()->getDate() != day)
		{
			if(rcur->testDate(day))
			{
				if(rcur->isAppointment())
				{
					tst = rcur->getAppointment()->getStartTime();
					tft = rcur->getAppointment()->getFinishTime();
					for(int i = ast; i <= aft; i++)
					{
						if((i == tst) || (i == tft))
						{
							return true;
						}
					}
				}
			}
		}
	}	// end of repeat for loop
	return false;
}

//
// read through the appointment/events/notes for a string
//
Day*
Calendar::searchForString(Search* search)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(search->goForward())
		{
			if(curr->getDate() > search->getDate())
			{
				if(curr->searchForString(search))
				{
					return curr;
				}
			}
		}
		else	// go backward
		{
			if(curr->getDate() < search->getDate())
			{
				if(curr->searchForString(search))
				{
					return curr;
				}
			}
		}
	}

	return (Day*)0;
}

//
// export the calendar
//
void
Calendar::export(QTextStream& fil, Export& expdata)
{
	Day *curr;
	QListIterator<Day> it(days);
	for(; it.current(); ++it)
	{
		curr = it.current();
		curr->export(fil,expdata);
	}
}
