//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "Day.h"

#include "Kdayview.h"
#include "Kweekview.h"
#include "Kmonthview.h"
#include "UserAccess.h"
#include "Kyearplan.h"
#include "Repeat.h"
#include "Calendar.h"
#include "Search.h"
#include "Export.h"


//
// Order appointment according to their start time
//
int
AppList::compareItems( GCI item1, GCI item2 )
{
Appointment *a1,*a2;

	a1 = (Appointment*)item1;
	a2 = (Appointment*)item2;

	return compareItems(a1,a2);
}

int
AppList::compareItems( Appointment* a1 , Appointment *a2 )
{
int i1,i2;

	i1 = a1->getStartTime();
	i2 = a2->getStartTime();

	return (i1 - i2);
}


//
// default constructor
//
Day::Day(Calendar* c)
{
	cal = c;
	dayNote = new QString("");
	list.setAutoDelete(true);
	events.setAutoDelete(true);
}

//
// create a new day and specify the date associated with it
//
Day::Day(Calendar* c,QDate& date)
{
	cal = c;
	day = date;
	dayNote = new QString("");
	list.setAutoDelete(true);
	events.setAutoDelete(true);
}

//
// create a new dat and specify the date associated with it
//
Day::Day(Calendar* c, QDate& date, QString& txt)
{
	cal = c;
	day = date;
	dayNote = new QString(txt);
	list.setAutoDelete(true);
	events.setAutoDelete(true);
}

//
// remove all appointments from this day
//
Day::~Day()
{
	delete dayNote;
	remove();
}

//
// return the day of the month
//
int
Day::getDay()
{
	return day.day();
}

//
// return the month of the year
//
int
Day::getMonth()
{
	return day.month();
}

//
// return the year
//
int
Day::getYear()
{
	return day.year();
}

//
// return a reference to the date associated with this day
//
QDate&
Day::getDate()
{
	return day;
}

//
// save the date associated with this day and all appointments on this day
//
void
Day::save(QDataStream& fil, bool hasChanged)
{
int recs,len;
char *add;

	if(hasChanged)
	{
		fil << day;
		len = dayNote->length();
		add = dayNote->data();
		fil << len;
		if(len)
		{
			fil.writeRawBytes(add,len);
		}
	}

	Appointment *acur;
	QListIterator<Appointment> ita(list);
	if(hasChanged)
	{
		recs = ita.count();
		fil << recs;
	}
	for(; ita.current(); ++ita)
	{
		acur = ita.current();
		acur->save(fil,hasChanged);
	}

	DayEvent *ecur;
	QListIterator<DayEvent> ite(events);
	if(hasChanged)
	{
		fil << countNonPublic();
	}
	for(; ite.current(); ++ite)
	{
		ecur = ite.current();
		ecur->save(fil,hasChanged);
	}
}

//
// load this day and all of its appointments
//
void
Day::load(QDataStream& fil, int version, Klook* p, bool me)
{
int recs,len;

	fil >> day;

	if(version > 2)
	{
		fil >> len;
		dayNote->resize(len);
		if(len)
		{
			fil.readRawBytes(dayNote->data(),len);
		}
		dayNote->truncate(len);
	}

	Appointment *aread;
	fil >> recs;
	for( int i = 0; i < recs; i++)
	{
		aread = new Appointment(p,this);
		aread->load(fil,version,p,me);
		list.inSort(aread);
	}

	if(version > 5)
	{
		DayEvent *eread;
		fil >> recs;
		for( int i = 0; i < recs; i++)
		{
			eread = new DayEvent(this);
			eread->load(fil,version);
			events.append(eread);
		}
	}
}

//
// remove all appointments from this day
//
bool
Day::remove(bool tidy)
{
bool cleared;
bool purge = true;

	if(!tidy)
	{
		list.clear();
		events.clear();
	}
	else
	{
		DayEvent *ecur;
		QListIterator<DayEvent> ite(events);
		for(; ite.current();)
		{
			cleared = false;
			ecur = ite.current();
			if(ecur->isRepeating())
			{
				if(!ecur->getRepeat()->isForever() && (ecur->getRepeat()->getEndDate() < day))
				{
					events.remove(ecur);
					cleared = true;
				}
				else
				{
					purge = false;
				}
			}
			else
			{
				events.remove(ecur);
				cleared = true;
			}
			if(!cleared)
			{
				++ite;
			}
		}

		Appointment *acur;
		QListIterator<Appointment> ita(list);
		for(; ita.current(); )
		{
			cleared = false;
			acur = ita.current();
			if(acur->isRepeating())
			{
				if(!acur->getRepeat()->isForever() && (acur->getRepeat()->getEndDate() < day))
				{
					list.remove(acur);
					cleared = true;
				}
				else
				{
					purge = false;
				}
			}
			else
			{
				list.remove(acur);
				cleared = true;
			}
			if(!cleared)
			{
				++ita;
			}
		}
	}
	return purge;
}

//
// display all appointments associated with this day into the day view
//
void
Day::listDay(Kdayview* v, UserAccess* u)
{
	v->clearEvents();
	DayEvent *ecur;
	QListIterator<DayEvent> ite(events);
	for(; ite.current(); ++ite)
	{
		ecur = ite.current();
		v->displayItem(ecur);
	}

	Appointment *acur;
	QListIterator<Appointment> ita(list);
	for(; ita.current(); ++ita)
	{
		acur = ita.current();
		int sense = acur->getSensitivity();
		if((sense == Appointment::public_access) ||
		  ((sense == Appointment::business_access) && (u->canViewBusiness())) ||
		  ((sense == Appointment::personnal_access) && (u->canViewPersonnal())))
		{
			v->displayItem(acur);
		}
	}
}

//
// display all appointments associated with this day into the week view
//
void
Day::listDay(Kweekview* v, UserAccess* u, int col)
{
	Appointment *acur;
	QListIterator<Appointment> ita(list);
	for(; ita.current(); ++ita)
	{
		acur = ita.current();
		int sense = acur->getSensitivity();
		if((sense == Appointment::public_access) ||
		  ((sense == Appointment::business_access) && (u->canViewBusiness())) ||
		  ((sense == Appointment::personnal_access) && (u->canViewPersonnal())))
		{
			v->displayItem(col,acur);
		}
	}
}

//
// display all appointments associated with this day into the month view
//
void
Day::listDay(Kmonthview* v, UserAccess* u, int col)
{
	DayEvent *ecur;
	QListIterator<DayEvent> ite(events);
	for(; ite.current(); ++ite)
	{
		ecur = ite.current();
		v->displayItem(col - 1,ecur);
	}

	Appointment *acur;
	QListIterator<Appointment> ita(list);
	for(; ita.current(); ++ita)
	{
		acur = ita.current();
		int sense = acur->getSensitivity();
		if((sense == Appointment::public_access) ||
		  ((sense == Appointment::business_access) && (u->canViewBusiness())) ||
		  ((sense == Appointment::personnal_access) && (u->canViewPersonnal())))
		{
			v->displayItem(col - 1,acur);
		}
	}
}


//
// display all appointments associated with this day into year plan
//
void
Day::listDay(Kyearplan* v, UserAccess* , int month, int day)
{
	int a = countAppointments();
	bool n = (getDayNote() == "") ? false : true;
	int e = countEvents();
	bool h = findHoliday();
	v->displayDay(month,day,a,n,e,h);
}


//
// find an appointment on this day based upon the the time specified.
// the value passed is in the range 0-TimeSlots::totalSlots (00:00 - 23:30 at half hour intervals)
// the method returns the appointment that is active during the specified
// time (between the start and finish time)
//
Appointment *
Day::findAppointment(int ind, Klook* w)
{
int s,f;

	Appointment *curr;
	QListIterator<Appointment> it(list);
	for(; it.current(); ++it)
	{
		curr = it.current();
		s = curr->getStartTime();
		f = curr->getFinishTime();
		for(int i = s; i <= f; i++)
		{
			if(i == ind)
			{
				return curr;
			}
		}
	}

	return createAppointment(w,ind);
}

//
// create a new appointment, and associate it with this day
//
Appointment*
Day::createAppointment(Klook* w, int ind)
{
	Appointment* a = new Appointment(w,this,ind);
	addAppointment(a);
	return a;
}

//
// remove an appointment from this day
//
void
Day::removeAppointment(Appointment* a)
{
	list.remove(a);
}

//
// associate an appointment with this day
//
void
Day::addAppointment(Appointment* a)
{
	list.inSort(a);
}

//
// check all other appointments on this day for a time range conflict
//
bool
Day::checkForConflict(int ast, int aft, Appointment* a)
{
int tst,tft;

	Appointment *curr;
	QListIterator<Appointment> it(list);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(curr == a)
		{
			continue;
		}
		tst = curr->getStartTime();
		tft = curr->getFinishTime();
		for(int i = ast; i <= aft; i++)
		{
			if((i == tst) || (i == tft))
			{
				return true;
			}
		}
	}
	return cal->checkForConflict(ast,aft,day);
}

//
// set the day notes
//
void
Day::setDayNote(QString& txt)
{
	dayNote->resize(0);
	dayNote->append(txt);
}

//
// return the day note
//
QString&
Day::getDayNote()
{
	return *dayNote;
}

int
Day::countAppointments()
{
	QListIterator<Appointment> it(list);
	int recs = it.count();
	return recs;
}


//
// find a day event
//
DayEvent *
Day::findEvent(int ind)
{
	DayEvent *curr;
	QListIterator<DayEvent> it(events);
	for(int i = 0; it.current(); ++it, i++)
	{
		curr = it.current();
		if(i == ind)
		{
			return curr;
		}
	}

	return createEvent();
}

//
// create a new event, and associate it with this day
//
DayEvent*
Day::createEvent()
{
	DayEvent* e = new DayEvent(this);
	addEvent(e);
	return e;
}

//
// remove an event from this day
//
void
Day::removeEvent(DayEvent* e)
{
	events.remove(e);
}

//
// associate an event with this day
//
void
Day::addEvent(DayEvent* e)
{
	events.append(e);
}

int
Day::countEvents()
{
	QListIterator<DayEvent> it(events);
	int recs = it.count();
	return recs;
}

int
Day::countNonPublic()
{
DayEvent *curr;
int recs = 0;

	QListIterator<DayEvent> it(events);
	for(; it.current(); ++it)
	{
		curr = it.current();
		if(!curr->isPublic())
		{
			recs++;
		}
	}
	return recs;
}

bool
Day::findHoliday()
{
	DayEvent *curr;
	QListIterator<DayEvent> it(events);
	for(int i = 0; it.current(); ++it, i++)
	{
		curr = it.current();
		if(curr->isHoliday())
		{
			return true;
		}
	}

	return false;
}


Calendar*
Day::getCalendar()
{
	return cal;
}


//
// search the daynote for a string
//
bool
Day::findString(QString& str)
{
	return (dayNote->find(str.data(),0,false) != -1) ? true : false;
}

//
// cycle through the items on this day for a string
//
bool
Day::searchForString(Search* search)
{
	if(search->doAppointments())
	{
		Appointment *acur;
		QListIterator<Appointment> ita(list);
		for(; ita.current(); ++ita)
		{
			acur = ita.current();
			if(acur->findString(search->getString()))
			{
				return true;
			}
		}
	}

	if(search->doDayEvents())
	{
		DayEvent *ecur;
		QListIterator<DayEvent> ite(events);
		for(; ite.current(); ++ite)
		{
			ecur = ite.current();
			if(ecur->findString(search->getString()))
			{
				return true;
			}
		}
	}

	if(search->doDayNotes())
	{
		if(findString(search->getString()))
		{
			return true;
		}
	}

	return false;
}

//
// export the calendar to a flat file
//
void
Day::export(QTextStream& fil, Export& expdata)
{
QString d;
int sense;

	if(expdata.dateFormat() == 0)
	{
		d.sprintf("%d%c%d%c%d",
			day.day(),
			expdata.delimiter(),
			day.month(),
			expdata.delimiter(),
			day.year());
	}
	else
	{
		d.sprintf("%d%c%d%c%d",
			day.month(),
			expdata.delimiter(),
			day.day(),
			expdata.delimiter(),
			day.year());
	}

	Appointment *acur;
	QListIterator<Appointment> ita(list);
	for(; ita.current(); ++ita)
	{
		acur = ita.current();
		sense = acur->getSensitivity();
		if((sense == Appointment::public_access && expdata.exportPublic()) ||
		   (sense == Appointment::business_access && expdata.exportBusiness()) ||
		   (sense == Appointment::personnal_access && expdata.exportPersonnal()))
		{
			fil << d;
			acur->export(fil,expdata);
			fil << '\n';
		}
	}

	if(expdata.exportDayEvent())
	{
		DayEvent *ecur;
		QListIterator<DayEvent> ite(events);
		for(; ite.current(); ++ite)
		{
			fil << d;
			ecur = ite.current();
			ecur->export(fil,expdata);
			fil << '\n';
		}
	}

}

