//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <kapp.h>
#include <klocale.h>

#include "DayEvent.h"

#include "Repeat.h"
#include "Day.h"
#include "Calendar.h"
#include "Export.h"


//
// create a blank Event item
//
DayEvent::DayEvent(Day* d)
{
	subject = new QString("");
	type = normal_event;
	valid = false;
	repeat = (Repeat*)0;
	day = d;
	publicEvent = false;
}

//
// create a Event item
//
DayEvent::DayEvent(Day* d, QString& sub, int t)
{
	subject = new QString(sub);
	type = t;
	valid = false;
	day = d;
	repeat = (Repeat*)0;
}

//
// tidy up
//
DayEvent::~DayEvent()
{
	delete subject;
	if(repeat != (Repeat*)0)
	{
		delete repeat;
	}
}

//
// set the subject text of this Event item
//
void
DayEvent::setSubject( QString& sub )
{
	subject->resize(0);
	subject->append(sub);
}

//
// return a reference to the subject text
//
QString&
DayEvent::getSubject()
{
	return *subject;
}

void
DayEvent::setType(int t)
{
	type = t;
}

int 
DayEvent::getType()
{
	return type;
}

bool
DayEvent::isHoliday()
{
	return (type == holiday_event) ? true : false;
}

bool
DayEvent::isStudy()
{
	return (type == study_event) ? true : false;
}

//
// save this item to a file
//
void
DayEvent::save(QDataStream& fil, bool hasChanged)
{
int len;
char *add;

	if(hasChanged && !publicEvent)
	{
		len = subject->length();
		add = subject->data();
		fil << len;
		if(len)
		{
			fil.writeRawBytes(add,len);
		}
		fil << type;

		if(repeat != (Repeat*)0)
		{
			fil << 1;
			repeat->save(fil,hasChanged);
		}
		else
		{
			fil << 0;
		}
	}
}

//
// load this item from a file
//
void
DayEvent::load(QDataStream& fil, int version)
{
int len;

	fil >> len;
	subject->resize(len);
	if(len)
	{
		fil.readRawBytes(subject->data(),len);
	}
	subject->truncate(len);
	if(version > 6)
	{
		fil >> type;
	}
	else
	{
		type = normal_event;
	}

	if(version > 7)
	{
	int rep;

		fil >> rep;
		if(rep == 1)
		{
			repeat = new Repeat(day,this);
			repeat->load(fil,version);
			day->getCalendar()->addRepeat(repeat);
		}
	}
	else
	{
		repeat = (Repeat*)0;
	}
	makeValid();
}

//
// return the status of this item.
// the EditTask class requires an existing instance of a Event class.
// if EditTask was creating a new Event item and quits, then this instance
// of the Event item is not required and should be deleted
//
bool
DayEvent::isNew()
{
	return !valid;
}

//
// mark this Event item as valid (ie. EditTask exited rather than quit)
//
void
DayEvent::makeValid()
{
	valid = true;
}

void
DayEvent::setRepeat(Repeat* r)
{
	repeat = r;
}

Repeat*
DayEvent::getRepeat()
{
	return repeat;
}

bool
DayEvent::isRepeating()
{
	return (repeat != (Repeat*)0) ? true : false;
}

void
DayEvent::clearRepeat()
{
	if(repeat != (Repeat*)0)
	{
		day->getCalendar()->removeRepeat(repeat);
		delete repeat;
	}
	repeat = (Repeat*)0;
}

Day*
DayEvent::getDay()
{
	return day;
}

//
// search the subject for a string
//
bool
DayEvent::findString(QString& str)
{
	return (subject->find(str.data(),0,false) != -1) ? true : false;
}


//
// export the day event
//
void
DayEvent::export(QTextStream& fil, Export& expdata)
{
int len;
char *add;
QString d;

	fil << expdata.delimiter();
	fil << "D";	// mark it as a day event
	fil << expdata.delimiter();

	len = subject->length();
	add = subject->data();
	if(len)
	{
		fil.writeRawBytes(add,len);
	}
	else
	{
		fil << " ";
	}
	fil << expdata.delimiter();

	d.sprintf("%d",type);
	fil << d;
}


void
DayEvent::makePublic()
{
	makeValid();
	publicEvent = true;
	type = holiday_event;
}

bool
DayEvent::isPublic()
{
	return publicEvent;
}
