//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <kapp.h>
#include <klocale.h>

#include "EditAppointment.h"

#include "Appointment.h"
#include "UserAccess.h"
#include "Alarm.h"
#include "Calendar.h"
#include "EditRepeat.h"
#include "Repeat.h"
#include "TimeSlots.h"

//
// create a new widget allowing editing of an appointment
//
EditAppointment::EditAppointment(Appointment* a, Calendar* c, UserAccess* u, QWidget* parent, const char* name)
	: ViewAppointment(a,c,parent,name)
{
	access = u;

	AllDayEvent = new QCheckBox(this);
	AllDayEvent->setGeometry( 310, 30, 70, 20 );
	AllDayEvent->setText(klocale->translate("All day"));
	AllDayEvent->setAutoRepeat(false);
	AllDayEvent->setAutoResize(false);
	setAllDay(false);

	Alarm = new QCheckBox(this);
	Alarm->setGeometry(0,205,50,20);
	Alarm->setText(klocale->translate("Alarm"));
	Alarm->setAutoRepeat(false);
	Alarm->setAutoResize(false);
	Alarm->setChecked(false);

	alarmDue = new QComboBox(false,this);
	alarmDue->setGeometry(60,200,70,30);
	setAlarm(false);
	populateAlarm();

	if(a->isAlarmed())
	{
		Alarm->setChecked(true);
		setAlarm(true);
		alarmDue->setCurrentItem(a->getAlarm().getDue());
	}

	cancel = new QPushButton(klocale->translate("Quit"),this);
	cancel->resize(40,30);
	cancel->setGeometry(100,240,60,30);

	connect(okay,SIGNAL(clicked()),SLOT(save()));
	connect(cancel,SIGNAL(clicked()),SLOT(quit()));
	connect(repeat,SIGNAL(clicked()),SLOT(makeRepeat()));

	connect(AllDayEvent,SIGNAL(toggled(bool)),SLOT(setAllDay(bool)));
	connect(Alarm,SIGNAL(toggled(bool)),SLOT(setAlarm(bool)));

	connect(StartTime,SIGNAL(activated(int)),SLOT(changeFinish(int)));

	editRep = (EditRepeat*)0;

	enable();

	Subject->setFocus();
}

//
// tidy up
//
EditAppointment::~EditAppointment()
{
	delete AllDayEvent;
	delete Alarm;
	delete alarmDue;
	delete cancel;
	if(editRep != (EditRepeat*)0)
	{
		delete editRep;
	}
}

//
// process the okay button being pressed
// emit a signal to say we wish to save this appointment
//
void
EditAppointment::save()
{
Day *curr;

	QString txt = AppointmentText->text();
	QString sub = Subject->text();
	int start = StartTime->currentItem();
	int finish = FinishTime->currentItem();
	int sens = Sensitivity->currentItem();

	if(start > finish)
	{
		return;
	}

	if(finish != start)
	{
		finish--;
	}

	curr = cal->findDay(baseDate);
	if(curr == (Day*)0)
	{
		return;	// but this is pretty hairy as a day object should exist
	}
	if(curr->checkForConflict(start,finish,event))
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("Appointment conflicts with another"));
		return;
	}

	QString s = Subject->text();
	if(s.isEmpty())
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("No subject entered"));
		return;
	}

	bool sensok = false;
	if(((sens == Appointment::public_access) && (access->canAddPublic())) ||
	   ((sens == Appointment::business_access) && (access->canAddBusiness())) ||
	   ((sens == Appointment::personnal_access) && (access->canAddPersonnal())))
	{
		sensok = true;
	}
	if(!sensok)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("Cannot add an appointment with this type of sensitivity"));
		return;
	}

	event->setSubject(sub);
	event->setText(txt);
	event->setStartTime(start);
	event->setFinishTime(finish);
	event->setSensitivity(sens);
	if(alarmed)
	{
		event->setAlarm(alarmDue->currentItem(),baseDate);
	}
	emit saveAppointment(event);
}

//
// signal emitted when the user presses the cancel button
//
void
EditAppointment::quit()
{
	emit abortAppointment(event);
}

//
// set the finish time (usually called when the start time is set to
// a value greater than the present finish time)
//
void
EditAppointment::changeFinish(int ind)
{
int cur;

	if(ind > TimeSlots::totalSlots)
	{
		ind = TimeSlots::totalSlots;
	}
	cur = FinishTime->currentItem();
	if(cur < ind)
	{
		FinishTime->setCurrentItem(ind);
	}
}

//
// process the 'all day' checkbox
//
void
EditAppointment::setAllDay(bool sts)
{
	if(sts == true)
	{
		StartTime->setCurrentItem(0);
		FinishTime->setCurrentItem(TimeSlots::totalSlots - 1);
	}
	StartTime->setEnabled(!sts);
	FinishTime->setEnabled(!sts);
}

//
// process the alarm checkbox
//
void
EditAppointment::setAlarm(bool sts)
{
	alarmDue->setEnabled(sts);
	alarmed = sts;
}

//
// enable widgets (these were disabled by our parent)
//
void
EditAppointment::enable()
{
	Subject->setReadOnly(false);
	AppointmentText->setReadOnly(false);
	StartTime->setEnabled(true);
	FinishTime->setEnabled(true);
	Sensitivity->setEnabled(true);
//	StartDate->setEnabled(true);
//	FinishDate->setEnabled(true);
}

//
// Populate the alarm due combo box with values
//
void
EditAppointment::populateAlarm()
{
	alarmDue->setSizeLimit(7);
	alarmDue->insertItem(klocale->translate("15 mins"));
	alarmDue->insertItem(klocale->translate("30 mins"));
	alarmDue->insertItem(klocale->translate("1 hour"));
	alarmDue->insertItem(klocale->translate("2 hours"));
	alarmDue->insertItem(klocale->translate("3 hours"));
	alarmDue->insertItem(klocale->translate("1 day"));
	alarmDue->insertItem(klocale->translate("2 days"));
	alarmDue->insertItem(klocale->translate("3 days"));
	alarmDue->insertItem(klocale->translate("1 week"));
}

//
// Edit a repeating event
//
void
EditAppointment::makeRepeat()
{
Repeat* r;
bool newr;

	if(event->isRepeating())
	{
		r = event->getRepeat();
		newr = false;
	}
	else
	{
		r = new Repeat(event->getDay(),event);
		newr = true;
	}
	editRep = new EditRepeat(r,newr,this);
	editRep->setGeometry(0,0,width(),height());
	editRep->resize(width(),height());
	editRep->show();
	connect(editRep,SIGNAL(saveRepeat(Repeat*)),SLOT(saveRep(Repeat*)));
	connect(editRep,SIGNAL(abortRepeat(Repeat*)),SLOT(abortRep(Repeat*)));
	connect(editRep,SIGNAL(killRepeat(Repeat*)),SLOT(killRep(Repeat*)));
}

void
EditAppointment::saveRep(Repeat* r)
{
	if(!event->isRepeating())
	{
		cal->addRepeat(r);
	}
	event->setRepeat(r);
	delete editRep;
	editRep = (EditRepeat*)0;
}

void
EditAppointment::abortRep(Repeat* r)
{
	if(!event->isRepeating())
	{
		delete r;
	}
	delete editRep;
	editRep = (EditRepeat*)0;
}

void
EditAppointment::killRep(Repeat*)
{
	if(event->isRepeating())
	{
		event->clearRepeat();
	}
	delete editRep;
	editRep = (EditRepeat*)0;
}

