//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <kapp.h>
#include <klocale.h>

#include "Kdayview.h"

#include "TabularList.h"
#include "Appointment.h"
#include "DayEvent.h"
#include "TimeSlots.h"

//
// create a day viewer
//
Kdayview::Kdayview(QWidget *parent, const char *name) 
	: QWidget(parent,name)
{
	EventList = new TabularList(this, "EventList",1);
	EventList->setGeometry(0,0,240,60);

	EventList->setNumCols(3);
	EventList->setColumn(0,klocale->translate("Type"),60);
	EventList->setColumn(1,klocale->translate("Rep"),30,KTabListBox::PixmapColumn);
	EventList->setColumn(2,klocale->translate("Event"),220);
	EventList->setSeparator('\t');
	EventList->setAutoUpdate(true);

	DayList = new TabularList(this, "DayList", 4);
	DayList->setGeometry(0,55,240,160);

	DayList->setNumCols(5);
	DayList->setColumn(0,klocale->translate("Time"),40);
	DayList->setColumn(1,"Alm",30,KTabListBox::PixmapColumn);
	DayList->setColumn(2,"Sen",30,KTabListBox::PixmapColumn);
	DayList->setColumn(3,"Rep",30,KTabListBox::PixmapColumn);
	DayList->setColumn(4,klocale->translate("Subject"),100);
	DayList->setSeparator('\t');
	DayList->setAutoUpdate(true);

TimeSlots ts;

	for(int i = 0; i < TimeSlots::totalSlots; i++)
	{
		DayList->insertItem(ts.whichTime(i));
	}

	connect(DayList,SIGNAL(selected(int,int)),SLOT(pickTimeItem(int,int)));
	connect(EventList,SIGNAL(selected(int,int)),SLOT(pickEvent(int,int)));

	DayList->setFocus();
}

//
// ensure the the view fills the available screen on a resize
//
void
Kdayview::resize(int w , int h)
{
	EventList->setAutoUpdate(false);
	EventList->setGeometry(0,0,w,60);
	EventList->setColumnWidth(2,w - 60 - 20);
	EventList->setAutoUpdate(true);

	DayList->setAutoUpdate(false);
	DayList->setGeometry(0,65,w,h - 65);
	DayList->setColumnWidth(4,w - 40 - 20  - 20 - 20);
	DayList->setAutoUpdate(true);

	EventList->update();
	DayList->update();
}

//
// signal emitted when a subject is clicked
// no signal is emitted if the user clicks on the time or the alarm
// (alarms are not implemented yet, but will emit a difference signal)
//
void
Kdayview::pickTimeItem(int ind , int)
{
	emit timePicked(ind);
}

//
// display a string of text over the date range requested
//
void
Kdayview::displayItem(Appointment* a)
{
	DayList->setAutoUpdate(false);
	DayList->changeItemPart(a->getSubject(),a->getStartTime(),4);
	if(a->getAlarmStatus() == Appointment::armed)
	{
		DayList->changeItemPart("A",a->getStartTime(),1);
	}
	if(a->getAlarmStatus() == Appointment::fired)
	{
		DayList->changeItemPart("a",a->getStartTime(),1);
	}
	if(a->getAlarmStatus() == Appointment::repeating)
	{
		DayList->changeItemPart("r",a->getStartTime(),1);
	}
	if(a->getSensitivity() == Appointment::public_access)
	{
		DayList->changeItemPart(" ",a->getStartTime(),2);
	}
	if(a->getSensitivity() == Appointment::business_access)
	{
		DayList->changeItemPart("B",a->getStartTime(),2);
	}
	if(a->getSensitivity() == Appointment::personnal_access)
	{
		DayList->changeItemPart("P",a->getStartTime(),2);
	}
	if(a->isRepeating())
	{
		DayList->changeItemPart("R",a->getStartTime(),3);
	}
	for(int i = a->getStartTime(); i <= a->getFinishTime(); i++)
	{
		DayList->highlightItem(i);
	}
	for(int i = a->getStartTime() + 1; i <= a->getFinishTime(); i++)
	{
		DayList->changeItemPart(" ...",i,4);
	}

	DayList->setAutoUpdate(true);
	DayList->repaint();
}

//
// clear all appointments from the list
//
void
Kdayview::clearItems()
{
	DayList->setAutoUpdate(false);
	for(int i = 0; i < TimeSlots::totalSlots; i++)
	{
		DayList->changeItemPart(" ",i,1);
		DayList->changeItemPart(" ",i,2);
		DayList->changeItemPart(" ",i,3);
		DayList->changeItemPart(" ",i,4);
	}
	DayList->setAutoUpdate(true);
	DayList->repaint();
}

//
// return the index of the selected item
//
int
Kdayview::getSelectedTime()
{
	return DayList->currentItem();
}

//
// set the top item visible
//
void
Kdayview::setTop()
{
QTime now = QTime::currentTime();

	int idx = (now.hour() * TimeSlots::hourUnits) + (now.minute() / TimeSlots::minuteUnits);
	DayList->setTopItem(idx);
}

//
// return the index of the selected day event
//
int
Kdayview::getSelectedEvent()
{
	return EventList->currentItem();
}


//
// clear all appointments from the list
//
void
Kdayview::clearEvents()
{
	for(; EventList->count();)
	{
		EventList->removeItem(0);
	}
	EventList->repaint();
}


void
Kdayview::pickEvent(int ind , int)
{
	emit eventPicked(ind);
}


//
// display a string of text over the date range requested
//
void
Kdayview::displayItem(DayEvent* de)
{
	QString t;
	switch(de->getType())
	{
	case DayEvent::normal_event  : t = klocale->translate("General");	break;
	case DayEvent::holiday_event : t = klocale->translate("Holiday");	break;
	case DayEvent::study_event   : t = klocale->translate("Study");	break;
	}

	QString r = " ";
	if(de->isRepeating())
	{
		r = "R";
	}

	QString s = t + "\t" + r + "\t" + de->getSubject();
	EventList->insertItem(s);
}


