//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <unistd.h>
#include <pwd.h>

#include <qfiledlg.h>

#include "Klook.h"

#include <klocale.h>

#include "ViewTask.h"
#include "EditTask.h"
#include "KalDatePicker.h"
#include "Calendar.h"
#include "TaskList.h"
#include "Kdayview.h"
#include "Kweekview.h"
#include "Kmonthview.h"
#include "Kyearplan.h"
#include "Ktodo.h"
#include "Kconfigure.h"
#include "ViewAppointment.h"
#include "EditAppointment.h"
#include "docking.h"
#include "UserList.h"
#include "UserAccess.h"
#include "EditUsers.h"
#include "EditEvent.h"
#include "FindText.h"
#include "Search.h"
#include "Export.h"
#include "ExportSelect.h"

Klook::Klook(QWidget *, const char *name) 
	: KTopLevelWidget(name)
{
	cal = new Calendar();
	tasks = new TaskList();
	users = new UserList();

	fullName = new QString("");
	eMail = new QString("");
	currFile = new QString("");

	allDone = false;
	isOpen = false;
	hasChanged = false;
	readOnly = false;

	setMinimumSize(600,360);
	resize(600,360);

	setCaption(PACKAGE);

	QFrame *vmain = new QFrame(this,"main");
	vmain->setFrameStyle(0);

	ThisMonth = new KalDatePicker(vmain,cal->getDate());
	ThisMonth->setGeometry(5,5,200,200);

	dayNote = new QMultiLineEdit(vmain,"todo");
	dayNote->setGeometry(5,210,200,110);
	dayNote->show();

	selDate = new QLabel(vmain,"selected");
	selDate->setGeometry(220,5,150,20);
	selDate->setFont(QFont("Arial",14,QFont::Bold));

	selName = new QLabel(vmain,"selected");
	selName->setGeometry(370,5,150,20);
	selName->setFont(QFont("Arial",14,QFont::Bold));
	selName->setAlignment(AlignRight);

	dayView = new Kdayview(vmain);
	dayView->setGeometry(220,30,280,220);
	dayView->show();

	weekView = new Kweekview(vmain);
	weekView->setGeometry(220,30,280,220);
	weekView->show();

	monthView = new Kmonthview(false,vmain);
	monthView->setGeometry(220,30,280,220);
	monthView->show();

	yearPlan = new Kyearplan(false,vmain);
	yearPlan->setGeometry(220,30,280,220);
	yearPlan->show();

	todoView = new Ktodo(vmain);
	todoView->setGeometry(220,30,280,220);
	todoView->hide();

	dock_widget = new DockWidget(this);
	dock_widget->dock();

connect(dayView,SIGNAL(timePicked(int)),SLOT(editDayItem(int)));
	connect(weekView,SIGNAL(timePicked(int,int)),SLOT(editWeekItem(int,int)));
	connect(monthView,SIGNAL(viewDay(int)),SLOT(selectDayView(int)));
	connect(yearPlan,SIGNAL(viewDay(int,int)),SLOT(selectDayView(int,int)));

	connect(todoView,SIGNAL(taskPicked(int)),SLOT(editTaskItem(int)));
	connect(todoView,SIGNAL(taskDone(int)),SLOT(finishTaskItem(int)));

	connect(ThisMonth,SIGNAL(dateSelected(QDate)),SLOT(selectDay(QDate)));
	connect(ThisMonth,SIGNAL(monthChanged(int,int)),SLOT(refreshMonth(int,int)));

	connect(dayNote,SIGNAL(textChanged()),SLOT(markAsChanged()));

	connect(KApplication::getKApplication(),SIGNAL(saveYourself()),SLOT(closeKalendar()));

	LoadRC();

	setupMenu();
	setupToolBar();
	setupStatusBar();

	setView(vmain);
	setMenu(Menu);

	viewApp = (ViewAppointment*)0;
	editApp = (EditAppointment*)0;
	editEvent = (EditEvent*)0;
	viewTask = (ViewTask*)0;
	editTask = (EditTask*)0;
	cfg = (Kconfigure*)0;
	usr = (EditUsers*)0;
	finder = (FindText*)0;
	exp = (ExportSelect*)0;

	search = (Search*)0;

	struct passwd *pwd = ::getpwuid(::geteuid());
	QString me(pwd->pw_name);
// my calendar security access (everything)
	baseUser = new UserAccess(me,true,true,true,true,true,true,true,true);

// a default for when I have not been given access by another user
// this one only allows me access to view public appointments
	nullUser = new UserAccess(me);

	ThisMonth->selectDate(cal->getDate().day());
	visibleView = v_day;

	openMyKalendar();
	ThisMonth->setDateFormat(dateFormat);
	ThisMonth->setStartWeek(startWeek);
	ThisMonth->selectDate(cal->getDate().day());

	monthView->setFit(sizeMonthToFit);
	yearPlan->setFit(sizeYearToFit);
}

Klook::~Klook()
{
	if(viewApp != (ViewAppointment*)0)
	{
		delete viewApp;
	}
	if(editApp != (EditAppointment*)0)
	{
		delete editApp;
	}
	if(editEvent != (EditEvent*)0)
	{
		delete editEvent;
	}
	if(viewTask != (ViewTask*)0)
	{
		delete viewTask;
	}
	if(editTask != (EditTask*)0)
	{
		delete editTask;
	}
	if(cfg != (Kconfigure*)0)
	{
		delete cfg;
	}
	if(usr != (EditUsers*)0)
	{
		delete usr;
	}
	if(finder != (FindText*)0)
	{
		delete finder;
	}
	if(search != (Search*)0)
	{
		delete search;
	}
	if(exp != (ExportSelect*)0)
	{
		delete exp;
	}

	closeKalendar();
}

//
// Display some brief help about the application
//
void
Klook::helpAbout()
{
	QString *txt = new QString();

	txt->append(PACKAGE);
	txt->append(klocale->translate(" Version "));
	txt->append(VERSION);
	txt->append(klocale->translate(", Copyright (C) 1999 Mark W J Redding"));
	txt->append("\n");
	txt->append(PACKAGE);
	txt->append(klocale->translate(" comes with ABSOLUTELY NO WARRANTY"));
	txt->append("\n");
	txt->append(klocale->translate("This is free software, and you are welcome to redistribute it."));
	txt->append("\n\n");
	txt->append(klocale->translate("Contact the author at <mark@grawlfang.demon.co.uk>"));

	QMessageBox::about(this,PACKAGE,txt->data());

	delete txt;
}

//
// invoke the standard KDE help utility
//
void
Klook::helpHelp()
{
	KApplication::getKApplication()->invokeHTMLHelp("","");
}

//
// resize the views (excluding appointment,task edit/view boxes)
//
void
Klook::resizeEvent(QResizeEvent *e)
{
// the - 1000 is to take account of the menu, toolbar & status bar
// What I need to find is a method somewhere that will tell where they are.


	int aw = width() - 220 - 10;
	int ah = height() - 100 - 10;
	int dh = height() - 100 - 10 + 30;

	selName->setGeometry(width() - 160 ,5,150,20);

	dayView->setGeometry(220,30,aw,ah);
	dayView->resize(aw,ah);
	weekView->setGeometry(220,30,aw,ah);
	weekView->resize(aw,ah);
	monthView->setGeometry(220,30,aw,ah);
	monthView->resize(aw,ah);
	yearPlan->setGeometry(220,30,aw,ah);
	yearPlan->resize(aw,ah);
	todoView->setGeometry(220,30,aw,ah);
	todoView->resize(aw,ah);

	dayNote->setGeometry(5,210,200,height() - 100 - 200);
	dayNote->resize(200,height() - 100 - 200);

	if(viewApp != (ViewAppointment*)0)
	{
		viewApp->setGeometry(220,30,aw,dh);
		viewApp->resize(aw,dh);
	}
	if(editApp != (EditAppointment*)0)
	{
		editApp->setGeometry(220,30,aw,dh);
		editApp->resize(aw,dh);
	}
	if(editEvent != (EditEvent*)0)
	{
		editEvent->setGeometry(220,30,aw,dh);
		editEvent->resize(aw,dh);
	}
	if(viewTask != (ViewTask*)0)
	{
		viewTask->setGeometry(220,30,aw,dh);
		viewTask->resize(aw,dh);
	}
	if(editTask != (EditTask*)0)
	{
		editTask->setGeometry(220,30,aw,dh);
		editTask->resize(aw,dh);
	}
	if(cfg != (Kconfigure*)0)
	{
		cfg->setGeometry(220,30,aw,dh);
		cfg->resize(aw,dh);
	}
	if(usr != (EditUsers*)0)
	{
		usr->setGeometry(220,30,aw,dh);
		usr->resize(aw,dh);
	}
	if(finder != (FindText*)0)
	{
		finder->setGeometry(220,30,aw,dh);
		finder->resize(aw,dh);
	}
	if(exp != (ExportSelect*)0)
	{
		exp->setGeometry(220,30,aw,dh);
		exp->resize(aw,dh);
	}

	KTopLevelWidget::resizeEvent(e);
}

void
Klook::closeEvent(QCloseEvent*)
{
	this->hide();
}

//
// exit from the application gracefully
//
void
Klook::closeKalendar()
{
	if(!allDone)
	{
		saveDayNote();
		allDone = true;
		save();
		SaveRC();
		emit quit();
	}
}

//
// Start up the configure dialog
//
void
Klook::configKalendar()
{
	disable();

	cfg = new Kconfigure(this,"cfg");
	cfg->setGeometry(220,60,width() - 220 , height() - 100 - 10);
	cfg->resize(width() - 220 - 10, height() - 100 - 10 + 30);
	cfg->setFullName(*fullName);
	cfg->setEMail(*eMail);
	cfg->setDateFormat(dateFormat);
	cfg->setStartWeek(startWeek);
	cfg->setMonthSize(sizeMonthToFit);
	cfg->setYearSize(sizeYearToFit);
	cfg->show();

	connect(cfg,SIGNAL(quitConfig()),SLOT(abortConfig()));
	connect(cfg,SIGNAL(exitConfig()),SLOT(storeConfig()));
}

//
// Start up the security dialog
//
void
Klook::secureKalendar()
{
	disable();

	usr = new EditUsers(baseUser->getUsername(),users,this,"usr");
	usr->setGeometry(220,60,width() - 220 , height() - 100 - 10);
	usr->resize(width() - 220 - 10, height() - 100 - 10 + 30);
	usr->show();

	connect(usr,SIGNAL(exitUsers()),SLOT(doneUsers()));
}

//
// Tidy old appointments from the calendar
//
void
Klook::tidyKalendar()
{
	int res = QMessageBox::warning(	this,
					PACKAGE,
					klocale->translate("Do you wish to delete items before ") + cal->getDate().toString(),
					QMessageBox::Yes,
					QMessageBox::No);
	if(res == 3)
	{
		cal->cleanCalendar();
		markAsChanged();
		selectView();
	}
}

void
Klook::exportKalendar()
{
	disable();

	exp = new ExportSelect(this);
	exp->setGeometry(220,60,width() - 220 , height() - 100 - 10);
	exp->resize(width() - 220 - 10, height() - 100 - 10 + 30);
	exp->show();

	connect(exp,SIGNAL(quitExport()),SLOT(abortExport()));
	connect(exp,SIGNAL(exitExport()),SLOT(doExport()));
}

//
// return a reference to the calendar object
//
Calendar&
Klook::getCalendar()
{
	return *cal;
}

//
// display the single day view
//
void
Klook::showDayView()
{
	enableAppointments();
	disableTasks();

	view->setItemChecked(vm_d,true);
	view->setItemChecked(vm_w,false);
	view->setItemChecked(vm_m,false);
	view->setItemChecked(vm_y,false);
	view->setItemChecked(vm_t,false);

	todoView->hide();
	weekView->hide();
	monthView->hide();
	yearPlan->hide();
	dayView->show();
	dayView->setTop();
	setCaption(klocale->translate("Kalendar - Day View"));
	visibleView = v_day;
}

//
// display the week view
//
void
Klook::showWeekView()
{
	enableAppointments();
	disableTasks();

	view->setItemChecked(vm_d,false);
	view->setItemChecked(vm_w,true);
	view->setItemChecked(vm_m,false);
	view->setItemChecked(vm_y,false);
	view->setItemChecked(vm_t,false);

	todoView->hide();
	dayView->hide();
	monthView->hide();
	yearPlan->hide();
	weekView->show();
	weekView->setTop();
	setCaption(klocale->translate("Kalendar - Week View"));
	visibleView = v_week;
}

//
// display the month view
//
void
Klook::showMonthView()
{
	enableAppointments();
	disableTasks();

	view->setItemChecked(vm_d,false);
	view->setItemChecked(vm_w,false);
	view->setItemChecked(vm_m,true);
	view->setItemChecked(vm_y,false);
	view->setItemChecked(vm_t,false);

	todoView->hide();
	dayView->hide();
	weekView->hide();
	yearPlan->hide();
	monthView->show();
	setCaption(klocale->translate("Kalendar - Month View"));
	visibleView = v_month;
}

//
// display the year planner
//
void
Klook::showYearPlan()
{
	enableAppointments();
	disableTasks();

	view->setItemChecked(vm_d,false);
	view->setItemChecked(vm_w,false);
	view->setItemChecked(vm_m,false);
	view->setItemChecked(vm_y,true);
	view->setItemChecked(vm_t,false);

	todoView->hide();
	dayView->hide();
	weekView->hide();
	monthView->hide();
	yearPlan->show();
	setCaption(klocale->translate("Kalendar - Year Plan"));
	visibleView = v_year;
}

//
// display the todo list
//
void
Klook::showToDoView()
{
	disableAppointments();
	enableTasks();

	view->setItemChecked(vm_d,false);
	view->setItemChecked(vm_w,false);
	view->setItemChecked(vm_m,false);
	view->setItemChecked(vm_y,false);
	view->setItemChecked(vm_t,true);

	dayView->hide();
	weekView->hide();
	monthView->hide();
	yearPlan->hide();
	todoView->show();
	setCaption(klocale->translate("Kalendar - ToDo List"));
	visibleView = v_todo;
}

//
// set the current date to a given day of the currently selected month
//
void
Klook::selectDay(int d)
{
	QDate now = cal->getDate();
	int m = now.month();
	int y = now.year();
	now.setYMD(y,m,d);
	selectDay(now);
	selectView(v_day);
}

//
// set the current date to a given date
//
void
Klook::selectDay(QDate d)
{
	saveDayNote();
	cal->setDate(d);
	cal->markCalendar(ThisMonth);
	selDate->setText(d.toString());
	selectDayNote();
	selectView();
}

//
// display the correct view
//
void
Klook::selectView(int i)
{
	visibleView = i;
	selectView();
}

void
Klook::selectView()
{
	if(visibleView == v_day)
	{
		showDayView();
		dayView->clearItems();
		dayView->clearEvents();
		cal->viewDate(currUser,dayView);
	}
	if(visibleView == v_week)
	{
		showWeekView();
		weekView->clearItems();
		cal->viewDate(currUser,weekView);
	}
	if(visibleView == v_month)
	{
		showMonthView();
		monthView->clearItems();
		cal->viewDate(currUser,monthView);
	}
	if(visibleView == v_year)
	{
		showYearPlan();
		yearPlan->clearItems();
		cal->viewDate(currUser,yearPlan);
	}
	if(visibleView == v_todo)
	{
		showToDoView();
		tasks->viewTasks(todoView);
	}
}

//
// load the users appointments,tasks etc
//
void
Klook::load(QString& user)
{
int len,ver;
bool isme = false;

	markAsUnchanged();

	currFile->resize(0);
	currFile->append(user);

	QString ufil = KApplication::kde_datadir() + "/kalendar/" + *currFile + ".kal";
	QFile fil(ufil);

	if(baseUser->getUsername() == *currFile)
	{
		isme = true;
	}

	lockKalendar(isme);

	if(fil.exists())
	{
		fil.open(IO_ReadOnly);
		QDataStream ds(&fil);

// see comments in save method below for reason for this

		ds >> ver;
		if(ver == -2)
		{
			ds >> ver;
			ds >> len;
		}
		else
		{
			len = ver;
			ver = 1;
		}

		fullName->resize(len);
		if(len)
		{
			ds.readRawBytes(fullName->data(),len);
		}
		fullName->truncate(len);

		ds >> len;
		eMail->resize(len);
		if(len)
		{
			ds.readRawBytes(eMail->data(),len);
		}
		eMail->truncate(len);

		if(version < 12)	// moved these to the config file
		{
			ds >> dateFormat;
			if(dateFormat > 2)
			{
				dateFormat = 0;
			}

			if(version > 8)
			{
				ds >> startWeek;
			}
			else
			{
				startWeek = 0;
			}

			if(version > 9)
			{
			int c;

				ds >> c; sizeMonthToFit = (bool)c;
				ds >> c; sizeYearToFit = (bool)c;
			}
			else
			{
				sizeMonthToFit = false;
				sizeYearToFit = false;
			}
		}

		cal->load(ds,ver,this,isme);
		tasks->load(ds,ver);

		if(ver > 1)
		{
			users->load(ds,ver);
		}

		fil.close();
	}
	isOpen = true;
	statusBar->changeItem(currFile->data(),0);
	markAsUnchanged();
	loadPublic();
}

//
// save the users appointment, tasks, etc
//
void
Klook::save()
{
int len;
char *add;

	if(hasChanged && !readOnly)
	{
		QString ufil = KApplication::kde_datadir() + "/kalendar/" + *currFile + ".kal";
		QString ulock = KApplication::kde_datadir() + "/kalendar/" + *currFile + ".LCK";

		QFile fil(ufil);

		fil.open(IO_WriteOnly);
		QDataStream ds(&fil);

// because I forgot to write the version number in 0.1, I have to place
// something at the start of the file that could not possible have been
// a valid length for 0.1

		ds << -2;
		ds << version;

		len = fullName->length();
		add = fullName->data();
		ds << len;
		if(len)
		{
			ds.writeRawBytes(add,len);
		}

		len = eMail->length();
		add = eMail->data();
		ds << len;
		if(len)
		{
			ds.writeRawBytes(add,len);
		}

// now in the config file
//		ds << dateFormat;
//		ds << startWeek;
//		ds << (int)sizeMonthToFit;
//		ds << (int)sizeYearToFit;

		cal->save(ds,hasChanged);
		tasks->save(ds,hasChanged);
		users->save(ds,hasChanged);

		fil.close();
		isOpen = false;

		// ensure that all users can access the calendar
		QString modcmd = "chmod 777 " + ufil;
		::system(modcmd.data());

	}

	unlockKalendar();
	cal->remove();
	tasks->remove();
	users->remove();
	statusBar->changeItem("",0);
}

//
// disable the main widgets (when editing appointments, tasks, etc)
//
void
Klook::disable()
{
	ThisMonth->setEnabled(false);
	dayView->setEnabled(false);
	weekView->setEnabled(false);
	monthView->setEnabled(false);
	yearPlan->setEnabled(false);
	todoView->setEnabled(false);
	disableMenus();
}

//
// enable the main widgets
//
void
Klook::enable()
{
	ThisMonth->setEnabled(true);
	dayView->setEnabled(true);
	weekView->setEnabled(true);
	monthView->setEnabled(true);
	yearPlan->setEnabled(true);
	todoView->setEnabled(true);
	enableMenus();
}

//
// find and display an appointment
// 
void
Klook::viewDayItem()
{
int ind = -1;

	if(dayView->isVisible())
	{
		ind = dayView->getSelectedTime();
	}

	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an appointment first"));
	}
	else
	{
		Appointment *a = cal->findAppointment(ind,this);
		if(a->isNew())
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an appointment first"));
			cal->removeAppointment(a);
		}
		else
		{
			disable();

			viewApp = new ViewAppointment(a,cal,this);
			viewApp->setGeometry(220,60,width() - 220 , height() - 100 - 10);
			viewApp->resize(width() - 220 - 10, height() - 100 - 10 + 30);
			viewApp->show();
			connect(viewApp,SIGNAL(exitAppointment()),SLOT(doneDayView()));
		}
	}
}

//
// create a new appointment
//
void
Klook::newDayItem()
{
	openEditor(-1);
}

//
// edit an existing appointment (accessed via the menu)
//
void
Klook::editDayItem()
{
int ind = -1;

	if(dayView->isVisible())
	{
		ind = dayView->getSelectedTime();
	}
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an appointment first"));
	}
	else
	{
		editDayItem(ind);
	}
}

//
// edit an appointment, when selected from week view
//
void
Klook::editWeekItem(int col, int ind)
{
	QDate now = cal->getDate();
	now = now.addDays(col);
	selectDay(now);
	ThisMonth->selectDate(cal->getDate().day());
	editDayItem(ind);
}

//
// edit an existing appointment (accessed via double click)
//
void
Klook::editDayItem(int ind)
{
	openEditor(ind);
}

//
// switch to appointment edit/create mode
//
void
Klook::openEditor(int ind)
{
	if(readOnly)
	{
		return;
	}

	Appointment *a = cal->findAppointment(ind,this);

//	if(a->isNew() && ind != -1)
//	{
//		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an appointment first"));
//		cal->removeAppointment(a);
//		return;
//	}
	
	bool editit = false;
	if(ind != -1)
	{
		int sense = a->getSensitivity();
		if(((sense == Appointment::public_access) && (currUser->canAddPublic())) ||
		   ((sense == Appointment::business_access) && (currUser->canAddBusiness())) ||
		   ((sense == Appointment::personnal_access) && (currUser->canAddPersonnal())))
		{
			editit = true;
		}
	}
	else
	{
		if(currUser->canAdd())
		{
			editit = true;
		}
	}

	if(editit)
	{
		disable();

		editApp = new EditAppointment(a,cal,currUser,this);
		editApp->setGeometry(220,60,width() - 220 , height() - 100 - 10);
		editApp->resize(width() - 220 - 10, height() - 100 - 10 + 30);
		editApp->show();

		connect(editApp,SIGNAL(saveAppointment(Appointment*)),SLOT(doneDayEdit(Appointment*)));
		connect(editApp,SIGNAL(abortAppointment(Appointment*)),SLOT(abortDayEdit(Appointment*)));
	}
	else
	{
		if(a->isNew())
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You cannot create appointment for this user"));
			cal->removeAppointment(a);
		}
		else
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You cannot edit this users appointments"));
		}
	}
}

//
// delete the currently selected appointment
//
void
Klook::deleteDayItem()
{
int ind = -1;

	if(dayView->isVisible())
	{
		ind = dayView->getSelectedTime();
	}
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an appointment first"));
	}
	else
	{
		Appointment *a = cal->findAppointment(ind,this);
		if(a->isNew())
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an appointment first"));
			cal->removeAppointment(a);
		}
		else
		{
			int sense = a->getSensitivity();
			if((sense == Appointment::public_access) && (currUser->canDelPublic()) ||
			   (sense == Appointment::business_access) && (currUser->canDelBusiness()) ||
			   (sense == Appointment::personnal_access) && (currUser->canDelPersonnal()))
			{
				cal->removeAppointment(a);
				markAsChanged();
			}
			else
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("You cannot delete this users appointments"));
			}
		}
	}
	cal->markCalendar(ThisMonth);
	selectView();
}

//
// exit from the appointment editor and save
//
void
Klook::doneDayEdit(Appointment* a)
{
	a->makeValid(this);
	doneDayEdit(true);
}

//
// close the appointment editor
//
void
Klook::doneDayEdit(bool changed)
{
	editApp->hide();
	disconnect(editApp);
	delete editApp;
	editApp = (EditAppointment*)0;
	enable();
	cal->markCalendar(ThisMonth);
	selectView();
	if(changed)
	{
		markAsChanged();
	}
}

//
// quit from the appointment editor and delete appointment if new
//
void
Klook::abortDayEdit(Appointment* a)
{
	if(a->isNew())
	{
		cal->removeAppointment(a);
	}
	doneDayEdit(false);
}

//
// close the appointment viewer
//
void
Klook::doneDayView()
{
	viewApp->hide();
	disconnect(viewApp);
	delete viewApp;
	viewApp = (ViewAppointment*)0;
	enable();
	selectView();
}

//
// edit a new task
//
void
Klook::newTaskItem()
{
	editTaskItem(-1);
}

//
// edit a task (accessed via the menu)
//
void
Klook::editTaskItem()
{
int ind;

	ind = todoView->currentTask();
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select a task first"));
	}
	else
	{
		editTaskItem(ind);
	}
}

//
// edit a task (accessed via double click)
//
void
Klook::editTaskItem(int ind)
{
ToDo* td;

	if(readOnly)
	{
		return;
	}

	td = tasks->findTask(ind);
	if(td == (ToDo*)0)
	{
		if(ind == -1)
		{
			td = tasks->addTask(cal->getDate());
		}
		else
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You must select a task first"));
			return;
		}
	}

	disable();

	editTask = new EditTask(td,this);
	editTask->setGeometry(220,60,width() - 220 , height() - 100 - 10);
	editTask->resize(width() - 220 - 10, height() - 100 - 10 + 30);
	editTask->show();

	connect(editTask,SIGNAL(saveTask()),SLOT(doneTaskEdit()));
	connect(editTask,SIGNAL(abortTask(ToDo*)),SLOT(abortTaskEdit(ToDo*)));
}

//
// mark a task as complete (accessed via double click)
//
void
Klook::finishTaskItem(int ind)
{
ToDo* td;

	td = tasks->findTask(ind);
	if(td != (ToDo*)0)
	{
		if(td->isComplete())
		{
			td->makeInComplete();
		}
		else
		{
			td->makeComplete();
		}
		selectView();
		markAsChanged();
	}
}

//
// close the task editor
//
void
Klook::doneTaskEdit()
{
	editTask->hide();
	disconnect(editTask);
	delete editTask;
	editTask = (EditTask*)0;
	enable();
	selectView();
	markAsChanged();
}

//
// abort from the task editor and delete if a new task
//
void
Klook::abortTaskEdit(ToDo* a)
{
	if(a->isNew())
	{
		tasks->removeTask(a);
	}
	bool mod = hasChanged;
	doneTaskEdit();
	if(!mod)
	{
		markAsUnchanged();
	}
}

//
// close the task viewer
//
void
Klook::doneTaskView()
{
	viewTask->hide();
	disconnect(viewTask);
	delete viewTask;
	viewTask = (ViewTask*)0;
	enable();
	selectView();
}

//
// Save data from the configurator
//
void
Klook::storeConfig()
{
	fullName->resize(0);
	fullName->append(cfg->getFullName());
	eMail->resize(0);
	eMail->append(cfg->getEMail());
	dateFormat = cfg->getDateFormat();
	startWeek = cfg->getStartWeek();
	monthView->setFit(sizeMonthToFit = cfg->getMonthSize());
	yearPlan->setFit(sizeYearToFit = cfg->getYearSize());
	abortConfig();
	ThisMonth->setDateFormat(dateFormat);
	ThisMonth->setStartWeek(startWeek);
	displayName();
	markAsChanged();
}

//
// close the configurator
//
void
Klook::abortConfig()
{
	cfg->hide();
	disconnect(cfg);
	delete cfg;
	cfg = (Kconfigure*)0;
	enable();
	selectView();
}

//
// close the security dialog
//
void
Klook::doneUsers()
{
	usr->hide();
	disconnect(usr);
	delete usr;
	usr = (EditUsers*)0;
	enable();
	selectView();
	markAsChanged();
}

//
// delete the currently selected task
//
void
Klook::deleteTaskItem()
{
int ind = -1;

	if(todoView->isVisible())
	{
		ind = todoView->currentTask();
	}
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select a task first"));
	}
	else
	{
		ToDo *a = tasks->findTask(ind);
		if(a != (ToDo*)0)
		{
			tasks->removeTask(a);
			markAsChanged();
		}
		else
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You must select a task first"));
		}
	}
	selectView();
}

//
// view the currently selected task
//
void
Klook::viewTaskItem()
{
ToDo* td;
int ind = -1;

	if(todoView->isVisible())
	{
		ind = todoView->currentTask();
	}
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select a task first"));
	}
	else
	{
		td = tasks->findTask(ind);
		if(td == (ToDo*)0)
		{
			QMessageBox::warning(this,PACKAGE,klocale->translate("You must select a task first"));
		}
		else
		{
			disable();

			viewTask = new ViewTask(td,this);
			viewTask->setGeometry(220,60,width() - 220 , height() - 100 - 10);
			viewTask->resize(width() - 220 - 10, height() - 100 - 10 + 30);
			viewTask->show();

			connect(viewTask,SIGNAL(exitTask()),SLOT(doneTaskView()));
		}
	}
}

//
// Build the main menu
//
void
Klook::setupMenu()
{
	file = new QPopupMenu();
	fm_m = file->insertItem(klocale->translate("Open"),this,SLOT(openMyKalendar()));
	fm_o = file->insertItem(klocale->translate("Open Other"),this,SLOT(openKalendar()));
	file->insertSeparator();
	fm_c = file->insertItem(klocale->translate("Configure"),this,SLOT(configKalendar()));
	fm_s = file->insertItem(klocale->translate("Security"),this,SLOT(secureKalendar()));
	fm_t = file->insertItem(klocale->translate("Tidy"),this,SLOT(tidyKalendar()));
	file->insertSeparator();
	fm_x = file->insertItem(klocale->translate("Export"),this,SLOT(exportKalendar()));
	file->insertSeparator();
	fm_e = file->insertItem(klocale->translate("Exit"),this,SLOT(closeKalendar()));

	view = new QPopupMenu();
	vm_d = view->insertItem(klocale->translate("Day"),this,SLOT(pickDayView()));
	vm_w = view->insertItem(klocale->translate("Week"),this,SLOT(pickWeekView()));
	vm_m = view->insertItem(klocale->translate("Month"),this,SLOT(pickMonthView()));
	vm_y = view->insertItem(klocale->translate("Year"),this,SLOT(pickYearPlan()));
	view->insertSeparator();
	vm_t = view->insertItem(klocale->translate("ToDo"),this,SLOT(pickToDoView()));
	view->setCheckable(true);

	appt = new QPopupMenu();
	am_n = appt->insertItem(klocale->translate("New"),this,SLOT(newDayItem()));
	am_e = appt->insertItem(klocale->translate("Edit"),this,SLOT(editDayItem()));
	am_d = appt->insertItem(klocale->translate("Delete"),this,SLOT(deleteDayItem()));
	am_v = appt->insertItem(klocale->translate("View"),this,SLOT(viewDayItem()));

	task = new QPopupMenu();
	tm_n = task->insertItem(klocale->translate("New"),this,SLOT(newTaskItem()));
	tm_e = task->insertItem(klocale->translate("Edit"),this,SLOT(editTaskItem()));
	tm_d = task->insertItem(klocale->translate("Delete"),this,SLOT(deleteTaskItem()));
	tm_v = task->insertItem(klocale->translate("View"),this,SLOT(viewTaskItem()));

	daye = new QPopupMenu();
	de_n = daye->insertItem(klocale->translate("New"),this,SLOT(newDayEvent()));
	de_e = daye->insertItem(klocale->translate("Edit"),this,SLOT(editDayEvent()));
	de_d = daye->insertItem(klocale->translate("Delete"),this,SLOT(deleteDayEvent()));

	find = new QPopupMenu();
	fd_f = find->insertItem(klocale->translate("Find..."),this,SLOT(Finder()));
	fd_a = find->insertItem(klocale->translate("Find Again"),this,SLOT(findAgain()));

	help = new QPopupMenu();
	help->insertItem(klocale->translate("Help"),this,SLOT(helpHelp()));
	help->insertItem(klocale->translate("About"),this,SLOT(helpAbout()));

	Menu = new KMenuBar(this,"File");
	Menu->setFont(QFont("Times",14,QFont::Normal,(int)0));

	Menu->insertItem(klocale->translate("File"),file);
	Menu->insertItem(klocale->translate("View"),view);
	Menu->insertItem(klocale->translate("Appointment"),appt);
	Menu->insertItem(klocale->translate("Events"),daye);
	Menu->insertItem(klocale->translate("Tasks"),task);
	Menu->insertItem(klocale->translate("Search"),find);
	Menu->insertSeparator();
	Menu->insertItem(klocale->translate("Help"),help);
}

//
// Disable the view menu
//
void
Klook::disableViews()
{
	Menu->setItemEnabled(vm_d,false);
	Menu->setItemEnabled(vm_w,false);
	Menu->setItemEnabled(vm_m,false);
	Menu->setItemEnabled(vm_y,false);
	Menu->setItemEnabled(vm_t,false);
}

//
// Disable the view menu
//
void
Klook::enableViews()
{
	bool cando = (currUser == baseUser) ? true : false;
	Menu->setItemEnabled(vm_d,true);
	Menu->setItemEnabled(vm_w,true);
	Menu->setItemEnabled(vm_m,true);
	Menu->setItemEnabled(vm_y,true);
	Menu->setItemEnabled(vm_t,cando);
}

//
// Disable the appointments menu
//
void
Klook::disableAppointments()
{
	Menu->setItemEnabled(am_n,false);
	Menu->setItemEnabled(am_e,false);
	Menu->setItemEnabled(am_v,false);
	Menu->setItemEnabled(am_d,false);
	Menu->setItemEnabled(de_n,false);
	Menu->setItemEnabled(de_e,false);
	Menu->setItemEnabled(de_d,false);
	appToolBar->enable(KToolBar::Hide);
}

//
// Enable the appointments menu
//
void
Klook::enableAppointments()
{
	bool cando = (currUser == baseUser) ? true : false;
	Menu->setItemEnabled(am_n,currUser->canAdd() & !readOnly);
	Menu->setItemEnabled(am_e,cando & !readOnly);
	Menu->setItemEnabled(am_v,true);
	Menu->setItemEnabled(am_d,currUser->canDelete() & !readOnly);
	Menu->setItemEnabled(de_n,cando & !readOnly);
	Menu->setItemEnabled(de_e,cando & !readOnly);
	Menu->setItemEnabled(de_d,cando & !readOnly);
	appToolBar->enable(KToolBar::Show);
	appToolBar->setBarPos(KToolBar::Top);
}

//
// Disable the to-do list menu
//
void
Klook::disableTasks()
{
	Menu->setItemEnabled(tm_n,false);
	Menu->setItemEnabled(tm_e,false);
	Menu->setItemEnabled(tm_v,false);
	Menu->setItemEnabled(tm_d,false);
	doToolBar->enable(KToolBar::Hide);
}

//
// Enable the to-do list menu
//
void
Klook::enableTasks()
{
	bool cando = (currUser == baseUser) ? true : false;
	Menu->setItemEnabled(tm_n,cando & !readOnly);
	Menu->setItemEnabled(tm_e,cando & !readOnly);
	Menu->setItemEnabled(tm_v,cando & !readOnly);
	Menu->setItemEnabled(tm_d,cando & !readOnly);
	doToolBar->enable(KToolBar::Show);
	doToolBar->setBarPos(KToolBar::Top);
}

//
// Disable all menus / toolbars
//
void
Klook::disableMenus()
{
	disableAppointments();
	disableTasks();
	disableViews();
	Menu->setItemEnabled(fm_m,false);
	Menu->setItemEnabled(fm_o,false);
	Menu->setItemEnabled(fm_c,false);
	Menu->setItemEnabled(fm_s,false);
	Menu->setItemEnabled(fm_t,false);
	Menu->setItemEnabled(fm_x,false);
	Menu->setItemEnabled(fm_e,false);
	Menu->setItemEnabled(fd_f,false);
	Menu->setItemEnabled(fd_a,false);
}

//
// Enable all menus / toolbars
//
void
Klook::enableMenus()
{
	bool cando = (currUser == baseUser) ? true : false;
	enableAppointments();
	enableTasks();
	enableViews();
	Menu->setItemEnabled(fm_m,true);
	Menu->setItemEnabled(fm_o,true);
	Menu->setItemEnabled(fm_c,cando & !readOnly);
	Menu->setItemEnabled(fm_s,cando & !readOnly);
	Menu->setItemEnabled(fm_t,cando & !readOnly);
	Menu->setItemEnabled(fm_x,cando);
	Menu->setItemEnabled(fm_e,true);
	Menu->setItemEnabled(fd_f,true);
	Menu->setItemEnabled(fd_a,true);
}

//
// Display the full name of the user
//
void
Klook::displayName()
{
	selName->setText(fullName->data());
}

//
// setup the toolbars
//
void
Klook::setupToolBar()
{
	KIconLoader *loader = kapp->getIconLoader();

	setIcon(loader->loadIcon("kalendar.xpm"));

	appToolBar = new KToolBar(this);

	at_n = appToolBar->insertButton(loader->loadIcon("filenew.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(newDayItem()),
					true,
					klocale->translate("New Appointment"));

	at_e = appToolBar->insertButton(loader->loadIcon("fileopen.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(editDayItem()),
					true,
					klocale->translate("Edit Appointment"));

	at_d = appToolBar->insertButton(loader->loadIcon("filedel.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(deleteDayItem()),
					true,
					klocale->translate("Delete Appointment"));

	at_v = appToolBar->insertButton(loader->loadIcon("viewzoom.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(viewDayItem()),
					true,
					klocale->translate("View Appointment"));

	appToolBar->setBarPos(KToolBar::Top);
	appToolBar->enable(KToolBar::Show);
	appToolBarID = addToolBar(appToolBar);

	doToolBar = new KToolBar(this);

	tt_n = doToolBar->insertButton(	loader->loadIcon("filenew.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(newTaskItem()),
					true,
					klocale->translate("New Task"));

	tt_e = doToolBar->insertButton(	loader->loadIcon("fileopen.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(editTaskItem()),
					true,
					klocale->translate("Edit Task"));

	tt_d = doToolBar->insertButton(	loader->loadIcon("filedel.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(deleteTaskItem()),
					true,
					klocale->translate("Delete Task"));

	tt_v = doToolBar->insertButton(	loader->loadIcon("viewzoom.xpm"),
					0,
					SIGNAL(clicked()),
					this,
					SLOT(viewTaskItem()),
					true,
					klocale->translate("View Task"));

	doToolBar->setBarPos(KToolBar::Top);
	doToolBar->enable(KToolBar::Show);
	doToolBarID = addToolBar(doToolBar);
}

//
// create the status bar
//
void
Klook::setupStatusBar()
{
	statusBar = new KStatusBar(this);
	setStatusBar(statusBar);
	statusBar->insertItem("                    ",0);	// username
	statusBar->insertItem("                    ",1);	// lock mode
	statusBar->insertItem("                    ",2);	// changed
	statusBar->insertItem("",3);				// messages
	statusBar->enable(KStatusBar::Show);
}

//
// return a reference to the dock widget
//
DockWidget&
Klook::getDock()
{
	return *dock_widget;
}

//
// return the currently visible view
//
int
Klook::whatIsVisible()
{
	return visibleView;
}

//
// open the users own calendar
//
void
Klook::openMyKalendar()
{
	if(isOpen)
	{
		save();
	}
	currUser = baseUser;
	load(baseUser->getUsername());
	selectDayNote();
	selectView();
	displayName();
	enableMenus();
	selectDay(cal->getDate());
	selectView();
}

//
// open another users calendar
//
void
Klook::openKalendar()
{
	QString udir = KApplication::kde_datadir() + "/kalendar/";
	QString gotfile = QFileDialog::getOpenFileName(udir.data(),"*.kal",this,"file");
	if(!gotfile.isEmpty())
	{
		int f1 = gotfile.findRev('/');
		int f2 = gotfile.findRev('.');
		if((f1 == -1) || (f2 == -1))
		{
			QMessageBox::critical(this,PACKAGE,klocale->translate("Invalid or missing calendar file"));
			return;
		}
		QString u = gotfile.mid(f1 + 1,f2 - f1 - 1);
		if(u != baseUser->getUsername())
		{
			if(isOpen)
			{
				save();
			}
			load(u);
			currUser = users->findUser(baseUser->getUsername());
			if(currUser == (UserAccess*)0)
			{
				currUser = nullUser;
			}
			if(!currUser->canAdd() && !currUser->canDelete())
			{
				unlockKalendar();
			}
			displayName();
			enableMenus();
			selectDay(cal->getDate());
			selectView();
		}
		else
		{
			openMyKalendar();
		}
	}
}

//
// view slots
//
void
Klook::pickDayView()
{
	selectView(v_day);
}

void
Klook::pickWeekView()
{
	selectView(v_week);
}

void
Klook::pickMonthView()
{
	selectView(v_month);
}

void
Klook::pickYearPlan()
{
	selectView(v_year);
}

void
Klook::pickToDoView()
{
	selectView(v_todo);
}

//
// lock the opened file, or mark as a read-only calendar
//
bool
Klook::lockKalendar(bool isme)
{
	QString ulock = KApplication::kde_datadir() + "/kalendar/" + *currFile + ".LCK";

	QFile lck(ulock);
	readOnly = (lck.exists()) ? true : false;
	if(!readOnly)
	{
		lck.open(IO_ReadWrite);
		lck.writeBlock(currFile->data(),currFile->length());
		lck.close();
		statusBar->changeItem(klocale->translate("Read/Write"),1);
		dayNote->setEnabled(isme);
	}
	else
	{
		statusBar->changeItem(klocale->translate("Read Only"),1);
		dayNote->setEnabled(false);
	}

	return !readOnly;
}

//
// unlock the kalendar
//
void
Klook::unlockKalendar()
{
	if(!readOnly)
	{
		QString ulock = KApplication::kde_datadir() + "/kalendar/" + *currFile + ".LCK";

		if(QFile::exists(ulock.data()))
		{
			QString delcmd = "rm " + ulock;
			::system(delcmd.data());
		}
	}
	statusBar->changeItem(klocale->translate("Read Only"),1);
	readOnly = true;
	dayNote->setEnabled(false);
}

//
// general messages on the status line
//
void
Klook::statusBarMessage(QString& msg)
{
	statusBar->changeItem(msg.data(),3);
}

//
// mark the calendar as changed (for when alarms fire)
//
void
Klook::markAsChanged()
{
	if(!readOnly)
	{
		hasChanged = true;
		statusBar->changeItem(klocale->translate("Changed"),2);
	}
}

//
// mark the file as unchanged
//
void
Klook::markAsUnchanged()
{
	statusBar->changeItem(" ",2);
	hasChanged = false;

}

//
// find and display a day note
//
void
Klook::selectDayNote()
{
QString txt;

	if(currUser != baseUser)
	{
		return;
	}

	dayNote->blockSignals(true);

	Day* d = cal->findDay();
	if(d != (Day*)0)
	{
		txt = d->getDayNote();
	}
	else
	{
		txt = "";
	}
	dayNote->setText(txt);

	dayNote->blockSignals(false);
}

//
// store a day note
//
void
Klook::saveDayNote()
{
Day *d;
QString txt;

	if(currUser != baseUser)
	{
		return;
	}

	txt = dayNote->text();
	d = cal->findDay();
	if(d != (Day*)0)
	{
		d->setDayNote(txt);
	}
	else
	{
		cal->addDay(txt);
	}
}


//
// select the view...today
//
void
Klook::selectDayView(int d)
{
	selectView(v_day);
	selectDay(d);
	ThisMonth->selectDate(d);
}

void
Klook::selectDayView(int m, int d)
{
	QDate now = cal->getDate();
	int y = now.year();
	now.setYMD(y,m,d);
	selectView(v_day);
	selectDay(now);
	ThisMonth->selectDate(d,m);
}

//
// edit a new day event
//
void
Klook::newDayEvent()
{
	editDayEvent(-1);
}

//
// edit an event (accessed via the menu)
//
void
Klook::editDayEvent()
{
int ind = -1;

	if(dayView->isVisible())
	{
		ind = dayView->getSelectedEvent();
	}
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an event first"));
	}
	else
	{
		editDayEvent(ind);
	}
}

//
// edit an event (accessed via double click)
//
void
Klook::editDayEvent(int ind)
{
DayEvent* de;

	if(readOnly)
	{
		return;
	}

	de = cal->findEvent(ind);
	if(de->isPublic())
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You cannot edit common events"));
		return;
	}

	disable();

	editEvent = new EditEvent(de,cal,this);
	editEvent->setGeometry(220,60,width() - 220 , height() - 100 - 10);
	editEvent->resize(width() - 220 - 10, height() - 100 - 10 + 30);
	editEvent->show();

	connect(editEvent,SIGNAL(saveEvent(DayEvent*)),SLOT(doneDayEvent(DayEvent*)));
	connect(editEvent,SIGNAL(abortEvent(DayEvent*)),SLOT(abortDayEvent(DayEvent*)));
}

void
Klook::doneDayEvent(DayEvent* e)
{
	e->makeValid();
	doneDayEvent(true);
}

//
// close the day event editor
//
void
Klook::doneDayEvent(bool changed)
{
	editEvent->hide();
	disconnect(editEvent);
	delete editEvent;
	editEvent = (EditEvent*)0;
	enable();
	cal->markCalendar(ThisMonth);
	selectView();
	if(changed)
	{
		markAsChanged();
	}
}

//
// abort from the day event editor and delete if a new event
//
void
Klook::abortDayEvent(DayEvent* a)
{
	if(a->isNew())
	{
		cal->removeEvent(a);
	}
	bool mod = hasChanged;
	doneDayEvent(false);
	if(!mod)
	{
		markAsUnchanged();
	}
}


void
Klook::refreshMonth(int m, int y)
{
	cal->markCalendar(ThisMonth,m,y);
//	if(visibleView == v_month)
//	{
//	}
	selectView();
}

//
// delete the currently selected task
//
void
Klook::deleteDayEvent()
{
int ind = -1;
DayEvent* de;

	if(dayView->isVisible())
	{
		ind = dayView->getSelectedEvent();
	}
	if(ind == -1)
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You must select an event first"));
		return;
	}

	de = cal->findEvent(ind);
	if(de->isPublic())
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("You cannot delete common events"));
		return;
	}

	cal->removeEvent(de);
	markAsChanged();
	selectView();
}

//
// Start up the find dialog
//
void
Klook::Finder()
{
	disable();

	if(search != (Search*)0)
	{
		delete search;
		search = (Search*)0;
	}

	finder = new FindText(this);
	finder->setGeometry(220,60,width() - 220 , height() - 100 - 10);
	finder->resize(width() - 220 - 10, height() - 100 - 10 + 30);
	finder->show();

	connect(finder,SIGNAL(goFind()),SLOT(beginFind()));
	connect(finder,SIGNAL(exitFind()),SLOT(abortFind()));
}

//
//	Look for another occurrence of the fidn string
//
void
Klook::findAgain()
{
	if(search != (Search*)0)
	{
		doFind(false);
	}
	else
	{
		QMessageBox::warning(	this,
					PACKAGE,
					klocale->translate("Nothing to find"));
	}
}

//
//	create the search object and find
//
void
Klook::beginFind()
{

	search = new Search(finder->searchString(),cal->getDate(),finder->searchAppointments(),finder->searchDayEvents(),finder->searchDayNotes());
	doFind(true);
}

//
//	locate the find string
//
void
Klook::doFind(bool clean)
{
Day* found;

	found = cal->searchForString(search);

	if(found != (Day*)0)
	{
		selectDay(found->getDate());
		search->setDate(found->getDate());
		ThisMonth->selectDate(found->getDay(),found->getMonth());
	}
	else
	{
		QMessageBox::warning(this,PACKAGE,klocale->translate("String not found"));
		delete search;
		search = (Search*)0;
	}
	if(clean)
	{
		abortFind();
	}
}

//
//	clean up the find dialog
//
void
Klook::abortFind()
{
	finder->hide();
	disconnect(finder);
	delete finder;
	finder = (FindText*)0;
	enable();
	selectView();
}

void
Klook::doExport()
{
Export export;

	export.setDateFormat(dateFormat);
	export.setPublic(exp->exportPublic());
	export.setBusiness(exp->exportBusiness());
	export.setPersonnal(exp->exportPersonnal());

	QFile expfil("export.kal");

	expfil.open(IO_WriteOnly);
	QTextStream es(&expfil);

	cal->export(es,export);

	expfil.close();

	QMessageBox::about(this,PACKAGE,"Calendar exported to export.kal");

	abortExport();
}

void
Klook::abortExport()
{
	exp->hide();
	disconnect(exp);
	delete exp;
	exp = (ExportSelect*)0;
	enable();
	selectView();
}

//
// load the users appointments,tasks etc
//
void
Klook::loadPublic()
{
char buf[256];
QString strbuf,sd,sm,sy,st,sr;
int p1,p2,p3,p4;
int y,m,d;
bool ok;
QDate tday;
Day *day;
DayEvent *evt;
Repeat* rep;

	QString ufil = KApplication::kde_datadir() + "/kalendar/kalendar.pub";
	QFile fil(ufil);

	if(fil.exists())
	{
		fil.open(IO_ReadOnly);

		while(!fil.atEnd())
		{
			fil.readLine(buf,sizeof(buf) - 1);

			// break the string up
			strbuf.setStr(buf);
			p1 = strbuf.find(',',0);
			if(p1 == -1)
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("Invalid Public Calendar Format"));
				break;
			}
			p2 = strbuf.find(',',p1 + 1);
			if(p2 == -1)
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("Invalid Public Calendar Format"));
				break;
			}
			p3 = strbuf.find(',',p2 + 1);
			if(p3 == -1)
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("Invalid Public Calendar Format"));
				break;
			}
			p4 = strbuf.find(',',p3 + 1);

			sy = strbuf.left(p1);
			sm = strbuf.mid(p1 + 1 , p2 - p1 - 1);
			sd = strbuf.mid(p2 + 1 , p3 - p2 - 1);
			if(p4 == -1)
			{
				st = strbuf.right(strbuf.length() - p3 - 1);
				st = st.left(st.length() - 1);	// remove the trailing \n
				sr = " ";
			}
			else
			{
				st = strbuf.mid(p3 + 1 , p4 - p3 - 1);
				sr = strbuf.right(strbuf.length() - p4 - 1);
				sr = sr.left(sr.length() - 1);	// remove the trailing \n
			}

			// convert to a real date
			d = sd.toInt(&ok);
			if(!ok)
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("Invalid Public Calendar Format"));
				break;
			}
			m = sm.toInt(&ok);
			if(!ok)
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("Invalid Public Calendar Format"));
				break;
			}
			y = sy.toInt(&ok);
			if(!ok)
			{
				QMessageBox::warning(this,PACKAGE,klocale->translate("Invalid Public Calendar Format"));
				break;
			}

			// locate the date
			tday.setYMD(y,m,d);
			day = cal->findDay(tday);
			if(day == (Day*)0)
			{
				day = cal->addDay(tday);
			}

			// create the day event
			evt = day->createEvent();
			evt->setSubject(st);
			evt->makePublic();

			// Repeating ?
			if(sr == "R")
			{
				rep = new Repeat(day,evt);
				evt->setRepeat(rep);
				cal->addRepeat(rep);
			}
		}
		fil.close();
	}

}

void
Klook::LoadRC()
{
KConfig* config;

	config=KApplication::getKApplication()->getConfig();
	config->setGroup("Dates");
	dateFormat = config->readNumEntry("DateFormat");
	startWeek = config->readNumEntry("StartOfWeek");
	config->setGroup("Sizes");
	sizeMonthToFit = config->readBoolEntry("SizeMonthToFit");
	sizeYearToFit = config->readBoolEntry("SizeYearToFit");
}

void
Klook::SaveRC()
{
KConfig* config;

	config=KApplication::getKApplication()->getConfig();
	config->setGroup("Dates");
	config->writeEntry("DateFormat",dateFormat);
	config->writeEntry("StartOfWeek",startWeek);
	config->setGroup("Sizes");
	config->writeEntry("SizeMonthToFit",sizeMonthToFit);
	config->writeEntry("SizeYearToFit",sizeYearToFit);
}

