//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


#include <qpainter.h>
#include <qkeycode.h>
#include <qprinter.h>

#include "MonthTable.h"

const int maxRows = 7;
const int maxCols = 5;

//
// create a table sized to allow a months worth of appointments to be viewed
//
MonthTable::MonthTable(bool fit, QWidget *parent, const char *name)
	: QTableView(parent,name)
{
	curRow = curCol = 0;
	setFocusPolicy(StrongFocus);
	setNumCols(maxRows);
	setNumRows(maxCols);
	setCellWidth(100);
	setCellHeight(100);
	setFit(fit);

	contents = new QString[maxRows * maxCols];

	curDay = 0;

	connect(parent,SIGNAL(daysInMonth(int,int)),SLOT(setDays(int,int)));
	connect(parent,SIGNAL(changeDay(int)),SLOT(setCurrentDay(int)));

	connect(this,SIGNAL(dayChanged(int)),SLOT(refreshDay(int)));
}

//
// tidy up the array of appointments visible
//
MonthTable::~MonthTable()
{
	delete[] contents;
}

void
MonthTable::resize(int w , int h)
{
	if(mustFit)
	{
		setCellWidth(w / maxRows);
		setCellHeight(h /maxCols);
	}
	else
	{
		setCellWidth(100);
		setCellHeight(100);
	}
}

//
// return a pointer to the appointment subjects for a given day
//
const char*
MonthTable::getDayEvents(int day) const
{
	return contents[day];
}

//
// append text to a days appointments
//
void
MonthTable::setDayEvent(int day, const char* txt, bool)
{
	if(contents[day].isEmpty())
	{
		contents[day] = txt;
	}
	else
	{
		contents[day] = contents[day] + '\n' + txt;
	}
	emit dayChanged(day);
 }

//
// draw a border around a day
//
void
MonthTable::paintCell(QPainter* p, int row, int col)
{
QPen cp;
QFont ft;

	int day = row * 7 + col - startOfWeek;

	if((day >= 0) && (day < daysOfMonth))
	{
		int w = cellWidth(col);
		int h = cellHeight(row);

		p->setPen(SolidLine);
		p->drawRect(0,0,w,h);

		QColor tc = p->pen().color();

		if((row == curRow) && (col == curCol))
		{
			cp.setColor(QColor(255,0,0));
		}
		else
		{
			cp.setColor(tc);
		}

		p->setPen(cp);

		p->drawRect(0,0,w,h);
		p->setPen(tc);

		QString tday;
		tday.setNum(day + 1);
		if(curDay == day + 1)
		{
			cp.setColor(QColor(0,0,255));
		}
		else
		{
			cp.setColor(tc);
		}
		p->setPen(cp);

		if(contents[day].isEmpty())
		{
			ft = QFont("Times",12,QFont::Normal);
		}
		else
		{
			ft = QFont("Times",12,QFont::Bold);
		}

		p->setFont(ft);
		p->drawText(5,1,20,15,AlignLeft,tday.data());
		p->drawLine(0,16,w,16);

		p->setFont(QFont("Times",11,QFont::Normal));
		p->drawText(0,15,w,h,AlignLeft,contents[day]);
	}
}

//
// alter the highlight with respect to the day of the month selected
// If the middle button was pressed then also emit a signal
//
//
void
MonthTable::mousePressEvent(QMouseEvent* e)
{
	int oldRow = curRow;
	int oldCol = curCol;

	QPoint clickedPos = e->pos();
	curRow = findRow(clickedPos.y());
	curCol = findCol(clickedPos.x());

	if((curRow != oldRow) || (curCol != oldCol))
	{
		updateCell(oldRow,oldCol);
		updateCell(curRow,curCol);
	}

	if(e->button() == MidButton)
	{
		int day = curRow * 7 + curCol + 1 - startOfWeek;
		emit dayPicked(day);
	}
}

//
// ditto above, but for keyboard events
//
void
MonthTable::keyPressEvent(QKeyEvent* e)
{
	int oldRow = curRow;
	int oldCol = curCol;
	int edge = 0;
	switch(e->key())
	{
	case Key_Left:
		if(curCol > 0)
		{
			curCol--;
			edge = leftCell();
			if(curCol < edge)
			{
				setLeftCell(edge - 1);
			}
		}
		break;
	case Key_Right:
		if(curCol < numCols() - 1)
		{
			curCol++;
			edge = lastColVisible();
			if (curCol >= edge)
			{
				setLeftCell(leftCell() + 1);
			}
		}
		break;
	case Key_Up:
		if(curRow > 0)
		{
			curRow--;
			edge = topCell();
			if(curRow < edge)
			{
				setTopCell(edge - 1);
			}
		}
		break;
	case Key_Down:
		if(curRow < numRows()-1)
		{
			curRow++;
			edge = lastRowVisible();
			if(curRow >= edge)
			{
				setTopCell(topCell() + 1);
			}
		}
		break;
	default:
		e->ignore();
		return;
	}

	if((curRow != oldRow) || (curCol != oldCol))
	{
		updateCell(oldRow,oldCol);
		updateCell(curRow,curCol);
	}
}

//
// give focus to the month list widget
//
void
MonthTable::focusInEvent(QFocusEvent*)
{
	updateCell(curRow,curCol);
}

//
// remove focus from the month list widget
//
void
MonthTable::focusOutEvent(QFocusEvent*)
{
	updateCell(curRow,curCol);
}

//
// select the number of days in the month
//
void
MonthTable::setDays(int d,int s)
{
	daysOfMonth = d;
	startOfWeek = s;
	repaint();
}

//
// reset the text array
//
void
MonthTable::clearMonth()
{
	for(int i = 0; i < (maxRows * maxCols); i++)
	{
		contents[i] = "";
	}
}

//
// signal emitted when the selected day is changed
//
void
MonthTable::setCurrentDay(int d)
{
	curDay = d;
	emit dayChanged(d);
}

//
// internal function to calculate which cell to update
//
void
MonthTable::refreshDay(int day)
{
	int r = day / 7;
	int c = day % 7;
	updateCell(r,c);
}

void
MonthTable::setFit(bool f)
{
	mustFit = f;
	if(mustFit)
	{
		setTableFlags(Tbl_clipCellPainting | Tbl_smoothScrolling);
		clearTableFlags(Tbl_vScrollBar | Tbl_hScrollBar);
	}
	else
	{
		setTableFlags(Tbl_vScrollBar | Tbl_hScrollBar | Tbl_clipCellPainting | Tbl_smoothScrolling);
		setCellWidth(100);
		setCellHeight(100);
	}
}

