//
// Copyright (C) 1998, Mark W J Redding <mark@grawlfang.demon.co.uk>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


#include <qpainter.h>
#include <qkeycode.h>
#include <qprinter.h>
#include <qdatetm.h>

#include <kapp.h>

#include "YearTable.h"

const int maxRows = 13;
const int maxCols = 42;

//
// create a table sized to allow a years worth of appointments to be viewed
//
YearTable::YearTable(bool fit, QWidget *parent, const char *name)
	: QTableView(parent,name)
{
	curRow = curCol = 0;
	setFocusPolicy(StrongFocus);
	setNumCols(maxCols);
	setNumRows(maxRows);
	setCellWidth(20);
	setCellHeight(20);
	setFit(fit);

	clearYear();

	connect(parent,SIGNAL(changeDay(int,int)),SLOT(setCurrentDay(int,int)));

	connect(this,SIGNAL(dayChanged(int,int)),SLOT(refreshDay(int,int)));
}

//
// tidy up the array of appointments visible
//
YearTable::~YearTable()
{}

void
YearTable::resize(int w , int h)
{
	if(mustFit)
	{
		setCellWidth(w / maxCols);
		setCellHeight(h / maxRows);
	}
	else
	{
		setCellWidth(20);
		setCellHeight(20);
	}
}

void
YearTable::setYear(int y)
{
	curYear = y;
}

//
// append text to a days appointments
//
void
YearTable::setDayEvent(int month, int day, int apps, bool dnote, int evt, bool hol)
{
	if(apps > 0)
	{
		events[month][day] |= appointment;
	}
	if(dnote)
	{
		events[month][day] |= daynote;
	}
	if(evt > 0)
	{
		events[month][day] |= dayevent;
	}
	if(hol)
	{
		events[month][day] |= holiday;
	}
	emit dayChanged(month,day);
 }

//
// draw a border around a day
//
void
YearTable::paintCell(QPainter* p, int month, int c)
{
QPen cp;
QFont ft;
QDate cd;
int w = cellWidth();
int h = cellHeight();
int s;
int f;

	if(mustFit)
	{
		s = h / 4;
		f = w / 2;
	}
	else
	{
		s = 5;
		f = 8;
	}

	if((c == 0 && month == 0) || (curYear == 0))
	{
		return;
	}

	QColor tc = p->pen().color();
	QColor tb = p->brush().color();

	if(c == 0)
	{
		p->setPen(black);
		p->setFont(QFont("Times",f));
		switch(month)
		{
		case  1: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Jan"));	break;
		case  2: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Feb"));	break;
		case  3: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Mar"));	break;
		case  4: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Apr"));	break;
		case  5: p->drawText(1,1,w,h,AlignCenter,klocale->translate("May"));	break;
		case  6: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Jun"));	break;
		case  7: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Jul"));	break;
		case  8: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Aug"));	break;
		case  9: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Sep"));	break;
		case 10: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Oct"));	break;
		case 11: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Nov"));	break;
		case 12: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Dec"));	break;
		}
		p->setPen(tc);
		return;
	}

	if(month == 0)
	{
		p->setPen(black);
		p->setFont(QFont("Times",f));
		switch((c - 1) % 7)
		{
		case 0: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Mon"));	break;
		case 1: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Tue"));	break;
		case 2: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Wed"));	break;
		case 3: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Thu"));	break;
		case 4: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Fri"));	break;
		case 5: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Sat"));	break;
		case 6: p->drawText(1,1,w,h,AlignCenter,klocale->translate("Sun"));	break;
		}
		p->setPen(tc);
		return;
	}

	cd.setYMD(curYear,month,1);

	int day = c - monthOffset(month);

	if(day >= 1 && day <= cd.daysInMonth())
	{
		p->setPen(SolidLine);
		p->drawRect(0,0,w - 1,h - 1);
		p->setPen(tc);

		p->setBrush(lightGray);
		p->drawRect(0,0,w,h);

		QString tday;
		tday.setNum(day);
		p->setPen(black);
		p->setFont(QFont("Times",f));
		p->drawText(1,1,w,h,AlignLeft,tday.data());

		if(events[month][day] & appointment)
		{
			p->setBrush(blue);
			p->drawRect(0,0,w,s);
		}
		if(events[month][day] & daynote)
		{
			p->setBrush(green);
			p->drawRect(0,s,w,s);
		}
		if(events[month][day] & dayevent)
		{
			p->setBrush(white);
			p->drawRect(0,s * 2,w,s);
		}
		if(events[month][day] & holiday)
		{
			p->setBrush(red);
			p->drawRect(0,s * 3,w,s);
		}
	}

	p->setBrush(tb);
	p->setPen(tc);
}

//
// alter the highlight with respect to the day of the month selected
// If the middle button was pressed then also emit a signal
//
//
void
YearTable::mousePressEvent(QMouseEvent* e)
{
	int oldRow = curRow;
	int oldCol = curCol;

	QPoint clickedPos = e->pos();
	curRow = findRow(clickedPos.y());
	curCol = findCol(clickedPos.x());

	if((curRow != oldRow) || (curCol != oldCol))
	{
		updateCell(oldRow,oldCol);
		updateCell(curRow,curCol);
	}

	if(e->button() == MidButton)
	{
		if(curRow > 0)
		{
			QDate cd(curYear,curRow,1);
			int day = curCol - monthOffset(curRow);
			if(day >= 1 && day <= cd.daysInMonth())
			{
				emit dayPicked(curRow,day);
			}
		}
	}
}

//
// reset the text array
//
void
YearTable::clearYear()
{
	for(int i = 0; i < 12; i++)
	{
		for(int j = 0; j < 42; j++)
		{
			events[i][j] = 0;
		}
	}

}

//
// signal emitted when the selected day is changed
//
void
YearTable::setCurrentDay(int month, int day)
{
	emit dayChanged(month,day);
}

//
// internal function to calculate which cell to update
//
void
YearTable::refreshDay(int month,int day)
{
	updateCell(month,day);
}

int
YearTable::monthOffset(int m)
{
QDate td;

	td.setYMD(curYear,m,1);
	return td.dayOfWeek() - 1;
}

void
YearTable::addToDay(int m , int d , int a , int e , bool h)
{
	if(a != 0)
	{
		events[m][d] |= appointment;
	}
	if(e != 0)
	{
		events[m][d] |= dayevent;
	}
	if(h)
	{
		events[m][d] |= holiday;
	}
	emit dayChanged(m,d);
}

void
YearTable::setFit(bool f)
{
	mustFit = f;
	if(mustFit)
	{
		setTableFlags( Tbl_clipCellPainting | Tbl_smoothScrolling);
		clearTableFlags(Tbl_vScrollBar | Tbl_hScrollBar);
	}
	else
	{
		setTableFlags( Tbl_vScrollBar | Tbl_hScrollBar | Tbl_clipCellPainting | Tbl_smoothScrolling);
		setCellWidth(20);
		setCellHeight(20);
	}
}

