/*
 *   ksendfax - a interface to fax-packages
 *   Copyright (C) 1998  Juergen Vigna
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>

#include <qtooltip.h>
#include <qapp.h>
#include <qstring.h>
#include <qfiledlg.h>
#include <qregexp.h>
#include <qmsgbox.h>
#include <qfile.h>
#include <qtstream.h>

#include <kapp.h>
#include <kkeyconf.h>
#include <kiconloader.h>

#include "ksendfax.h"
#include "loglistbox.h"
#include "myprocess.h"

#define Inherited KSendFaxData
/* Global Variables */

extern QString debugFlag;
extern int oneShotFlag;

#define LEN_PHONE	20
#define LEN_NAME	25

KSendFax::KSendFax(QWidget* parent, const char* name)
	:Inherited( parent, name )
{
    readConfig();
    debugFlag = getenv("DEBUG");
    setCaption( "KSendFax V" VERSION);
    connect(browseButton, SIGNAL(clicked()), SLOT(BrowseFileCB()));
    // add some tooltips
    QToolTip::add(browseButton, i18n("Browse a file"));
    QToolTip::add(buttonAddFile, i18n("Add file to list"));
    QToolTip::add(buttonRemoveFile, i18n("Remove file from list"));
    current_directory = QDir::currentDirPath();
    error_listbox = new LogListBox(0,i18n("Fax Send Log"));
    error_listbox->setIcon(kapp->getIconLoader()->loadIcon("faxsend.xpm"));
    show_listbox = new LogListBox(0,i18n("Show Fax Queue"));
    show_listbox->setIcon(kapp->getIconLoader()->loadIcon("faxshow.xpm"));
    remove_listbox = new LogListBox(0,i18n("Remove Log"));
    remove_listbox->setIcon(kapp->getIconLoader()->loadIcon("faxremove.xpm"));
    // set labels for i18n
    labelFileName->setAutoResize(true);
    labelDestination->setAutoResize(true);
    labelFaxNo->setAutoResize(true);
    labelDestName->setAutoResize(true);
    labelEnterprise->setAutoResize(true);
    labelComment->setAutoResize(true);
    labelPhonebook->setAutoResize(true);
    labelFileName->setText(i18n("File Name:"));
    labelDestination->setText(i18n("Destination:"));
    labelFaxNo->setText(i18n("Fax no.:"));
    labelDestName->setText(i18n("Dest. Name:"));
    labelEnterprise->setText(i18n("Enterprise:"));
    labelComment->setText(i18n("Comment:"));
    labelPhonebook->setText(i18n("Phonebook"));
    browseButton->setText(i18n("&Browse"));
    setPos(labelFileName,InputFName);
    setPos(labelFaxNo,InputFaxNumber);
    setPos(labelDestName,InputDestName);
    setPos(labelEnterprise,InputEnterprise);
    buttonAddFile->setPixmap(kapp->getIconLoader()->loadIcon("add.xpm"));
    buttonRemoveFile->setPixmap(kapp->getIconLoader()->loadIcon("remove.xpm"));
    connect(buttonAddFile, SIGNAL(clicked()), SLOT(AddFileCB()));
    connect(buttonRemoveFile, SIGNAL(clicked()), SLOT(RemoveFileCB()));
    numsent=0;
}


KSendFax::~KSendFax()
{
}

void KSendFax::SendFaxCB()
{
    SendFax();
}

int KSendFax::SendFax()
{
    QString
        fname = InputFName->text(),
        name = InputDestName->text(),
        phone = InputFaxNumber->text(),
        enterprise = InputEnterprise->text(),
        comment = InputComment->text(),
        host = getenv("FAXSERVER"),
        message = "Error: Trasformation to Postscript failed",
        filename,
        logfile,
        cmd;
    int
        ret;
    QFile
        ff,fl;

    ++numsent;
    if (phone.isEmpty()) {
        return false;
    }
    if (listboxFileNames->count() > 0) {
        fname="";
        for(int i=listboxFileNames->count();i>0;i--) {
            filename = listboxFileNames->text(listboxFileNames->count()-i);
            if (filename.isEmpty())
                continue;
            if (!SetFileName(filename))
                return false;
            fname += " " + filename;
        }
    } else if (fname.isEmpty() || !SetFileName(fname)) {
        return false;
    }
    if (fname.isEmpty())
        return false;
    logfile.sprintf("%s/ksendfax%d.log",logdir.data(),getpid());
    filename = fname.data();
    ret=system("exit 1");
    cmd.sprintf("if [ `file %s | grep -c roff` -gt 0 ];then exit 1;fi;exit 0",
                fname.data());
    ff.setName(filename);
    fl.setName(logfile);
    if (system(cmd) == ret) {
        // preprocess by groff-cmd
        filename.sprintf("%s_%d_%d.tmp",fname.data(),getpid(),numsent);
        cmd.sprintf("%s >%s 2>%s",groffcmd.data(),filename.data(),
                    logfile.data());
        cmd.replace(QRegExp("@@FName"),fname);
        ff.setName(filename);
        if (!debugFlag.isEmpty())
            fprintf(stderr,"GROFF-CMD: %s\n",(const char *)cmd);
        system(cmd);
        if (ff.size() <= 0) {
            if ((fl.size() > 0) && fl.open(IO_ReadOnly)) {
                QTextStream t(&fl);
                message+= "\n(" + t.readLine() + ")";
                fl.close();
            }
            QMessageBox::warning(this,"Trasformation to Postscript failed.",
                                 message);
            remove(logfile);
            remove(filename);
            return false;
        }
        remove(logfile);
    }
    cmd = sendcmd.data();
    if (!debugFlag.isEmpty())
        fprintf(stderr,"CMD: %s:%s:%s\n",cmd.data(),host.data(),name.data());
    cmd.replace(QRegExp("@@Host"),host);
    cmd.replace(QRegExp("@@Comment"),comment);
    cmd.replace(QRegExp("@@Enterprise"),enterprise);
    cmd.replace(QRegExp("@@Name"),name);
    cmd.replace(QRegExp("@@Phone"),phone);
    cmd.replace(QRegExp("@@FName"),filename);
    if (!debugFlag.isEmpty())
        fprintf(stderr,"CMD: %s\n",cmd.data());
    MyProcess
        *proc = new MyProcess(0,false);

    *proc << cmd.data();
    bool result;
    if (oneShotFlag) {
        result = proc->start(KProcess::Block, KProcess::NoCommunication);
        if(!result){
            QString str;
            str.sprintf(i18n("Cannot start a new program\nfork() failed."));
            QMessageBox::critical(this,"KSendFax",str.data());
            if (proc) {
                if(proc->isRunning())
                    proc->kill();
                delete proc;
                proc = 0;
            }
        }
        if (proc)
            delete proc;
        emit ExitKSendFax();
        return true;
    }
    connect(proc, SIGNAL(processExited(KProcess *)),this,
            SLOT(SendFaxDone(KProcess*)));
    connect(proc, SIGNAL(receivedStderr(KProcess *,char*,int)),this, 
            SLOT(ReadStderr(KProcess*,char*,int)));
    connect(proc, SIGNAL(receivedStdout(KProcess *,char*,int)),this, 
            SLOT(ReadStdout(KProcess*,char*,int)));
    if (!error_listbox->isVisible())
        error_listbox->show();
    if (fname != filename)
        proc->setLocalData(filename.data());
    result = proc->start(KProcess::NotifyOnExit , KProcess::All);
    if(!result){
        QString str;
        str.sprintf(i18n("Cannot start a new program\nfork() failed."));
        QMessageBox::critical(this,"KSendFax",str.data());
        if(proc->isRunning())
            proc->kill();
        delete proc;
        return false;
    }
    return true;
}

int KSendFax::ShowJobs()
{
    QString
        host = getenv("FAXSERVER"),
        filename,
        logfile,
        cmd;
    QFile
        ff,fl;

    cmd = showcmd.data();
    cmd.replace(QRegExp("@@Host"),host);
    if (!debugFlag.isEmpty())
        fprintf(stderr,"CMD: %s\n",cmd.data());
    show_listbox->setTimerUpdate(updateSeconds*1000);
    show_listbox->setCommand(cmd);
    show_listbox->activateCommand();
    return true;
}

int KSendFax::RemoveJob()
{
    QString
        host = getenv("FAXSERVER"),
        filename,
        logfile,
        cmd;
    QFile
        ff,fl;

    cmd = removecmd.data();
    cmd.replace(QRegExp("@@Host"),host);
    if (cmd.find("@@ID") >= 0) { // ask for the faxid
        QString
            id = askID();
        if (id.isEmpty())
            return false;
        cmd.replace(QRegExp("@@ID"),id.data());
    }
    if (!debugFlag.isEmpty())
        fprintf(stderr,"CMD: %s\n",cmd.data());
    remove_listbox->setCommand(cmd);
    remove_listbox->activateCommand(false);
    return true;
}

void KSendFax::BrowseFileCB()
{
    (void)SetFileName();
}

QFileDialog* KSendFax::getFileDialog(const char* captiontext)
{
    static QFileDialog *file_dialog=0;

    if(!file_dialog) {
        file_dialog = new QFileDialog(current_directory.data(), "*",
                                      this, "file_dialog", TRUE);
    }
    file_dialog->setCaption(captiontext);
    file_dialog->rereadDir();

    return file_dialog;
}

int KSendFax::SetFileName()
{
    QString fname;
    QFileDialog *box;
            
    box = getFileDialog(i18n("Select Document to Open"));
    box->show();
    
    if (!box->result())   /* cancelled */
        return FALSE;
    if(box->selectedFile().isEmpty()) {  /* no selection */
        return FALSE;
    }
    fname =  box->selectedFile();
    InputFName->setText(fname);

    return TRUE;
}

int KSendFax::SetFileName(const char *fname)
{
    QFileInfo ff(fname);

    if (!ff.exists() || !ff.isFile()) {
        QString message;

        if (!ff.isFile())
            message.sprintf("File %s is not a regular file",fname);
        else
            message.sprintf("File %s empty or does not exists",fname);
        QMessageBox::warning(this,"Parameter Error",message);
        return false;
    } else
        InputFName->setText(fname);
    return true;

}

void KSendFax::readConfig()
{
    KConfig
        *conf = kapp->getConfig();

    conf->setGroup("ksendfax");
    groffcmd = conf->readEntry("groff_command");
    if (groffcmd.isEmpty())
        groffcmd = "groff -t -p -ms @@FName";
    sendcmd = conf->readEntry("send_command");
    if (sendcmd.isEmpty())
        sendcmd = "sendfax -n -h '@@Host' -c '@@Comment' -x '@@Enterprise'" \
                  " -d '@@Name'@'@@Phone' @@FName";
    showcmd = conf->readEntry("show_command");
    if (showcmd.isEmpty())
        showcmd = "faxstat -h '@@Host'";
    removecmd = conf->readEntry("remove_command");
    if (removecmd.isEmpty())
        removecmd = "faxrm -h '@@Host' @@ID";
    if (debugFlag) {
        fprintf(stderr,"SendCMD: %s\n",sendcmd.data());
        fprintf(stderr,"ShowCMD: %s\n",showcmd.data());
        fprintf(stderr,"RemoveCMD: %s\n",removecmd.data());
        fprintf(stderr,"GroffCMD: %s\n",groffcmd.data());
    }
    updateSeconds = conf->readNumEntry("update_seconds",0);
    logdir = conf->readEntry("log_directory");
    if (logdir.isEmpty())
        logdir = "/tmp";
    phonebook = conf->readEntry("phonebook");
    if (phonebook.isEmpty()) {
        phonebook = getenv("PHONEBOOK");
        if (phonebook.isEmpty()) {
            phonebook = getenv("HOME");
            phonebook += "/.phonebook";
        }
    }
    pbLoad(phonebook);
    showPB = conf->readNumEntry("show_phonebook",true);
    int
        height = conf->readNumEntry("height_phonebook",pbListBox->height()),
        width = conf->readNumEntry("width_phonebook",pbListBox->width());
    pbListBox->resize(width,height);
    if (pbListBox->currentItem()>=0)
        pbListBox->setCurrentItem(0);
    conf->sync();
    adjustSize();
}

void KSendFax::saveOptions()
{
    QString
        str;

    KConfig *conf = kapp->getConfig();
    conf->setGroup("ksendfax");
    str = groffcmd;
    conf->writeEntry("groff_command",str);
    str = sendcmd;
    conf->writeEntry("send_command",str);
    conf->writeEntry("log_directory",logdir);
    conf->writeEntry("phonebook",phonebook);
    conf->writeEntry("show_phonebook",showPB);
    conf->writeEntry("width_phonebook",pbListBox->width());
    conf->writeEntry("height_phonebook",pbListBox->height());
    conf->sync();
}

void KSendFax::ReadStderr(KProcess *proc, char *buffer, int buflen)
{
    if (!proc)
        return;
    QString
        str,
        s(buffer,buflen);
    int
        pos;
    MyProcess
        *mproc = (MyProcess*)proc;

    if (mproc->row > 0) {
        str = error_listbox->text(mproc->row);
        error_listbox->setCurrentItem(mproc->row);
        mproc->row = 0;
    }
    str += s;
    pos = str.find('\n');
    while((str.length() > 1) && pos >= 0) {
        pos++;
        s = str.left(pos);
        error_listbox->insertItem(s);
        str = str.right(str.length()-pos);
        pos = str.find('\n');
    }
    if (str.length() > 0) {
        mproc->row = error_listbox->currentItem();
        error_listbox->insertItem(str);
    }
}

void KSendFax::ReadStdout(KProcess *proc, char *buffer, int buflen)
{
    if (!proc)
        return;
    QString
        str,
        s(buffer,buflen);
    int
        pos;
    MyProcess
        *mproc = (MyProcess*)proc;

    if (mproc->row > 0) {
        str = error_listbox->text(mproc->row);
        error_listbox->setCurrentItem(mproc->row);
        mproc->row = 0;
    }
    str += s;
    pos = str.find('\n');
    while((str.length() > 1) && pos >= 0) {
        pos++;
        s = str.left(pos);
        error_listbox->insertItem(s);
        str = str.right(str.length()-pos);
        pos = str.find('\n');
    }
    if (str.length() > 0) {
        mproc->row = error_listbox->currentItem();
        error_listbox->insertItem(str);
    }
}

void KSendFax::SendFaxDone(KProcess *proc)
{
    if (!proc)
        return;
    MyProcess *mproc = (MyProcess*)proc;

    if (mproc->localData())
        remove(mproc->localData());
    if (!debugFlag.isEmpty())
        printf("Called SendFaxDone\n");
    if(proc->isRunning()) {
        proc->kill();
    }
    delete mproc;
}

void KSendFax::resizeEvent (QResizeEvent *rse)
{
    if (!showPB)
        return;

    QSize
        sz;
    int
        width,height,minh;

    sz = rse->size();
    minh = pbListBox->minimumSize().height();
    width = sz.width() - 20;
    height = sz.height() - pbListBox->y() - 10;
    if (height < minh)
        height = minh;
    pbListBox->setGeometry(pbListBox->x(), pbListBox->y(), width, height);
}

int KSendFax::totMinimumHeight() 
{
    return pbListBox->y() + pbListBox->minimumSize().height();
}

void KSendFax::pbLoad(const char *fname)
{
    QFile f(fname);
    pbListBox->clear();
    if (!f.open( IO_ReadOnly ))
        return;

    pbListBox->setAutoUpdate( FALSE );
    QTextStream t(&f);
    QString
        line,a,b,c,
        str;
    while (!t.eof()) {
        str = t.readLine();
        a = (str.mid(LEN_PHONE,LEN_NAME)).simplifyWhiteSpace();
        if (a.isNull())
            a="";
        b = (str.mid(0,LEN_PHONE)).simplifyWhiteSpace();
        if (b.isNull())
            b="";
        c = (str.mid(LEN_PHONE+LEN_NAME,str.length())).simplifyWhiteSpace();
        if (c.isNull())
            c="";
        line="";
        line.sprintf("%-*.*s %-*.*s %-s",LEN_NAME,LEN_NAME,a.data(),
                     LEN_PHONE,LEN_PHONE,b.data(),c.data());
        if ((line.stripWhiteSpace()).length() > 1)
            pbListBox->inSort(line);
    }
    f.close();
    pbListBox->setAutoUpdate( TRUE );
    pbListBox->repaint();
}

void KSendFax::pbSelectCB(const char *item)
{
    QString
        str=item;

    InputDestName->setText((str.mid(0,LEN_NAME)).simplifyWhiteSpace());
    InputFaxNumber->setText((str.mid(LEN_NAME,LEN_PHONE)).simplifyWhiteSpace());
    InputEnterprise->setText((str.mid(LEN_PHONE+LEN_NAME,str.length())).simplifyWhiteSpace());
}

bool KSendFax::pbAdd()
{
    QString
        a,b,c,str,line;
    uint
        i;

    a = ((QString)(InputFaxNumber->text())).simplifyWhiteSpace();
    b = ((QString)(InputDestName->text())).simplifyWhiteSpace();
    c = ((QString)(InputEnterprise->text())).simplifyWhiteSpace();
    line.sprintf("%-*.*s %-*.*s %-s",LEN_NAME,LEN_NAME,b.data(),
                 LEN_PHONE,LEN_PHONE,a.data(),c.data());
    for(i=0;(i < pbListBox->count());i++) {
        if (line == (QString)pbListBox->text(i))
            break;
    }
    if ((i>=pbListBox->count() && ((line.stripWhiteSpace()).length() > 1))) {
        pbListBox->inSort(line);
        return true;
    }
    return false;
}

bool KSendFax::pbModify()
{
    QString
        a,b,c,str,line;

    a = ((QString)(InputFaxNumber->text())).simplifyWhiteSpace();
    b = ((QString)(InputDestName->text())).simplifyWhiteSpace();
    c = ((QString)(InputEnterprise->text())).simplifyWhiteSpace();
    line.sprintf("%-*.*s %-*.*s %-s",LEN_NAME,LEN_NAME,b.data(),
                 LEN_PHONE,LEN_PHONE,a.data(),c.data());
    if ((line.stripWhiteSpace()).length() > 1) {
        pbListBox->changeItem(line,pbListBox->currentItem());
        return true;
    }
    return false;
}

bool KSendFax::pbDelete()
{
    pbListBox->removeItem(pbListBox->currentItem());
    return true;
}

bool KSendFax::pbSave()
{
    QFile
        f(phonebook);
    if (!f.open( IO_WriteOnly ))
        return false;

    uint
        i;
    QString
        line,a,b,c,
        str;
    for(i=0;(i < pbListBox->count());i++) {
        str = pbListBox->text(i);
        a = (str.mid(0,LEN_NAME)).simplifyWhiteSpace();
        if (a.isNull())
            a="";
        b = (str.mid(LEN_NAME,LEN_PHONE)).simplifyWhiteSpace();
        if (b.isNull())
            b="";
        c = (str.mid(LEN_PHONE+LEN_NAME,str.length())).simplifyWhiteSpace();
        if (c.isNull())
            c="";
        line="";
        line.sprintf("%-*.*s %-*.*s %-s\n",
                     LEN_PHONE,LEN_PHONE,b.data(),
                     LEN_NAME,LEN_NAME,a.data(),
                     c.data());
        f.writeBlock(line.data(),line.length());
    }
    f.close();
    return true;
}

void KSendFax::setPos(QLabel *lab, QLineEdit *le)
{
    int pos, diff;

    pos = lab->x() + lab->width() + 5;
    diff = pos - le->x();
    if (diff > 0)
        le->setGeometry(pos, le->y(), le->width() - diff, le->height());
}

const char *KSendFax::askID()
{
    static QDialog *askIDdiag = 0;
    static QLabel *label;
    static QLineEdit *inputID;
    static QPushButton *ok;
    static QPushButton *cancel;

    if (!askIDdiag) {
        askIDdiag = new QDialog(0,i18n("Input Fax ID"),true);
        CHECK_PTR(askIDdiag);
        label = new QLabel(askIDdiag, "Label");
        CHECK_PTR(label);
        inputID = new QLineEdit(askIDdiag,"LineEdit");
        CHECK_PTR(inputID);
        ok = new QPushButton(askIDdiag,"OKButton");
        CHECK_PTR(ok);
        cancel = new QPushButton(askIDdiag,"CancelButton");
        CHECK_PTR(cancel);
        
        label->setGeometry( 10, 10, 50, 30 );
        label->setText(i18n("Fax ID:"));
        label->setAlignment(289);
        label->setMargin(-1);
        
        inputID->setGeometry( 70, 10, 100, 30 );
        inputID->setMaxLength(30);
        
        ok->setGeometry( 55, 45, 60, 25 );
        ok->setText(i18n("Ok"));
        ok->setDefault(TRUE);
        connect(ok, SIGNAL(clicked()), askIDdiag, SLOT(accept()));
        
        cancel->setGeometry( 125, 45, 60, 25 );
        cancel->setText(i18n("&Cancel"));
        connect(cancel, SIGNAL(clicked()), askIDdiag, SLOT(reject()));
        
        askIDdiag->resize( 190,75 );
        askIDdiag->setMinimumSize( 190, 75 );
        askIDdiag->setMaximumSize( 190, 75 );
    }
    inputID->setText("");
    if (!askIDdiag->exec())
        return "";
    else
        return inputID->text();
}

void KSendFax::AddFileCB()
{
    QString
        fname = InputFName->text();
    if (fname.isEmpty())
        return;
    for(int i=listboxFileNames->count()-1;i>=0;i--)
        if (listboxFileNames->text(i) == fname)
            return;
    listboxFileNames->insertItem(fname.data());
}

void KSendFax::RemoveFileCB()
{
    if (listboxFileNames->currentItem() >= 0)
        listboxFileNames->removeItem(listboxFileNames->currentItem());
}

void moveNewPos(QWidget *obj, int dy)
{
    int
        x = obj->x(),
        y = obj->y();

    obj->move(x,y+dy);
}

void KSendFax::showFilelist(bool flag)
{
    int
        minw = minimumSize().width(),
        minh = minimumSize().height(),
        dy = listboxFileNames->height();
    bool
        domove = false;
    
    if (flag && !listboxFileNames->isVisible()) {
        listboxFileNames->show();
        buttonAddFile->show();
        buttonRemoveFile->show();
        domove = true;
    } else if (!flag && listboxFileNames->isVisible()) {
        dy = -dy;
        listboxFileNames->hide();
        buttonAddFile->hide();
        buttonRemoveFile->hide();
        domove = true;
    }
    if (domove) {
        // Destination Box
        moveNewPos(frameDestination,dy);
        moveNewPos(labelDestination,dy);
        moveNewPos(labelFaxNo,dy);
        moveNewPos(InputFaxNumber,dy);
        moveNewPos(labelDestName,dy);
        moveNewPos(InputDestName,dy);
        moveNewPos(labelEnterprise,dy);
        moveNewPos(InputEnterprise,dy);
        // Comment Box
        moveNewPos(frameComment,dy);
        moveNewPos(labelComment,dy);
        moveNewPos(InputComment,dy);
        // Phonebook Box 
        moveNewPos(framePhonebook,dy);
        moveNewPos(labelPhonebook,dy);
        moveNewPos(pbListBox,dy);
        // adjust size
        setMinimumSize(minw,minh+dy);
        adjustSize();
    }
}
