#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <dirent.h>
#include <fcntl.h>

#include "forms.h"

#include "xfaxdefs.h"
#include "version.h"

/* This function converts a string into an array like 'argv' , as is used
 * to start a subprocess with exec.
 */
char **BuildArgv(char *string)
{
   int x,count=0;
   char **argv,*ptr;
   
   /* first count the number of arguments */
   
   ptr = string;
   while(*ptr != '\0')
   {
      while (*ptr != ' ' && *ptr != '\0') ptr++;
      while(*ptr == ' ') ptr++; /* skip spaces */
      count ++;
   }
   count++; /* extra space for NULL pointer */
   
   /* And turn them into an array of separate strings */
   argv = (char **)malloc(count * sizeof(char *));

   argv[0] = ptr = string;
   for(x=1;x<count - 1;x++)
   {
      while (*ptr != ' ') ptr++;
      *ptr++='\0';
      while(*ptr == ' ') ptr++; /* skip spaces */
      argv[x]=ptr;
   }
   argv[x]=NULL;
   return argv;
}

void SpawnCmd(char *cmdbuffer,int flag)
{  char **argv;
   pid_t pid;
   
   if ((pid=fork()) == 0)
   {
      fprintf(stderr,"Executing: %s\n",cmdbuffer);
      argv = BuildArgv(cmdbuffer);
      execvp(argv[0],argv);
      free(argv);
      exit(0);
   }
   else
   {
      if(flag)
	wait(NULL);
      else
	signal(SIGCHLD,SIG_IGN);
   }
}

int CountFilesInDir(char *dirname)
{
   DIR *faxdir;
   struct dirent *g3file;
   int count=0;
   
   faxdir = opendir(dirname);
   while((g3file = readdir(faxdir)) != NULL)
   {
      count++;
   }
   closedir(faxdir);

   return count - 2; /* forget . and .. entries */
}

int CheckForExistence(char *filename)
{
   FILE *fp;
   
   if((fp=fopen(filename,"r")) != NULL)
   {
      fclose(fp);
      return 0;
   }
   else
     return 1;
}

void RemoveTmpdir(char *dirname)
{
   char filename[MAXSTRING];
   struct dirent *g3file;
   DIR *faxdir;
   int count=0;
   
   faxdir = opendir(dirname);
   while((g3file = readdir(faxdir)) != NULL)
   {
      if(count++ <2) continue; /* skip . and .. entries ! */
      strncpy(filename,dirname,MAXSTRING / 2);
      strcat(filename,"/");
      strncat(filename,g3file->d_name,MAXSTRING / 2 -1);
      remove(filename);
   }
   closedir(faxdir);

   rmdir(dirname);
}

const char *BuildOutputName(int number,const char *dirname)
{  static char newname[MAXSTRING];
   
   if(number == 0)
     sprintf(newname,"%s/" FRONTPAGE,dirname);
   else
     sprintf(newname,"%s/" G3FAXPAGE "%04d",dirname,number);

   return newname;
}
   
void MakeFaxPage(const char *filename, char *outputdir,int number)
{
   FILE *page;
   char string[MAXSTRING];
   char cmdbuffer[MAXCMD];
   char *AsciiCmd;
   char *GifCmd;
   char *PSCmd;
   
   /* provide fallback defaults for conversions: crucial for good operation */
   
   AsciiCmd = (strlen(info->AsciiCmd) == 0) ? ASCIITOG3 : info->AsciiCmd;
   GifCmd   = (strlen(info->GifCmd)   == 0) ? GIFTOG3   : info->GifCmd;
   PSCmd    = (strlen(info->PSCmd)    == 0) ? PSTOG3    : info->PSCmd;

   if((page = fopen(filename,"r")) == NULL)
   {
      fl_show_message("","Can't open file:", filename);
      return;
   }
   fgets(string,MAXSTRING,page);
   fclose(page);

   if(! strncmp(string,"%!",2))
   {
      /* Postscript document */
      sprintf(cmdbuffer,PSCmd,BuildOutputName(number,outputdir),
	      filename);
      SpawnCmd(cmdbuffer,1);
   }
   else if(! strncmp(string,"GIF8",4))
   {
      /* GIF picture */
      sprintf(cmdbuffer,GifCmd,BuildOutputName(number,outputdir),
	      filename);
      SpawnCmd(cmdbuffer,1);
   }
   else
   {
      /* ASCII document */
      sprintf(cmdbuffer,AsciiCmd,BuildOutputName(number,outputdir),
	      filename);
      SpawnCmd(cmdbuffer,1);
   }
}

char *BuildFrontPage(FD_Xfax *fd_Xfax,char *outputdir)
{
   FILE *tmpfile,*header;
   char *tmpname = tempnam(NULL,"fp");
   char c;
   time_t ti;
   
   if(fl_get_button(fd_Xfax->FrontPage) == 0)
     return NULL;
     
   tmpfile = fopen(tmpname,"w");
   fputc('\n',tmpfile);
   if(strlen(info->Header) > 0)
   {
      if((header = fopen(info->Header,"r")) == NULL)
	fl_show_message("","Can't open specified headerfile :",info->Header);
      else
      {
	 while(c= fgetc(header),!feof(header))
	   fputc(c,tmpfile);
	 fclose(header);
      }
   }
   
   fprintf(tmpfile,"\nFrom       : %s\n",info->Name);
   fprintf(tmpfile,  "Title      : %s\n",info->Title);
   fprintf(tmpfile,  "Department : %s\n",info->Department);
   fprintf(tmpfile,  "Email      : %s\n\n\n",info->Email);
   
   fprintf(tmpfile,"To    : %s\n",fl_get_input(fd_Xfax->Name));
   fprintf(tmpfile,"Faxnr : %s\n",fl_get_input(fd_Xfax->Number)); 
   ti = time(NULL);
   fprintf(tmpfile,"Date  : %s\n\n",asctime(localtime(&ti)));
   
   fprintf(tmpfile,"Number of pages (including this page) : %d\n\n",
	   CountFilesInDir(outputdir) + 1);
   
   fprintf(tmpfile,"Message :\n");
   fprintf(tmpfile,"---------\n\n");

   /* count  number of [pages !!! */
   
   fputs(fl_get_input(fd_Xfax->Message),tmpfile);

   fclose(tmpfile);
   
   MakeFaxPage(tmpname,outputdir,0);
   remove(tmpname);
   return (tmpname);
}

void BuildFaxPages(FD_Xfax *fd_Xfax,char *outputdir)
{  int x;

   for(x=1;x<=fl_get_browser_maxline(fd_Xfax->FileList);x++)
   {
      MakeFaxPage(fl_get_browser_line(fd_Xfax->FileList,x),outputdir,x);
   }
}

/* callbacks for form Xfax */

void Send(FL_OBJECT *ob, long data)
{
   char *faxnumber;
   char cmdbuffer[MAXCMD];
   char filename[MAXSTRING];
   char dirname[MAXSTRING];
   char *ptr;
   FILE *tmp;
   char *SendCmd;
   char *SpoolDir;
   int km;
   FL_Coord x,y;
   
   SendCmd  = (strlen(info->FaxSend ) == 0) ? FAXSEND  : info->FaxSend;
   SpoolDir = (strlen(info->SpoolDir) == 0) ? SPOOLDIR : info->SpoolDir;
   
   if((faxnumber = (char *)fl_get_input(fd_Xfax->Number)) == NULL || *faxnumber == 0)
   {
      fl_show_message("","Can't send without a fax (telephone) number !!","");
      return;
   }
   
   /* remove spaces from faxnumber (confuses the argument parser !) */
   ptr = faxnumber;
   while(*++ptr != 0)
     if(*ptr == ' ') *ptr = '-';
   
   if(strstr(info->EmailFlag,"True") != NULL && strlen(info->Email) == 0)
   {
      fl_show_message("","Can't mail status without an emailaddress :",
		      "Please set your emailaddress in the From info first.");
      return;
   }
   
   strncpy(dirname,SpoolDir,MAXSTRING / 2);
   ptr = strrchr(tempnam(NULL,"dir"),'/'); /* strip prefix */
   strncat(dirname,ptr,MAXSTRING / 2);
   mkdir(dirname,0755);
 
   BuildFaxPages(fd_Xfax,dirname);
   BuildFrontPage(fd_Xfax,dirname);
   
   strncpy(filename,dirname,MAXSTRING / 2 -1);
   strcat(filename,"/");
   strncat(filename,FAXINFO,MAXSTRING / 2 -1);
   tmp = fopen(filename,"w");

   fprintf(tmp,"EmailAddress: %s\n",info->Email);
   fprintf(tmp,"FaxNumber: %s\n",   faxnumber);

   fprintf(tmp,"Email: %s\n",info->EmailFlag);
   fclose(tmp);

   sprintf(cmdbuffer,"%s %s %s/",SendCmd,faxnumber,dirname);
   
   SpawnCmd(cmdbuffer,0); /* fork, no wait !! */
   
   fl_get_mouse(&x,&y,&km);
   fl_show_oneliner("Your fax was submitted using the command \n"
		    "set in the defaults section",x,y);
   fl_check_forms();
   sleep(4);
   fl_hide_oneliner();
   /* provide for a remove of the spool directory ! */
}

void Print(FL_OBJECT *ob, long data)
{
   char cmdbuffer[MAXCMD];
   char dirname[MAXSTRING];
   char *ptr;
   char *PrintCmd;
   char *SpoolDir;
   int km;
   FL_Coord x,y;
   
   PrintCmd  = (strlen(info->FaxPrint ) == 0) ? FAXPRINT : info->FaxPrint;
   SpoolDir = (strlen(info->SpoolDir) == 0) ? SPOOLDIR : info->SpoolDir;
   
   strncpy(dirname,SpoolDir,MAXSTRING / 2);
   ptr = strrchr(tempnam(NULL,"dir"),'/'); /* strip prefix */
   strncat(dirname,ptr,MAXSTRING / 2);
   mkdir(dirname,0755);
 
   BuildFaxPages(fd_Xfax,dirname);
   BuildFrontPage(fd_Xfax,dirname);
   
   sprintf(cmdbuffer,"%s %s/",PrintCmd,dirname);
   
   SpawnCmd(cmdbuffer,0); /* fork, no wait !! */
   
   fl_get_mouse(&x,&y,&km);
   fl_show_oneliner("Your fax was printed using the command \n"
		    "set in the defaults section",x,y);
   fl_check_forms();
   sleep(4);
   fl_hide_oneliner();
   /* provide for a remove of the spool directory ! */
}

void LoadMessageFile()
{  const char *file;
   FILE *in;
   int nlcount,c,count =0;
   char buffer[MAXMESSAGE];
   
   if((file = fl_show_fselector("Select a file to load",
				       "","*","")) == NULL)
     return;
   if ((in = fopen(file,"r")) != NULL) 
   {
      nlcount =0;
      while(c=fgetc(in),count < MAXMESSAGE - 1 && ! feof(in))
      {
	 if(c == '\n') nlcount++;
	 if(nlcount < MAXMSGLINES)
	   buffer[count++] = c;
      }
      buffer[count] = 0; /* last character */
      fclose(in);
      fl_set_input(fd_Xfax->Message,buffer);
   }
}

void SaveMessageFile()
{  const char *file;
   FILE *out;
   int count =0;
   const char *buffer;
   
   if((file = fl_show_fselector("Enter filename to write to",
					"","*","")) == NULL)
     return;
   if ((out = fopen(file,"w")) != NULL) 
   {
      buffer = fl_get_input(fd_Xfax->Message);
      while(count < MAXMESSAGE - 1 && buffer[count] != 0)
	fputc(buffer[count++],out);
      fclose(out);
   }
}

void FileMenu(FL_OBJECT *ob, long data)
{  const char *tmp;
   
   switch(fl_get_menu(ob))
   {
    case 1:
      InitFromWin(fd_FromWin); break;
    case 2: 
      tmp = fl_show_input("New frontpage Headerfile :",info->Header);
      if(tmp != NULL)
      {
	 strncpy(info->Header,tmp,MAXSTRING);
	 SaveUserInfo(info);
      }
      break;
      
    case 3: LoadMessageFile(); break;
    case 4: SaveMessageFile(); break;
    case 5: fl_call_object_callback(fd_Xfax->Exit); break;
    default: /* Can't happen !! */ break;
   }
}

void OptionsMenu(FL_OBJECT *ob, long data)
{
   switch(fl_get_menu(ob))
   {
    case 1: 
      if(fl_get_menu_item_mode(ob,1) & FL_PUP_CHECK)
	strcpy(info->EmailFlag,"True");
      else
	strcpy(info->EmailFlag,"False");
      SaveUserInfo(info);
      break;
    case 2:
      if(fl_get_menu_item_mode(ob,2) & FL_PUP_CHECK)
      {
	 fl_set_button(fd_Xfax->FrontPage, 1);
	 strcpy(info->FrontPageFlag,"True");
      }
      else
      {
	 fl_set_button(fd_Xfax->FrontPage, 0);
	 strcpy(info->FrontPageFlag,"False");
      }
      SaveUserInfo(info);
      break;
    case 3:
      InitSystemWin(fd_SystemWin); break;
    default: /* Can't happen !! */ break;
   }
}

void Exit(FL_OBJECT *ob, long data)
{
#ifdef FORMS081
   if(! fl_show_question("Do you want to quit ?","",""))
     return;
#else
   if(! fl_show_question("Do you want to quit ?",1))
     return;
#endif
   exit(0);
}

void AddFile(FL_OBJECT *ob, long data)
{  char *file;
   
   file = (char *)fl_show_fselector("Select a file to send","","*","");
   fl_freeze_form(fd_Xfax->Xfax);
   if (file != NULL) fl_addto_browser(fd_Xfax->FileList,file);
   fl_unfreeze_form(fd_Xfax->Xfax); /* necessary to force a redraw */ 
}

void RemoveFile(FL_OBJECT *ob, long data)
{
   int linenr;
   
   if((linenr = fl_get_browser(fd_Xfax->FileList)) > 0)
     fl_delete_browser_line(fd_Xfax->FileList,linenr);
   else
     fl_show_message("","No file selected !","");
}

void StartEditor(FL_OBJECT *ob, long data)
{  char cmdbuffer[MAXSTRING];
   const char *tmpname;
   int linenr;
   char *EditCmd;
   
   EditCmd = (strlen(info->Editor) == 0) ? EDITOR : info->Editor;
   
   if((linenr = fl_get_browser(fd_Xfax->FileList)) > 0)
     tmpname = fl_get_browser_line(fd_Xfax->FileList,linenr);
   else
     tmpname = tempnam(NULL,"fe");
   
   sprintf(cmdbuffer,"%s %s",EditCmd,tmpname);
   SpawnCmd(cmdbuffer,1); /* just spawn : wait for return */
   if(linenr == 0)
   {
      fl_freeze_form(fd_Xfax->Xfax);
      fl_addto_browser(fd_Xfax->FileList,tmpname);
      fl_unfreeze_form(fd_Xfax->Xfax);
   }
}

void Info(FL_OBJECT *ob, long data)
{
   fl_show_message("Xfax: a frontend for efax build using the Xforms library",
		   "by Kees Lemmens, The Netherlands (C) 1996",
		   VERSION ", Faculty TWI, TU Delft");
}

