/*
 *  Flasher Source File
 *
 *  GNU Copyright (C) 2003 Gaspar Sinai <gsinai@yudit.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef generic_flasher_h
#define generic_flasher_h

#include "memory_model.h"

namespace FLASHER
{

  /*!
   * \interface Flasher.
   * \brief A generic interface to a flasher.
   */
  class Flasher
  {
  public:
    /*!
     * \brief Check the ID.
     * \param id is the id string.
     */
    virtual bool checkID(const char* id)=0;

    /*!
     * \brief Read the chips's memory and return it.
     * \param from is the starting address.
     * \param to is the non-inclusive ending address.
     * \param ie is true if we are supposed to ignore read-errors.
     * \return a memory model.
     */
    virtual MemoryModel* read (unsigned long from, unsigned long until, bool ie)=0; 

    /*!
     * \brief Read the chips's boot memory and return it.
     * \param from is the starting address.
     * \param to is the non-inclusive ending address.
     * \param ie is true if we are supposed to ignore read-errors.
     * \return a memory model.
     */
    virtual MemoryModel* readBoot (unsigned long from, unsigned long until, bool ie)=0; 

    /*!
     * \brief Load the memory-model into the RAM and execute it.
     *        The program is expected to be re-locatable, as we don't 
     *        know where the program will be loaded. Most likely, it will
     *        be loaded to 0x400 RAM area. The firt byte will be the first 
     *        byte set in memory model.
     * \param model is the memory model.
     */
    virtual void execute (const MemoryModel& model)=0; 

    /*!
     * \brief Lock bit program. The 6th bit (0x40) should be set for lock.
     * \param model is the memory model.
     */
    virtual void writeLock (const MemoryModel& model)=0; 

    /*!
     * \brief write pages.
     * \param model is the memory model.
     */
    virtual void write (const MemoryModel& model)=0; 

    /*!
     * \brief Read the chips's memory and return it.
     * \param ie is true if we are supposed to ignore read-errors.
     * \return a memory model with lock bits.
     */
    virtual MemoryModel* readLock (bool ie)=0; 
  };
}

#endif/* generic_flasher_h */
