/*
 * @(#)$Id: wx200.c,v 1.27 1998/09/06 02:58:24 twitham Rel $
 *
 * Copyright (C) 1998 Tim Witham <twitham@pcocd2.intel.com>
 *
 * (see the files README and COPYING for more details)
 *
 * This file implements a simple command-line wx200d client
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include "wx200.h"

#define MAXARGS 20		/* maximum CGI args to parse */
#define TMPLEN 128		/* max length of CGI */

char error[80];
char *progname;
char *query;

void
usage(int ret)
{
  if (query)
      printf("Content-type: text/plain\nStatus: 200\n\n");
  printf("usage: %s [options] [[host][:port]]
  -h      --help		show this help and exit
  -r      --refresh		continuously refresh instead of exiting
  --dm --md --12hr --24hr	date order and time format
  --C --celsius --F --fahrenheit	temperature format
  --inhg --mmhg --mbar --hpa	barometric pressure format
  --mm --mm/h --in --in/h	rain rate and accumulation format
  --mph --knots --mps --kph	wind speed format
  --intemp --inhumidity --indewpoint	show one indoor variable
  --temp --humidity --dewpoint		show one outdoor variable
  --sea --local				show one barometric pressure
  --gust --average			show one wind variable
  --rate --yesterday --total		show one rain variable
units may be uniquely abbreviated; they default to those of the WX200
", progname);
  exit(ret);
}

int
main(int argc, char **argv)
{
  const char *flags = "Hhr";
  int opt, num = EOF, socket, done = 0, opt_r = 0;
  int index, ftemp = -1, fbaro = -1, fdepth = -1, fspeed = -1, ftd = 0;
  char *hostport = NULL, buf[300], *string, *arg[MAXARGS + 2];
  char *tmp, tmpbuf[TMPLEN];
  char *disp[] = {"Clock", "Temp", "Humidity", "Dew",
		  "Baro", "Wind", "W.Chill", "Rain"};
  char *acdc[] = {"AC", "DC"};
  char *batt[] = {"ok ", "LOW"};
  char format[] = "%11s:"
    "%7s Hi%7s %s %7s Hi%7s %s\n"
    "            "
    "        Lo%7s %s         Lo%7s %s\n"
    "                    "
    "Al%7s %7s %3s           Al%7s %7s %3s\n\n";
  char *trend[] = {"Rising", "Steady", "", "Falling"};
  char *pred[] = {"Sunny", "Cloudy", "", "Partly Cloudy",
		  "", "", "", "Rain"};
  char *offon[] = {"Off", "On "};

  int which_current = -1;	/* which current value should be returned? */

  struct option longopt[] = {
    {"refresh", 0, 0, 'r'},
    {"celsius", 0, 0, 1}, {"fahrenheit", 0, 0, 2},
    {"C", 0, 0, 1}, {"F", 0, 0, 2},
    {"inhg", 0, 0, 3}, {"mmhg", 0, 0, 4}, {"mbar", 0, 0, 5}, {"hpa", 0, 0, 6},
    {"mm", 0, 0, 7}, {"mm/h", 0, 0, 7}, {"in", 0, 0, 8}, {"in/h", 0, 0, 8},
    {"mph", 0, 0, 9}, {"knots", 0, 0, 10}, {"mps", 0, 0, 11}, {"kph", 0, 0, 12},
    {"dm", 0, 0, 13}, {"md", 0, 0, 14},
    {"12hr", 0, 0, 15}, {"24hr", 0, 0, 16},
    {"help", 0, 0, 17},

    /* added by trs: 12 Feb 1998, names changed by twitham 16 Apr 1998 */
    {"intemp", 0, &which_current, 0},
    {"temp", 0, &which_current, 1},
    {"inhumidity", 0, &which_current, 2},
    {"humidity", 0, &which_current, 3},
    {"indewpoint", 0, &which_current, 4},
    {"dewpoint", 0, &which_current, 5},
    {"sea", 0, &which_current, 6},
    {"local", 0, &which_current, 7},
    {"gust", 0, &which_current, 8},
    {"average", 0, &which_current, 9},
    {"rate", 0, &which_current, 10},
    {"yesterday", 0, &which_current, 11},
    {"total", 0, &which_current, 12},

    {0, 0, 0, 0}
  };

  progname = strrchr(argv[0], '/');
  if (progname == NULL)
    progname = argv[0];
  else
    progname++;

  if ((query = getenv("QUERY_STRING"))) { /* we running as a web CGI ? */
    strncpy(tmpbuf, query, TMPLEN);
    tmp = tmpbuf;
    argv = arg;			/* get args from CGI env */
    argv[argc = 0] = progname;
    while ((argc < MAXARGS) && (string = strtok(argc ? NULL : tmp, "&="))) {
      if (string[0] != '\0') {
	argc++;
	argv[argc] = string;
      }
    }
    argv[++argc] = NULL;
  }

  while ((opt = getopt_long(argc, argv, flags, longopt, &index)) != EOF) {
    switch (opt) {		/* parse command-line or CGI options */
    case '?':
      usage(1);
    case 'h':
    case 'H':
      usage(0);
    case 0:
      break;
    case 1:
    case 2:
      ftemp = opt - 1;
      break;
    case 3:
    case 4:
    case 5:
    case 6:
      fbaro = opt - 3;
      break;
    case 7:
    case 8:
      fdepth = opt - 7;
      break;
    case 9:
    case 10:
    case 11:
    case 12:
      fspeed = opt - 9;
      break;
    case 13:
    case 14:			/* bit 7 is flag, bit 3 is value */
      ftd |= 0x80 | ((opt - 13) << 3);
      break;
    case 15:
    case 16:			/* bit 6 is flag, bit 2 is value */
      ftd |= 0x40 | ((opt - 15) << 2);
      break;
    case 'r':
      opt_r = 1;
      break;
    case 17:
      usage(0);
    }
  }
  if (optind < argc && argv[optind] != NULL) {
    hostport = argv[optind];
    if ((tmp = strchr(hostport, '%')) != NULL) {
      if (*(tmp + 1) == '3' && (*(tmp + 2) == 'A' || *(tmp + 2) == 'a'))
	*tmp = *(tmp + 1) = *(tmp + 2) = ':';
    }
  }

  if ((socket = wx200open(hostport)) == -1)
    exit(1);
  if (query) {
    printf("Content-type: text/html\n");
    if (opt_r)
      printf("Refresh: 5; URL=%s?%s\n",
	     (tmp = getenv("SCRIPT_NAME")) ? tmp : "", query);
    opt_r = 0;
    printf("Status: 200

<html><head>
<title>%s @ %s:%d</title>
</head><body>
<h1>%s @ %s:%d</h1>
<hr>
<pre>
", progname, wx200host, wx200port, progname, wx200host, wx200port);
  }

  wx200bufinit();		/* get current conditions first */
  while ((done < 2) && ((num = wx200bufread(socket)) != EOF)) {
    wx200parse(num);
    if ((num == GROUP8) || (num == GROUPF)) done++;
  }

  if (ftemp == -1) ftemp = temp.format;	/* user prefs override wx200 settings */
  if (fbaro == -1) fbaro = baro.format;
  if (fdepth == -1) fdepth = rain.format;
  if (fspeed == -1) fspeed = wind.format;
  if (ftd & 0x80) ftd |= (ftd & 0x08) >> 2; /* m/d preference? */
  else ftd |= td.format & 0x02;
  if (ftd & 0x40) ftd |= (ftd & 0x04) >> 2; /* 12/24 preference? */
  else ftd |= td.format & 0x01;

  do {				/* forever if -r or --refresh, once otherwise */
    wx200parse(num);

    format_data(F_TIME, buf, &td.clock, ftd);	/* hack to add seconds */
    strcpy(buf + 20, buf + 11);			/* save am/pm to temp area */
    sprintf(buf + 11, ":%02d", td.clock.sec);	/* overwrite with seconds */
    strcpy(buf + 14, buf + 20);			/* append am/pm */
    sprintf(buf + 100, "%2d:%02d%s", ftd & 0x01 ? td.a.hi
	    : (td.a.hi % 12 ? td.a.hi % 12 : 12), td.a.lo,
	    ftd & 0x01 ? "  " : (td.a.hi >= 12 ? "pm" : "am"));


    if (which_current != -1) {	/* just give one value, then quit or loop */
      switch (which_current) {
      case 0:
	printf("%s\n", format_data(F_TEMP, buf, &temp.in, ftemp));
	break;
      case 1:
	printf("%s\n", format_data(F_TEMP, buf, &temp.out, ftemp));
	break;
      case 2:
	printf("%s\n", format_data(F_HUMID, buf, &humid.in, 0));
	break;
      case 3:
	printf("%s\n", format_data(F_HUMID, buf, &humid.out, 0));
	break;
      case 4:
	printf("%s\n", format_data(F_TEMP1, buf, &dew.in, ftemp));
	break;
      case 5:
	printf("%s\n", format_data(F_TEMP1, buf, &dew.out, ftemp));
	break;
      case 6:
	printf("%s\n", format_data(F_PRESS, buf, &baro.sea, fbaro));
	break;
      case 7:
	printf("%s\n", format_data(F_PRESS, buf, &baro.local, fbaro));
	break;
      case 8:
	printf("%s\t%s\n",  format_data(F_SPEED, buf, &wind.gspeed, fspeed),
	       format_data(F_DIR, buf + 20, &wind.gdir, 0));
	break;
      case 9:
	printf("%s\t%s\n",  format_data(F_SPEED, buf, &wind.aspeed, fspeed),
	       format_data(F_DIR, buf + 20, &wind.adir, 0));
	break;
      case 10:
	printf("%s\n", format_data(F_RATE, buf, &rain.rate, fdepth));
	break;
      case 11:
	printf("%s\n", format_data(F_DEPTH, buf, &rain.yest, fdepth));
	break;
      case 12:
	printf("%s\n", format_data(F_DEPTH, buf, &rain.total, fdepth));
	break;
      default:
	fprintf(stderr, "bad current option\n");
	exit (1);
      }

      continue;			/* skip the rest of the loop */
    }

    printf("WX200 %xf: %20s:%-5d %s    Alarm: %s %s\n"
	   "%5s             Display: %8s/%d/%d"
	   "          Power: %s  Battery: %s\n"
	   "             "
	   "----------- Indoor -----------  ---------- Outdoor -----------\n",
	   num,
	   wx200host, wx200port, buf, buf + 100, offon[td.a.on],
	   VER, disp[gen.section], gen.screen, gen.subscreen,
	   acdc[gen.power], batt[gen.lowbat]);

    printf(format, "Temperature",
	   format_data(F_TEMP, buf, &temp.in, ftemp),
	   format_data(F_TEMP, buf + 20, &temp.inhi, ftemp),
	   format_data(F_TIME, buf + 40, &temp.inhi, ftd),
	   format_data(F_TEMP, buf + 60, &temp.out, ftemp),
	   format_data(F_TEMP, buf + 80, &temp.outhi, ftemp),
	   format_data(F_TIME, buf + 100, &temp.outhi, ftd),
	   format_data(F_TEMP, buf + 120, &temp.inlo, ftemp),
	   format_data(F_TIME, buf + 140, &temp.inlo, ftd),
	   format_data(F_TEMP, buf + 160, &temp.outlo, ftemp),
	   format_data(F_TIME, buf + 180, &temp.outlo, ftd),
	   format_val(F_TEMP1, buf + 200, temp.ina.nlo, ftemp),
	   format_val(F_TEMP1, buf + 220, temp.ina.nhi, ftemp),
	   offon[temp.ina.on],
	   format_val(F_TEMP1, buf + 240, temp.outa.nlo, ftemp),
	   format_val(F_TEMP1, buf + 260, temp.outa.nhi, ftemp),
	   offon[temp.outa.on]);

    printf(format, "Humidity",
	   format_data(F_HUMID, buf, &humid.in, 0),
	   format_data(F_HUMID, buf + 20, &humid.inhi, 0),
	   format_data(F_TIME, buf + 40, &humid.inhi, ftd),
	   format_data(F_HUMID, buf + 60, &humid.out, 0),
	   format_data(F_HUMID, buf + 80, &humid.outhi, 0),
	   format_data(F_TIME, buf + 100, &humid.outhi, ftd),
	   format_data(F_HUMID, buf + 120, &humid.inlo, 0),
	   format_data(F_TIME, buf + 140, &humid.inlo, ftd),
	   format_data(F_HUMID, buf + 160, &humid.outlo, 0),
	   format_data(F_TIME, buf + 180, &humid.outlo, ftd),
	   format_val(F_HUMID, buf + 200, humid.ina.nlo, 0),
	   format_val(F_HUMID, buf + 220, humid.ina.nhi, 0),
	   offon[humid.ina.on],
	   format_val(F_HUMID, buf + 240, humid.outa.nlo, 0),
	   format_val(F_HUMID, buf + 260, humid.outa.nhi, 0),
	   offon[humid.outa.on]);

    printf(format, "Dew Point",
	   format_data(F_TEMP1, buf, &dew.in, ftemp),
	   format_data(F_TEMP1, buf + 20, &dew.inhi, ftemp),
	   format_data(F_TIME, buf + 40, &dew.inhi, ftd),
	   format_data(F_TEMP1, buf + 60, &dew.out, ftemp),
	   format_data(F_TEMP1, buf + 80, &dew.outhi, ftemp),
	   format_data(F_TIME, buf + 100, &dew.outhi, ftd),
	   format_data(F_TEMP1, buf + 120, &dew.inlo, ftemp),
	   format_data(F_TIME, buf + 140, &dew.inlo, ftd),
	   format_data(F_TEMP1, buf + 160, &dew.outlo, ftemp),
	   format_data(F_TIME, buf + 180, &dew.outlo, ftd),
	   format_val(F_DIFF1, buf + 200, dew.ina.nlo, ftemp), "spread ",
	   offon[dew.ina.on],
	   format_val(F_DIFF1, buf + 220, dew.outa.nlo, ftemp), "spread ",
	   offon[dew.outa.on]);

    printf("Wind Gust:%s@%s  Wind Gust Hi:%s@%s %s\n"
	   "      Avg:%s@%s    Wind Chill:%s   Lo:%s %s\n"
	   "    Alarm:%12s %s           Alarm:%s %s\n\n",
	   format_data(F_DIR, buf, &wind.gdir, 0),
	   format_data(F_SPEED, buf + 20, &wind.gspeed, fspeed),
	   format_data(F_DIR, buf + 40, &wind.dirhi, 0),
	   format_data(F_SPEED, buf + 60, &wind.speedhi, fspeed),
	   format_data(F_TIME, buf + 80, &wind.speedhi, ftd),
	   format_data(F_DIR, buf + 100, &wind.adir, 0),
	   format_data(F_SPEED, buf + 120, &wind.aspeed, fspeed),
	   format_data(F_TEMP1, buf + 140, &chill.temp, ftemp),
	   format_data(F_TEMP1, buf + 160, &chill.low, ftemp),
	   format_data(F_TIME, buf + 180, &chill.low, ftd),
	   format_val(F_SPEED, buf + 200,  wind.a.nhi, fspeed),
	   offon[wind.a.on],
	   format_val(F_TEMP1, buf + 220,  chill.a.nlo, ftemp),
	   offon[chill.a.on]);

    printf("Rain Rate:%s  Yesterday:%s Total:%s since %s\n"
	   "    Alarm:%s %s\n\n",
	   format_data(F_RATE, buf, &rain.rate, fdepth),
	   format_data(F_DEPTH, buf + 20, &rain.yest, fdepth),
	   format_data(F_DEPTH, buf + 40, &rain.total, fdepth),
	   format_data(F_TIME, buf + 60, &rain.total, ftd),
	   format_val(F_RATE, buf + 80, rain.a.nhi, fdepth),
	   offon[rain.a.on]);

    printf("Barometer: %s at %s %s sea; 12-24hr forecast: %s\n"
	   "    Alarm:%s %s"
	   "      *=Out of Range, !=Sensor Error, #=Both * and !\n",
	   trend[baro.trend - 1],
	   format_data(F_PRESS, buf, &baro.local, fbaro),
	   format_data(F_PRESS, buf + 20, &baro.sea, fbaro),
	   pred[baro.pred - 1],
	   format_val(F_PRESS, buf + 40, baro.a.nhi, fbaro),
	   offon[rain.a.on]);

  } while (opt_r && (num = wx200bufread(socket)) != EOF);

  if (query)
    printf("</pre>
<hr>
Weather by <a href=\"http://www.bobsplace.com/~twitham/wx200d/\">
wx200d</a> version <b>%s</b>.
</body></html>
", VER);

  exit(wx200close(socket));
}
