/*
 * @(#)$Id: wx200.h,v 1.18 1998/05/03 06:27:14 twitham Rel $
 *
 * Copyright (C) 1998 Tim Witham <twitham@pcocd2.intel.com>
 *
 * (see the files README and COPYING for more details)
 *
 * This file defines common WX200 data structures and function prototypes.
 *
 */

/* get build-time config options */
#include "config.h"

#define GROUPS	16		/* number of group buffers, indexes: */

#define GROUP0	0
#define GROUP1	1
#define GROUP2	2
#define GROUP3	3
#define GROUP4	4
#define GROUP5	5
#define GROUP6	6
#define GROUP7	7
#define GROUP8	8		/* Time, Humidity */
#define GROUP9	9		/* Temperature */
#define GROUPA	10		/* Barometer, Dew Point */
#define GROUPB	11		/* Rain */
#define GROUPC	12		/* Wind, Wind Chill */
#define GROUPD	13		/* (reserved place holder) */
#define GROUPE	14		/* (reserved place holder) */
#define GROUPF	15		/* Time */

#define LENMAX	38		/* max buffer lenth, lengths: */
#define LENGTHS	{0, 0, 0, 0, 0, 0, 0, 0, 35, 34, 31, 14, 27, 0, 0, 5}

extern char *progname;		/* files with main() should provide these */
extern char error[];		/* for error messages */

extern int init_serial(char *);	/* serial.c */
extern void cleanup_serial(int);

extern int wx200open(char *);	/* client.c */
extern int wx200close(int);
extern char wx200host[];
extern int wx200port;

extern void wx200bufinit();	/* buffer.c */
extern int wx200bufread(int);
extern unsigned char groupbuf[GROUPS * 2][LENMAX];
extern int grouplen[GROUPS];

extern int wx200parse(int);	/* parse.c */

/* byte converting macros for improved readability and efficiency */
#define LO(byte)	(byte & 0x0f)
#define HI(byte)	((byte & 0xf0) >> 4)
#define NUM(byte)	(10 * HI(byte) + LO(byte))
#define BIT(byte, bit)	((byte & (1 << bit)) >> bit)
#define ERROR(mem, byte, value)	if (!((mem.err) = (byte == 0xee))) \
				    (mem.val) = (value);

/* raw int to base float unit conversions */
#define TENTHS(val)	(val * 0.1)
#define F2C(val)	((val - 32.0) / 1.8)
#define IN2MM(val)	(val * 25.4)
#define MPH2MPS(val)	(val * 0.44704)

extern int wx200dserver;	/* whether we're a server (1) or client (0) */

typedef struct SENS {		/* sensor or hi or lo memory */
  int val;			/* raw integer value of sensor or memory */
  float n;			/* base float unit value */
  int or;			/* out of range flag */
  int err;			/* sensor error flag, unused by memories */
  int sec;			/* second used only by the clock */
  int min;			/* time of memory, unused by sensors */
  int hour;
  int day;
  int mon;
} SENS;
#define HILO struct SENS

typedef struct ALRM {		/* alarm */
  int hi;
  float nhi;
  int lo;
  float nlo;
  int on;			/* on flag */
} ALRM;

typedef struct Time {		/* Time */
  SENS clock;
  ALRM a;			/* hi=HH, lo=MM */
  int format;
} Time;
extern Time td;

typedef struct Humid {		/* Humidity, n = percent */
  SENS in;			/* 10<val<97 % @ 1 */
  HILO inhi;
  HILO inlo;
  ALRM ina;
  SENS out;			/* 10<val<97 % @ 1 */
  HILO outhi;
  HILO outlo;
  ALRM outa;
} Humid;
extern Humid humid;

typedef struct Temp {		/* Temperature, n = degrees Celsius */
  SENS in;			/* 0<val<500 .1 degrees C @ 1 */
  HILO inhi;
  HILO inlo;
  ALRM ina;			/* 32<lo/hi<122 degrees F @ 1 */
  SENS out;			/* -40<val<600 .1 degrees C @ 1 */
  HILO outhi;
  HILO outlo;
  ALRM outa;			/* -40<val<140 degrees F @ 1 */
  int format;
} Temp;
extern Temp temp;

typedef struct Baro {		/* Barometer, n = millibars */
  SENS local;			/* 7950<val<10500 .1 mb @ 10 */
  SENS sea;			/* 7950<val<10500 .1 mb @ 1 */
  int format;
  int pred;
  int trend;
  ALRM a;			/* 1<hi<16 mb @ 1, lo unused */
} Baro;
extern Baro baro;

typedef struct Dew {		/* Dew Point, n = degrees Celsius */
  SENS in;			/* 0<val<47 degrees C @ 1 */
  HILO inhi;
  HILO inlo;
  ALRM ina;			/* 1<lo<16 degrees C @ 1, hi unused */
  SENS out;			/* 0<val<56 degrees C @ 1 */
  HILO outhi;
  HILO outlo;
  ALRM outa;			/* 1<lo<16 degrees C @ 1, hi unused */
} Dew;
extern Dew dew;

typedef struct Rain {		/* Rain, n = mm/hr or mm */
  SENS rate;			/* 0<val<998 mm/hr @ 1 */
  HILO yest;			/* 0<val<9999 mm @ 1, timestamp not used */
  HILO total;			/* 0<val<9999 mm @ 1 */
  int format;
  ALRM a;			/* 0<hi<393 .1 in/hr @ 1, lo unused */
} Rain;
extern Rain rain;

typedef struct Wind {		/* Wind, n = meters per second or degrees */
  SENS gspeed;			/* 0<val<560 .1 m/s @ 2 */
  SENS gdir;			/* 0<val<359 compass degrees @ 1 */
  SENS aspeed;
  SENS adir;
  HILO speedhi;
  HILO dirhi;			/* timestamp not used, get it from speedhi */
  int format;
  ALRM a;			/* 0<hi<56 m/s @ 1, lo unused */
} Wind;
extern Wind wind;

typedef struct Chill {		/* Wind Chill, n = degrees Celsius */
  SENS temp;			/* -85<val<60 degrees C @ 1 */
  HILO low;
  ALRM a;			/* -121<lo<140 degrees F @ 1, hi unused */
} Chill;
extern Chill chill;

typedef struct Gen {
  int power;			/* 0=AC, 1=DC */
  int lowbat;			/* 0=good, 1=low */
  int section;			/* 0<section<7 */
  int screen;			/* 0<screen<3 */
  int subscreen;		/* 0<subscreen<3 */
} Gen;
extern Gen gen;

typedef struct Unk {		/* bytes that may contain some unknown bits */
  unsigned char eightf5;
  unsigned char eightf32;
  unsigned char eightf33;
  unsigned char ninef13;
  unsigned char ninef15;
  unsigned char ninef28;
  unsigned char ninef30;
  unsigned char ninef31;
  unsigned char af5;
  unsigned char af6;
  unsigned char af28;
  unsigned char af29;
  unsigned char bf2;
  unsigned char bf10;
  unsigned char bf12;
  unsigned char cf14;
  unsigned char cf15;
  unsigned char cf21;
  unsigned char cf23;
  unsigned char cf24;
  unsigned char cf25;
} Unk;
extern Unk unk;

#define TABLEN	256		/* tab.c */
extern char tabbuf[TABLEN];
extern void wx200tabinit();
extern int wx200tab(int);

extern float unit_humid(float, int); /* unit.c */
extern float unit_temp(float, int);
extern float unit_press(float, int);
extern float unit_depth(float, int);
extern float unit_speed(float, int);
extern int unit_dir(float, int);

#define F_HUMID	0		/* format.c */
#define F_TEMP	1		/* tenths for temp */
#define F_TEMP1	2		/* units for dew point and wind chill */
#define F_DIFF1	3		/* temperature diff for dew point alarm */
#define F_PRESS	4
#define F_DEPTH	5
#define F_RATE	6
#define F_SPEED	7
#define F_DIR	8
#define F_TIME	9

typedef struct UNIT {
  int w;			/* width */
  int p;			/* precision */
  char *l;			/* wx200 display label */
  char *d;			/* text description */
} UNIT;

extern UNIT humid_unit[];
extern UNIT temp_unit[];
extern UNIT temp1_unit[];
extern UNIT press_unit[];
extern UNIT depth_unit[];
extern UNIT rate_unit[];
extern UNIT speed_unit[];
extern char *wind_unit4[];
extern char *wind_unit8[];
extern char *wind_unit16[];
extern char *format_data(int, char *, SENS *, int);
extern char *format_val(int, char *, float, int);
