#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <unistd.h>

#include "mdm_lib.h"


int mdm_modem;		/* el fd del modem */
static struct termios old;
static char mdm_dev[10];
int mdm_verboso= 0;
FILE *mdm_log= stdout;

#define MDM_CAD_RESET "ATZ"
#define MDM_SEG_ESPERA_DTR 2
#define MDM_DLE 0x10


/**********************************/
/* APERTURA Y CIERRE DE mdm_modem */
/**********************************/

/* dev es el nombre del dispositivo _sin_ /dev/, es decir, 'ttyS1' por
   ejemplo. vel_puerto est en mdm_lib.h */
   
int mdm_inicia(char *dev, int vel_puerto)
	{
	struct termios termConfig;
	char fich_lock[255], fich_dev[255];
	FILE *f;
	
	if(mdm_verboso)
		fprintf(mdm_log,"iniciando\n");
	
	sprintf( fich_lock, "/var/lock/LCK..%s", dev );
	if( !access(fich_lock, F_OK) ) {
		fprintf( stderr, "dispositivo %s bloqueado (%s)\n", dev, fich_lock );
		return(MDM_ERR_BLOQUEADO);
		}
		
	if( (f= fopen(fich_lock, "wt" ))==NULL ) {
		fprintf( stderr, 
			"fallo al abrir fichero de bloqueo (en /var/lock): %s\n",
			strerror(errno) );
		return(MDM_ERR_FICH_LOCK);
		}
		
	fprintf( f, "%10d\n", getpid() );
	fclose(f);

	sprintf( fich_dev, "/dev/%s", dev );
	if( (mdm_modem= open(fich_dev, O_RDWR | O_NOCTTY) )< 0) {
		fprintf( stderr, "fallo al abrir el dispositivo del modem: %s\n",
			strerror(errno) );
		return MDM_ERR_FICH_DEV;
		}

	strcpy(mdm_dev, dev);
	
	tcgetattr(mdm_modem, &old); /* salvar config. */

	termConfig.c_iflag= IGNBRK;
	termConfig.c_oflag= 0;
	termConfig.c_lflag= 0;
/*  termConfig.c_cflag= CSTOPB | CREAD | CLOCAL | CS8;*/
	termConfig.c_cflag= CSTOPB | CREAD | CLOCAL | CRTSCTS | CS8;
	
	termConfig.c_cc[VMIN]= 1;
	termConfig.c_cc[VTIME]= 0;
	
	if (cfsetispeed(&termConfig,vel_puerto)< 0)	{
		fprintf( stderr, "fallo en cfsetispeed()\n" );
		return MDM_ERR_SET;
		}

	if (cfsetospeed(&termConfig,vel_puerto)< 0) {
		fprintf( stderr, "fallo en cfsetospeed()\n" );	
		return MDM_ERR_SET;
		}

	tcflush(mdm_modem, TCIFLUSH);

	if(tcsetattr(mdm_modem,TCSANOW,&termConfig)< 0) {
		fprintf( stderr, "fallo en tcsetattr()\n" );
		return MDM_ERR_SET;
		}

	return MDM_ERR_OK;
	}


int mdm_termina(void)
	{
	char buffer[80];
	char fich_lock[255], fich_dev[255];
	int bits= TIOCM_DTR;
	
	if(mdm_verboso)
		fprintf( mdm_log, "terminando\n");

	tcflush(mdm_modem, TCIOFLUSH);
	
	/* echamos la DTR abajo (MDM_SEG_ESPERA_DTR) segundos y arriba */
	ioctl( mdm_modem, TIOCMBIC, &bits );	
	sleep(MDM_SEG_ESPERA_DTR);
	ioctl( mdm_modem, TIOCMBIS, &bits );	
	
	sleep(1);
	
	if( mdm_escribe_espera( MDM_CAD_RESET, 5, buffer, "*") ) {
		fprintf( stderr, "error al resetear en mdm_termina()\n" );
		return MDM_ERR_RESET;
		}
	
	tcsetattr(mdm_modem, TCSANOW, &old);
	close(mdm_modem);
	
	sprintf( fich_lock, "/var/lock/LCK..%s", mdm_dev );
	if( access(fich_lock, F_OK) )
		fprintf( stderr, "fichero de bloqueo no hallado (%s) :-?", fich_lock );
	else
		unlink(fich_lock);
		
	strcpy(mdm_dev, "");

	return MDM_ERR_OK;
	}


/**********************************/
/* ESCRITURA Y LECTURA BAJO NIVEL */
/**********************************/

void mdm_escribe( char *cad )
	{
	int l;
	
	if(mdm_verboso)
		fprintf(mdm_log, "-> '%s'\n", cad);
	
	l= strlen(cad);
	
	write( mdm_modem, cad, l );
	write( mdm_modem, "\r", 1 );
/*	tcdrain(mdm_modem);*/
	}


void mdm_escribebyte( char car )
	{
	if(mdm_verboso)
		fprintf(mdm_log, "-> '%c'[%d]\n", car, car);
	
	write( mdm_modem, &car, 1 );
	}
	

/* mdm_lee: plazo en segundos. Buf (debe estar ya reservado) almacenar
	la lnea leda.
   Retorno: 0 si OK, MDM_ERR_TIMEOUT si timeout, MDM_ERR_SELECT si error
	en select
*/
int mdm_lee( int plazo, char *buf )
	{
	struct timeval timeout;
	fd_set readset;
	int r;
	char *p= buf;

	while(1) {
		timeout.tv_sec = plazo;
		timeout.tv_usec = 0;
							
		FD_ZERO(&readset);
		FD_SET(mdm_modem, &readset);

		if(select(mdm_modem+1,&readset,0,0,&timeout) == -1) {
			fprintf( stderr, "error en mdm_lee(): %s\n", 
				strerror(errno) );
			return MDM_ERR_SELECT;
			}
		
		if(FD_ISSET(mdm_modem, &readset)) {
			r= read( mdm_modem, p++, 1 );
			if( *(p-1)=='\n' ) {
				*p= 0;
				return MDM_ERR_OK;
				}
			}
		else
			return MDM_ERR_TIMEOUT; 
		}	
	}


/* mdm_leebyte: retorna en *byte el car. ledo.
   Retorno: 0 si OK, MDM_ERR_TIMEOUT si timeout, MDM_ERR_SELECT si error
	en select
*/
int mdm_leebyte( int plazo, char *byte )
	{
	struct timeval timeout;
	fd_set readset;
	int r;
	char b;

	timeout.tv_sec = plazo;
	timeout.tv_usec = 0;
						
	FD_ZERO(&readset);
	FD_SET(mdm_modem, &readset);

	if (select(mdm_modem+1,&readset,0,0,&timeout) == -1) {
			fprintf( stderr, "error en mdm_leebyte(): %s\n", 
				strerror(errno) );
			return MDM_ERR_SELECT;
			}
		
	if (FD_ISSET(mdm_modem, &readset)) {
		r= read( mdm_modem, &b, 1 );
		*byte= b;

		if(mdm_verboso)
			fprintf( mdm_log, "<- '%c' [%d]\n", b, b);
		
		return MDM_ERR_OK;
		}
	else 
		return MDM_ERR_TIMEOUT;
	}


/***********/
/* DIALOGO */
/***********/


/* mdm_espera: plazo en segundos, buf contendr el resultado, cad es
	"<c1>|<c2>|...|<cn>" o "*".
   Retorno: 0 si OK, MDM_ERR_TIMEOUT si timeout.
*/
int mdm_espera( int plazo, char *buf, char *cad )
	{
	int res;
	char *r, cad_esperada[100];
	
	if(mdm_verboso)
		fprintf(mdm_log, "esperando '%s'...", cad);
	
	while( (res=mdm_lee(plazo, buf))==0 ) {
		if( !strcmp(cad,"*") )	/* aceptar lo que sea */
			return MDM_ERR_OK;
		
		strncpy(cad_esperada,cad,100); /* hay que 
			restaurarla cada vez */
	
		r= strtok(cad_esperada,"|");
		while(r!=NULL) {
			if( !strncmp(buf,r,strlen(r)) ) {
				if(mdm_verboso)
					fprintf( mdm_log, "la tengo\n" );
		
				return MDM_ERR_OK;
				}
	
			r= strtok(NULL,"|");
			}
		}

	if(mdm_verboso)
		fprintf( mdm_log, "no hallada\n");
	
	return MDM_ERR_TIMEOUT;
	}


int mdm_escribe_espera( char *escribe, int plazo, char *buf, char *espera )
	{
	mdm_escribe(escribe);
	return( mdm_espera( plazo, buf, espera ) );
	}
	

/* mdm_dialoga: dlg es estructura definida en mdm_lib.h, buf contendr 
	la lnea recibida
   Retorno: 0 si OK, n+1 si error en la lnea n del dilogo (la primera
	es la n 1)
*/   
int mdm_dialoga( struct mdm_dialogo dlg[], char *buf )
	{
	int n= 0;
	
	while(dlg[n].escribe!=NULL) {
		if( mdm_escribe_espera( dlg[n].escribe, dlg[n].plazo, 
			buf, dlg[n].espera ) )
			return(n+1);
		n++;
		}
	
	return 0;
	}
	
	
/************/	
/* FICHEROS */
/************/

/* mdm_graba: 
	f: FD del fichero a grabar.
	dtmfs: cadena (de longitud adecuada) que contendr los 
	DTMFS que se reciban. 
	para_si_tecla: determina si se retorna al pulsar el usuario una tecla. 
	dtmfs_parar: si se recibe un DTMF que est en esta cadena, se 
	retornar.
   Retorno: nmero de bytes ledos.
*/
unsigned long mdm_graba(int f, char *dtmfs, int para_si_tecla,
	char *dtmfs_parar)
	{
	char c, cant, aux[2];
	int anterior_fue_DLE= 0, ok;
	unsigned long cuenta= 0L;

	dtmfs[0]= 0;
	
	while( read(mdm_modem,&c,1) ) {
		if(anterior_fue_DLE) {
			anterior_fue_DLE= 0;

			if(c==MDM_DLE)
				write(f, &c, 1);
			else {
				printf("DLE ");
				sprintf(dtmfs, "%s%c", dtmfs, c);
				if(dtmfs_parar)
					if( strchr(dtmfs_parar, c) )
						return cuenta;
				}
			}
		else
			if(c==MDM_DLE)
				anterior_fue_DLE= 1;
			else
				write(f, &c, 1);

		cuenta++;
		}

/*	cant= 0;
	ok= read( mdm_modem, &c, 1 );
	while(ok==1) {
		write( f, &c, 1 );
		cuenta++;
			
		if( cant==0x10 && c=='*') {
			printf( "//FIN DE RECEPCION//\n");
			break;
			}

		cant= c;
		ok= read( mdm_modem, &c, 1 );
		}
*/
	return cuenta;
	}


/* mdm_reproduce: 
	f: FD del fichero a reproducir. 
	dtmfs: cadena (de longitud adecuada) que contendr los 
	DTMFS que se reciban. 
	para_si_tecla: determina si se retorna al pulsar el usuario una tecla. 
	dtmfs_parar: si se recibe un DTMF que est en esta cadena, se 
	retornar.
   Retorno: nmero de bytes escritos.
*/
unsigned long mdm_reproduce(int f, char *dtmfs, int para_si_tecla,
	char *dtmfs_parar)
	{
	unsigned long cuenta= 0L;
	int r;
	char c; 

	dtmfs[0]= 0;

	r= read(f, &c, 1);
	
	while(r==1) {
		if( write(mdm_modem,&c,1)==1 ) {
			r= read(f, &c, 1);
			cuenta++;
			}
		else 
			if(errno==EAGAIN)
				printf( "EAGAIN\n" );
			else {
				perror("en mdm_reproduce");
				return  cuenta;
				}
		}

	return cuenta;
	}
