/*
 * $Id: libconf.c,v 1.10 1998/02/17 09:51:55 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/libconf/libconf.c,v $
 * $Revision: 1.10 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>

#include "libconf.h"

#define MAXBUF   512

static char buf[MAXBUF];   /* Global buffer */

static Bool  readLine( FILE* f );
static void  add_comment( char**, char* );
static Bool  is_comment( char* s );
static Bool  is_group( char* s );
static Bool  is_item( char* s );
static char* strip_spaces( char* s );
static Bool  read_groups( FILE* f, conf_file* cf );
static Bool  read_items( FILE* f, conf_file* cf, grp* group );
static int   compare( conf_file* cf, char* s1, char* s2 );
static int   Strlen( char *s );
static void  clear_error( conf_file* cf );
static void  set_error( conf_file* cf, confErr err );

/*    Open Close Write configuration                              */
 
/*
 * Function: open_conf
 *
 * Description:
 *   This function opens the configuration file 'file'
 *
 * Arguments:
 *   - const char*   file   The configuration file to open
 *
 * Returns:
 *   - conf_file* f      The opened configuration file
 *   - conf_file* NULL   On error
 *
 * Remarks:
 * This function opens the configuration file and stores all
 * groups and items in datastructure. After reading the
 * file, the 'cursor' is located at the first item in the
 * first group.
 *
 */
conf_file* open_conf( const char* file )
{
   FILE* f;
   conf_file* cf;

   cf = (conf_file*)malloc( sizeof( conf_file) );
   if( cf == NULL )
      return NULL;
      
   clear_error( cf );
   cf->fileName = strdup( file );
   if( cf->fileName == NULL )
   {
      free( cf );
      return NULL;
   }
   
   case_significant( cf );
   cf->currGroup = NULL;
   cf->currItem  = NULL;
   cf->groups    = NULL;
   cf->comment   = NULL;
      
   f = fopen( file, "r" );
   if( f == NULL )
   {
      free( cf->fileName );
      free( cf );
      return NULL;
   }

   if( read_groups( f, cf ) == False )
   {
      close_conf( cf );
      return NULL;
   }      
   fclose( f );
   
   return cf;
}

/*
 * Function:   close_conf
 *
 * Description:
 *   This function closes the configuration file
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file to close
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error or cf == NULL
 *
 * Remarks:
 *   This funcion also free's the memory used to store
 *   items and groups, and the memory 
 *   used to store the conf_file structure.
 *
 */
Bool close_conf( conf_file* cf )
{
   if( cf == NULL )
      return False;

      
   while( !conf_empty( cf ) )
      del_group( cf, group_name( current_group( cf ) ) );

   if( cf->comment != NULL )
      free( cf->comment );
   cf->comment = NULL;
   cf->currGroup = NULL;
   cf->currItem= NULL;

   free( cf->fileName );
   free( cf );
   return True;
}


/*
 * Function:   write_conf
 *
 * Description:
 *   This function writes all groups and items
 *   stored in memory to the config_file.
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file to write
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error
 *
 * Remarks:
 *   This function only writes the data stored in memory,
 *   which means that comments, additional whitespaces and
 *   newlines are lost!
 *
 */
Bool write_conf( conf_file* cf )
{
   FILE* f;
   grp* gPtr;
   item* iPtr;
   
   if( cf == NULL )
      return False;
   
   f = fopen( cf->fileName, "w" );
   if( f == NULL )
      return False;
   if( cf->comment != NULL )
      fprintf( f, "%s\n", cf->comment );
   gPtr = goto_first_group( cf );
   while( gPtr != NULL )
   {
      fprintf( f, "[%s]\n", group_name( current_group( cf ) ) );
      if( gPtr->comment != NULL )
         fprintf( f, "%s\n", gPtr->comment );
      iPtr = goto_first_item( cf );
      while( iPtr != NULL )
      {
         fprintf( f, "   %s = %s\n", item_key( current_item( cf ) ),
                                     item_value( current_item( cf ) ) );
         if( iPtr->comment != NULL )
            fprintf( f, "%s\n", iPtr->comment );
         iPtr = goto_next_item( cf );
      }
      gPtr = goto_next_group( cf );
   }
   fclose( f );
   
   return True;
}

/*  Init conf: read groups and items                              */
 
/*
 * Function: read_groups
 *
 * Description:
 *   This function reads all groups and items 
 *   from configuration file, and stores them in memory
 *
 * Arguments:
 *   - FILE*        f   The Config. file to read from
 *   - conf_file*   cf  The structure to store data in
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error
 *
 */
static Bool read_groups( FILE* f, conf_file* cf )
{
   grp* tmp;
   char* ptr;

   assert( f != NULL && cf != NULL );
   
   fseek( f, SEEK_SET, 0 );
   
   tmp = NULL;
   
   if( readLine( f ) == False )
      return False;
   
   while( True )
   {
      if( is_group( buf ) )
      {
         if( cf->groups == NULL )
         {
            cf->groups = ( grp*) malloc( sizeof( grp ) );
            if( cf->groups == NULL )
               return False;
            tmp = cf->groups;

         }
         else
         {
            tmp->next = (grp*)malloc( sizeof( grp ) );
            if( tmp->next == NULL )
               return False;
            tmp = tmp->next;
         }
         tmp->next = NULL;
         tmp->items = NULL;
         tmp->comment = NULL;
         ptr = strchr( buf, ']' );
         *ptr = '\0';
         tmp->grpName = strdup( buf + 1 );
         if( tmp->grpName == NULL )
            return False;
         if( read_items( f, cf, tmp ) == False )
            return False;
      }
      else
      {
         if( is_comment( buf ) )
         {
            if( cf->groups == NULL )
               add_comment( &cf->comment, buf );
            else
               add_comment( &tmp->comment, buf );
         }
         if( readLine( f ) == False )
            break;
      }

   }
   goto_first_group( cf );
   return True;
}

/*
 * Function: read_items
 *
 * Description:
 *   This function reads all items of current
 *   group, and stores them in memory
 *
 * Arguments:
 *   - FILE*        f       Config. file to read from
 *   - conf_file*   cf      Structure to store data in
 *   - grp*         group   Group where items should be stored
 *
 * Returns:
 *   - Bool   True   On success, all items are stored
 *   - Bool   False  On error
 *
 */
static Bool read_items( FILE* f, conf_file* cf, grp* group )
{
   item* tmp;
   char* ptr;

   assert( f != NULL && cf != NULL && group != NULL );
   
   tmp = NULL;
   
   while( readLine( f ) == True )
   {
      if( is_group( buf ) )
         return True;
         
      if( is_item( buf ) )
      {
         if( tmp == NULL )
         {
            group->items = (item*)malloc(sizeof( item ) );
            if( group->items == NULL )
               return False;
            tmp = group->items;
         }      
         else
         {
            assert( tmp != NULL );
            tmp->next = (item*)malloc( sizeof( item ) );
            if( tmp->next == NULL )
               return False;
            tmp = tmp->next;
         }
         
         tmp->next = NULL;
          tmp->comment = NULL;
            
         ptr = strchr( buf, '=' );
         if( ptr == NULL )
            return False;
            
         *ptr = '\0';
         
         tmp->value = strdup( strip_spaces( ptr + 1 ) );
         if( tmp->value == NULL )
            return False;
         tmp->key = strdup( strip_spaces( buf ) );
         if( tmp->key == NULL )
            return False;
      }
      else
      if( is_comment( buf ) )
      {
         if( tmp == NULL )
            add_comment( &group->comment, buf );
         else
            add_comment( &tmp->comment, buf );
      }
   }
   
   return True;
}

/* Goto:  group/item                                              */
/* Query: current group/item, group/conf empty                    */

/*
 * Function:   conf_empty
 *
 * Description:
 *   This function checks if configuration file is empty
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file we check
 *
 * Returns:
 *   - Bool   True   Config. file is empty
 *   - Bool   False  Config. file is not empty
 *
 * Remarks:
 *   When cf == NULL we also return True
 *
 */
Bool conf_empty( conf_file* cf )
{
   if( cf == NULL )
      return True;
   return ( cf->groups == NULL );
}

/*
 * Function:   group_empty
 *
 * Description:
 *   This functions checks if current group
 *   has one or more items stored
 *
 * Arguments:
 *   - conf_file* cf   Config. file we check
 *
 * Returns:
 *   - Bool   True   Group is empty
 *   - Bool   False  Group is not empty
 *
 * Remarks
 *   If cf == NULL, or no groups are stored,
 *    we also return True
 *
 */
Bool group_empty( conf_file* cf )
{
   if( cf == NULL )
      return True;
   if( cf->currGroup == NULL )
      return True;
   
   return cf->currGroup->items == NULL;
}

/*
 * Function:   current_group
 *
 * Description:
 *   This functions returns a pointer to
 *   the current group
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file we're talking about
 *
 * Returns:
 *   - grp*   cg   Pointer to current group
 *   - grp*   NULL On error (Means, cf = empty )
 *
 */
grp* current_group( conf_file* cf )
{
   if( cf == NULL )
      return NULL;
      
   return cf->currGroup;
}

/*
 * Function: current_item
 *
 * Description:
 *   This function returns pointer to
 *   current item in current group
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file we're using
 *
 * Returns:
 *   - item*   ci   Pointer to current item in current group
 *   - item*   NULL On error, which means current group is empty
 *                  or configuration file is empty
 *
 */
item* current_item( conf_file* cf )
{
   if( cf == NULL )
      return NULL;
      
   return cf->currItem;
}

/*
 * Function: goto_group
 *
 * Description:
 *   This function changes the current group to gName
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *   - char*       gName The group name
 *
 * Returns:
 *   - grp*   ng   Pointer to the new group
 *   - grp*   NULL On error, gName does not exists
 *
 */
grp* goto_group( conf_file* cf, char* gName )
{
   grp* tmp;

   if( cf == NULL || gName == NULL )
      return NULL;
   if( cf->currGroup != NULL )
   if( compare( cf, group_name( cf->currGroup ), gName ) == 0 )
   {
      goto_first_item( cf );
      return cf->currGroup;
   }   
   tmp = cf->groups;
   
   while( tmp != NULL )
   {
      if( compare( cf,  group_name( tmp ), gName ) == 0 )
      {
         cf->currGroup = tmp;
         goto_first_item( cf );
         return tmp;
      }
      tmp = tmp->next;
   }

   return NULL;
}

/*
 * Function:   goto_item
 *
 * Description:
 *   This function changes the current item to iName
 *   in the current group
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *   - char*       iName The name of the item
 *
 * Returns:
 *   - item*   ni   The new current item
 *   - item*   NULL On error, item does not exists
 *
 */
item* goto_item( conf_file* cf, char* iName )
{
   item* tmp;
   
   if( cf == NULL || iName == NULL )
      return NULL;
      
   assert( cf->currGroup != NULL );
   
   if( cf->currItem != NULL )
      if( compare( cf, item_key( cf->currItem) , iName ) == 0 )
         return cf->currItem;
   
   tmp = cf->currGroup->items;
   
   while( tmp != NULL )
   {
      if( compare( cf,  item_key( tmp ), iName ) == 0 )
      {
         cf->currItem = tmp;
         return tmp;
      }
      tmp = tmp->next;
   }
   return NULL;
}

/*
 * Function:   goto_first_group
 *
 * Description:
 *   This functions changes current group
 *   to the first group in the configuration file
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * Returns:
 *   - grp*   ng   Pointer to the new current group
 *   - grp*   NULL On error, cf = empty
 *
 */
 
grp* goto_first_group( conf_file* cf )
{
   if( cf == NULL )
      return NULL;
      
   cf->currGroup = cf->groups;
   goto_first_item( cf );

   return cf->groups;
}

/*
 * Function: goto_next_group
 *
 * Description:
 *   This function changes the current group
 *   to the next group.
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file 
 *
 * Returns:
 *   - grp*   ng   The new current group
 *   - grp*   NULL On error, cf is empty
 *                 or no more groups
 *
 */
grp* goto_next_group( conf_file* cf )
{
   if( cf == NULL || cf->currGroup == NULL )
      return NULL;
   
   if( cf->currGroup->next == NULL )
      return NULL;

   cf->currGroup = cf->currGroup->next;
   goto_first_item( cf );

   return cf->currGroup;
}

/*
 * Function: goto_first_item
 *
 * Description:
 *   This function changes the current item
 *   to the first item in the current group
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * Returns:
 *   - item*   ni   Pointer to new current item
 *   - item*   NULL On error, cf or group is empty
 *
 */
item* goto_first_item( conf_file* cf )
{
   if( cf == NULL || cf->currGroup == NULL )
      return NULL;
   
   cf->currItem = cf->currGroup->items;
   return cf->currItem;
}

/*
 * Function:   goto_next_item
 *
 * Description:
 *   This function changes the current item
 *   to the next one.
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * Returns:
 *   - item*   ni   Pointer to new current item
 *   - item*   NULL On error, cf is empty 
 *                  or no more items
 *
 */
item* goto_next_item( conf_file* cf )
{
   if( cf== NULL )
      return NULL;
   if( cf->currItem == NULL )
      return NULL;
   
   cf->currItem = cf->currItem->next;
   
   return cf->currItem;
}


/* Get: groupName, itemKey, itemValue                             */
/*      int, string                                               */
 
/*
 * Function:   group_name
 *
 * Description:
 *   This function returns the name of group 'group'
 *
 * Arguments:
 *   - grp*   group   Pointer to group
 *
 * Returns:
 *   - char*  gName   Name of the group
 *   - char*  NULL    On error
 *
 */
char* group_name( grp* group )
{   
   if( group == NULL )
      return NULL;
   return group->grpName;
}

/*
 * Function:   item_key
 *
 * Description:
 *   This function returns the item name
 *   of the item
 *
 * Arguments:
 *   - item*   i   Pointer to item structure
 *
 * Returns:
 *   - char*   in   Name of item
 *   - char*   NULL On error
 *
 */
char* item_key( item* i )
{   
   if( i == NULL )
      return NULL;
   return i->key;
}

/*
 * Function:   item_value
 *
 * Description:
 *   This function returns the value of the item
 *
 * Arguments:
 *   - char*   iv   Value of item
 *   - char*   NULL On error
 *
 * Remarks:
 *   This function only returns string, no
 *   integers.
 *
 */
char* item_value( item* i )
{
   if( i == NULL )
      return NULL;
   return i->value;
}

/*
 * Function:   get_str
 *
 * Description:
 *   This function returns the string value that belongs
 *   to iName in the group 'gName'.
 *
 * Arguments:
 *   - conf_file*   cf      The configuration file
 *   - char*        gName   Name of group in which to search
 *   - char*        iName   Name of item to search for
 *   - char*        default Default value to return
 *
 * Returns:
 *   - char*   value   Value that was found
 *   - char*   default On error, or when item could not be found
 *
 * Remarks:
 *   When gName == NULL, iName is searched in current group
 *   When iName == NULL, Value of current item is returned
 *
 *   When default is returned (on error, or when item could
 *   not be found), err will be set to indicate the error.
 *   Otherwise err is set to noErr.
 *
 */
char* get_str( conf_file* cf, char* gName, char* iName, char* Default )
{
   clear_error( cf );
   
   if( cf == NULL )
   {
      set_error( cf, notFound );
      return Default;
   }
      
   if( gName != NULL )
      if( goto_group( cf, gName ) == NULL )
      {
         set_error( cf, notFound );
         return Default;
      }
   
   if( iName != NULL )
      if( goto_item( cf, iName ) == NULL )
      {
         set_error( cf, notFound );
         return Default;
      }
   return item_value( cf->currItem );
}

/*
 * Function:   get_int
 *
 * Description:
 *   This function returns the integer value that belongs
 *   to iName in the group 'gName'.
 * Arguments:
 *   - conf_file*   cf      The configuration file
 *   - char*        gName   Name of group in which to search
 *   - char*        iName   Name of item to search fot
 *   - int          default Default value to return
 *
 * Returns:
 *   - int     value   Value that was found
 *   - int     default On error, or when item could not be found
 *
 * Remarks:
 *   This function simply calls get_str( cg, gName, iName, NULL )
 *   And next, converts the returned string to an integer using
 *   strtol.
 *   See get_str for more details.
 *   Numbers in the configuration file can be in base 10 (default)
 *   base 8 (prefex 0) and base 16 (prefix 0x), just like
 *   the C language.
 *
 */
int get_int( conf_file* cf, char* gName, char* iName, int Default )   
{
   char* ptr;
   
   ptr = get_str( cf, gName, iName, NULL );
   if( ptr == NULL )
      return Default;

   return strtol( ptr, NULL, 0 );
}

/* Set: string, int                                               */

/*
 * Function:   set_str 
 *
 * Description:
 *   This function sets the string value of iName in
 *   gName to value. If gName does not exists it is created
 *   If iName does not exists it is created.
 *
 * Arguments:
 *   - conf_file*   cf      The configuration file
 *   - char*        gName   The group name to store value in
 *   - char*        iName   The item name which value sould be set
 *   - char*        value   The value to set
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error
 *
 * Remarks:
 *   If gName == NULL, iName will be searched in current group
 *   If iName == NULL, value of current item will be set
 *
 *   When needed, a new group or item is created.
 *
 */
Bool set_str( conf_file* cf, char* gName, char* iName, char* value )
{
   if( cf == NULL )
      return False;
      
   if( gName != NULL )
   {
      if( goto_group( cf, gName ) == NULL )
      {
         if( create_new_group( cf, gName ) == NULL )
            return False;
         goto_group( cf, gName );
      }
   }
   
   if( iName != NULL )
   {

      if( goto_item( cf, iName ) == NULL )
      {

         if( create_new_item( cf, iName ) == NULL )
            return False;
         goto_item( cf, iName );
      }
   }
   
   if( value != NULL   )
   {
      if( cf->currItem->value != NULL )
         free( cf->currItem->value );
      cf->currItem->value = strdup( value );
      if( cf->currItem->value == NULL )
         return False;
   }
   
   return True;
}

/*
 * Function:   set_int
 *
 * Description:
 *   This function sets the integer value of iName in
 *   gName to value. If gName does not exists it is created
 *   If iName does not exists it is created.
 *
 * Arguments:
 *   - conf_file*   cf      The configuration file
 *   - char*        gName   The group name to store value in
 *   - char*        iName   The item name which value sould be set
 *   - int          value   The value to set
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error
 *
 * Remarks:
 *   This function simply prints the value to a string, and
 *   calls set_str( cf, gName, iName, buf )
 *
 *   see set_str for more details
 *
 */
Bool set_int( conf_file* cf, char* gName, char* iName, int value )
{
   char buf[20];
   
   sprintf( buf, "%d", value );
   
   return set_str( cf, gName, iName, buf );
}

/* Create/Delete Group/Item                                        */

/*
 * Function:   create_new_group
 *
 * Description:
 *   This function creates a new group with name gName
 *   in front of all other groups.
 *
 * Arguments:
 *   - conf_file*   cf    The configuration file
 *   - char*        gName The group to create
 *
 * Returns:
 *   - grp*   ng   Pointer to new created group
 *   - grp*   NULL On error
 *
 * Remarks:
 *   This function does no checking wheter or not
 *   the group already exists.
 *
 */
grp* create_new_group( conf_file* cf, char* gName )
{
   grp* nGrp;

   if( cf == NULL )
      return NULL;
      
   nGrp = (grp*)malloc( sizeof( grp ) );
   if( nGrp == NULL )
      return NULL;

   nGrp->grpName = strdup( gName );
   if( nGrp->grpName == NULL )
   {
      free( nGrp );
      return NULL;
   }
      
   nGrp->items = NULL;
         
   nGrp->next = cf->groups;
   cf->groups = nGrp;
   
   return nGrp;
}

/*
 * Function:   create_new_item
 *
 * Description
 *   This function creates a new item with name iName
 *   in the current group in front of all other items
 *
 * Arguments:
 *   - conf_file*   cf      The configuration file
 *   - char*        iName   The item to create
 *
 * Returns:
 *   - item*   ni   The new created item
 *   - item*   NULL On error
 *
 * Remarks:
 *   This function does no checking wheter or not
 *   the item already exists.
 *
 */
item* create_new_item( conf_file* cf, char* iName )
{
   item* nItem;
   
   if( cf == NULL )
      return NULL;
   if( cf->currGroup == NULL )
      return NULL;
      
   nItem = (item*)malloc( sizeof( item ) );
   if( nItem == NULL )
      return NULL;

   nItem->key = strdup( iName );
   if( nItem->key == NULL )
   {
      free( nItem );
      return NULL;
   }
      
   nItem->value = NULL;
         
   nItem->next = cf->currGroup->items;
   cf->currGroup->items = nItem;

   return nItem;
}


/*
 * Function: del_group
 *
 * Description:
 * This function removes group with name gName and
 * all items stored in it from the configuration file.
 *
 * Arguments:
 *   - config_file*   cf     The configuration file
 *   - char*          gName  Name of group to remove
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error   (when gName could not be found)
 *
 * Remarks:
 *   If gName == NULL, the current group will be removed.
 *
 */
Bool del_group( conf_file* cf, char* gName )
{
   grp* ptr, *ptr1;

   if( cf == NULL )
      return False;
      
   if( gName != NULL )
      if( goto_group( cf, gName ) == NULL )
         return False;
      
   ptr = ptr1 = cf->groups;
   
   while( ptr && compare( cf, ptr->grpName, gName ) != 0 )
   {
      ptr1 = ptr;
      ptr = ptr->next;
   }
   
   if( ptr == NULL )
      return False;
   
   if( ptr->comment != NULL )
      free( ptr->comment );
   ptr->comment = NULL;
   
   while( goto_first_item( cf ) != NULL )
   {
      if( del_item( cf, NULL, item_key( goto_first_item( cf ) ) ) == False )
         break;
   }
   free( ptr->grpName );
            
   if( ptr != ptr1 )
      ptr1->next = ptr->next;
   else
      cf->groups = ptr->next;
      
   if( goto_group( cf, group_name( ptr1->next ) ) == NULL )
      if( goto_group( cf, group_name( ptr1 ) ) == NULL )
         goto_first_group( cf );
   
   free( ptr );
   
   return True;
}   
   
/*
 * Function:   del_item
 *
 * Description:
 *   This function removes an item from the configuration file.
 *
 * Arguments:
 *   - conf_file*   cf      The configuration file
 *   - char*        gName   The group from which to remove
 *   - char*        iName   The item to remove
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error (The item could not be found ).
 *
 * Remarks:
 *   If gName == NULL, iName will be searched in current group
 *   If iName == NULL, Current item will be removed
 *
 */
Bool del_item( conf_file* cf, char* gName, char* iName )
{
   item* ptr, *ptr1;

   clear_error( cf );
      
   if( cf == NULL )
      return False;
   
   if( cf->currGroup == NULL )
      return False;
      
   if( gName != NULL )
      if( goto_group( cf, gName ) == NULL )
         return False;
   
   ptr = ptr1 = cf->currGroup->items;
   
   while( ptr && compare( cf, ptr->key, iName ) != 0 )
   {
      ptr1 = ptr;
      ptr = ptr->next;
   }
   
   if( ptr == NULL )
      return False;
   free( ptr->key );
   free( ptr->value );
   
   if( ptr != ptr1 )
   {
      if( ptr->comment != NULL )
         add_comment( &ptr1->comment, ptr->comment );
      ptr1->next = ptr->next;
   }
   else
   {
      if( ptr->comment != NULL )
         add_comment( &cf->currGroup->comment, ptr->comment );
      cf->currGroup->items = ptr->next;
   }
   if( ptr->comment != NULL )
      free( ptr->comment );
   
   if( goto_item( cf, item_key( ptr1->next ) ) == NULL )
      if( goto_item( cf, item_key( ptr1 ) ) == NULL )
         goto_first_item( cf );
         
   free( ptr );
   
   return True;
}


/* Misc.                                                          */


/*
 * Function: readLine
 *
 * Description:
 *   This function reads a line from the file f,
 *   and removes spaces at the beginning and the end of the line.
 *   The line is stored in global buffer buf
 *
 * Arguments:
 *   - FILE*   f   The file to read from
 *
 * Returns:
 *   - Bool   True   On success
 *   - Bool   False  On error
 *
 */ 
static Bool readLine( FILE* f )
{
   buf[0] = EOF;
   if( fgets( buf, MAXBUF, f ) == NULL )
      return False;
   
   buf[Strlen(buf) - 1 ] = '\0';
   strip_spaces( buf );
   return True;
}

/*
 * Function:   is_comment
 *
 * Description:
 *   This functions determines if a line
 *   is a comment (first no white character is '#')
 *
 * Arguments:
 *   char*   s   The line to test
 *
 * Returns:
 *   Bool   True   Yes, the line is comment
 *   Bool   False  No, the line is not a comment
 *
 */
static Bool is_comment( char* s )
{
   if( s && 
      (s[0] == '#' ||
       strlen( s ) == 0 ) ) return True;
   return False;
}

/*
 * Function: is_group
 *
 * Description:
 *   This function determines if a line is a group
 *   ( first non white character '[' last character ']' )
 *
 * Arguments:
 *   - char*   s   The line to check
 *
 * Returns:
 *   - Bool   True   Yes, the line is a group
 *   - Bool   false  No, the line is not a group
 *
 */
static Bool is_group( char* s )
{
   int n;
   
   n = strlen( s );
   
   
   if( s && n &&
       s[0] == '[' &&
       s[n - 1] == ']'
      )
      return True;
   return False;
}

/*
 * Function:   is_item
 *
 * Description:
 *   This function determines if a line is an item.
 *   A line that is not a group or comment, and has
 *   an equal sign ('=') is an item.
 *
 * Arguments:
 *   - char*   s   The line to check
 *
 * Returns:
 *   - Bool   True   Yes, the line is an item
 *   - Bool   False  No, the line is not an item
 *
 */
static Bool is_item( char* s )
{
   if( !is_comment( s ) &&
       !is_group( s ) &&
       strchr( s, '=' ) != NULL
      )
      return True;
   return False;
}

/*
 * Function:   strip_spaces
 *
 * Description:
 *   This function removes all white-spaces at
 *   the beginning and the end of the string s/
 *
 * Arguments:
 *   char*   s   The string we strip
 *
 * Returns:
 *   char*   s   Pointer to stripped string
 *
 */
static char* strip_spaces( char* s )
{
   int count;
   int i;
   char* ptr;
   count = 0;
   ptr = s;
   
   while( *ptr && isspace( *ptr ) )
   {
      ptr++;
      count++;
   }

   if( count > 0 )
      for( i = 0; i < Strlen( s ) - count + 1; i++ )
         s[i] = s[i+count];
   if( Strlen( s ) == 0 )
      return s;
      
   ptr = s + Strlen( s ) - 1;
   while( ptr > s && isspace( *ptr ) )
      *ptr-- = '\0';
   return s;
}    

/*
 * Function:   compare
 *
 * Description:
 *   This function compares two strings according
 *   to case_significant in the configuration file cf
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *   - char*        s1   The first string
 *   - char*        s2   The second string
 *
 * Returns:
 *   - int    0   s1 == s2
 *   - int    >0  s1 >  s2
 *   - int    <0  s1 < s2
 *
 * Remarks:
 *   when cf->case_significant == TRUE, strcmp is used
 *   otherwise, strcasecmp is used
 *
 */
static int compare( conf_file* cf, char* s1, char* s2 )
{
   if( cf == NULL || cf->case_significant )
      return strcmp( s1, s2 );
   else
      return strcasecmp( s1, s2 );
      
}

/*
 * Function:   case_significant
 *
 * Description:
 *   This function sets cf->case_significant = True
 *   Wich means that all string compares are case 
 *   sensitive
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * Returns:
 *   - None
 *
 */
void case_significant( conf_file* cf )
{
   if( cf != NULL )
      cf->case_significant = True;
}

/*
 * Function:   case_insignificant
 *
 * Description:
 *   This function sets cf->case_significant = False
 *   Wich means that all string compares are not case 
 *   sensitive
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * Returns:
 *   - None
 *
 */
void case_insignificant( conf_file* cf )
{
   if( cf != NULL )
      cf->case_significant = False;
}

/* 
 * Function:   Strlen
 *
 * Description:
 *   On my computer, strcmp(NULL) returns 3.
 *   Since few functions test for NULL pointers 
 *   before calling strcmp, I wrote a new strlen
 *   function that returns 0
 *
 * Arguments:
 *   - char*   s   The string which length we count
 *
 * Returns:
 *   - int     l   The length of the string
 *
 */
static int Strlen( char *s )
{
   int count = 0;

   if( s != NULL )
      count = strlen( s );

   return count;
}

/*
 * Function:   clear_error
 *
 * Description:
 *   This function sets cf->err to noErr.
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * returns:
 *   - None
 *
 */ 
static void clear_error( conf_file* cf )
{
   if( cf != NULL )
      cf->err = noErr;
}

/*
 * Function:   set_error
 *
 * Description:
 *   This function sets cf->err to err
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *   - confErr      err  The error to set
 *
 * Returns:
 *   - None
 *
 */
static void set_error( conf_file* cf, confErr err )
{
   if( cf != NULL )
      cf->err = err;
}

/*
 * Function:   get_error
 *
 * Description:
 *   This function returns the error value
 *   of configuration file cf
 *
 * Arguments:
 *   - conf_file*   cf   The configuration file
 *
 * Returns:
 *   - confErr   err   The error
 *
 */
confErr get_error( conf_file* cf )
{
   if( cf == NULL )
      return noErr;
   return cf->err;
}

static void add_comment( char**ptr, char* comment )
{
   if( *ptr == NULL )
      *ptr = strdup( comment );
   else
   {
      *ptr = (char*) realloc( *ptr, strlen(*ptr) + strlen( comment ) + 2 );
      strcat( *ptr, "\n" );
      strcat( *ptr, comment );
   }
}

/*
 * EOF libconf/libconf.c
 */
